// SPDX-License-Identifier: MIT
/*
 * Copyright 2025 Advanced Micro Devices, Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER(S) OR AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 *
 */
#include "ras.h"
#include "ras_gfx_v9_0.h"
#include "ras_core_status.h"

enum ta_gfx_v9_subblock {
	/*CPC*/
	TA_GFX_V9__GFX_CPC_INDEX_START = 0,
	TA_GFX_V9__GFX_CPC_SCRATCH = TA_GFX_V9__GFX_CPC_INDEX_START,
	TA_GFX_V9__GFX_CPC_UCODE,
	TA_GFX_V9__GFX_DC_STATE_ME1,
	TA_GFX_V9__GFX_DC_CSINVOC_ME1,
	TA_GFX_V9__GFX_DC_RESTORE_ME1,
	TA_GFX_V9__GFX_DC_STATE_ME2,
	TA_GFX_V9__GFX_DC_CSINVOC_ME2,
	TA_GFX_V9__GFX_DC_RESTORE_ME2,
	TA_GFX_V9__GFX_CPC_INDEX_END = TA_GFX_V9__GFX_DC_RESTORE_ME2,
	/* CPF*/
	TA_GFX_V9__GFX_CPF_INDEX_START,
	TA_GFX_V9__GFX_CPF_ROQ_ME2 = TA_GFX_V9__GFX_CPF_INDEX_START,
	TA_GFX_V9__GFX_CPF_ROQ_ME1,
	TA_GFX_V9__GFX_CPF_TAG,
	TA_GFX_V9__GFX_CPF_INDEX_END = TA_GFX_V9__GFX_CPF_TAG,
	/* CPG*/
	TA_GFX_V9__GFX_CPG_INDEX_START,
	TA_GFX_V9__GFX_CPG_DMA_ROQ = TA_GFX_V9__GFX_CPG_INDEX_START,
	TA_GFX_V9__GFX_CPG_DMA_TAG,
	TA_GFX_V9__GFX_CPG_TAG,
	TA_GFX_V9__GFX_CPG_INDEX_END = TA_GFX_V9__GFX_CPG_TAG,
	/* GDS*/
	TA_GFX_V9__GFX_GDS_INDEX_START,
	TA_GFX_V9__GFX_GDS_MEM = TA_GFX_V9__GFX_GDS_INDEX_START,
	TA_GFX_V9__GFX_GDS_INPUT_QUEUE,
	TA_GFX_V9__GFX_GDS_OA_PHY_CMD_RAM_MEM,
	TA_GFX_V9__GFX_GDS_OA_PHY_DATA_RAM_MEM,
	TA_GFX_V9__GFX_GDS_OA_PIPE_MEM,
	TA_GFX_V9__GFX_GDS_INDEX_END = TA_GFX_V9__GFX_GDS_OA_PIPE_MEM,
	/* SPI*/
	TA_GFX_V9__GFX_SPI_SR_MEM,
	/* SQ*/
	TA_GFX_V9__GFX_SQ_INDEX_START,
	TA_GFX_V9__GFX_SQ_SGPR = TA_GFX_V9__GFX_SQ_INDEX_START,
	TA_GFX_V9__GFX_SQ_LDS_D,
	TA_GFX_V9__GFX_SQ_LDS_I,
	TA_GFX_V9__GFX_SQ_VGPR, /* VGPR = SP*/
	TA_GFX_V9__GFX_SQ_INDEX_END = TA_GFX_V9__GFX_SQ_VGPR,
	/* SQC (3 ranges)*/
	TA_GFX_V9__GFX_SQC_INDEX_START,
	/* SQC range 0*/
	TA_GFX_V9__GFX_SQC_INDEX0_START = TA_GFX_V9__GFX_SQC_INDEX_START,
	TA_GFX_V9__GFX_SQC_INST_UTCL1_LFIFO =
		TA_GFX_V9__GFX_SQC_INDEX0_START,
	TA_GFX_V9__GFX_SQC_DATA_CU0_WRITE_DATA_BUF,
	TA_GFX_V9__GFX_SQC_DATA_CU0_UTCL1_LFIFO,
	TA_GFX_V9__GFX_SQC_DATA_CU1_WRITE_DATA_BUF,
	TA_GFX_V9__GFX_SQC_DATA_CU1_UTCL1_LFIFO,
	TA_GFX_V9__GFX_SQC_DATA_CU2_WRITE_DATA_BUF,
	TA_GFX_V9__GFX_SQC_DATA_CU2_UTCL1_LFIFO,
	TA_GFX_V9__GFX_SQC_INDEX0_END =
		TA_GFX_V9__GFX_SQC_DATA_CU2_UTCL1_LFIFO,
	/* SQC range 1*/
	TA_GFX_V9__GFX_SQC_INDEX1_START,
	TA_GFX_V9__GFX_SQC_INST_BANKA_TAG_RAM =
		TA_GFX_V9__GFX_SQC_INDEX1_START,
	TA_GFX_V9__GFX_SQC_INST_BANKA_UTCL1_MISS_FIFO,
	TA_GFX_V9__GFX_SQC_INST_BANKA_MISS_FIFO,
	TA_GFX_V9__GFX_SQC_INST_BANKA_BANK_RAM,
	TA_GFX_V9__GFX_SQC_DATA_BANKA_TAG_RAM,
	TA_GFX_V9__GFX_SQC_DATA_BANKA_HIT_FIFO,
	TA_GFX_V9__GFX_SQC_DATA_BANKA_MISS_FIFO,
	TA_GFX_V9__GFX_SQC_DATA_BANKA_DIRTY_BIT_RAM,
	TA_GFX_V9__GFX_SQC_DATA_BANKA_BANK_RAM,
	TA_GFX_V9__GFX_SQC_INDEX1_END =
		TA_GFX_V9__GFX_SQC_DATA_BANKA_BANK_RAM,
	/* SQC range 2*/
	TA_GFX_V9__GFX_SQC_INDEX2_START,
	TA_GFX_V9__GFX_SQC_INST_BANKB_TAG_RAM =
		TA_GFX_V9__GFX_SQC_INDEX2_START,
	TA_GFX_V9__GFX_SQC_INST_BANKB_UTCL1_MISS_FIFO,
	TA_GFX_V9__GFX_SQC_INST_BANKB_MISS_FIFO,
	TA_GFX_V9__GFX_SQC_INST_BANKB_BANK_RAM,
	TA_GFX_V9__GFX_SQC_DATA_BANKB_TAG_RAM,
	TA_GFX_V9__GFX_SQC_DATA_BANKB_HIT_FIFO,
	TA_GFX_V9__GFX_SQC_DATA_BANKB_MISS_FIFO,
	TA_GFX_V9__GFX_SQC_DATA_BANKB_DIRTY_BIT_RAM,
	TA_GFX_V9__GFX_SQC_DATA_BANKB_BANK_RAM,
	TA_GFX_V9__GFX_SQC_INDEX2_END =
		TA_GFX_V9__GFX_SQC_DATA_BANKB_BANK_RAM,
	TA_GFX_V9__GFX_SQC_INDEX_END = TA_GFX_V9__GFX_SQC_INDEX2_END,
	/* TA*/
	TA_GFX_V9__GFX_TA_INDEX_START,
	TA_GFX_V9__GFX_TA_FS_DFIFO = TA_GFX_V9__GFX_TA_INDEX_START,
	TA_GFX_V9__GFX_TA_FS_AFIFO,
	TA_GFX_V9__GFX_TA_FL_LFIFO,
	TA_GFX_V9__GFX_TA_FX_LFIFO,
	TA_GFX_V9__GFX_TA_FS_CFIFO,
	TA_GFX_V9__GFX_TA_INDEX_END = TA_GFX_V9__GFX_TA_FS_CFIFO,
	/* TCA*/
	TA_GFX_V9__GFX_TCA_INDEX_START,
	TA_GFX_V9__GFX_TCA_HOLE_FIFO = TA_GFX_V9__GFX_TCA_INDEX_START,
	TA_GFX_V9__GFX_TCA_REQ_FIFO,
	TA_GFX_V9__GFX_TCA_INDEX_END = TA_GFX_V9__GFX_TCA_REQ_FIFO,
	/* TCC (5 sub-ranges)*/
	TA_GFX_V9__GFX_TCC_INDEX_START,
	/* TCC range 0*/
	TA_GFX_V9__GFX_TCC_INDEX0_START = TA_GFX_V9__GFX_TCC_INDEX_START,
	TA_GFX_V9__GFX_TCC_CACHE_DATA = TA_GFX_V9__GFX_TCC_INDEX0_START,
	TA_GFX_V9__GFX_TCC_CACHE_DATA_BANK_0_1,
	TA_GFX_V9__GFX_TCC_CACHE_DATA_BANK_1_0,
	TA_GFX_V9__GFX_TCC_CACHE_DATA_BANK_1_1,
	TA_GFX_V9__GFX_TCC_CACHE_DIRTY_BANK_0,
	TA_GFX_V9__GFX_TCC_CACHE_DIRTY_BANK_1,
	TA_GFX_V9__GFX_TCC_HIGH_RATE_TAG,
	TA_GFX_V9__GFX_TCC_LOW_RATE_TAG,
	TA_GFX_V9__GFX_TCC_INDEX0_END = TA_GFX_V9__GFX_TCC_LOW_RATE_TAG,
	/* TCC range 1*/
	TA_GFX_V9__GFX_TCC_INDEX1_START,
	TA_GFX_V9__GFX_TCC_IN_USE_DEC = TA_GFX_V9__GFX_TCC_INDEX1_START,
	TA_GFX_V9__GFX_TCC_IN_USE_TRANSFER,
	TA_GFX_V9__GFX_TCC_INDEX1_END =
		TA_GFX_V9__GFX_TCC_IN_USE_TRANSFER,
	/* TCC range 2*/
	TA_GFX_V9__GFX_TCC_INDEX2_START,
	TA_GFX_V9__GFX_TCC_RETURN_DATA = TA_GFX_V9__GFX_TCC_INDEX2_START,
	TA_GFX_V9__GFX_TCC_RETURN_CONTROL,
	TA_GFX_V9__GFX_TCC_UC_ATOMIC_FIFO,
	TA_GFX_V9__GFX_TCC_WRITE_RETURN,
	TA_GFX_V9__GFX_TCC_WRITE_CACHE_READ,
	TA_GFX_V9__GFX_TCC_SRC_FIFO,
	TA_GFX_V9__GFX_TCC_SRC_FIFO_NEXT_RAM,
	TA_GFX_V9__GFX_TCC_CACHE_TAG_PROBE_FIFO,
	TA_GFX_V9__GFX_TCC_INDEX2_END =
		TA_GFX_V9__GFX_TCC_CACHE_TAG_PROBE_FIFO,
	/* TCC range 3*/
	TA_GFX_V9__GFX_TCC_INDEX3_START,
	TA_GFX_V9__GFX_TCC_LATENCY_FIFO = TA_GFX_V9__GFX_TCC_INDEX3_START,
	TA_GFX_V9__GFX_TCC_LATENCY_FIFO_NEXT_RAM,
	TA_GFX_V9__GFX_TCC_INDEX3_END =
		TA_GFX_V9__GFX_TCC_LATENCY_FIFO_NEXT_RAM,
	/* TCC range 4*/
	TA_GFX_V9__GFX_TCC_INDEX4_START,
	TA_GFX_V9__GFX_TCC_WRRET_TAG_WRITE_RETURN =
		TA_GFX_V9__GFX_TCC_INDEX4_START,
	TA_GFX_V9__GFX_TCC_ATOMIC_RETURN_BUFFER,
	TA_GFX_V9__GFX_TCC_INDEX4_END =
		TA_GFX_V9__GFX_TCC_ATOMIC_RETURN_BUFFER,
	TA_GFX_V9__GFX_TCC_INDEX_END = TA_GFX_V9__GFX_TCC_INDEX4_END,
	/* TCI*/
	TA_GFX_V9__GFX_TCI_WRITE_RAM,
	/* TCP*/
	TA_GFX_V9__GFX_TCP_INDEX_START,
	TA_GFX_V9__GFX_TCP_CACHE_RAM = TA_GFX_V9__GFX_TCP_INDEX_START,
	TA_GFX_V9__GFX_TCP_LFIFO_RAM,
	TA_GFX_V9__GFX_TCP_CMD_FIFO,
	TA_GFX_V9__GFX_TCP_VM_FIFO,
	TA_GFX_V9__GFX_TCP_DB_RAM,
	TA_GFX_V9__GFX_TCP_UTCL1_LFIFO0,
	TA_GFX_V9__GFX_TCP_UTCL1_LFIFO1,
	TA_GFX_V9__GFX_TCP_INDEX_END = TA_GFX_V9__GFX_TCP_UTCL1_LFIFO1,
	/* TD*/
	TA_GFX_V9__GFX_TD_INDEX_START,
	TA_GFX_V9__GFX_TD_SS_FIFO_LO = TA_GFX_V9__GFX_TD_INDEX_START,
	TA_GFX_V9__GFX_TD_SS_FIFO_HI,
	TA_GFX_V9__GFX_TD_CS_FIFO,
	TA_GFX_V9__GFX_TD_INDEX_END = TA_GFX_V9__GFX_TD_CS_FIFO,
	/* EA (3 sub-ranges)*/
	TA_GFX_V9__GFX_EA_INDEX_START,
	/* EA range 0*/
	TA_GFX_V9__GFX_EA_INDEX0_START = TA_GFX_V9__GFX_EA_INDEX_START,
	TA_GFX_V9__GFX_EA_DRAMRD_CMDMEM = TA_GFX_V9__GFX_EA_INDEX0_START,
	TA_GFX_V9__GFX_EA_DRAMWR_CMDMEM,
	TA_GFX_V9__GFX_EA_DRAMWR_DATAMEM,
	TA_GFX_V9__GFX_EA_RRET_TAGMEM,
	TA_GFX_V9__GFX_EA_WRET_TAGMEM,
	TA_GFX_V9__GFX_EA_GMIRD_CMDMEM,
	TA_GFX_V9__GFX_EA_GMIWR_CMDMEM,
	TA_GFX_V9__GFX_EA_GMIWR_DATAMEM,
	TA_GFX_V9__GFX_EA_INDEX0_END = TA_GFX_V9__GFX_EA_GMIWR_DATAMEM,
	/* EA range 1*/
	TA_GFX_V9__GFX_EA_INDEX1_START,
	TA_GFX_V9__GFX_EA_DRAMRD_PAGEMEM = TA_GFX_V9__GFX_EA_INDEX1_START,
	TA_GFX_V9__GFX_EA_DRAMWR_PAGEMEM,
	TA_GFX_V9__GFX_EA_IORD_CMDMEM,
	TA_GFX_V9__GFX_EA_IOWR_CMDMEM,
	TA_GFX_V9__GFX_EA_IOWR_DATAMEM,
	TA_GFX_V9__GFX_EA_GMIRD_PAGEMEM,
	TA_GFX_V9__GFX_EA_GMIWR_PAGEMEM,
	TA_GFX_V9__GFX_EA_INDEX1_END = TA_GFX_V9__GFX_EA_GMIWR_PAGEMEM,
	/* EA range 2*/
	TA_GFX_V9__GFX_EA_INDEX2_START,
	TA_GFX_V9__GFX_EA_MAM_D0MEM = TA_GFX_V9__GFX_EA_INDEX2_START,
	TA_GFX_V9__GFX_EA_MAM_D1MEM,
	TA_GFX_V9__GFX_EA_MAM_D2MEM,
	TA_GFX_V9__GFX_EA_MAM_D3MEM,
	TA_GFX_V9__GFX_EA_INDEX2_END = TA_GFX_V9__GFX_EA_MAM_D3MEM,
	TA_GFX_V9__GFX_EA_INDEX_END = TA_GFX_V9__GFX_EA_INDEX2_END,
	/* UTC VM L2 bank*/
	TA_GFX_V9__UTC_VML2_BANK_CACHE,
	/* UTC VM walker*/
	TA_GFX_V9__UTC_VML2_WALKER,
	/* UTC ATC L2 2MB cache*/
	TA_GFX_V9__UTC_ATCL2_CACHE_2M_BANK,
	/* UTC ATC L2 4KB cache*/
	TA_GFX_V9__UTC_ATCL2_CACHE_4K_BANK,
	TA_GFX_V9__GFX_MAX
};

struct ras_gfx_subblock_t {
	unsigned char *name;
	int ta_subblock;
	int hw_supported_error_type;
	int sw_supported_error_type;
};

#define RAS_GFX_SUB_BLOCK(subblock, a, b, c, d, e, f, g, h)                             \
	[RAS_GFX_V9__##subblock] = {                                     \
		#subblock,                                                     \
		TA_GFX_V9__##subblock,                                      \
		((a) | ((b) << 1) | ((c) << 2) | ((d) << 3)),                  \
		(((e) << 1) | ((f) << 3) | (g) | ((h) << 2)),                  \
	}

const struct ras_gfx_subblock_t ras_gfx_v9_0_subblocks[] = {
	RAS_GFX_SUB_BLOCK(GFX_CPC_SCRATCH, 0, 1, 1, 1, 1, 0, 0, 1),
	RAS_GFX_SUB_BLOCK(GFX_CPC_UCODE, 0, 1, 1, 1, 1, 0, 0, 1),
	RAS_GFX_SUB_BLOCK(GFX_DC_STATE_ME1, 1, 0, 0, 1, 0, 0, 1, 0),
	RAS_GFX_SUB_BLOCK(GFX_DC_CSINVOC_ME1, 1, 0, 0, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_DC_RESTORE_ME1, 1, 0, 0, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_DC_STATE_ME2, 1, 0, 0, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_DC_CSINVOC_ME2, 1, 0, 0, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_DC_RESTORE_ME2, 1, 0, 0, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_CPF_ROQ_ME2, 1, 0, 0, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_CPF_ROQ_ME1, 1, 0, 0, 1, 0, 0, 1, 0),
	RAS_GFX_SUB_BLOCK(GFX_CPF_TAG, 0, 1, 1, 1, 1, 0, 0, 1),
	RAS_GFX_SUB_BLOCK(GFX_CPG_DMA_ROQ, 1, 0, 0, 1, 0, 0, 1, 0),
	RAS_GFX_SUB_BLOCK(GFX_CPG_DMA_TAG, 0, 1, 1, 1, 0, 1, 0, 1),
	RAS_GFX_SUB_BLOCK(GFX_CPG_TAG, 0, 1, 1, 1, 1, 1, 0, 1),
	RAS_GFX_SUB_BLOCK(GFX_GDS_MEM, 0, 1, 1, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_GDS_INPUT_QUEUE, 1, 0, 0, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_GDS_OA_PHY_CMD_RAM_MEM, 0, 1, 1, 1, 0, 0, 0,
			     0),
	RAS_GFX_SUB_BLOCK(GFX_GDS_OA_PHY_DATA_RAM_MEM, 1, 0, 0, 1, 0, 0, 0,
			     0),
	RAS_GFX_SUB_BLOCK(GFX_GDS_OA_PIPE_MEM, 0, 1, 1, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_SPI_SR_MEM, 1, 0, 0, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_SQ_SGPR, 0, 1, 1, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_SQ_LDS_D, 0, 1, 1, 1, 1, 0, 0, 1),
	RAS_GFX_SUB_BLOCK(GFX_SQ_LDS_I, 0, 1, 1, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_SQ_VGPR, 0, 1, 1, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_SQC_INST_UTCL1_LFIFO, 0, 1, 1, 1, 0, 0, 0, 1),
	RAS_GFX_SUB_BLOCK(GFX_SQC_DATA_CU0_WRITE_DATA_BUF, 0, 1, 1, 1, 0, 0,
			     0, 0),
	RAS_GFX_SUB_BLOCK(GFX_SQC_DATA_CU0_UTCL1_LFIFO, 0, 1, 1, 1, 0, 0, 0,
			     0),
	RAS_GFX_SUB_BLOCK(GFX_SQC_DATA_CU1_WRITE_DATA_BUF, 0, 1, 1, 1, 0, 0,
			     0, 0),
	RAS_GFX_SUB_BLOCK(GFX_SQC_DATA_CU1_UTCL1_LFIFO, 0, 1, 1, 1, 1, 0, 0,
			     0),
	RAS_GFX_SUB_BLOCK(GFX_SQC_DATA_CU2_WRITE_DATA_BUF, 0, 1, 1, 1, 0, 0,
			     0, 0),
	RAS_GFX_SUB_BLOCK(GFX_SQC_DATA_CU2_UTCL1_LFIFO, 0, 1, 1, 1, 0, 0, 0,
			     0),
	RAS_GFX_SUB_BLOCK(GFX_SQC_INST_BANKA_TAG_RAM, 0, 1, 1, 1, 1, 0, 0,
			     1),
	RAS_GFX_SUB_BLOCK(GFX_SQC_INST_BANKA_UTCL1_MISS_FIFO, 1, 0, 0, 1, 0,
			     0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_SQC_INST_BANKA_MISS_FIFO, 1, 0, 0, 1, 0, 0, 0,
			     0),
	RAS_GFX_SUB_BLOCK(GFX_SQC_INST_BANKA_BANK_RAM, 0, 1, 1, 1, 0, 0, 0,
			     0),
	RAS_GFX_SUB_BLOCK(GFX_SQC_DATA_BANKA_TAG_RAM, 0, 1, 1, 1, 0, 0, 0,
			     0),
	RAS_GFX_SUB_BLOCK(GFX_SQC_DATA_BANKA_HIT_FIFO, 1, 0, 0, 1, 0, 0, 0,
			     0),
	RAS_GFX_SUB_BLOCK(GFX_SQC_DATA_BANKA_MISS_FIFO, 1, 0, 0, 1, 0, 0, 0,
			     0),
	RAS_GFX_SUB_BLOCK(GFX_SQC_DATA_BANKA_DIRTY_BIT_RAM, 1, 0, 0, 1, 0, 0,
			     0, 0),
	RAS_GFX_SUB_BLOCK(GFX_SQC_DATA_BANKA_BANK_RAM, 0, 1, 1, 1, 0, 0, 0,
			     0),
	RAS_GFX_SUB_BLOCK(GFX_SQC_INST_BANKB_TAG_RAM, 0, 1, 1, 1, 1, 0, 0,
			     0),
	RAS_GFX_SUB_BLOCK(GFX_SQC_INST_BANKB_UTCL1_MISS_FIFO, 1, 0, 0, 1, 0,
			     0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_SQC_INST_BANKB_MISS_FIFO, 1, 0, 0, 1, 0, 0, 0,
			     0),
	RAS_GFX_SUB_BLOCK(GFX_SQC_INST_BANKB_BANK_RAM, 0, 1, 1, 1, 0, 0, 0,
			     0),
	RAS_GFX_SUB_BLOCK(GFX_SQC_DATA_BANKB_TAG_RAM, 0, 1, 1, 1, 0, 0, 0,
			     0),
	RAS_GFX_SUB_BLOCK(GFX_SQC_DATA_BANKB_HIT_FIFO, 1, 0, 0, 1, 0, 0, 0,
			     0),
	RAS_GFX_SUB_BLOCK(GFX_SQC_DATA_BANKB_MISS_FIFO, 1, 0, 0, 1, 0, 0, 0,
			     0),
	RAS_GFX_SUB_BLOCK(GFX_SQC_DATA_BANKB_DIRTY_BIT_RAM, 1, 0, 0, 1, 0, 0,
			     0, 0),
	RAS_GFX_SUB_BLOCK(GFX_SQC_DATA_BANKB_BANK_RAM, 0, 1, 1, 1, 0, 0, 0,
			     0),
	RAS_GFX_SUB_BLOCK(GFX_TA_FS_DFIFO, 0, 1, 1, 1, 1, 0, 0, 1),
	RAS_GFX_SUB_BLOCK(GFX_TA_FS_AFIFO, 1, 0, 0, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_TA_FL_LFIFO, 1, 0, 0, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_TA_FX_LFIFO, 1, 0, 0, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_TA_FS_CFIFO, 1, 0, 0, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_TCA_HOLE_FIFO, 1, 0, 0, 1, 0, 1, 1, 0),
	RAS_GFX_SUB_BLOCK(GFX_TCA_REQ_FIFO, 1, 0, 0, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_TCC_CACHE_DATA, 0, 1, 1, 1, 1, 0, 0, 1),
	RAS_GFX_SUB_BLOCK(GFX_TCC_CACHE_DATA_BANK_0_1, 0, 1, 1, 1, 1, 0, 0,
			     1),
	RAS_GFX_SUB_BLOCK(GFX_TCC_CACHE_DATA_BANK_1_0, 0, 1, 1, 1, 1, 0, 0,
			     1),
	RAS_GFX_SUB_BLOCK(GFX_TCC_CACHE_DATA_BANK_1_1, 0, 1, 1, 1, 1, 0, 0,
			     1),
	RAS_GFX_SUB_BLOCK(GFX_TCC_CACHE_DIRTY_BANK_0, 0, 1, 1, 1, 0, 0, 0,
			     0),
	RAS_GFX_SUB_BLOCK(GFX_TCC_CACHE_DIRTY_BANK_1, 0, 1, 1, 1, 0, 0, 0,
			     0),
	RAS_GFX_SUB_BLOCK(GFX_TCC_HIGH_RATE_TAG, 0, 1, 1, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_TCC_LOW_RATE_TAG, 0, 1, 1, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_TCC_IN_USE_DEC, 1, 0, 0, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_TCC_IN_USE_TRANSFER, 1, 0, 0, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_TCC_RETURN_DATA, 1, 0, 0, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_TCC_RETURN_CONTROL, 1, 0, 0, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_TCC_UC_ATOMIC_FIFO, 1, 0, 0, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_TCC_WRITE_RETURN, 1, 0, 0, 1, 0, 1, 1, 0),
	RAS_GFX_SUB_BLOCK(GFX_TCC_WRITE_CACHE_READ, 1, 0, 0, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_TCC_SRC_FIFO, 0, 1, 1, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_TCC_SRC_FIFO_NEXT_RAM, 1, 0, 0, 1, 0, 0, 1, 0),
	RAS_GFX_SUB_BLOCK(GFX_TCC_CACHE_TAG_PROBE_FIFO, 1, 0, 0, 1, 0, 0, 0,
			     0),
	RAS_GFX_SUB_BLOCK(GFX_TCC_LATENCY_FIFO, 1, 0, 0, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_TCC_LATENCY_FIFO_NEXT_RAM, 1, 0, 0, 1, 0, 0, 0,
			     0),
	RAS_GFX_SUB_BLOCK(GFX_TCC_WRRET_TAG_WRITE_RETURN, 1, 0, 0, 1, 0, 0,
			     0, 0),
	RAS_GFX_SUB_BLOCK(GFX_TCC_ATOMIC_RETURN_BUFFER, 1, 0, 0, 1, 0, 0, 0,
			     0),
	RAS_GFX_SUB_BLOCK(GFX_TCI_WRITE_RAM, 1, 0, 0, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_TCP_CACHE_RAM, 0, 1, 1, 1, 1, 0, 0, 1),
	RAS_GFX_SUB_BLOCK(GFX_TCP_LFIFO_RAM, 0, 1, 1, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_TCP_CMD_FIFO, 1, 0, 0, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_TCP_VM_FIFO, 0, 1, 1, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_TCP_DB_RAM, 1, 0, 0, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_TCP_UTCL1_LFIFO0, 0, 1, 1, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_TCP_UTCL1_LFIFO1, 0, 1, 1, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_TD_SS_FIFO_LO, 0, 1, 1, 1, 1, 0, 0, 1),
	RAS_GFX_SUB_BLOCK(GFX_TD_SS_FIFO_HI, 0, 1, 1, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_TD_CS_FIFO, 1, 0, 0, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_EA_DRAMRD_CMDMEM, 0, 1, 1, 1, 1, 0, 0, 1),
	RAS_GFX_SUB_BLOCK(GFX_EA_DRAMWR_CMDMEM, 0, 1, 1, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_EA_DRAMWR_DATAMEM, 0, 1, 1, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_EA_RRET_TAGMEM, 0, 1, 1, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_EA_WRET_TAGMEM, 0, 1, 1, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_EA_GMIRD_CMDMEM, 0, 1, 1, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_EA_GMIWR_CMDMEM, 0, 1, 1, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_EA_GMIWR_DATAMEM, 0, 1, 1, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_EA_DRAMRD_PAGEMEM, 1, 0, 0, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_EA_DRAMWR_PAGEMEM, 1, 0, 0, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_EA_IORD_CMDMEM, 1, 0, 0, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_EA_IOWR_CMDMEM, 1, 0, 0, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_EA_IOWR_DATAMEM, 1, 0, 0, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_EA_GMIRD_PAGEMEM, 1, 0, 0, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_EA_GMIWR_PAGEMEM, 1, 0, 0, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_EA_MAM_D0MEM, 1, 0, 0, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_EA_MAM_D1MEM, 1, 0, 0, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_EA_MAM_D2MEM, 1, 0, 0, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(GFX_EA_MAM_D3MEM, 1, 0, 0, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(UTC_VML2_BANK_CACHE, 0, 1, 1, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(UTC_VML2_WALKER, 0, 1, 1, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(UTC_ATCL2_CACHE_2M_BANK, 1, 0, 0, 1, 0, 0, 0, 0),
	RAS_GFX_SUB_BLOCK(UTC_ATCL2_CACHE_4K_BANK, 0, 1, 1, 1, 0, 0, 0, 0),
};

static int gfx_v9_0_get_ta_subblock(struct ras_core_context *ras_core,
			uint32_t error_type, uint32_t subblock, uint32_t *ta_subblock)
{
	const struct ras_gfx_subblock_t *gfx_subblock;

	if (subblock >= ARRAY_SIZE(ras_gfx_v9_0_subblocks))
		return -EINVAL;

	gfx_subblock = &ras_gfx_v9_0_subblocks[subblock];
	if (!gfx_subblock->name)
		return -EPERM;

	if (!(gfx_subblock->hw_supported_error_type & error_type)) {
		RAS_DEV_ERR(ras_core->dev, "GFX Subblock %s, hardware do not support type 0x%x\n",
			gfx_subblock->name, error_type);
		return -EPERM;
	}

	if (!(gfx_subblock->sw_supported_error_type & error_type)) {
		RAS_DEV_ERR(ras_core->dev, "GFX Subblock %s, driver do not support type 0x%x\n",
			gfx_subblock->name, error_type);
		return -EPERM;
	}

	*ta_subblock = gfx_subblock->ta_subblock;

	return 0;
}

const struct ras_gfx_ip_func gfx_ras_func_v9_0 = {
	.get_ta_subblock = gfx_v9_0_get_ta_subblock,
};
