package Spidy::Database;

use strict;
use Carp;

# ignore subroutine warnings until end of
# this file being parsed;
my $WARNSUB;
BEGIN {
  $WARNSUB = $SIG{__WARN__};
  $SIG{__WARN__} = sub {
    return if $_[0] =~ /ubroutine \w+ redefined/;
    warn @_;
  };
}

sub insert {
  my $self = shift;
  my $tree = shift;
  my $gb = shift;
  $self->insert_sizes($gb) if $gb;
  $self->insert_tree( $tree ) if $tree;
  $self->insert_hierarchy($tree, $gb);
}

sub insert_tree {
  my $self = shift;
  my $tree = shift;
  my $gb = shift;
  $self->insert_group($tree);
  $self->insert_links($tree);

  $self->insert_tree($_) for $tree->children();
  return;
}

sub format_insert {
  my $self  = shift;
  my $table = shift;
  my $obj   = shift;
  my $size  = shift;
  
  my $name = $table->get_name();
  my $col_names = join ",\n       ", $self->format_insert_cols($table);
  my @values = $self->format_insert_values($table, $obj, $size);
  my $col_values = join ",\n       ", ('?') x scalar(@values);
  
  my $sql =<<EOF;
INSERT INTO $name (
       $col_names
       ) VALUES ( 
       $col_values
       )
EOF
  return $sql, \@values;
}

sub format_insert_cols {
  my $self = shift;
  my $table = shift;
  return map {
    $_->{'name'}
  } $self->filter_insert_cols( $table->get_cols() );
}

sub format_insert_values {
  my $self  = shift;
  my $table = shift;
  my $obj   = shift;
  
  # if the 'object' is a plain hash, then do not call set_param_data
  if( UNIVERSAL::isa($obj, "Spidy::Group") ) {
    my $size  = shift;
    return map {
      $self->format_value( $_, $obj->set_param_data($_->{'name'}, $size) )
    } $self->filter_insert_cols( $table->get_cols() );
  }
  return map {
    $self->format_value( $_, $obj->{$_->{'name'}} )
  } $self->filter_insert_cols( $table->get_cols() );
  
}

sub filter_insert_cols { 
  my $self = shift;
  return @_;
};
  
sub format_value {
  my $self = shift;
  my $col = shift;
  my $data = shift;
  
  return "''" if $col->{'value'} eq 'auto';
  return $col->{'default'} unless defined $data;
  if( $col->{'type'} =~ /^(?:int|bool)$/ ) {
    return $data if $data =~ /\d/;
    return undef;
  }
  return $data;
#  return $self->get_dbh()->quote( $data );
}
 
sub insert_image {
  my $self = shift;
  my $image = shift;

  my $image_id = $self->get_image_id( $image );
  return $image_id if $image_id;

  my $dbh = $self->get_dbh();
  
  my $db = Spidy::Database::Structure->new();
  my $image_table = $db->get_table('images');
  $self->query( $self->format_insert($image_table, $image) );

  $image_id = $self->get_image_id( $image );
  return $image_id;
}

sub insert_group {
  my $self = shift;
  my $group = shift;

  $group->{'type'} = ref($group);
  
  if( $self->isa("Spidy::Image") ) {
    $group->{'image_id'} = $self->insert_image($group);
  } else {
    my $image = $group->first_image();
    $group->{'image_id'} = $self->insert_image($image);
  }

  $group->{'color_id'}  = $self->insert_color($group);
  $group->{'comment_id'} = $self->insert_comment($group);
  
  my $db = Spidy::Database::Structure->new();
  my $table = $db->get_table('groups');
  $self->query( $self->format_insert($table, $group) );
  
  $group->{'group_id'} = $self->get_group_id( $group );
  
  return;
}

sub insert_hierarchy {
  my $self = shift;
  my $group = shift;
  my $gb = shift;

  my $group_id = $self->get_group_id( $group );
  my $db = Spidy::Database::Structure->new();
  my $table = $db->get_table('hierarchy');
  
  # start with the 0th generation, just so we
  # can get all the color information for the
  # current object.  We dont actually
  # insert the 0th generation into the hierarchy 
  # though.
  my $generation = 0;
  my $elder = $group;
  my $colors = {};
  
  #
  # will need to display color for current group
  #
  $colors->{$group->{'color_id'}} = $group->set_param_data("bg_color");
  
  #
  # will need to display current group with parents color
  #
  $colors->{$group->{'color_id'}} =
    $group->{'parent'}->set_param_data("bg_color")
    if $group->{'parent'};
    
  #
  # figure out the previous group
  #
  my $prev;
  
  # check to see if it just a sibling
  if( $group->{'previous'} ) {
    $prev = $group->{'previous'};
  } else {
    # no prev sibling so try to get the last image 
    # from the previous group
    my $prev_group = $group->previous_group();
    # get the previous image if this is an image
    if( $group->isa("Spidy::Image") && $prev_group ) {
      $prev = $prev_group->last_image();
      # forget it if it they should be isolated
      undef $prev if $group->isolated( $prev, $group );
    } else {
      $prev = $prev_group unless $group->isolated( $prev_group, $group );    
    }
  }
  
  #
  # will need to display current group on the previous
  # groups page (with prev group colors)
  #
  $colors->{$prev->{'color_id'}} = 
    $prev->set_param_data("bg_color")
    if $prev;
  
  #
  # figure out the next group
  #
  my $next;
  
  # check to see if it just a sibling
  if( $group->{'next'} ) {
    $next = $group->{'next'};
  } else {
    # no next sibling so try to get the last image 
    # from the next group
    my $next_group = $group->next_group();
    # get the next image if this is an image
    if( $group->isa("Spidy::Image") && $next_group ) {
      $next = $next_group->first_image();
      # forget it if it they should be isolated
      undef $next if $group->isolated( $next, $group );
    } else {
      $next = $next_group unless $group->isolated( $next_group, $group );    
    }
  }
  
  #
  # will need to display current group on the next
  # groups page (with next group colors)
  #
  $colors->{$next->{'color_id'}} = 
    $next->set_param_data("bg_color")
    if $next;
    
    
  #
  # if this is an image and there is no previous sibling,
  # then we need to find all the parent colors to we can
  # correctly be displayed in that color
  #
  if( $group->isa("Spidy::Image") && !$group->{'previous'} ) {
    my $elder = $group->{'parent'};
    while( $elder ) {
      $colors->{$elder->{'color_id'}} = $next->set_param_data("bg_color");
      $elder = $elder->{'parent'};
    }
  }
  
  #
  # insert this group and its ancestors into the hierarchy table:
  #
  my $prev_id = $self->get_group_id( $prev )
    if $prev;
  my $next_id = $self->get_group_id( $next )
    if $next;
  my $elder = $group->{'parent'};
  my $generation = 1;
  while( $elder ) {
    $self->query(
      $self->format_insert( $table, {
        'group_id'   => $group_id,
        'elder_id'   => $self->get_group_id($elder),
        'left_id'    => $prev_id,
        'right_id'   => $next_id,
        'generation' => $generation++,
        'pos'        => $group->pos(),
      })
    );
    $elder = $elder->{'parent'};
  }
  
  # recurse through all the children trying
  # to find  any one with a different color
  my @children = $group->children();
  while( my $cur = shift @children ) {
    $colors->{$cur->{'color_id'}} = $cur->set_param_data("bg_color");
  }
  

  my $gs_table = $db->get_table('group_sizes');
  foreach my $size_name ( $gb->get_size_names() ) {
    my $size = $gb->get_size($size_name);
    my $size_id = $self->get_size_id( $size );

    my $width  = $group->set_param_data('width', $size);
    my $height = $group->set_param_data('height', $size);
    my $src    = $group->set_param_data('src', $size );
    my $color  = $group->set_param_data('bg_color', $size);

    for my $color_id ( keys %$colors ) {
      my $src_copy = $src;
      if ( $size->{'padded'} ) {
        $src_copy =~ s/$color/$colors->{$color_id}/g;
      }
      $self->query( $self->format_insert( $gs_table, {
        group_id => $group->{'group_id'},
        size_id  => $size_id,
        color_id => $color_id,
        width    => $width,
        height   => $height,
        src      => $src_copy,
      }));
    }
  }


  $self->insert_hierarchy($_, $gb) for $group->children();
}    

sub insert_sizes {
  my $self = shift;
  my $gb = shift;
  
  my $db = Spidy::Database::Structure->new();
  my $table = $db->get_table('sizes');

  for my $size_name ($gb->get_size_names()) {
    my $size = $gb->get_size($size_name);
    $self->query( $self->format_insert($table, $size) );
  }
}

sub insert_links {
  my $self = shift;
  my $group = shift;

  return unless $group->{'links'};
  my $db = Spidy::Database::Structure->new();
  my $link_table = $db->get_table('links');
  my $nl_table = $db->get_table('group_links');
  
  my $group_id = $self->get_group_id($group);
  foreach my $link ( @{$group->{'links'}} ) {
    next unless $link->{'text'};
    my $link_id = $self->get_link_id($link);
    unless( $link_id ) {
      $self->query( $self->format_insert($link_table, $link) );
      $link_id = $self->get_link_id($link);
    }
    $self->query( $self->format_insert( $nl_table, {
      'group_id' => $group_id,
      'link_id' => $link_id,
    }));
  }
}

sub insert_comment {
  my $self = shift;
  my $group = shift;
  my $comment = $group->{'comment'};
  return unless $comment;

  my $comment_id = $self->get_comment_id($comment);
  return $comment_id if $comment_id;
  
  my $db = Spidy::Database::Structure->new();
  my $table = $db->get_table('comments');
  
  $self->query( $self->format_insert($table, $comment) );
  return $self->get_comment_id($comment);
}

sub insert_color {
  my $self = shift;
  my $group = shift;

  my $color_id = $self->get_color_id($group);
  return $color_id if $color_id;
  
  my $db = Spidy::Database::Structure->new();
  my $table = $db->get_table('colors');
  
  $self->query( $self->format_insert($table, $group) );
  return $self->get_color_id($group);
}
    

# after file is parsed
# reinstall default warning sub.
$SIG{__WARN__} = $WARNSUB;
1;
