package Spidy::Database;
use Spidy::Object;
@ISA = qw( Spidy::Object );

use Spidy::Group;
use Spidy::Image;
use Spidy::Comment;
use Spidy::Link;

use strict;
use Carp;
use DBI;

sub autoload_members {
  return {
    'warning_sql' => "scalar=$ENV{'SPIDY_WARN_SQL'}",
    'database'   => "scalar=spider",
    'type'       => "scalar=Pg",
    'server'     => "scalar",
    'user'       => "scalar",
    'password'   => "scalar",
    'port'       => "scalar",
    'dbh'        => "scalar",
    'group'      => "scalar",
    'image'      => "scalar",
  }
}

sub new {
  my $package = shift;
  my %args = @_;
  $args{'type'} = ucfirst($args{'type'});
  my $db_package = "Spidy::Database::$args{'type'}";
  eval "require $db_package";
  croak "Error loaded database module $db_package: $@" if $@;
  #call the parent class of the database package should go to
  # Spidy::Object::new
  return $db_package->SUPER::new(%args);
}

sub DESTROY {
  my $self = shift;
  my $dbh = $self->{'dbh'};
  $dbh->disconnect() if $dbh;
}

#
# This should be overloaded in DBD specific modules.
#
sub get_dbh {
  my $self = shift;
  confess "Unrecognized Database Type: \"$self->{'type'}\"";
}

sub create {
  require Spidy::Database::Create;
  goto &create;
}

sub drop {
  require Spidy::Database::Create;
  goto &drop;
}

sub insert {
  require Spidy::Database::Insert;
  goto &insert;
}

sub ping {
  return shift->get_dbh()->ping();
}

sub query {
  my $self = shift;
  my $sql  = shift;
  
  my @params;
  my $return_type;
  if( ref($_[0]) eq 'ARRAY' ) { 
    @params = @{+shift};
  } else {
    $return_type = shift;
  }
  $return_type = shift unless $return_type;

  my $DBstatus;
  my @sql_results;
  
  if( $self->{'warning_sql'} ) {
    warn "#"x50, "\n", $sql, "\n", "#"x50, "\n";
    if( @params ) {
      warn "BINDING $_ -> \"$params[$_]\"\n" for 0 .. $#params;
    }
  };

  my $dbh = $self->get_dbh();

  my $sth = $dbh->prepare($sql) 
    or confess "Error: $DBI::errstr\nSQL: $sql\n";

  #
  # do not execute if there was an error from prepare
  # 
  $DBstatus = $sth->execute(@params)
    or confess "Error: $DBI::errstr\nSQL: $sql\n"
    unless $DBI::errstr;

  unless( $sth->{NUM_OF_FIELDS} == 0 ) {
    if( $return_type =~ /ARRAY/i ) {
      if( $DBstatus && !$DBI::errstr ) {
        do { 
          while( my $data = $sth->fetchrow_arrayref ) {
            if( $DBI::errstr ) {  
              confess "Error: $DBI::errstr\nSQL: $sql\n";
              next;
            }
            #
            # The same row pointer is returned for each row,
            # so we need to copy the data out into a new array
            # so that every row will not be pointing to the same
            # data structure.
            #
            push @sql_results, [@$data];
          }
        } while( $sth->{syb_more_results}); #sybase hack
      }
    } elsif( $return_type =~ /SIMPLE/i ) {
      if( $DBstatus && !$DBI::errstr ) {
        do { 
          while( my @data = $sth->fetchrow_array ) {
            if( $DBI::errstr ) {  
              carp "Error: $DBI::errstr\nSQL: $sql\n";
              next;
            }
            push @sql_results, @data;
          }
        } while( $sth->{syb_more_results}); #sybase hack
      }
    } else {
      if( $DBstatus && !$DBI::errstr ) {
        do { 
          while( my $data = $sth->fetchrow_hashref ) {
            if( $DBI::errstr ) {  
              carp "Error: $DBI::errstr\nSQL: $sql\n";
              next;
            }
            #
            # The same row pointer is returned for each row,
            # so we need to copy the data out into a new hash
            # so that every row will not be pointing to the same
            # data structure.
            #
            # Actually this is a lie, but it is in 
            # their to-do list to make this work exactly as the
            # fetchrow_arrayref, so I am preparing for it now.
            #
            push @sql_results, {%$data};
          }
        } while( $sth->{syb_more_results});
      }
    }
  }
  return wantarray ? @sql_results : \@sql_results;
}

sub get_group_id {
  my $self = shift;
  my $group = shift;

  my $sql =<<EOM;
SELECT group_id
  FROM groups
 WHERE name = ?
   AND xml_id = ?
EOM

  my @results = $self->query( $sql, [ $group->{'name'}, $group->{'id'} ] );
  
  confess("SQL:\n$sql\n\n more than one returned from get_group_id for $group->{'name'} $group->{'id'}") if @results > 1;
  confess("SQL:\n$sql\n\n no results returned from get_group_id for $group->{'name'} $group->{'id'}") if @results == 0;
  return $results[0]->{'group_id'};
}

sub get_image_id {
  my $self = shift;
  my $image = shift;

  my $sql =<<EOM;
SELECT image_id
  FROM images
 WHERE input_image = ?
   AND input_image_path = ?
EOM
  my @results = $self->query($sql, [ $image->{'input_image'}, $image->{'input_image_path'} ] );
  confess("SQL:\n$sql\n\n_more than one returned from get_image_id for $image->{'file_path'} $image->{'output_image'}") if @results > 1;
  return $results[0]->{'image_id'};
}

sub get_color_id {
  my $self = shift;
  my $group = shift;

  my $sql =<<EOM;
SELECT color_id
  FROM colors
 WHERE bg_color = ?
   AND fg_color = ?
   AND vlink_color = ?
   AND link_color = ?
EOM
  my @results = $self->query($sql, [
    $group->set_param_data('bg_color'),
    $group->set_param_data('fg_color'),
    $group->set_param_data('vlink_color'),
    $group->set_param_data('link_color'),
  ]);
  return $results[0]->{'color_id'};
}   
    
sub get_link_id{ 
  my $self = shift;
  my $link = shift;
  my $sql =<<EOF;
SELECT link_id 
  FROM links 
 WHERE text = ?
EOF
  my @results = $self->query( $sql,  [ $link->{'text'} ] );
  return $results[0]->{'link_id'};
}

sub get_comment_id{ 
  my $self = shift;
  my $comment = shift;
  my $sql =<<EOF;
SELECT comment_id 
  FROM comments 
 WHERE text = ?
EOF
  my @results = $self->query( $sql, [ $comment->{'text'} ] );
  return $results[0]->{'comment_id'};
}

sub get_size_id {
  my $self = shift;
  my $size = shift;
  my $sql =<<EOF;
SELECT size_id
  FROM sizes
 WHERE name = ?
EOF
  my @results = $self->query( $sql, [ $size->{'name'} ] );
  return $results[0]->{'size_id'};
}

sub get_group {
  my $self = shift;
  my $path = shift;
  my $size = shift;
  my $type = shift;
  
  return $self->get_image($path, $size, $type) if $type eq 'image';
  
  my $dbh = $self->get_dbh();
  my $and;
  $path = "." unless $path;
  my $size_name = $size->first()->{'name'};
  my $sql =<<EOF;
SELECT g.group_id,
       g.title, 
       g.short_title,
       g.copyright,
       g.name, 
       g.path,
       g.bundle_download,
       g.bundle_size,
       g.bundle_href,
       g.column_count,
       g.rating,
       g.xml_id,
       g.type,
       gs.height,
       gs.width,
       gs.src,
       i.input_image,
       s.max_height,
       s.max_width,
       s.size_id,
       c.color_id,
       c.bg_color,
       c.fg_color,
       c.vlink_color,
       c.link_color
  FROM groups g,
       images i,
       group_sizes gs,
       sizes s,
       colors c
 WHERE g.group_id = gs.group_id
   AND gs.size_id = s.size_id
   AND gs.color_id = c.color_id
   AND g.color_id = c.color_id
   AND g.image_id = i.image_id
   AND g.path = ?
   AND g.type LIKE '%Group'
   AND s.name = ?
EOF
  my $results = $self->query($sql, [ $path, $size_name ] );
  croak "No image found for path = \"$path\", size = \"$size_name\"" unless @$results;
  
  my $group = $results->[0];

  bless $group, $group->{'type'} and eval "require $group->{'type'}";

  $sql =<<EOF;
SELECT g.name,
       gs.src,
       g.path
  FROM groups g,
       group_sizes gs,
       sizes s,
       hierarchy h
 WHERE g.group_id = gs.group_id
   AND gs.size_id = s.size_id
   AND h.right_id = g.group_id
   AND g.type LIKE '%Group'
   AND gs.color_id = ?
   AND s.name = ?
   AND h.group_id = ?
   AND h.generation = 1
EOF
  $results = $self->query($sql, [ $group->{color_id}, 'thumb', $group->{group_id} ] );       

  if( @$results ) {  
    $group->{next} = $results->[0];
    bless $group->{next}, $group->{'type'};
  }
  
  $sql =<<EOF;
SELECT g.name,
       gs.src,
       g.path
  FROM groups g,
       group_sizes gs,
       sizes s,
       hierarchy h
 WHERE g.group_id = gs.group_id
   AND gs.size_id = s.size_id
   AND h.left_id = g.group_id
   AND g.type LIKE '%Group'
   AND gs.color_id = ?
   AND s.name = ?
   AND h.group_id = ?
   AND h.generation = 1
EOF
  $results = $self->query($sql, [ $group->{color_id}, 'thumb', $group->{group_id} ] );       
  
  if( @$results ) {
    $group->{previous} = $results->[0];
    bless $group->{previous}, $group->{'type'};
  }

  $sql =<<EOF;
SELECT g.name,
       g.group_id,
       gs.src,
       g.path,
       g.title,
       g.short_title,
       g.type
  FROM groups g,
       group_sizes gs,
       sizes s,
       hierarchy h
 WHERE g.group_id = gs.group_id
   AND gs.size_id = s.size_id
   AND h.elder_id = g.group_id
   AND g.type LIKE '%Group'
   AND gs.color_id = ?
   AND s.name = ?
   AND h.group_id = ?
 ORDER BY generation ASC
EOF
  $results = $self->query($sql, [ $group->{color_id}, 'thumb', $group->{group_id} ] );       

  my $cursor = $group;
  for my $p ( @$results ) {
    bless $p, $p->{'type'};
    $cursor->{'parent'} = $p;
    $cursor = $p;
  }
  
  $sql =<<EOF;
SELECT g.name,
       g.group_id,
       gs.src,
       g.path,
       g.title,
       g.short_title,
       g.type,
       gs.width,
       gs.height,
       i.input_image
  FROM groups g,
       images i,
       group_sizes gs,
       sizes s,
       hierarchy h
 WHERE g.group_id = gs.group_id
   AND gs.size_id = s.size_id
   AND h.group_id = g.group_id
   AND g.image_id = i.image_id
   AND gs.color_id = ?
   AND s.name = ?
   AND h.elder_id = ?
   AND h.generation = 1
 ORDER BY h.pos
EOF
  $results = $self->query($sql, [ $group->{color_id}, 'thumb', $group->{group_id} ] );       
  
  my %objs;
  $objs{$_->{type}}++ for @$results;
  eval "require $_" for keys %objs;

  my @groups = grep { $_->{type} =~ /Group/ } @$results;
  my @images = grep { $_->{type} =~ /Image/ } @$results;
  
  if( @groups ) {
    bless $_, $_->{'type'} for @groups;
    $group->{'groups'} = Spidy::List->link(@groups);
  }

  if( @images ) {
    bless $_, $_->{'type'} for @images;
    $group->{'images'} = Spidy::List->link(@images);
  }

  if( $group->{'groups'} ) {
    my $count = $group->{'groups'}->count();
    my $vars = join (',', ("?")x$count);
  $sql=<<EOF;
SELECT elder_id, COUNT(*) AS "count"
  FROM hierarchy h
 WHERE elder_id IN ($vars)
 GROUP BY elder_id
EOF
    my @ids = map { $_->{'group_id'} } $group->children();
    my $results = $self->query( $sql, [ @ids ] );
    my %results = map { $_->{'elder_id'} => $_ } @$results;
    $_->{'count'} = $results{$_->{'group_id'}}->{'count'} for $group->children();
  }
  
  $sql=<<EOF;
SELECT l.text
  FROM links l,
       group_links nl
 WHERE nl.link_id = l.link_id
   AND nl.group_id = ?
EOF
  my @links = map { Spidy::Link->new( %$_) } $self->query( $sql, [ $group->{'group_id'} ] );
  $group->{'links'} = \@links;
    
  $sql=<<EOF;
SELECT co.text
  FROM comments co,
       groups n
 WHERE n.comment_id = co.comment_id
   AND n.group_id = ?
EOF
  my @comments = $self->query( $sql, [ $group->{'group_id'} ] );
  #there should only be one comment;
  $group->{'comment'} = Spidy::Comment->new( %{$comments[0]} );

  return $group;
}

sub get_image {
  my $self = shift;
  my $path = shift;
  my $size = shift;
  my $type = shift;
  
  my($path, $image_name ) = ( $path =~ /(.*?)\/([^\/]+)$/ );
  
  my $group;
  my $dbh = $self->get_dbh();

  my $size_name = $size->{'name'};
  my $sql =<<EOF;
SELECT g.group_id,
       g.title, 
       g.short_title,
       g.copyright,
       g.name, 
       g.path,
       g.bundle_download,
       g.bundle_size,
       g.bundle_href,
       g.column_count,
       g.rating,
       g.xml_id,
       g.type,
       gs.height,
       gs.width,
       gs.src,
       s.max_height,
       s.max_width,
       s.size_id,
       c.color_id,
       c.bg_color,
       c.fg_color,
       c.vlink_color,
       c.link_color
  FROM groups g,
       group_sizes gs,
       sizes s,
       colors c
 WHERE g.group_id = gs.group_id
   AND gs.size_id = s.size_id
   AND gs.color_id = c.color_id
   AND g.color_id = c.color_id
   AND g.path = ?
   AND g.type LIKE '%Image'
   AND g.name = ?
   AND s.name = ?
EOF
  my $results = $self->query($sql, [ $path, $image_name, $size_name ] );
  croak "No image found for path = \"$path\", image_name = \"$image_name\", size = \"$size_name\"" unless @$results;
  
  my $group = $results->[0];

  bless $group, $group->{'type'} and eval "require $group->{'type'}";

  $sql =<<EOF;
SELECT g.name,
       gs.src,
       g.path
  FROM groups g,
       group_sizes gs,
       sizes s,
       hierarchy h
 WHERE g.group_id = gs.group_id
   AND gs.size_id = s.size_id
   AND h.right_id = g.group_id
   AND g.type LIKE '%Image'
   AND gs.color_id = ?
   AND s.name = ?
   AND h.group_id = ?
   AND h.generation = 1
EOF
  $results = $self->query($sql, [ $group->{color_id}, 'thumb', $group->{group_id} ] );       

  if( @$results ) {  
    $group->{next} = $results->[0];
    bless $group->{next}, $group->{'type'};
  }
  
  $sql =<<EOF;
SELECT g.name,
       gs.src,
       g.path
  FROM groups g,
       group_sizes gs,
       sizes s,
       hierarchy h
 WHERE g.group_id = gs.group_id
   AND gs.size_id = s.size_id
   AND h.left_id = g.group_id
   AND g.type LIKE '%Image'
   AND gs.color_id = ?
   AND s.name = ?
   AND h.group_id = ?
   AND h.generation = 1
EOF
  $results = $self->query($sql, [ $group->{color_id}, 'thumb', $group->{group_id} ] );       
  
  if( @$results ) {
    $group->{previous} = $results->[0];
    bless $group->{previous}, $group->{'type'};
  }

  $sql =<<EOF;
SELECT g.name,
       g.group_id,
       gs.src,
       g.path,
       g.title,
       g.short_title,
       g.type
  FROM groups g,
       group_sizes gs,
       sizes s,
       hierarchy h
 WHERE g.group_id = gs.group_id
   AND gs.size_id = s.size_id
   AND h.elder_id = g.group_id
   AND g.type LIKE '%Group'
   AND gs.color_id = ?
   AND s.name = ?
   AND h.group_id = ?
 ORDER BY generation ASC
EOF
  $results = $self->query($sql, [ $group->{color_id}, 'thumb', $group->{group_id} ] );       

  my $cursor = $group;
  for my $p ( @$results ) {
    bless $p, $p->{'type'};
    $cursor->{'parent'} = $p;
    $cursor = $p;
  }
  
  if( $group->{'parent'} ) {
    my $parent = $group->{'parent'};
    $sql =<<EOF;
SELECT g.name,
       gs.src,
       g.path,
       g.type,
       i.input_image
  FROM groups g,
       group_sizes gs,
       sizes s,
       images i,
       hierarchy h
 WHERE g.group_id = gs.group_id
   AND gs.size_id = s.size_id
   AND h.right_id = g.group_id
   AND i.image_id = g.image_id
   AND g.type LIKE '%Group'
   AND gs.color_id = ?
   AND s.name = ?
   AND h.group_id = ?
   AND h.generation = 1
EOF
    $results = $self->query($sql, [ $group->{color_id}, 'thumb', $parent->{group_id} ] );

    if( @$results ) {
      $parent->{next} = $results->[0];
      bless $parent->{next}, $parent->{next}->{'type'};
    }

    $sql =<<EOF;
SELECT g.name,
       gs.src,
       g.path,
       g.type,
       i.input_image
  FROM groups g,
       group_sizes gs,
       sizes s,
       images i,
       hierarchy h
 WHERE g.group_id = gs.group_id
   AND gs.size_id = s.size_id
   AND h.left_id = g.group_id
   AND i.image_id = g.image_id
   AND g.type LIKE '%Group'
   AND gs.color_id = ?
   AND s.name = ?
   AND h.group_id = ?
   AND h.generation = 1
EOF
    $results = $self->query($sql, [ $group->{color_id}, 'thumb', $parent->{group_id} ] );       

    if( @$results ) {
      $parent->{previous} = $results->[0];
      bless $parent->{previous}, $parent->{previous}->{'type'};
    }
  }

  $sql=<<EOF;
SELECT l.text
  FROM links l,
       group_links nl
 WHERE nl.link_id = l.link_id
   AND nl.group_id = ?
EOF
  my @links = map { Spidy::Link->new(%$_) } $self->query( $sql, [$group->{'group_id'}] );
  $group->{'links'} = \@links;

  $sql=<<EOF;
SELECT co.text
  FROM comments co,
       groups n
 WHERE n.comment_id = co.comment_id
   AND n.group_id = ?
EOF
  my @comments = $self->query( $sql, [$group->{'group_id'}] );
  #there should be only one.
  $group->{'comment'} = Spidy::Comment->new( %{$comments[0]} );

  return $group;
}

1;
