/**
 * @file davis.h
 *
 * DAVIS specific configuration defines and information structures.
 */

#ifndef LIBCAER_DEVICES_DAVIS_H_
#define LIBCAER_DEVICES_DAVIS_H_

#include "usb.h"
#include "../events/polarity.h"
#include "../events/special.h"
#include "../events/frame.h"
#include "../events/imu6.h"
#include "../events/sample.h"

#ifdef __cplusplus
extern "C" {
#endif

/**
 * Device type definition for iniLabs DAVIS FX2-based boards, like DAVIS240a/b/c.
 * Deprecated in favor of CAER_DEVICE_DAVIS.
 */
#define CAER_DEVICE_DAVIS_FX2 1
/**
 * Device type definition for iniLabs DAVIS FX3-based boards, like DAVIS640.
 * Deprecated in favor of CAER_DEVICE_DAVIS.
 */
#define CAER_DEVICE_DAVIS_FX3 2
/**
 * Device type definition for iniLabs DAVIS boards, supporting both FX2 and FX3
 * generation devices. This is the preferred way to access cameras now.
 */
#define CAER_DEVICE_DAVIS     4
/**
 * Device type definition for iniLabs Raspberry Pi-based DAVIS boards.
 */
#define CAER_DEVICE_DAVIS_RPI 6

/**
 * DAVIS240A chip identifier.
 * 240x180, no color, no global shutter.
 */
#define DAVIS_CHIP_DAVIS240A 0
/**
 * DAVIS240B chip identifier.
 * 240x180, no color, 50 test columns left-side.
 */
#define DAVIS_CHIP_DAVIS240B 1
/**
 * DAVIS240C chip identifier.
 * 240x180, no color.
 */
#define DAVIS_CHIP_DAVIS240C 2
/**
 * DAVIS128 chip identifier.
 * 128x128, color possible, internal ADC.
 */
#define DAVIS_CHIP_DAVIS128  3
/**
 * DAVIS346A chip identifier.
 * 346x260, color possible, internal ADC.
 */
#define DAVIS_CHIP_DAVIS346A 4
/**
 * DAVIS346B chip identifier.
 * 346x260, color possible, internal ADC.
 */
#define DAVIS_CHIP_DAVIS346B 5
/**
 * DAVIS640 chip identifier.
 * 640x480, color possible, internal ADC.
 */
#define DAVIS_CHIP_DAVIS640  6
/**
 * DAVISRGB chip identifier.
 * 640x480 APS, 320x240 DVS, color possible, internal ADC.
 */
#define DAVIS_CHIP_DAVISRGB  7
/**
 * DAVIS208 chip identifier.
 * 208x192, special sensitive test pixels, color possible, internal ADC.
 */
#define DAVIS_CHIP_DAVIS208  8
/**
 * DAVIS346C chip identifier.
 * 346x260, BSI, color possible, internal ADC.
 */
#define DAVIS_CHIP_DAVIS346C 9

//@{
/**
 * Macros to check a chip identifier integer against the known chip types.
 * Returns true if a chip identifier matches, false otherwise.
 */
#define IS_DAVIS128(chipID)  ((chipID) == DAVIS_CHIP_DAVIS128)
#define IS_DAVIS208(chipID)  ((chipID) == DAVIS_CHIP_DAVIS208)
#define IS_DAVIS240A(chipID) ((chipID) == DAVIS_CHIP_DAVIS240A)
#define IS_DAVIS240B(chipID) ((chipID) == DAVIS_CHIP_DAVIS240B)
#define IS_DAVIS240C(chipID) ((chipID) == DAVIS_CHIP_DAVIS240C)
#define IS_DAVIS240(chipID)  (IS_DAVIS240A(chipID) || IS_DAVIS240B(chipID) || IS_DAVIS240C(chipID))
#define IS_DAVIS346A(chipID) ((chipID) == DAVIS_CHIP_DAVIS346A)
#define IS_DAVIS346B(chipID) ((chipID) == DAVIS_CHIP_DAVIS346B)
#define IS_DAVIS346C(chipID) ((chipID) == DAVIS_CHIP_DAVIS346C)
#define IS_DAVIS346(chipID)  (IS_DAVIS346A(chipID) || IS_DAVIS346B(chipID) || IS_DAVIS346C(chipID))
#define IS_DAVIS640(chipID)  ((chipID) == DAVIS_CHIP_DAVIS640)
#define IS_DAVISRGB(chipID)  ((chipID) == DAVIS_CHIP_DAVISRGB)
//@}

/**
 * Module address: device-side Multiplexer configuration.
 * The Multiplexer is responsible for mixing, timestamping and outputting
 * (via USB) the various event types generated by the device. It is also
 * responsible for timestamp generation and synchronization.
 */
#define DAVIS_CONFIG_MUX      0
/**
 * Module address: device-side DVS configuration.
 * The DVS state machine handshakes with the chip's AER bus and gets the
 * polarity events from it. It supports various configurable delays, as
 * well as advanced filtering capabilities on the polarity events.
 */
#define DAVIS_CONFIG_DVS      1
/**
 * Module address: device-side APS (Frame) configuration.
 * The APS (Active-Pixel-Sensor) is responsible for getting the normal,
 * synchronous frame from the camera chip. It supports various options
 * for very precise timing control, as well as Region of Interest imaging.
 */
#define DAVIS_CONFIG_APS      2
/**
 * Module address: device-side IMU (Inertial Measurement Unit) configuration.
 * The IMU module connects to the external IMU chip and sends data on the
 * device's movement in space. It can configure various options on the external
 * chip, such as accelerometer range or gyroscope refresh rate.
 */
#define DAVIS_CONFIG_IMU      3
/**
 * Module address: device-side External Input (signal detector/generator) configuration.
 * The External Input module is used to detect external signals on the external input
 * jack and inject an event into the event stream when this happens. It can detect pulses
 * of a specific length or rising and falling edges.
 * On some systems, a signal generator module is also present, which can generate
 * PWM-like pulsed signals with configurable timing.
 */
#define DAVIS_CONFIG_EXTINPUT 4
/**
 * Module address: device-side chip bias configuration.
 * Shared with DAVIS_CONFIG_CHIP.
 * This state machine is responsible for configuring the chip's bias generator.
 */
#define DAVIS_CONFIG_BIAS     5
/**
 * Module address: device-side chip control configuration.
 * Shared with DAVIS_CONFIG_BIAS.
 * This state machine is responsible for configuring the chip's internal
 * control shift registers, to set special options.
 */
#define DAVIS_CONFIG_CHIP     5
/**
 * Module address: device-side system information.
 * The system information module provides various details on the device, such
 * as currently installed logic revision or clock speeds.
 * All its parameters are read-only.
 * This is reserved for internal use and should not be used by
 * anything other than libcaer. Please see the 'struct caer_davis_info'
 * documentation for more details on what information is available.
 */
#define DAVIS_CONFIG_SYSINFO  6
/**
 * Module address: device-side microphone configuration.
 * The Microphone module enables the use of InvenSense stereo
 * microphones to capture samples of sound from devices that
 * support is, such as the miniDAVIS346.
 */
#define DAVIS_CONFIG_MICROPHONE 7
/**
 * Module address: device-side USB output configuration.
 * The USB output module forwards the data from the device and the FPGA/CPLD to
 * the USB chip, usually a Cypress FX2 or FX3.
 */
#define DAVIS_CONFIG_USB      9
/**
 * Module address: device-side DDR-AER output configuration.
 * The DDR-AER output module forwards the data from the device and the FPGA/CPLD to
 * some external device using a 4-phase handshake with data on both flanks.
 */
#define DAVIS_CONFIG_DDRAER   9

/**
 * Parameter address for module DAVIS_CONFIG_MUX:
 * run the Multiplexer state machine, which is responsible for
 * mixing the various event types at the device level, timestamping
 * them and outputting them via USB or other connectors.
 */
#define DAVIS_CONFIG_MUX_RUN                             0
/**
 * Parameter address for module DAVIS_CONFIG_MUX:
 * run the Timestamp Generator inside the Multiplexer state machine,
 * which will provide microsecond accurate timestamps to the
 * events passing through.
 */
#define DAVIS_CONFIG_MUX_TIMESTAMP_RUN                   1
/**
 * Parameter address for module DAVIS_CONFIG_MUX:
 * reset the Timestamp Generator to zero. This also sends a reset
 * pulse to all connected slave devices, resetting their timestamp too.
 */
#define DAVIS_CONFIG_MUX_TIMESTAMP_RESET                 2
/**
 * Parameter address for module DAVIS_CONFIG_MUX:
 * under normal circumstances, the chip's bias generator is only powered
 * up when either the DVS or the APS state machines are running, to save
 * power. This flag forces the bias generator to be powered up all the time,
 * which may be useful when one wants to shut-down both APS and DVS
 * temporarily, but still have a quick and well-defined resume behavior.
 */
#define DAVIS_CONFIG_MUX_FORCE_CHIP_BIAS_ENABLE          3
/**
 * Parameter address for module DAVIS_CONFIG_MUX:
 * drop DVS events if the USB output FIFO is full, instead of having
 * them pile up at the input FIFOs.
 */
#define DAVIS_CONFIG_MUX_DROP_DVS_ON_TRANSFER_STALL      4
/**
 * Parameter address for module DAVIS_CONFIG_MUX:
 * drop APS events if the USB output FIFO is full, instead of having
 * them pile up at the input FIFOs.
 * This normally should not be enabled to guarantee complete, coherent
 * frame events, though small timing differences may cause a reduction
 * in observed image quality.
 */
#define DAVIS_CONFIG_MUX_DROP_APS_ON_TRANSFER_STALL      5
/**
 * Parameter address for module DAVIS_CONFIG_MUX:
 * drop IMU events if the USB output FIFO is full, instead of having
 * them pile up at the input FIFOs.
 * This normally should not be enabled to guarantee complete, coherent
 * IMU events, and not get incomplete or wrong IMU information.
 */
#define DAVIS_CONFIG_MUX_DROP_IMU_ON_TRANSFER_STALL      6
/**
 * Parameter address for module DAVIS_CONFIG_MUX:
 * drop External Input events if the USB output FIFO is full, instead of having
 * them pile up at the input FIFOs.
 */
#define DAVIS_CONFIG_MUX_DROP_EXTINPUT_ON_TRANSFER_STALL 7
/**
 * Parameter address for module DAVIS_CONFIG_MUX:
 * drop Microphone sample events if the USB output FIFO is full, instead of having
 * them pile up at the input FIFOs.
 */
#define DAVIS_CONFIG_MUX_DROP_MIC_ON_TRANSFER_STALL      8
/**
 * Parameter address for module DAVIS_CONFIG_MUX:
 * read-only parameter, information about the presence of the
 * statistics feature.
 * This is reserved for internal use and should not be used by
 * anything other than libcaer. Please see the 'struct caer_davis_info'
 * documentation to get this information.
 */
#define DAVIS_CONFIG_MUX_HAS_STATISTICS              10
/**
 * Parameter address for module DAVIS_CONFIG_MUX:
 * read-only parameter, representing the number of dropped
 * DVS events on the device due to full USB buffers.
 * This is a 64bit value, and should always be read using the
 * function: caerDeviceConfigGet64().
 */
#define DAVIS_CONFIG_MUX_STATISTICS_DVS_DROPPED      11
/**
 * Parameter address for module DAVIS_CONFIG_MUX:
 * read-only parameter, representing the number of dropped
 * APS events on the device due to full USB buffers.
 * This is a 64bit value, and should always be read using the
 * function: caerDeviceConfigGet64().
 */
#define DAVIS_CONFIG_MUX_STATISTICS_APS_DROPPED      13
/**
 * Parameter address for module DAVIS_CONFIG_MUX:
 * read-only parameter, representing the number of dropped
 * IMU events on the device due to full USB buffers.
 * This is a 64bit value, and should always be read using the
 * function: caerDeviceConfigGet64().
 */
#define DAVIS_CONFIG_MUX_STATISTICS_IMU_DROPPED      15
/**
 * Parameter address for module DAVIS_CONFIG_MUX:
 * read-only parameter, representing the number of dropped
 * External Input events on the device due to full USB buffers.
 * This is a 64bit value, and should always be read using the
 * function: caerDeviceConfigGet64().
 */
#define DAVIS_CONFIG_MUX_STATISTICS_EXTINPUT_DROPPED 17
/**
 * Parameter address for module DAVIS_CONFIG_MUX:
 * read-only parameter, representing the number of dropped
 * Microphone sample events on the device due to full USB buffers.
 * This is a 64bit value, and should always be read using the
 * function: caerDeviceConfigGet64().
 */
#define DAVIS_CONFIG_MUX_STATISTICS_MIC_DROPPED      19

/**
 * Parameter address for module DAVIS_CONFIG_DVS:
 * read-only parameter, contains the X axis resolution of the
 * DVS events returned by the camera.
 * This is reserved for internal use and should not be used by
 * anything other than libcaer. Please see the 'struct caer_davis_info'
 * documentation to get proper size information that already
 * considers the rotation and orientation settings.
 */
#define DAVIS_CONFIG_DVS_SIZE_COLUMNS           0
/**
 * Parameter address for module DAVIS_CONFIG_DVS:
 * read-only parameter, contains the Y axis resolution of the
 * DVS events returned by the camera.
 * This is reserved for internal use and should not be used by
 * anything other than libcaer. Please see the 'struct caer_davis_info'
 * documentation to get proper size information that already
 * considers the rotation and orientation settings.
 */
#define DAVIS_CONFIG_DVS_SIZE_ROWS              1
/**
 * Parameter address for module DAVIS_CONFIG_DVS:
 * read-only parameter, contains information on the orientation
 * of the X/Y axes, whether they should be inverted or not on
 * the host when parsing incoming events.
 * Bit 2: dvsInvertXY
 * Bit 1: reserved
 * Bit 0: reserved
 * This is reserved for internal use and should not be used by
 * anything other than libcaer. Please see the 'struct caer_davis_info'
 * documentation to get proper size information that already
 * considers the rotation and orientation settings.
 */
#define DAVIS_CONFIG_DVS_ORIENTATION_INFO       2
/**
 * Parameter address for module DAVIS_CONFIG_DVS:
 * run the DVS state machine and get polarity events from the chip by
 * handshaking with its AER bus.
 */
#define DAVIS_CONFIG_DVS_RUN                    3
/**
 * Parameter address for module DAVIS_CONFIG_DVS:
 * delay capturing the data and acknowledging it on the AER bus for
 * the row events (serial AER protocol) by this many LogicClock cycles.
 */
#define DAVIS_CONFIG_DVS_ACK_DELAY_ROW          4
/**
 * Parameter address for module DAVIS_CONFIG_DVS:
 * delay capturing the data and acknowledging it on the AER bus for
 * the column events (serial AER protocol) by this many LogicClock cycles.
 */
#define DAVIS_CONFIG_DVS_ACK_DELAY_COLUMN       5
/**
 * Parameter address for module DAVIS_CONFIG_DVS:
 * extend the length of the acknowledge on the AER bus for
 * the row events (serial AER protocol) by this many LogicClock cycles.
 */
#define DAVIS_CONFIG_DVS_ACK_EXTENSION_ROW      6
/**
 * Parameter address for module DAVIS_CONFIG_DVS:
 * extend the length of the acknowledge on the AER bus for
 * the column events (serial AER protocol) by this many LogicClock cycles.
 */
#define DAVIS_CONFIG_DVS_ACK_EXTENSION_COLUMN   7
/**
 * Parameter address for module DAVIS_CONFIG_DVS:
 * if the output FIFO for this module is full, stall the AER handshake with
 * the chip and wait until it's free again, instead of just continuing
 * the handshake and dropping the resulting events.
 */
#define DAVIS_CONFIG_DVS_WAIT_ON_TRANSFER_STALL 8
/**
 * Parameter address for module DAVIS_CONFIG_DVS:
 * enable row-only event filter, to eliminate spurious row events with no
 * following columns events. This can happen on DAVIS240 chips, or
 * following the various pixel and background-activity filtering stages,
 * which drop column events to achieve their effect.
 * This should always be enabled!
 */
#define DAVIS_CONFIG_DVS_FILTER_ROW_ONLY_EVENTS 9
/**
 * Parameter address for module DAVIS_CONFIG_DVS:
 * enable external AER control. This ensures the chip and the DVS pixel
 * array are running, but doesn't do the handshake and leaves the ACK
 * pin in high-impedance, to allow for an external system to take
 * over the AER communication with the chip.
 * DAVIS_CONFIG_DVS_RUN has to be turned off for this to work.
 */
#define DAVIS_CONFIG_DVS_EXTERNAL_AER_CONTROL   10
/**
 * Parameter address for module DAVIS_CONFIG_DVS:
 * read-only parameter, information about the presence of the
 * pixel filter feature.
 * This is reserved for internal use and should not be used by
 * anything other than libcaer. Please see the 'struct caer_davis_info'
 * documentation to get this information.
 */
#define DAVIS_CONFIG_DVS_HAS_PIXEL_FILTER       11
/**
 * Parameter address for module DAVIS_CONFIG_DVS:
 * the pixel filter completely suppresses up to eight pixels in the
 * DVS array, filtering out all events produced by them.
 * This is the pixel 0, Y axis setting.
 */
#define DAVIS_CONFIG_DVS_FILTER_PIXEL_0_ROW     12
/**
 * Parameter address for module DAVIS_CONFIG_DVS:
 * the pixel filter completely suppresses up to eight pixels in the
 * DVS array, filtering out all events produced by them.
 * This is the pixel 0, X axis setting.
 */
#define DAVIS_CONFIG_DVS_FILTER_PIXEL_0_COLUMN  13
/**
 * Parameter address for module DAVIS_CONFIG_DVS:
 * the pixel filter completely suppresses up to eight pixels in the
 * DVS array, filtering out all events produced by them.
 * This is the pixel 1, Y axis setting.
 */
#define DAVIS_CONFIG_DVS_FILTER_PIXEL_1_ROW     14
/**
 * Parameter address for module DAVIS_CONFIG_DVS:
 * the pixel filter completely suppresses up to eight pixels in the
 * DVS array, filtering out all events produced by them.
 * This is the pixel 1, X axis setting.
 */
#define DAVIS_CONFIG_DVS_FILTER_PIXEL_1_COLUMN  15
/**
 * Parameter address for module DAVIS_CONFIG_DVS:
 * the pixel filter completely suppresses up to eight pixels in the
 * DVS array, filtering out all events produced by them.
 * This is the pixel 2, Y axis setting.
 */
#define DAVIS_CONFIG_DVS_FILTER_PIXEL_2_ROW     16
/**
 * Parameter address for module DAVIS_CONFIG_DVS:
 * the pixel filter completely suppresses up to eight pixels in the
 * DVS array, filtering out all events produced by them.
 * This is the pixel 2, X axis setting.
 */
#define DAVIS_CONFIG_DVS_FILTER_PIXEL_2_COLUMN  17
/**
 * Parameter address for module DAVIS_CONFIG_DVS:
 * the pixel filter completely suppresses up to eight pixels in the
 * DVS array, filtering out all events produced by them.
 * This is the pixel 3, Y axis setting.
 */
#define DAVIS_CONFIG_DVS_FILTER_PIXEL_3_ROW     18
/**
 * Parameter address for module DAVIS_CONFIG_DVS:
 * the pixel filter completely suppresses up to eight pixels in the
 * DVS array, filtering out all events produced by them.
 * This is the pixel 3, X axis setting.
 */
#define DAVIS_CONFIG_DVS_FILTER_PIXEL_3_COLUMN  19
/**
 * Parameter address for module DAVIS_CONFIG_DVS:
 * the pixel filter completely suppresses up to eight pixels in the
 * DVS array, filtering out all events produced by them.
 * This is the pixel 4, Y axis setting.
 */
#define DAVIS_CONFIG_DVS_FILTER_PIXEL_4_ROW     20
/**
 * Parameter address for module DAVIS_CONFIG_DVS:
 * the pixel filter completely suppresses up to eight pixels in the
 * DVS array, filtering out all events produced by them.
 * This is the pixel 4, X axis setting.
 */
#define DAVIS_CONFIG_DVS_FILTER_PIXEL_4_COLUMN  21
/**
 * Parameter address for module DAVIS_CONFIG_DVS:
 * the pixel filter completely suppresses up to eight pixels in the
 * DVS array, filtering out all events produced by them.
 * This is the pixel 5, Y axis setting.
 */
#define DAVIS_CONFIG_DVS_FILTER_PIXEL_5_ROW     22
/**
 * Parameter address for module DAVIS_CONFIG_DVS:
 * the pixel filter completely suppresses up to eight pixels in the
 * DVS array, filtering out all events produced by them.
 * This is the pixel 5, X axis setting.
 */
#define DAVIS_CONFIG_DVS_FILTER_PIXEL_5_COLUMN  23
/**
 * Parameter address for module DAVIS_CONFIG_DVS:
 * the pixel filter completely suppresses up to eight pixels in the
 * DVS array, filtering out all events produced by them.
 * This is the pixel 6, Y axis setting.
 */
#define DAVIS_CONFIG_DVS_FILTER_PIXEL_6_ROW     24
/**
 * Parameter address for module DAVIS_CONFIG_DVS:
 * the pixel filter completely suppresses up to eight pixels in the
 * DVS array, filtering out all events produced by them.
 * This is the pixel 6, X axis setting.
 */
#define DAVIS_CONFIG_DVS_FILTER_PIXEL_6_COLUMN  25
/**
 * Parameter address for module DAVIS_CONFIG_DVS:
 * the pixel filter completely suppresses up to eight pixels in the
 * DVS array, filtering out all events produced by them.
 * This is the pixel 7, Y axis setting.
 */
#define DAVIS_CONFIG_DVS_FILTER_PIXEL_7_ROW     26
/**
 * Parameter address for module DAVIS_CONFIG_DVS:
 * the pixel filter completely suppresses up to eight pixels in the
 * DVS array, filtering out all events produced by them.
 * This is the pixel 7, X axis setting.
 */
#define DAVIS_CONFIG_DVS_FILTER_PIXEL_7_COLUMN  27
/**
 * Parameter address for module DAVIS_CONFIG_DVS:
 * read-only parameter, information about the presence of the
 * background-activity filter feature.
 * This is reserved for internal use and should not be used by
 * anything other than libcaer. Please see the 'struct caer_davis_info'
 * documentation to get this information.
 */
#define DAVIS_CONFIG_DVS_HAS_BACKGROUND_ACTIVITY_FILTER  28
/**
 * Parameter address for module DAVIS_CONFIG_DVS:
 * enable the background-activity filter, which tries to remove events
 * caused by transistor leakage, by rejecting uncorrelated events.
 */
#define DAVIS_CONFIG_DVS_FILTER_BACKGROUND_ACTIVITY      29
/**
 * Parameter address for module DAVIS_CONFIG_DVS:
 * specify the time difference constant for the background-activity
 * filter in microseconds. Events that do correlated within this
 * time-frame are let through, while others are filtered out.
 */
#define DAVIS_CONFIG_DVS_FILTER_BACKGROUND_ACTIVITY_TIME 30
/**
 * Parameter address for module DAVIS_CONFIG_DVS:
 * read-only parameter, information about the presence of the
 * test event generator feature.
 * This is reserved for internal use and should not be used by
 * anything other than libcaer. Please see the 'struct caer_davis_info'
 * documentation to get this information.
 */
#define DAVIS_CONFIG_DVS_HAS_TEST_EVENT_GENERATOR        31
/**
 * Parameter address for module DAVIS_CONFIG_DVS:
 * enable the test event generator for debugging purposes.
 * This generates fake events that appear to originate from all rows
 * sequentially, and for each row going through all its columns,
 * first with an ON polarity and then with an OFF polarity.
 * Both DAVIS_CONFIG_DVS_RUN and DAVIS_CONFIG_DVS_EXTERNAL_AER_CONTROL
 * have to be turned off for this to work.
 */
#define DAVIS_CONFIG_DVS_TEST_EVENT_GENERATOR_ENABLE     32
/**
 * Parameter address for module DAVIS_CONFIG_DVS:
 * enable the refractory period filter, which limits the firing rate
 * of pixels. This is supported together with the background-activity
 * filter.
 */
#define DAVIS_CONFIG_DVS_FILTER_REFRACTORY_PERIOD        33
/**
 * Parameter address for module DAVIS_CONFIG_DVS:
 * specify the time constant for the refractory period filter.
 * Pixels will be inhibited from generating new events during this
 * time after the last even has fired.
 */
#define DAVIS_CONFIG_DVS_FILTER_REFRACTORY_PERIOD_TIME   34
/**
 * Parameter address for module DAVIS_CONFIG_DVS:
 * read-only parameter, information about the presence of the
 * ROI filter feature.
 * This is reserved for internal use and should not be used by
 * anything other than libcaer. Please see the 'struct caer_davis_info'
 * documentation to get this information.
 */
#define DAVIS_CONFIG_DVS_HAS_ROI_FILTER          35
/**
 * Parameter address for module DAVIS_CONFIG_DVS:
 * start position on the X axis for Region of Interest.
 * Must be between 0 and DVS_SIZE_X-1, and be smaller
 * or equal to DAVIS_CONFIG_DVS_FILTER_ROI_END_COLUMN.
 */
#define DAVIS_CONFIG_DVS_FILTER_ROI_START_COLUMN 36
/**
 * Parameter address for module DAVIS_CONFIG_DVS:
 * start position on the Y axis for Region of Interest.
 * Must be between 0 and DVS_SIZE_Y-1, and be smaller
 * or equal to DAVIS_CONFIG_DVS_FILTER_ROI_END_ROW.
 */
#define DAVIS_CONFIG_DVS_FILTER_ROI_START_ROW    37
/**
 * Parameter address for module DAVIS_CONFIG_DVS:
 * end position on the X axis for Region of Interest.
 * Must be between 0 and DVS_SIZE_X-1, and be greater
 * or equal to DAVIS_CONFIG_DVS_FILTER_ROI_START_COLUMN.
 */
#define DAVIS_CONFIG_DVS_FILTER_ROI_END_COLUMN   38
/**
 * Parameter address for module DAVIS_CONFIG_DVS:
 * end position on the Y axis for Region of Interest.
 * Must be between 0 and DVS_SIZE_Y-1, and be greater
 * or equal to DAVIS_CONFIG_DVS_FILTER_ROI_START_ROW.
 */
#define DAVIS_CONFIG_DVS_FILTER_ROI_END_ROW      39
/**
 * Parameter address for module DAVIS_CONFIG_DVS:
 * read-only parameter, information about the presence of the
 * statistics feature.
 * This is reserved for internal use and should not be used by
 * anything other than libcaer. Please see the 'struct caer_davis_info'
 * documentation to get this information.
 */
#define DAVIS_CONFIG_DVS_HAS_STATISTICS             40
/**
 * Parameter address for module DAVIS_CONFIG_DVS:
 * read-only parameter, representing the number of row event
 * transactions completed on the device.
 * This is a 64bit value, and should always be read using the
 * function: caerDeviceConfigGet64().
 */
#define DAVIS_CONFIG_DVS_STATISTICS_EVENTS_ROW      41
/**
 * Parameter address for module DAVIS_CONFIG_DVS:
 * read-only parameter, representing the number of column event
 * transactions completed on the device.
 * This is a 64bit value, and should always be read using the
 * function: caerDeviceConfigGet64().
 */
#define DAVIS_CONFIG_DVS_STATISTICS_EVENTS_COLUMN   43
/**
 * Parameter address for module DAVIS_CONFIG_DVS:
 * read-only parameter, representing the number of dropped
 * transaction sequences on the device due to full buffers.
 * This is a 64bit value, and should always be read using the
 * function: caerDeviceConfigGet64().
 */
#define DAVIS_CONFIG_DVS_STATISTICS_EVENTS_DROPPED  45
/**
 * Parameter address for module DAVIS_CONFIG_DVS:
 * read-only parameter, representing the number of dropped
 * events due to the pixel filter.
 * This is a 64bit value, and should always be read using the
 * function: caerDeviceConfigGet64().
 */
#define DAVIS_CONFIG_DVS_STATISTICS_FILTERED_PIXELS 47
/**
 * Parameter address for module DAVIS_CONFIG_DVS:
 * read-only parameter, representing the number of dropped
 * events due to the background-activity filter.
 * This is a 64bit value, and should always be read using the
 * function: caerDeviceConfigGet64().
 */
#define DAVIS_CONFIG_DVS_STATISTICS_FILTERED_BACKGROUND_ACTIVITY 49
/**
 * Parameter address for module DAVIS_CONFIG_DVS:
 * read-only parameter, representing the number of dropped
 * events due to the refractory period filter.
 * This is a 64bit value, and should always be read using the
 * function: caerDeviceConfigGet64().
 */
#define DAVIS_CONFIG_DVS_STATISTICS_FILTERED_REFRACTORY_PERIOD   51

/**
 * Parameter address for module DAVIS_CONFIG_APS:
 * read-only parameter, contains the X axis resolution of the
 * APS frames returned by the camera.
 * This is reserved for internal use and should not be used by
 * anything other than libcaer. Please see the 'struct caer_davis_info'
 * documentation to get proper size information that already
 * considers the rotation and orientation settings.
 */
#define DAVIS_CONFIG_APS_SIZE_COLUMNS           0
/**
 * Parameter address for module DAVIS_CONFIG_APS:
 * read-only parameter, contains the Y axis resolution of the
 * APS frames returned by the camera.
 * This is reserved for internal use and should not be used by
 * anything other than libcaer. Please see the 'struct caer_davis_info'
 * documentation to get proper size information that already
 * considers the rotation and orientation settings.
 */
#define DAVIS_CONFIG_APS_SIZE_ROWS              1
/**
 * Parameter address for module DAVIS_CONFIG_APS:
 * read-only parameter, contains information on the orientation
 * of the X/Y axes, whether they should be inverted or not on
 * the host when parsing incoming pixels, as well as if the X
 * or Y axes need to be flipped when reading the pixels.
 * Bit 2: apsInvertXY
 * Bit 1: apsFlipX
 * Bit 0: apsFlipY
 * This is reserved for internal use and should not be used by
 * anything other than libcaer. Please see the 'struct caer_davis_info'
 * documentation to get proper size information that already
 * considers the rotation and orientation settings.
 */
#define DAVIS_CONFIG_APS_ORIENTATION_INFO       2
/**
 * Parameter address for module DAVIS_CONFIG_APS:
 * read-only parameter, contains information on the type of
 * color filter present on the device.
 * This is reserved for internal use and should not be used by
 * anything other than libcaer. Please see the 'struct caer_davis_info'
 * documentation to get proper color filter information.
 */
#define DAVIS_CONFIG_APS_COLOR_FILTER           3
/**
 * Parameter address for module DAVIS_CONFIG_APS:
 * enable the APS module and take intensity images of the scene.
 * While this parameter is enabled, frames will be taken continuously.
 * To slow down the frame-rate, see DAVIS_CONFIG_APS_FRAME_DELAY.
 * To only take snapshots, see DAVIS_CONFIG_APS_SNAPSHOT.
 */
#define DAVIS_CONFIG_APS_RUN                    4
/**
 * Parameter address for module DAVIS_CONFIG_APS:
 * enable the reset read phase in addition to the signal read,
 * to allow for correlated double sampling schemes. This heavily
 * improves image quality and should always be turned on. In special
 * cases, especially when the camera is perfectly stationary, this
 * can be turned off for longer periods of time to achieve a higher
 * frame-rate and significantly faster frame capture.
 */
#define DAVIS_CONFIG_APS_RESET_READ             5
/**
 * Parameter address for module DAVIS_CONFIG_APS:
 * if the output FIFO for this module is full, stall the APS state machine
 * and wait until it's free again, instead of just dropping the pixels as
 * they are being read out. This guarantees a complete frame readout, at the
 * possible cost of slight timing differences between pixels. If disabled,
 * incomplete frames may be transmitted and will then be dropped on the host,
 * resulting in lower frame-rates, especially during high DVS traffic.
 */
#define DAVIS_CONFIG_APS_WAIT_ON_TRANSFER_STALL 6
/**
 * Parameter address for module DAVIS_CONFIG_APS:
 * read-only parameter, information about the presence of the
 * global shutter feature.
 * This is reserved for internal use and should not be used by
 * anything other than libcaer. Please see the 'struct caer_davis_info'
 * documentation to get this information.
 */
#define DAVIS_CONFIG_APS_HAS_GLOBAL_SHUTTER     7
/**
 * Parameter address for module DAVIS_CONFIG_APS:
 * enable Global Shutter mode instead of Rolling Shutter.
 * The Global Shutter eliminates motion artifacts, but
 * is noisier than the Rolling Shutter (worse quality).
 */
#define DAVIS_CONFIG_APS_GLOBAL_SHUTTER         8
/**
 * Parameter address for module DAVIS_CONFIG_APS:
 * start position on the X axis for Region of Interest 0.
 * Must be between 0 and APS_SIZE_X-1, and be smaller
 * or equal to DAVIS_CONFIG_APS_END_COLUMN_0.
 */
#define DAVIS_CONFIG_APS_START_COLUMN_0         9
/**
 * Parameter address for module DAVIS_CONFIG_APS:
 * start position on the Y axis for Region of Interest 0.
 * Must be between 0 and APS_SIZE_Y-1, and be smaller
 * or equal to DAVIS_CONFIG_APS_END_ROW_0.
 */
#define DAVIS_CONFIG_APS_START_ROW_0            10
/**
 * Parameter address for module DAVIS_CONFIG_APS:
 * end position on the X axis for Region of Interest 0.
 * Must be between 0 and APS_SIZE_X-1, and be greater
 * or equal to DAVIS_CONFIG_APS_START_COLUMN_0.
 */
#define DAVIS_CONFIG_APS_END_COLUMN_0           11
/**
 * Parameter address for module DAVIS_CONFIG_APS:
 * end position on the Y axis for Region of Interest 0.
 * Must be between 0 and APS_SIZE_Y-1, and be greater
 * or equal to DAVIS_CONFIG_APS_START_ROW_0.
 */
#define DAVIS_CONFIG_APS_END_ROW_0              12
/**
 * Parameter address for module DAVIS_CONFIG_APS:
 * frame exposure time in microseconds, up to about
 * one second maximum.
 * Very precise for Global Shutter, slightly less exact for
 * Rolling Shutter due to column-based timing constraints.
 */
#define DAVIS_CONFIG_APS_EXPOSURE               13
/**
 * Parameter address for module DAVIS_CONFIG_APS:
 * delay between consecutive frames in microseconds,
 * up to about one second maximum.
 * This can be used to achieve slower frame-rates, down
 * to about 1 Hertz.
 */
#define DAVIS_CONFIG_APS_FRAME_DELAY            14
/**
 * Parameter address for module DAVIS_CONFIG_APS:
 * column reset settle time in ADCClock cycles.
 */
#define DAVIS_CONFIG_APS_RESET_SETTLE           15
/**
 * Parameter address for module DAVIS_CONFIG_APS:
 * column settle time in ADCClock cycles.
 */
#define DAVIS_CONFIG_APS_COLUMN_SETTLE          16
/**
 * Parameter address for module DAVIS_CONFIG_APS:
 * row settle time in ADCClock cycles.
 */
#define DAVIS_CONFIG_APS_ROW_SETTLE             17
/**
 * Parameter address for module DAVIS_CONFIG_APS:
 * null (between states) settle time in ADCClock cycles.
 */
#define DAVIS_CONFIG_APS_NULL_SETTLE            18
/**
 * Parameter address for module DAVIS_CONFIG_APS:
 * read-only parameter, information about the presence of the
 * Quadruple Region of Interest feature.
 * This is reserved for internal use and should not be used by
 * anything other than libcaer. Please see the 'struct caer_davis_info'
 * documentation to get this information.
 */
#define DAVIS_CONFIG_APS_HAS_QUAD_ROI           19
/**
 * Parameter address for module DAVIS_CONFIG_APS:
 * start position on the X axis for Region of Interest 1.
 * Must be between 0 and APS_SIZE_X-1, and be smaller
 * or equal to DAVIS_CONFIG_APS_END_COLUMN_1.
 */
#define DAVIS_CONFIG_APS_START_COLUMN_1         20
/**
 * Parameter address for module DAVIS_CONFIG_APS:
 * start position on the Y axis for Region of Interest 1.
 * Must be between 0 and APS_SIZE_Y-1, and be smaller
 * or equal to DAVIS_CONFIG_APS_END_ROW_1.
 */
#define DAVIS_CONFIG_APS_START_ROW_1            21
/**
 * Parameter address for module DAVIS_CONFIG_APS:
 * end position on the X axis for Region of Interest 1.
 * Must be between 0 and APS_SIZE_X-1, and be greater
 * or equal to DAVIS_CONFIG_APS_START_COLUMN_1.
 */
#define DAVIS_CONFIG_APS_END_COLUMN_1           22
/**
 * Parameter address for module DAVIS_CONFIG_APS:
 * end position on the Y axis for Region of Interest 1.
 * Must be between 0 and APS_SIZE_Y-1, and be greater
 * or equal to DAVIS_CONFIG_APS_START_ROW_1.
 */
#define DAVIS_CONFIG_APS_END_ROW_1              23
/**
 * Parameter address for module DAVIS_CONFIG_APS:
 * start position on the X axis for Region of Interest 2.
 * Must be between 0 and APS_SIZE_X-1, and be smaller
 * or equal to DAVIS_CONFIG_APS_END_COLUMN_2.
 */
#define DAVIS_CONFIG_APS_START_COLUMN_2         24
/**
 * Parameter address for module DAVIS_CONFIG_APS:
 * start position on the Y axis for Region of Interest 2.
 * Must be between 0 and APS_SIZE_Y-1, and be smaller
 * or equal to DAVIS_CONFIG_APS_END_ROW_2.
 */
#define DAVIS_CONFIG_APS_START_ROW_2            25
/**
 * Parameter address for module DAVIS_CONFIG_APS:
 * end position on the X axis for Region of Interest 2.
 * Must be between 0 and APS_SIZE_X-1, and be greater
 * or equal to DAVIS_CONFIG_APS_START_COLUMN_2.
 */
#define DAVIS_CONFIG_APS_END_COLUMN_2           26
/**
 * Parameter address for module DAVIS_CONFIG_APS:
 * end position on the Y axis for Region of Interest 2.
 * Must be between 0 and APS_SIZE_Y-1, and be greater
 * or equal to DAVIS_CONFIG_APS_START_ROW_2.
 */
#define DAVIS_CONFIG_APS_END_ROW_2              27
/**
 * Parameter address for module DAVIS_CONFIG_APS:
 * start position on the X axis for Region of Interest 3.
 * Must be between 0 and APS_SIZE_X-1, and be smaller
 * or equal to DAVIS_CONFIG_APS_END_COLUMN_3.
 */
#define DAVIS_CONFIG_APS_START_COLUMN_3         28
/**
 * Parameter address for module DAVIS_CONFIG_APS:
 * start position on the Y axis for Region of Interest 3.
 * Must be between 0 and APS_SIZE_Y-1, and be smaller
 * or equal to DAVIS_CONFIG_APS_END_ROW_3.
 */
#define DAVIS_CONFIG_APS_START_ROW_3            29
/**
 * Parameter address for module DAVIS_CONFIG_APS:
 * end position on the X axis for Region of Interest 3.
 * Must be between 0 and APS_SIZE_X-1, and be greater
 * or equal to DAVIS_CONFIG_APS_START_COLUMN_3.
 */
#define DAVIS_CONFIG_APS_END_COLUMN_3           30
/**
 * Parameter address for module DAVIS_CONFIG_APS:
 * end position on the Y axis for Region of Interest 3.
 * Must be between 0 and APS_SIZE_Y-1, and be greater
 * or equal to DAVIS_CONFIG_APS_START_ROW_3.
 */
#define DAVIS_CONFIG_APS_END_ROW_3              31
/**
 * Parameter address for module DAVIS_CONFIG_APS:
 * read-only parameter, information about the presence of an
 * internal, on-chip ADC to read the pixel values.
 * This is reserved for internal use and should not be used by
 * anything other than libcaer. Please see the 'struct caer_davis_info'
 * documentation to get this information.
 */
#define DAVIS_CONFIG_APS_HAS_INTERNAL_ADC       33
/**
 * Parameter address for module DAVIS_CONFIG_APS:
 * enable sampling of pixel voltage by the internal ADC circuitry.
 * Must always be enabled to get proper frame values.
 */
#define DAVIS_CONFIG_APS_SAMPLE_ENABLE          35
/**
 * Parameter address for module DAVIS_CONFIG_APS:
 * sample settle time in ADCClock cycles.
 */
#define DAVIS_CONFIG_APS_SAMPLE_SETTLE          36
/**
 * Parameter address for module DAVIS_CONFIG_APS:
 * ramp reset time in ADCClock cycles.
 */
#define DAVIS_CONFIG_APS_RAMP_RESET             37
/**
 * Parameter address for module DAVIS_CONFIG_APS:
 * only perform a short ramp (half length) during reset reads, given that
 * the voltage should always be close to the top of the range.
 * This increases the frame-rate, but may have impacts on image quality,
 * especially in very bright regions.
 */
#define DAVIS_CONFIG_APS_RAMP_SHORT_RESET       38
/**
 * Parameter address for module DAVIS_CONFIG_APS:
 * put all APS pixels into reset, while keeping everything else running.
 * This is only useful for testing and characterizing the internal
 * ADC, to minimize noise.
 */
#define DAVIS_CONFIG_APS_ADC_TEST_MODE          39
/**
 * Parameter address for module DAVIS_CONFIG_APS:
 * Enable/disable ROI region 0.
 * ROI region 0 is always present.
 */
#define DAVIS_CONFIG_APS_ROI0_ENABLED           40
/**
 * Parameter address for module DAVIS_CONFIG_APS:
 * Enable/disable ROI region 1.
 * ROI region 1 is only available when apsHasQuadROI=true,
 * see 'struct caer_davis_info' for more information.
 */
#define DAVIS_CONFIG_APS_ROI1_ENABLED           41
/**
 * Parameter address for module DAVIS_CONFIG_APS:
 * Enable/disable ROI region 2.
 * ROI region 2 is only available when apsHasQuadROI=true,
 * see 'struct caer_davis_info' for more information.
 */
#define DAVIS_CONFIG_APS_ROI2_ENABLED           42
/**
 * Parameter address for module DAVIS_CONFIG_APS:
 * Enable/disable ROI region 3.
 * ROI region 3 is only available when apsHasQuadROI=true,
 * see 'struct caer_davis_info' for more information.
 */
#define DAVIS_CONFIG_APS_ROI3_ENABLED           43

// Extra timing settings for DAVISRGB APS module.
/**
 * Parameter address for module DAVIS_CONFIG_APS (only for DAVIS RGB chip):
 * charge transfer time in ADCClock cycles.
 */
#define DAVISRGB_CONFIG_APS_TRANSFER    50
/**
 * Parameter address for module DAVIS_CONFIG_APS (only for DAVIS RGB chip):
 * Rolling Shutter FD settle time in ADCClock cycles.
 */
#define DAVISRGB_CONFIG_APS_RSFDSETTLE  51
/**
 * Parameter address for module DAVIS_CONFIG_APS (only for DAVIS RGB chip):
 * Global Shutter PD reset time in ADCClock cycles.
 */
#define DAVISRGB_CONFIG_APS_GSPDRESET   52
/**
 * Parameter address for module DAVIS_CONFIG_APS (only for DAVIS RGB chip):
 * Global Shutter Reset Fall time in ADCClock cycles.
 */
#define DAVISRGB_CONFIG_APS_GSRESETFALL 53
/**
 * Parameter address for module DAVIS_CONFIG_APS (only for DAVIS RGB chip):
 * Global Shutter Transfer Fall time in ADCClock cycles.
 */
#define DAVISRGB_CONFIG_APS_GSTXFALL    54
/**
 * Parameter address for module DAVIS_CONFIG_APS (only for DAVIS RGB chip):
 * Global Shutter FD reset time in ADCClock cycles.
 */
#define DAVISRGB_CONFIG_APS_GSFDRESET   55

/**
 * Parameter address for module DAVIS_CONFIG_APS:
 * takes a snapshot (one frame), like a photo-camera.
 * More efficient implementation that just toggling
 * the DAVIS_CONFIG_APS_RUN parameter.
 * The APS module should not be running prior to calling this,
 * as it only makes sense if frames are not being generated
 * at the time. Also, DAVIS_CONFIG_APS_FRAME_DELAY should
 * be set to zero if only doing snapshots, to ensure a
 * quicker readiness for the next one, since the delay is
 * always observed after taking a frame.
 */
#define DAVIS_CONFIG_APS_SNAPSHOT       80

/**
 * Parameter address for module DAVIS_CONFIG_APS:
 * automatic exposure control, tries to set the exposure value
 * automatically to an appropriate value to maximize information
 * in the scene and minimize under- and over-exposure.
 */
#define DAVIS_CONFIG_APS_AUTOEXPOSURE   81


/**
 * Parameter address for module DAVIS_CONFIG_IMU:
 * run the IMU state machine to get information about the
 * movement and position of the device. This takes the
 * IMU chip out of sleep.
 */
#define DAVIS_CONFIG_IMU_RUN                     0
/**
 * Parameter address for module DAVIS_CONFIG_IMU:
 * put the temperature sensor in standby, disabling it.
 */
#define DAVIS_CONFIG_IMU_TEMP_STANDBY            1
/**
 * Parameter address for module DAVIS_CONFIG_IMU:
 * put the accelerometer sensor in standby, disabling it.
 */
#define DAVIS_CONFIG_IMU_ACCEL_STANDBY           2
/**
 * Parameter address for module DAVIS_CONFIG_IMU:
 * put the gyroscope sensor in standby, disabling it.
 */
#define DAVIS_CONFIG_IMU_GYRO_STANDBY            3
/**
 * Parameter address for module DAVIS_CONFIG_IMU:
 * put the IMU into Cycle Mode. In Cycle Mode, the device
 * cycles between sleep mode and waking up to take a single
 * sample of data from the accelerometer at a rate determined
 * by DAVIS_CONFIG_IMU_LP_WAKEUP.
 */
#define DAVIS_CONFIG_IMU_LP_CYCLE                4
/**
 * Parameter address for module DAVIS_CONFIG_IMU:
 * rate at which the IMU takes an accelerometer sample while
 * in Cycle Mode (see DAVIS_CONFIG_IMU_LP_CYCLE).
 * Valid values are:
 * 0 - 1.25 Hz wake-up frequency
 * 1 - 5 Hz wake-up frequency
 * 2 - 20 Hz wake-up frequency
 * 3 - 40 Hz wake-up frequency
 */
#define DAVIS_CONFIG_IMU_LP_WAKEUP               5
/**
 * Parameter address for module DAVIS_CONFIG_IMU:
 * this specifies the divider from the Gyroscope Output Rate used
 * to generate the Sample Rate for the IMU.
 * Valid values are from 0 to 255.
 * The Sample Rate is generated like this:
 * Sample Rate = Gyroscope Output Rate / (1 + DAVIS_CONFIG_IMU_SAMPLE_RATE_DIVIDER)
 * where Gyroscope Output Rate = 8 kHz when DAVIS_CONFIG_IMU_DIGITAL_LOW_PASS_FILTER
 * is disabled (set to 0 or 7), and 1 kHz when enabled.
 * Note: the accelerometer output rate is 1 kHz. This means that for a Sample Rate
 * greater than 1 kHz, the same accelerometer sample may be output multiple times.
 */
#define DAVIS_CONFIG_IMU_SAMPLE_RATE_DIVIDER     6
/**
 * Parameter address for module DAVIS_CONFIG_IMU:
 * this configures the digital low-pass filter for both the
 * accelerometer and the gyroscope.
 * Valid values are from 0 to 7 and have the following meaning:
 * 0 - Accel: BW=260Hz, Delay=0ms,    FS=1kHz - Gyro: BW=256Hz, Delay=0.98ms, FS=8kHz
 * 1 - Accel: BW=184Hz, Delay=2.0ms,  FS=1kHz - Gyro: BW=188Hz, Delay=1.9ms,  FS=1kHz
 * 2 - Accel: BW=94Hz,  Delay=3.0ms,  FS=1kHz - Gyro: BW=98Hz,  Delay=2.8ms,  FS=1kHz
 * 3 - Accel: BW=44Hz,  Delay=4.9ms,  FS=1kHz - Gyro: BW=42Hz,  Delay=4.8ms,  FS=1kHz
 * 4 - Accel: BW=21Hz,  Delay=8.5ms,  FS=1kHz - Gyro: BW=20Hz,  Delay=8.3ms,  FS=1kHz
 * 5 - Accel: BW=10Hz,  Delay=13.8ms, FS=1kHz - Gyro: BW=10Hz,  Delay=13.4ms, FS=1kHz
 * 6 - Accel: BW=5Hz,   Delay=19.0ms, FS=1kHz - Gyro: BW=5Hz,   Delay=18.6ms, FS=1kHz
 * 7 - Accel: RESERVED,               FS=1kHz - Gyro: RESERVED,               FS=8kHz
 */
#define DAVIS_CONFIG_IMU_DIGITAL_LOW_PASS_FILTER 7
/**
 * Parameter address for module DAVIS_CONFIG_IMU:
 * select the full scale range of the accelerometer outputs.
 * Valid values are:
 * 0 - +- 2 g
 * 1 - +- 4 g
 * 2 - +- 8 g
 * 3 - +- 16 g
 */
#define DAVIS_CONFIG_IMU_ACCEL_FULL_SCALE        8
/**
 * Parameter address for module DAVIS_CONFIG_IMU:
 * select the full scale range of the gyroscope outputs.
 * Valid values are:
 * 0 - +- 250 °/s
 * 1 - +- 500 °/s
 * 2 - +- 1000 °/s
 * 3 - +- 2000 °/s
 */
#define DAVIS_CONFIG_IMU_GYRO_FULL_SCALE         9
/**
 * Parameter address for module DAVIS_CONFIG_IMU:
 * read-only parameter, contains information on the orientation
 * of the X/Y/Z axes, whether they should be flipped or not on
 * the host when parsing incoming IMU data samples.
 * Bit 2: imuFlipX
 * Bit 1: imuFlipY
 * Bit 0: imuFlipZ
 * This is reserved for internal use and should not be used by
 * anything other than libcaer. Generated IMU events are already
 * properly flipped when returned to the user.
 */
#define DAVIS_CONFIG_IMU_ORIENTATION_INFO        10

/**
 * Parameter address for module DAVIS_CONFIG_EXTINPUT:
 * enable the signal detector module. It generates events
 * when it sees certain types of signals, such as edges or
 * pulses of a defined length, on the IN JACK signal.
 * This can be useful to inject events into the event
 * stream in response to external stimuli or controls,
 * such as turning on a LED lamp.
 */
#define DAVIS_CONFIG_EXTINPUT_RUN_DETECTOR               0
/**
 * Parameter address for module DAVIS_CONFIG_EXTINPUT:
 * send a special EXTERNAL_INPUT_RISING_EDGE event when a
 * rising edge is detected (transition from low voltage to high).
 */
#define DAVIS_CONFIG_EXTINPUT_DETECT_RISING_EDGES        1
/**
 * Parameter address for module DAVIS_CONFIG_EXTINPUT:
 * send a special EXTERNAL_INPUT_FALLING_EDGE event when a
 * falling edge is detected (transition from high voltage to low).
 */
#define DAVIS_CONFIG_EXTINPUT_DETECT_FALLING_EDGES       2
/**
 * Parameter address for module DAVIS_CONFIG_EXTINPUT:
 * send a special EXTERNAL_INPUT_PULSE event when a pulse, of
 * a specified, configurable polarity and length, is detected.
 * See DAVIS_CONFIG_EXTINPUT_DETECT_PULSE_POLARITY and
 * DAVIS_CONFIG_EXTINPUT_DETECT_PULSE_LENGTH for more details.
 */
#define DAVIS_CONFIG_EXTINPUT_DETECT_PULSES              3
/**
 * Parameter address for module DAVIS_CONFIG_EXTINPUT:
 * the polarity the pulse must exhibit to be detected as such.
 * '1' means active high; a pulse will start when the signal
 * goes from low to high and will continue to be seen as the
 * same pulse as long as it stays high.
 * '0' means active low; a pulse will start when the signal
 * goes from high to low and will continue to be seen as the
 * same pulse as long as it stays low.
 */
#define DAVIS_CONFIG_EXTINPUT_DETECT_PULSE_POLARITY      4
/**
 * Parameter address for module DAVIS_CONFIG_EXTINPUT:
 * the minimal length that a pulse must have to trigger the
 * sending of a special event. This is measured in cycles
 * at LogicClock frequency (see 'struct caer_davis_info' for
 * details on how to get the frequency).
 */
#define DAVIS_CONFIG_EXTINPUT_DETECT_PULSE_LENGTH        5
/**
 * Parameter address for module DAVIS_CONFIG_EXTINPUT:
 * read-only parameter, information about the presence of the
 * signal generator feature.
 * This is reserved for internal use and should not be used by
 * anything other than libcaer. Please see the 'struct caer_davis_info'
 * documentation to get this information.
 */
#define DAVIS_CONFIG_EXTINPUT_HAS_GENERATOR              6
/**
 * Parameter address for module DAVIS_CONFIG_EXTINPUT:
 * enable the signal generator module. It generates a
 * PWM-like signal based on configurable parameters and
 * outputs it on the OUT JACK signal.
 */
#define DAVIS_CONFIG_EXTINPUT_RUN_GENERATOR              7
/**
 * Parameter address for module DAVIS_CONFIG_EXTINPUT:
 * instead of generating a PWM-like signal by using the
 * configured parameters, use a signal on the FPGA/CPLD
 * that's passed as an input to the External Input module.
 * By default this is disabled and tied to ground, but
 * it can be useful for customized logic designs.
 */
#define DAVIS_CONFIG_EXTINPUT_GENERATE_USE_CUSTOM_SIGNAL 8
/**
 * Parameter address for module DAVIS_CONFIG_EXTINPUT:
 * polarity of the PWM-like signal to be generated.
 * '1' means active high, '0' means active low.
 */
#define DAVIS_CONFIG_EXTINPUT_GENERATE_PULSE_POLARITY    9
/**
 * Parameter address for module DAVIS_CONFIG_EXTINPUT:
 * the interval between the start of two consecutive pulses,
 * expressed in cycles at LogicClock frequency (see
 * 'struct caer_davis_info' for details on how to get the frequency).
 * This must be bigger or equal to DAVIS_CONFIG_EXTINPUT_GENERATE_PULSE_LENGTH.
 * To generate a signal with 50% duty cycle, this would
 * have to be exactly double of DAVIS_CONFIG_EXTINPUT_GENERATE_PULSE_LENGTH.
 */
#define DAVIS_CONFIG_EXTINPUT_GENERATE_PULSE_INTERVAL    10
/**
 * Parameter address for module DAVIS_CONFIG_EXTINPUT:
 * the length a pulse stays active, expressed in cycles at
 * LogicClock frequency (see 'struct caer_davis_info' for
 * details on how to get the frequency). This must be
 * smaller or equal to DAVIS_CONFIG_EXTINPUT_GENERATE_PULSE_INTERVAL.
 * To generate a signal with 50% duty cycle, this would
 * have to be exactly half of DAVIS_CONFIG_EXTINPUT_GENERATE_PULSE_INTERVAL.
 */
#define DAVIS_CONFIG_EXTINPUT_GENERATE_PULSE_LENGTH      11
/**
 * Parameter address for module DAVIS_CONFIG_EXTINPUT:
 * enables event injection when a rising edge occurs in the
 * generated signal; a special event EXTERNAL_GENERATOR_RISING_EDGE
 * is emitted into the event stream.
 */
#define DAVIS_CONFIG_EXTINPUT_GENERATE_INJECT_ON_RISING_EDGE  12
/**
 * Parameter address for module DAVIS_CONFIG_EXTINPUT:
 * enables event injection when a falling edge occurs in the
 * generated signal; a special event EXTERNAL_GENERATOR_FALLING_EDGE
 * is emitted into the event stream.
 */
#define DAVIS_CONFIG_EXTINPUT_GENERATE_INJECT_ON_FALLING_EDGE 13
/**
 * Parameter address for module DAVIS_CONFIG_EXTINPUT:
 * read-only parameter, information about the presence of the
 * extra detectors feature.
 * This is reserved for internal use and should not be used by
 * anything other than libcaer. Please see the 'struct caer_davis_info'
 * documentation to get this information.
 */
#define DAVIS_CONFIG_EXTINPUT_HAS_EXTRA_DETECTORS        14
/**
 * Parameter address for module DAVIS_CONFIG_EXTINPUT:
 * enable the signal detector module. It generates events
 * when it sees certain types of signals, such as edges or
 * pulses of a defined length, on the B1P20 input pin.
 * This can be useful to inject events into the event
 * stream in response to external stimuli or controls,
 * such as turning on a LED lamp.
 */
#define DAVIS_CONFIG_EXTINPUT_RUN_DETECTOR1              15
/**
 * Parameter address for module DAVIS_CONFIG_EXTINPUT:
 * send a special EXTERNAL_INPUT1_RISING_EDGE event when a
 * rising edge is detected (transition from low voltage to high).
 */
#define DAVIS_CONFIG_EXTINPUT_DETECT_RISING_EDGES1       16
/**
 * Parameter address for module DAVIS_CONFIG_EXTINPUT:
 * send a special EXTERNAL_INPUT1_FALLING_EDGE event when a
 * falling edge is detected (transition from high voltage to low).
 */
#define DAVIS_CONFIG_EXTINPUT_DETECT_FALLING_EDGES1      17
/**
 * Parameter address for module DAVIS_CONFIG_EXTINPUT:
 * send a special EXTERNAL_INPUT1_PULSE event when a pulse, of
 * a specified, configurable polarity and length, is detected.
 * See DAVIS_CONFIG_EXTINPUT_DETECT_PULSE_POLARITY1 and
 * DAVIS_CONFIG_EXTINPUT_DETECT_PULSE_LENGTH1 for more details.
 */
#define DAVIS_CONFIG_EXTINPUT_DETECT_PULSES1             18
/**
 * Parameter address for module DAVIS_CONFIG_EXTINPUT:
 * the polarity the pulse must exhibit to be detected as such.
 * '1' means active high; a pulse will start when the signal
 * goes from low to high and will continue to be seen as the
 * same pulse as long as it stays high.
 * '0' means active low; a pulse will start when the signal
 * goes from high to low and will continue to be seen as the
 * same pulse as long as it stays low.
 */
#define DAVIS_CONFIG_EXTINPUT_DETECT_PULSE_POLARITY1     19
/**
 * Parameter address for module DAVIS_CONFIG_EXTINPUT:
 * the minimal length that a pulse must have to trigger the
 * sending of a special event. This is measured in cycles
 * at LogicClock frequency (see 'struct caer_davis_info' for
 * details on how to get the frequency).
 */
#define DAVIS_CONFIG_EXTINPUT_DETECT_PULSE_LENGTH1       20
/**
 * Parameter address for module DAVIS_CONFIG_EXTINPUT:
 * enable the signal detector module. It generates events
 * when it sees certain types of signals, such as edges or
 * pulses of a defined length, on the B1P21 input pin.
 * This can be useful to inject events into the event
 * stream in response to external stimuli or controls,
 * such as turning on a LED lamp.
 */
#define DAVIS_CONFIG_EXTINPUT_RUN_DETECTOR2              21
/**
 * Parameter address for module DAVIS_CONFIG_EXTINPUT:
 * send a special EXTERNAL_INPUT2_RISING_EDGE event when a
 * rising edge is detected (transition from low voltage to high).
 */
#define DAVIS_CONFIG_EXTINPUT_DETECT_RISING_EDGES2       22
/**
 * Parameter address for module DAVIS_CONFIG_EXTINPUT:
 * send a special EXTERNAL_INPUT2_FALLING_EDGE event when a
 * falling edge is detected (transition from high voltage to low).
 */
#define DAVIS_CONFIG_EXTINPUT_DETECT_FALLING_EDGES2      23
/**
 * Parameter address for module DAVIS_CONFIG_EXTINPUT:
 * send a special EXTERNAL_INPUT2_PULSE event when a pulse, of
 * a specified, configurable polarity and length, is detected.
 * See DAVIS_CONFIG_EXTINPUT_DETECT_PULSE_POLARITY2 and
 * DAVIS_CONFIG_EXTINPUT_DETECT_PULSE_LENGTH2 for more details.
 */
#define DAVIS_CONFIG_EXTINPUT_DETECT_PULSES2             24
/**
 * Parameter address for module DAVIS_CONFIG_EXTINPUT:
 * the polarity the pulse must exhibit to be detected as such.
 * '1' means active high; a pulse will start when the signal
 * goes from low to high and will continue to be seen as the
 * same pulse as long as it stays high.
 * '0' means active low; a pulse will start when the signal
 * goes from high to low and will continue to be seen as the
 * same pulse as long as it stays low.
 */
#define DAVIS_CONFIG_EXTINPUT_DETECT_PULSE_POLARITY2     25
/**
 * Parameter address for module DAVIS_CONFIG_EXTINPUT:
 * the minimal length that a pulse must have to trigger the
 * sending of a special event. This is measured in cycles
 * at LogicClock frequency (see 'struct caer_davis_info' for
 * details on how to get the frequency).
 */
#define DAVIS_CONFIG_EXTINPUT_DETECT_PULSE_LENGTH2       26

/**
 * Parameter address for module DAVIS_CONFIG_SYSINFO:
 * read-only parameter, the version of the logic currently
 * running on the device's FPGA/CPLD.
 * This is reserved for internal use and should not be used by
 * anything other than libcaer. Please see the 'struct caer_davis_info'
 * documentation to get this information.
 */
#define DAVIS_CONFIG_SYSINFO_LOGIC_VERSION    0
/**
 * Parameter address for module DAVIS_CONFIG_SYSINFO:
 * read-only parameter, an integer used to identify the different
 * types of sensor chips used on the device.
 * This is reserved for internal use and should not be used by
 * anything other than libcaer. Please see the 'struct caer_davis_info'
 * documentation to get this information.
 */
#define DAVIS_CONFIG_SYSINFO_CHIP_IDENTIFIER  1
/**
 * Parameter address for module DAVIS_CONFIG_SYSINFO:
 * read-only parameter, whether the device is currently a timestamp
 * master or slave when synchronizing multiple devices together.
 * This is reserved for internal use and should not be used by
 * anything other than libcaer. Please see the 'struct caer_davis_info'
 * documentation to get this information.
 */
#define DAVIS_CONFIG_SYSINFO_DEVICE_IS_MASTER 2
/**
 * Parameter address for module DAVIS_CONFIG_SYSINFO:
 * read-only parameter, the frequency in MHz at which the main
 * FPGA/CPLD logic is running.
 * This is reserved for internal use and should not be used by
 * anything other than libcaer. Please see the 'struct caer_davis_info'
 * documentation to get this information.
 */
#define DAVIS_CONFIG_SYSINFO_LOGIC_CLOCK      3
/**
 * Parameter address for module DAVIS_CONFIG_SYSINFO:
 * read-only parameter, the frequency in MHz at which the FPGA/CPLD
 * logic related to APS frame grabbing is running.
 * This is reserved for internal use and should not be used by
 * anything other than libcaer. Please see the 'struct caer_davis_info'
 * documentation to get this information.
 */
#define DAVIS_CONFIG_SYSINFO_ADC_CLOCK        4

/**
 * Parameter address for module DAVIS_CONFIG_MICROPHONE:
 * enable the Microphone module, which provides stereo samples
 * of sound recorded by on-board InvenSense microphones.
 */
#define DAVIS_CONFIG_MICROPHONE_RUN              0
/**
 * Parameter address for module DAVIS_CONFIG_MICROPHONE:
 * allows setting the sample frequency of the stereo microphones,
 * by specifying the length of an SCK clock cycle in LogicClock cycles.
 * Value can be between 30 and 219 inclusive.
 * The desired value can be calculated in the following way:
 * floor((100.8 * 1000)/(64 * <desired frequency in KHz>))
 * For example for 48 KHz sampling frequency, this would be 32.
 * For 44.1 KHz it would be 35, and for 16 KHz it would be 98.
 */
#define DAVIS_CONFIG_MICROPHONE_SAMPLE_FREQUENCY 1

/**
 * Parameter address for module DAVIS_CONFIG_USB:
 * enable the USB FIFO module, which transfers the data from the
 * FPGA/CPLD to the USB chip, to be then sent to the host.
 * Turning this off will suppress any USB data communication!
 */
#define DAVIS_CONFIG_USB_RUN                0
/**
 * Parameter address for module DAVIS_CONFIG_USB:
 * the time delay after which a packet of data is committed to
 * USB, even if it is not full yet (short USB packet).
 * The value is in 125µs time-slices, corresponding to how
 * USB schedules its operations (a value of 4 for example
 * would mean waiting at most 0.5ms until sending a short
 * USB packet to the host).
 */
#define DAVIS_CONFIG_USB_EARLY_PACKET_DELAY 1

/**
 * Parameter address for module DAVIS_CONFIG_DDRAER:
 * enable the DDR-AER output module, which transfers the data from the
 * FPGA/CPLD to some external device like a Raspberry Pi.
 */
#define DAVIS_CONFIG_DDRAER_RUN       0
/**
 * Parameter address for module DAVIS_CONFIG_DDRAER:
 * delay the request by this many cycles after having output the data.
 */
#define DAVIS_CONFIG_DDRAER_REQ_DELAY 1
/**
 * Parameter address for module DAVIS_CONFIG_DDRAER:
 * wait this many cycles after having received the acknowledge.
 */
#define DAVIS_CONFIG_DDRAER_ACK_DELAY 2

//@{
/**
 * Parameter address for module DAVIS128_CONFIG_BIAS:
 * DAVIS128 chip biases.
 * Bias configuration values must be generated using the proper
 * functions, which are:
 * - caerBiasVDACGenerate() for VDAC (voltage) biases.
 * - caerBiasCoarseFineGenerate() for coarse-fine (current) biases.
 * - caerBiasShiftedSourceGenerate() for shifted-source biases.
 * See 'http://inilabs.com/support/biasing/' for more details.
 */
#define DAVIS128_CONFIG_BIAS_APSOVERFLOWLEVEL 0
#define DAVIS128_CONFIG_BIAS_APSCAS           1
#define DAVIS128_CONFIG_BIAS_ADCREFHIGH       2
#define DAVIS128_CONFIG_BIAS_ADCREFLOW        3
#define DAVIS128_CONFIG_BIAS_LOCALBUFBN       8
#define DAVIS128_CONFIG_BIAS_PADFOLLBN        9
#define DAVIS128_CONFIG_BIAS_DIFFBN           10
#define DAVIS128_CONFIG_BIAS_ONBN             11
#define DAVIS128_CONFIG_BIAS_OFFBN            12
#define DAVIS128_CONFIG_BIAS_PIXINVBN         13
#define DAVIS128_CONFIG_BIAS_PRBP             14
#define DAVIS128_CONFIG_BIAS_PRSFBP           15
#define DAVIS128_CONFIG_BIAS_REFRBP           16
#define DAVIS128_CONFIG_BIAS_READOUTBUFBP     17
#define DAVIS128_CONFIG_BIAS_APSROSFBN        18
#define DAVIS128_CONFIG_BIAS_ADCCOMPBP        19
#define DAVIS128_CONFIG_BIAS_COLSELLOWBN      20
#define DAVIS128_CONFIG_BIAS_DACBUFBP         21
#define DAVIS128_CONFIG_BIAS_LCOLTIMEOUTBN    22
#define DAVIS128_CONFIG_BIAS_AEPDBN           23
#define DAVIS128_CONFIG_BIAS_AEPUXBP          24
#define DAVIS128_CONFIG_BIAS_AEPUYBP          25
#define DAVIS128_CONFIG_BIAS_IFREFRBN         26
#define DAVIS128_CONFIG_BIAS_IFTHRBN          27
#define DAVIS128_CONFIG_BIAS_BIASBUFFER       34
#define DAVIS128_CONFIG_BIAS_SSP              35
#define DAVIS128_CONFIG_BIAS_SSN              36
//@}

//@{
/**
 * Parameter address for module DAVIS128_CONFIG_CHIP:
 * DAVIS128 chip configuration.
 * These are for expert control and should never be used
 * or changed unless for advanced debugging purposes.
 * To change the Global Shutter configuration, please use
 * DAVIS_CONFIG_APS_GLOBAL_SHUTTER instead.
 */
#define DAVIS128_CONFIG_CHIP_DIGITALMUX0       128
#define DAVIS128_CONFIG_CHIP_DIGITALMUX1       129
#define DAVIS128_CONFIG_CHIP_DIGITALMUX2       130
#define DAVIS128_CONFIG_CHIP_DIGITALMUX3       131
#define DAVIS128_CONFIG_CHIP_ANALOGMUX0        132
#define DAVIS128_CONFIG_CHIP_ANALOGMUX1        133
#define DAVIS128_CONFIG_CHIP_ANALOGMUX2        134
#define DAVIS128_CONFIG_CHIP_BIASMUX0          135
#define DAVIS128_CONFIG_CHIP_RESETCALIBNEURON  136
#define DAVIS128_CONFIG_CHIP_TYPENCALIBNEURON  137
#define DAVIS128_CONFIG_CHIP_RESETTESTPIXEL    138
#define DAVIS128_CONFIG_CHIP_AERNAROW          140
#define DAVIS128_CONFIG_CHIP_USEAOUT           141
#define DAVIS128_CONFIG_CHIP_GLOBAL_SHUTTER    142
#define DAVIS128_CONFIG_CHIP_SELECTGRAYCOUNTER 143
//@}

//@{
/**
 * Parameter address for module DAVIS208_CONFIG_BIAS:
 * DAVIS208 chip biases.
 * Bias configuration values must be generated using the proper
 * functions, which are:
 * - caerBiasVDACGenerate() for VDAC (voltage) biases.
 * - caerBiasCoarseFineGenerate() for coarse-fine (current) biases.
 * - caerBiasShiftedSourceGenerate() for shifted-source biases.
 * See 'http://inilabs.com/support/biasing/' for more details.
 */
#define DAVIS208_CONFIG_BIAS_APSOVERFLOWLEVEL 0
#define DAVIS208_CONFIG_BIAS_APSCAS           1
#define DAVIS208_CONFIG_BIAS_ADCREFHIGH       2
#define DAVIS208_CONFIG_BIAS_ADCREFLOW        3
#define DAVIS208_CONFIG_BIAS_RESETHIGHPASS    6
#define DAVIS208_CONFIG_BIAS_REFSS            7
#define DAVIS208_CONFIG_BIAS_LOCALBUFBN       8
#define DAVIS208_CONFIG_BIAS_PADFOLLBN        9
#define DAVIS208_CONFIG_BIAS_DIFFBN           10
#define DAVIS208_CONFIG_BIAS_ONBN             11
#define DAVIS208_CONFIG_BIAS_OFFBN            12
#define DAVIS208_CONFIG_BIAS_PIXINVBN         13
#define DAVIS208_CONFIG_BIAS_PRBP             14
#define DAVIS208_CONFIG_BIAS_PRSFBP           15
#define DAVIS208_CONFIG_BIAS_REFRBP           16
#define DAVIS208_CONFIG_BIAS_READOUTBUFBP     17
#define DAVIS208_CONFIG_BIAS_APSROSFBN        18
#define DAVIS208_CONFIG_BIAS_ADCCOMPBP        19
#define DAVIS208_CONFIG_BIAS_COLSELLOWBN      20
#define DAVIS208_CONFIG_BIAS_DACBUFBP         21
#define DAVIS208_CONFIG_BIAS_LCOLTIMEOUTBN    22
#define DAVIS208_CONFIG_BIAS_AEPDBN           23
#define DAVIS208_CONFIG_BIAS_AEPUXBP          24
#define DAVIS208_CONFIG_BIAS_AEPUYBP          25
#define DAVIS208_CONFIG_BIAS_IFREFRBN         26
#define DAVIS208_CONFIG_BIAS_IFTHRBN          27
#define DAVIS208_CONFIG_BIAS_REGBIASBP        28
#define DAVIS208_CONFIG_BIAS_REFSSBN          30
#define DAVIS208_CONFIG_BIAS_BIASBUFFER       34
#define DAVIS208_CONFIG_BIAS_SSP              35
#define DAVIS208_CONFIG_BIAS_SSN              36
//@}

//@{
/**
 * Parameter address for module DAVIS208_CONFIG_CHIP:
 * DAVIS208 chip configuration.
 * These are for expert control and should never be used
 * or changed unless for advanced debugging purposes.
 * To change the Global Shutter configuration, please use
 * DAVIS_CONFIG_APS_GLOBAL_SHUTTER instead.
 */
#define DAVIS208_CONFIG_CHIP_DIGITALMUX0       128
#define DAVIS208_CONFIG_CHIP_DIGITALMUX1       129
#define DAVIS208_CONFIG_CHIP_DIGITALMUX2       130
#define DAVIS208_CONFIG_CHIP_DIGITALMUX3       131
#define DAVIS208_CONFIG_CHIP_ANALOGMUX0        132
#define DAVIS208_CONFIG_CHIP_ANALOGMUX1        133
#define DAVIS208_CONFIG_CHIP_ANALOGMUX2        134
#define DAVIS208_CONFIG_CHIP_BIASMUX0          135
#define DAVIS208_CONFIG_CHIP_RESETCALIBNEURON  136
#define DAVIS208_CONFIG_CHIP_TYPENCALIBNEURON  137
#define DAVIS208_CONFIG_CHIP_RESETTESTPIXEL    138
#define DAVIS208_CONFIG_CHIP_AERNAROW          140
#define DAVIS208_CONFIG_CHIP_USEAOUT           141
#define DAVIS208_CONFIG_CHIP_GLOBAL_SHUTTER    142
#define DAVIS208_CONFIG_CHIP_SELECTGRAYCOUNTER 143
#define DAVIS208_CONFIG_CHIP_SELECTPREAMPAVG   145
#define DAVIS208_CONFIG_CHIP_SELECTBIASREFSS   146
#define DAVIS208_CONFIG_CHIP_SELECTSENSE       147
#define DAVIS208_CONFIG_CHIP_SELECTPOSFB       148
#define DAVIS208_CONFIG_CHIP_SELECTHIGHPASS    149
//@}

//@{
/**
 * Parameter address for module DAVIS240_CONFIG_BIAS:
 * DAVIS240chip biases.
 * Bias configuration values must be generated using the proper
 * functions, which are:
 * - caerBiasCoarseFineGenerate() for coarse-fine (current) biases.
 * - caerBiasShiftedSourceGenerate() for shifted-source biases.
 * See 'http://inilabs.com/support/biasing/' for more details.
 */
#define DAVIS240_CONFIG_BIAS_DIFFBN             0
#define DAVIS240_CONFIG_BIAS_ONBN               1
#define DAVIS240_CONFIG_BIAS_OFFBN              2
#define DAVIS240_CONFIG_BIAS_APSCASEPC          3
#define DAVIS240_CONFIG_BIAS_DIFFCASBNC         4
#define DAVIS240_CONFIG_BIAS_APSROSFBN          5
#define DAVIS240_CONFIG_BIAS_LOCALBUFBN         6
#define DAVIS240_CONFIG_BIAS_PIXINVBN           7
#define DAVIS240_CONFIG_BIAS_PRBP               8
#define DAVIS240_CONFIG_BIAS_PRSFBP             9
#define DAVIS240_CONFIG_BIAS_REFRBP             10
#define DAVIS240_CONFIG_BIAS_AEPDBN             11
#define DAVIS240_CONFIG_BIAS_LCOLTIMEOUTBN      12
#define DAVIS240_CONFIG_BIAS_AEPUXBP            13
#define DAVIS240_CONFIG_BIAS_AEPUYBP            14
#define DAVIS240_CONFIG_BIAS_IFTHRBN            15
#define DAVIS240_CONFIG_BIAS_IFREFRBN           16
#define DAVIS240_CONFIG_BIAS_PADFOLLBN          17
#define DAVIS240_CONFIG_BIAS_APSOVERFLOWLEVELBN 18
#define DAVIS240_CONFIG_BIAS_BIASBUFFER         19
#define DAVIS240_CONFIG_BIAS_SSP                20
#define DAVIS240_CONFIG_BIAS_SSN                21
//@}

//@{
/**
 * Parameter address for module DAVIS240_CONFIG_CHIP:
 * DAVIS240 chip configuration.
 * These are for expert control and should never be used
 * or changed unless for advanced debugging purposes.
 * To change the Global Shutter configuration, please use
 * DAVIS_CONFIG_APS_GLOBAL_SHUTTER instead.
 * On DAVIS240B cameras, DAVIS240_CONFIG_CHIP_SPECIALPIXELCONTROL
 * can be used to enable the test pixel array.
 */
#define DAVIS240_CONFIG_CHIP_DIGITALMUX0         128
#define DAVIS240_CONFIG_CHIP_DIGITALMUX1         129
#define DAVIS240_CONFIG_CHIP_DIGITALMUX2         130
#define DAVIS240_CONFIG_CHIP_DIGITALMUX3         131
#define DAVIS240_CONFIG_CHIP_ANALOGMUX0          132
#define DAVIS240_CONFIG_CHIP_ANALOGMUX1          133
#define DAVIS240_CONFIG_CHIP_ANALOGMUX2          134
#define DAVIS240_CONFIG_CHIP_BIASMUX0            135
#define DAVIS240_CONFIG_CHIP_RESETCALIBNEURON    136
#define DAVIS240_CONFIG_CHIP_TYPENCALIBNEURON    137
#define DAVIS240_CONFIG_CHIP_RESETTESTPIXEL      138
#define DAVIS240_CONFIG_CHIP_SPECIALPIXELCONTROL 139
#define DAVIS240_CONFIG_CHIP_AERNAROW            140
#define DAVIS240_CONFIG_CHIP_USEAOUT             141
#define DAVIS240_CONFIG_CHIP_GLOBAL_SHUTTER      142
//@}

//@{
/**
 * Parameter address for module DAVIS346_CONFIG_BIAS:
 * DAVIS346 chip biases.
 * Bias configuration values must be generated using the proper
 * functions, which are:
 * - caerBiasVDACGenerate() for VDAC (voltage) biases.
 * - caerBiasCoarseFineGenerate() for coarse-fine (current) biases.
 * - caerBiasShiftedSourceGenerate() for shifted-source biases.
 * See 'http://inilabs.com/support/biasing/' for more details.
 */
#define DAVIS346_CONFIG_BIAS_APSOVERFLOWLEVEL 0
#define DAVIS346_CONFIG_BIAS_APSCAS           1
#define DAVIS346_CONFIG_BIAS_ADCREFHIGH       2
#define DAVIS346_CONFIG_BIAS_ADCREFLOW        3
#define DAVIS346_CONFIG_BIAS_ADCTESTVOLTAGE   4
#define DAVIS346_CONFIG_BIAS_LOCALBUFBN       8
#define DAVIS346_CONFIG_BIAS_PADFOLLBN        9
#define DAVIS346_CONFIG_BIAS_DIFFBN           10
#define DAVIS346_CONFIG_BIAS_ONBN             11
#define DAVIS346_CONFIG_BIAS_OFFBN            12
#define DAVIS346_CONFIG_BIAS_PIXINVBN         13
#define DAVIS346_CONFIG_BIAS_PRBP             14
#define DAVIS346_CONFIG_BIAS_PRSFBP           15
#define DAVIS346_CONFIG_BIAS_REFRBP           16
#define DAVIS346_CONFIG_BIAS_READOUTBUFBP     17
#define DAVIS346_CONFIG_BIAS_APSROSFBN        18
#define DAVIS346_CONFIG_BIAS_ADCCOMPBP        19
#define DAVIS346_CONFIG_BIAS_COLSELLOWBN      20
#define DAVIS346_CONFIG_BIAS_DACBUFBP         21
#define DAVIS346_CONFIG_BIAS_LCOLTIMEOUTBN    22
#define DAVIS346_CONFIG_BIAS_AEPDBN           23
#define DAVIS346_CONFIG_BIAS_AEPUXBP          24
#define DAVIS346_CONFIG_BIAS_AEPUYBP          25
#define DAVIS346_CONFIG_BIAS_IFREFRBN         26
#define DAVIS346_CONFIG_BIAS_IFTHRBN          27
#define DAVIS346_CONFIG_BIAS_BIASBUFFER       34
#define DAVIS346_CONFIG_BIAS_SSP              35
#define DAVIS346_CONFIG_BIAS_SSN              36
//@}

//@{
/**
 * Parameter address for module DAVIS346_CONFIG_CHIP:
 * DAVIS346 chip configuration.
 * These are for expert control and should never be used
 * or changed unless for advanced debugging purposes.
 * To change the Global Shutter configuration, please use
 * DAVIS_CONFIG_APS_GLOBAL_SHUTTER instead.
 */
#define DAVIS346_CONFIG_CHIP_DIGITALMUX0       128
#define DAVIS346_CONFIG_CHIP_DIGITALMUX1       129
#define DAVIS346_CONFIG_CHIP_DIGITALMUX2       130
#define DAVIS346_CONFIG_CHIP_DIGITALMUX3       131
#define DAVIS346_CONFIG_CHIP_ANALOGMUX0        132
#define DAVIS346_CONFIG_CHIP_ANALOGMUX1        133
#define DAVIS346_CONFIG_CHIP_ANALOGMUX2        134
#define DAVIS346_CONFIG_CHIP_BIASMUX0          135
#define DAVIS346_CONFIG_CHIP_RESETCALIBNEURON  136
#define DAVIS346_CONFIG_CHIP_TYPENCALIBNEURON  137
#define DAVIS346_CONFIG_CHIP_RESETTESTPIXEL    138
#define DAVIS346_CONFIG_CHIP_AERNAROW          140
#define DAVIS346_CONFIG_CHIP_USEAOUT           141
#define DAVIS346_CONFIG_CHIP_GLOBAL_SHUTTER    142
#define DAVIS346_CONFIG_CHIP_SELECTGRAYCOUNTER 143
#define DAVIS346_CONFIG_CHIP_TESTADC           144
//@}

//@{
/**
 * Parameter address for module DAVIS640_CONFIG_BIAS:
 * DAVIS640 chip biases.
 * Bias configuration values must be generated using the proper
 * functions, which are:
 * - caerBiasVDACGenerate() for VDAC (voltage) biases.
 * - caerBiasCoarseFineGenerate() for coarse-fine (current) biases.
 * - caerBiasShiftedSourceGenerate() for shifted-source biases.
 * See 'http://inilabs.com/support/biasing/' for more details.
 */
#define DAVIS640_CONFIG_BIAS_APSOVERFLOWLEVEL 0
#define DAVIS640_CONFIG_BIAS_APSCAS           1
#define DAVIS640_CONFIG_BIAS_ADCREFHIGH       2
#define DAVIS640_CONFIG_BIAS_ADCREFLOW        3
#define DAVIS640_CONFIG_BIAS_ADCTESTVOLTAGE   4
#define DAVIS640_CONFIG_BIAS_LOCALBUFBN       8
#define DAVIS640_CONFIG_BIAS_PADFOLLBN        9
#define DAVIS640_CONFIG_BIAS_DIFFBN           10
#define DAVIS640_CONFIG_BIAS_ONBN             11
#define DAVIS640_CONFIG_BIAS_OFFBN            12
#define DAVIS640_CONFIG_BIAS_PIXINVBN         13
#define DAVIS640_CONFIG_BIAS_PRBP             14
#define DAVIS640_CONFIG_BIAS_PRSFBP           15
#define DAVIS640_CONFIG_BIAS_REFRBP           16
#define DAVIS640_CONFIG_BIAS_READOUTBUFBP     17
#define DAVIS640_CONFIG_BIAS_APSROSFBN        18
#define DAVIS640_CONFIG_BIAS_ADCCOMPBP        19
#define DAVIS640_CONFIG_BIAS_COLSELLOWBN      20
#define DAVIS640_CONFIG_BIAS_DACBUFBP         21
#define DAVIS640_CONFIG_BIAS_LCOLTIMEOUTBN    22
#define DAVIS640_CONFIG_BIAS_AEPDBN           23
#define DAVIS640_CONFIG_BIAS_AEPUXBP          24
#define DAVIS640_CONFIG_BIAS_AEPUYBP          25
#define DAVIS640_CONFIG_BIAS_IFREFRBN         26
#define DAVIS640_CONFIG_BIAS_IFTHRBN          27
#define DAVIS640_CONFIG_BIAS_BIASBUFFER       34
#define DAVIS640_CONFIG_BIAS_SSP              35
#define DAVIS640_CONFIG_BIAS_SSN              36
//@}

//@{
/**
 * Parameter address for module DAVIS640_CONFIG_CHIP:
 * DAVIS640 chip configuration.
 * These are for expert control and should never be used
 * or changed unless for advanced debugging purposes.
 * To change the Global Shutter configuration, please use
 * DAVIS_CONFIG_APS_GLOBAL_SHUTTER instead.
 */
#define DAVIS640_CONFIG_CHIP_DIGITALMUX0       128
#define DAVIS640_CONFIG_CHIP_DIGITALMUX1       129
#define DAVIS640_CONFIG_CHIP_DIGITALMUX2       130
#define DAVIS640_CONFIG_CHIP_DIGITALMUX3       131
#define DAVIS640_CONFIG_CHIP_ANALOGMUX0        132
#define DAVIS640_CONFIG_CHIP_ANALOGMUX1        133
#define DAVIS640_CONFIG_CHIP_ANALOGMUX2        134
#define DAVIS640_CONFIG_CHIP_BIASMUX0          135
#define DAVIS640_CONFIG_CHIP_RESETCALIBNEURON  136
#define DAVIS640_CONFIG_CHIP_TYPENCALIBNEURON  137
#define DAVIS640_CONFIG_CHIP_RESETTESTPIXEL    138
#define DAVIS640_CONFIG_CHIP_AERNAROW          140
#define DAVIS640_CONFIG_CHIP_USEAOUT           141
#define DAVIS640_CONFIG_CHIP_GLOBAL_SHUTTER    142
#define DAVIS640_CONFIG_CHIP_SELECTGRAYCOUNTER 143
#define DAVIS640_CONFIG_CHIP_TESTADC           144
//@}

//@{
/**
 * Parameter address for module DAVISRGB_CONFIG_BIAS:
 * DAVISRGB chip biases.
 * Bias configuration values must be generated using the proper
 * functions, which are:
 * - caerBiasVDACGenerate() for VDAC (voltage) biases.
 * - caerBiasCoarseFineGenerate() for coarse-fine (current) biases.
 * - caerBiasShiftedSourceGenerate() for shifted-source biases.
 * See 'http://inilabs.com/support/biasing/' for more details.
 */
#define DAVISRGB_CONFIG_BIAS_APSCAS             0
#define DAVISRGB_CONFIG_BIAS_OVG1LO             1
#define DAVISRGB_CONFIG_BIAS_OVG2LO             2
#define DAVISRGB_CONFIG_BIAS_TX2OVG2HI          3
#define DAVISRGB_CONFIG_BIAS_GND07              4
#define DAVISRGB_CONFIG_BIAS_ADCTESTVOLTAGE     5
#define DAVISRGB_CONFIG_BIAS_ADCREFHIGH         6
#define DAVISRGB_CONFIG_BIAS_ADCREFLOW          7
#define DAVISRGB_CONFIG_BIAS_IFREFRBN           8
#define DAVISRGB_CONFIG_BIAS_IFTHRBN            9
#define DAVISRGB_CONFIG_BIAS_LOCALBUFBN         10
#define DAVISRGB_CONFIG_BIAS_PADFOLLBN          11
#define DAVISRGB_CONFIG_BIAS_PIXINVBN           13
#define DAVISRGB_CONFIG_BIAS_DIFFBN             14
#define DAVISRGB_CONFIG_BIAS_ONBN               15
#define DAVISRGB_CONFIG_BIAS_OFFBN              16
#define DAVISRGB_CONFIG_BIAS_PRBP               17
#define DAVISRGB_CONFIG_BIAS_PRSFBP             18
#define DAVISRGB_CONFIG_BIAS_REFRBP             19
#define DAVISRGB_CONFIG_BIAS_ARRAYBIASBUFFERBN  20
#define DAVISRGB_CONFIG_BIAS_ARRAYLOGICBUFFERBN 22
#define DAVISRGB_CONFIG_BIAS_FALLTIMEBN         23
#define DAVISRGB_CONFIG_BIAS_RISETIMEBP         24
#define DAVISRGB_CONFIG_BIAS_READOUTBUFBP       25
#define DAVISRGB_CONFIG_BIAS_APSROSFBN          26
#define DAVISRGB_CONFIG_BIAS_ADCCOMPBP          27
#define DAVISRGB_CONFIG_BIAS_DACBUFBP           28
#define DAVISRGB_CONFIG_BIAS_LCOLTIMEOUTBN      30
#define DAVISRGB_CONFIG_BIAS_AEPDBN             31
#define DAVISRGB_CONFIG_BIAS_AEPUXBP            32
#define DAVISRGB_CONFIG_BIAS_AEPUYBP            33
#define DAVISRGB_CONFIG_BIAS_BIASBUFFER         34
#define DAVISRGB_CONFIG_BIAS_SSP                35
#define DAVISRGB_CONFIG_BIAS_SSN                36
//@}

//@{
/**
 * Parameter address for module DAVISRGB_CONFIG_CHIP:
 * DAVISRGB chip configuration.
 * These are for expert control and should never be used
 * or changed unless for advanced debugging purposes.
 * To change the Global Shutter configuration, please use
 * DAVIS_CONFIG_APS_GLOBAL_SHUTTER instead.
 */
#define DAVISRGB_CONFIG_CHIP_DIGITALMUX0       128
#define DAVISRGB_CONFIG_CHIP_DIGITALMUX1       129
#define DAVISRGB_CONFIG_CHIP_DIGITALMUX2       130
#define DAVISRGB_CONFIG_CHIP_DIGITALMUX3       131
#define DAVISRGB_CONFIG_CHIP_ANALOGMUX0        132
#define DAVISRGB_CONFIG_CHIP_ANALOGMUX1        133
#define DAVISRGB_CONFIG_CHIP_ANALOGMUX2        134
#define DAVISRGB_CONFIG_CHIP_BIASMUX0          135
#define DAVISRGB_CONFIG_CHIP_RESETCALIBNEURON  136
#define DAVISRGB_CONFIG_CHIP_TYPENCALIBNEURON  137
#define DAVISRGB_CONFIG_CHIP_RESETTESTPIXEL    138
#define DAVISRGB_CONFIG_CHIP_AERNAROW          140
#define DAVISRGB_CONFIG_CHIP_USEAOUT           141
#define DAVISRGB_CONFIG_CHIP_SELECTGRAYCOUNTER 143
#define DAVISRGB_CONFIG_CHIP_TESTADC           144
#define DAVISRGB_CONFIG_CHIP_ADJUSTOVG1LO      145
#define DAVISRGB_CONFIG_CHIP_ADJUSTOVG2LO      146
#define DAVISRGB_CONFIG_CHIP_ADJUSTTX2OVG2HI   147
//@}

/**
 * DAVIS: maximum number of APS Regions-of-Interest that
 * can be produced by any DAVIS camera. See 'struct caer_davis_info'
 * for actual number: 4 if apsHasQuadROI=true, else 1.
 */
#define DAVIS_APS_ROI_REGIONS_MAX 4

/**
 * DAVIS device-related information.
 */
struct caer_davis_info {
	/// Unique device identifier. Also 'source' for events.
	int16_t deviceID;
	/// Device serial number.
	char deviceSerialNumber[8 + 1];
	/// Device USB bus number.
	uint8_t deviceUSBBusNumber;
	/// Device USB device address.
	uint8_t deviceUSBDeviceAddress;
	/// Device information string, for logging purposes.
	char *deviceString;
	/// Logic (FPGA/CPLD) version.
	int16_t logicVersion;
	/// Whether the device is a time-stamp master or slave.
	bool deviceIsMaster;
	/// Clock in MHz for main logic (FPGA/CPLD).
	int16_t logicClock;
	/// Clock in MHz for ADC/APS logic (FPGA/CPLD).
	int16_t adcClock;
	/// Chip identifier/type.
	int16_t chipID;
	/// DVS X axis resolution.
	int16_t dvsSizeX;
	/// DVS Y axis resolution.
	int16_t dvsSizeY;
	/// Feature test: DVS pixel-level filtering.
	bool dvsHasPixelFilter;
	/// Feature test: DVS Background Activity filter (and Refractory Period filter).
	bool dvsHasBackgroundActivityFilter;
	/// Feature test: fake event generator (testing/debug).
	bool dvsHasTestEventGenerator;
	/// APS X axis resolution.
	int16_t apsSizeX;
	/// APS Y axis resolution.
	int16_t apsSizeY;
	/// APS color filter type.
	enum caer_frame_event_color_filter apsColorFilter;
	/// Feature test: APS supports Global Shutter.
	bool apsHasGlobalShutter;
	/// Feature test: APS supports Quadruple Region-of-Interest readout.
	bool apsHasQuadROI;
	/// Feature test: APS supports External ADC for getting the image.
	bool apsHasExternalADC;
	/// Feature test: APS supports Internal (on-chip) ADC for getting the image.
	bool apsHasInternalADC;
	/// Feature test: External Input module supports Signal-Generation.
	bool extInputHasGenerator;
	/// Feature test: External Input module supports extra detectors (1 & 2).
	bool extInputHasExtraDetectors;
	/// Feature test: DVS ROI filter.
	bool dvsHasROIFilter;
	/// Feature test: DVS statistics support.
	bool dvsHasStatistics;
	/// Feature test: Multiplexer statistics support (event drops).
	bool muxHasStatistics;
};

/**
 * Return basic information on the device, such as its ID, its
 * resolution, the logic version, and so on. See the 'struct
 * caer_davis_info' documentation for more details.
 *
 * @param handle a valid device handle.
 *
 * @return a copy of the device information structure if successful,
 *         an empty structure (all zeros) on failure.
 */
struct caer_davis_info caerDavisInfoGet(caerDeviceHandle handle);

/**
 * On-chip voltage digital-to-analog converter configuration.
 * See 'http://inilabs.com/support/biasing/' for more details.
 */
struct caer_bias_vdac {
	/// Voltage, between 0 and 63, as a fraction of 1/64th of VDD=3.3V.
	uint8_t voltageValue;
	/// Current, between 0 and 7, that drives the voltage.
	uint8_t currentValue;
};

/**
 * Transform VDAC bias structure into internal integer representation,
 * suited for sending directly to the device via caerDeviceConfigSet().
 *
 * @param vdacBias VDAC bias structure.
 *
 * @return internal integer representation for device configuration.
 */
uint16_t caerBiasVDACGenerate(const struct caer_bias_vdac vdacBias);
/**
 * Transform internal integer representation, as received by calls to
 * caerDeviceConfigGet(), into a VDAC bias structure, for easier
 * handling and understanding of the various parameters.
 *
 * @param vdacBias internal integer representation from device.
 *
 * @return VDAC bias structure.
 */
struct caer_bias_vdac caerBiasVDACParse(const uint16_t vdacBias);

/**
 * On-chip coarse-fine bias current configuration.
 * See 'http://inilabs.com/support/biasing/' for more details.
 */
struct caer_bias_coarsefine {
	/// Coarse current, from 0 to 7, creates big variations in output current.
	uint8_t coarseValue;
	/// Fine current, from 0 to 255, creates small variations in output current.
	uint8_t fineValue;
	/// Whether this bias is enabled or not.
	bool enabled;
	/// Bias sex: true for 'N' type, false for 'P' type.
	bool sexN;
	/// Bias type: true for 'Normal', false for 'Cascode'.
	bool typeNormal;
	/// Bias current level: true for 'Normal, false for 'Low'.
	bool currentLevelNormal;
};

/**
 * Transform coarse-fine bias structure into internal integer representation,
 * suited for sending directly to the device via caerDeviceConfigSet().
 *
 * @param coarseFineBias coarse-fine bias structure.
 *
 * @return internal integer representation for device configuration.
 */
uint16_t caerBiasCoarseFineGenerate(const struct caer_bias_coarsefine coarseFineBias);
/**
 * Transform internal integer representation, as received by calls to
 * caerDeviceConfigGet(), into a coarse-fine bias structure, for easier
 * handling and understanding of the various parameters.
 *
 * @param coarseFineBias internal integer representation from device.
 *
 * @return coarse-fine bias structure.
 */
struct caer_bias_coarsefine caerBiasCoarseFineParse(const uint16_t coarseFineBias);

/**
 * Shifted-source bias operating mode.
 */
enum caer_bias_shiftedsource_operating_mode {
	/// Standard mode.
	SHIFTED_SOURCE = 0,
	/// High impedance (driven from outside).
	HI_Z = 1,
	/// Tied to ground (SSN) or VDD (SSP).
	TIED_TO_RAIL = 2,
};

/**
 * Shifted-source bias voltage level.
 */
enum caer_bias_shiftedsource_voltage_level {
	/// Standard mode (200-400mV).
	SPLIT_GATE = 0,
	/// Higher shifted-source voltage (one cascode).
	SINGLE_DIODE = 1,
	/// Even higher shifted-source voltage (two cascodes).
	DOUBLE_DIODE = 2,
};

/**
 * On-chip shifted-source bias current configuration.
 * See 'http://inilabs.com/support/biasing/' for more details.
 */
struct caer_bias_shiftedsource {
	/// Shifted-source bias level, from 0 to 63.
	uint8_t refValue;
	/// Shifted-source bias current for buffer amplifier, from 0 to 63.
	uint8_t regValue;
	/// Shifted-source operating mode (see 'enum caer_bias_shiftedsource_operating_mode').
	enum caer_bias_shiftedsource_operating_mode operatingMode;
	/// Shifted-source voltage level (see 'enum caer_bias_shiftedsource_voltage_level').
	enum caer_bias_shiftedsource_voltage_level voltageLevel;
};

/**
 * Transform shifted-source bias structure into internal integer representation,
 * suited for sending directly to the device via caerDeviceConfigSet().
 *
 * @param shiftedSourceBias shifted-source bias structure.
 *
 * @return internal integer representation for device configuration.
 */
uint16_t caerBiasShiftedSourceGenerate(const struct caer_bias_shiftedsource shiftedSourceBias);
/**
 * Transform internal integer representation, as received by calls to
 * caerDeviceConfigGet(), into a shifted-source bias structure, for easier
 * handling and understanding of the various parameters.
 *
 * @param shiftedSourceBias internal integer representation from device.
 *
 * @return shifted-source bias structure.
 */
struct caer_bias_shiftedsource caerBiasShiftedSourceParse(const uint16_t shiftedSourceBias);

/**
 * Configure an APS ROI region in one step. This function guarantees efficiency and
 * atomicity (no intermediate-sized results possible).
 *
 * @param handle a valid device handle.
 * @param roiRegion which ROI region to configure, 0 or [0,3] if 'apsHasQuadROI' is defined.
 * @param enable whether to enable or disable this ROI region.
 * @param startX start corner X coordinate (0, 0 is upper left of frame).
 * @param startY start corner Y coordinate (0, 0 is upper left of frame).
 * @param endX end corner X coordinate (0, 0 is upper left of frame). Must be bigger than start.
 * @param endY end corner Y coordinate (0, 0 is upper left of frame). Must be bigger than start.
 *
 * @return true on success, false otherwise.
 */
bool caerDavisROIConfigure(caerDeviceHandle handle, uint8_t roiRegion, bool enable, uint16_t startX, uint16_t startY,
	uint16_t endX, uint16_t endY);

#ifdef __cplusplus
}
#endif

#endif /* LIBCAER_DEVICES_DAVIS_H_ */
