//===-- X866432ThunkReferencePass.cpp - Lower func refs to thunks ---------===//
//
//                     The LLVM Compiler Infrastructure
//
// This file is distributed under the University of Illinois Open Source
// License. See LICENSE.TXT for details.
//
//===----------------------------------------------------------------------===//
//
// This file defines a function pass that replaces all references to functions
// with 32-bit calling conventions other than direct calls with references to
// the thunks that will be generated by the X866432InteropThunkInserter pass.
//
//===----------------------------------------------------------------------===//

#include "X86.h"
#include "llvm/Pass.h"
#include "llvm/PassAnalysisSupport.h"
#include "llvm/ADT/DenseMap.h"
#include "llvm/IR/CallingConv.h"
#include "llvm/IR/Constants.h"
#include "llvm/IR/Function.h"
#include "llvm/IR/Instructions.h"
#include "llvm/IR/Module.h"

using namespace llvm;

namespace {

// Unfortunately, this has to be a module pass, since it can insert new IR
// functions into the module.
class X866432ThunkReferencePass : public ModulePass {
public:
  X866432ThunkReferencePass() : ModulePass(ID) {}

  bool runOnModule(Module &M) override;

  void getAnalysisUsage(AnalysisUsage &AU) const override;

private:
  Value *getReplacementValue(Value *Old, Function &Fn, Function &ThunkFn);
  bool replaceNonCallUsesWith(Function &Fn, Value *Old, Function &ThunkFn,
                              bool &Restart);

  DenseMap<Value *, Value *> ReplacementValues;

  StringRef getPassName() const override {
    return "X86 64/32 Interop Thunk Reference Inserter";
  }
  static char ID;
};

char X866432ThunkReferencePass::ID = 0;

} // end anonymous namespace

ModulePass *llvm::createX866432ThunkReferencePass() {
  return new X866432ThunkReferencePass();
}

void X866432ThunkReferencePass::getAnalysisUsage(AnalysisUsage &AU) const {
}

static bool isPointerCast(unsigned Opcode) {
  return Opcode == Instruction::BitCast ||
      Opcode == Instruction::AddrSpaceCast || Opcode == Instruction::PtrToInt;
}

Value *X866432ThunkReferencePass::getReplacementValue(
    Value *Old, Function &Fn, Function &ThunkFn) {
  auto I = ReplacementValues.find(Old);
  if (I != ReplacementValues.end())
    return I->second;

  // Base case.
  if (Old == &Fn) {
    ReplacementValues[Old] = &ThunkFn;
    return &ThunkFn;
  }

  // ConstantExpr case: get a new constant with the operand replaced by
  // its replacement value.
  if (auto *CE = dyn_cast<ConstantExpr>(Old)) {
    assert(isPointerCast(CE->getOpcode()));
    auto *NewCE = CE->getWithOperandReplaced(
        0, cast<Constant>(getReplacementValue(CE->getOperand(0), Fn, ThunkFn)));
    ReplacementValues[Old] = NewCE;
    return NewCE;
  }

  // CastInst case: insert a new instruction with the operand replaced by
  // its replacement value.
  auto *Cast = cast<CastInst>(Old);
  assert(isPointerCast(Cast->getOpcode()));
  auto *NewCast = Cast->clone();
  NewCast->setOperand(0, getReplacementValue(Cast->getOperand(0), Fn, ThunkFn));
  NewCast->insertAfter(Cast);
  ReplacementValues[Old] = NewCast;
  return NewCast;
}

bool X866432ThunkReferencePass::replaceNonCallUsesWith(
    Function &Fn, Value *Old, Function &ThunkFn, bool &Restart) {
  bool Modified = false;
  for (auto UI = Old->materialized_use_begin(), UE = Old->use_end(); UI != UE;){
    // Increment the iterator before mutating the list.
    Use &U = *UI++;
    if (auto *CE = dyn_cast<ConstantExpr>(U.getUser())) {
      if (isPointerCast(CE->getOpcode()))
        Modified |= replaceNonCallUsesWith(Fn, CE, ThunkFn, Restart);
      if (Restart)
        return Modified;
      continue;
    }
    if (auto *Cast = dyn_cast<CastInst>(U.getUser())) {
      if (isPointerCast(Cast->getOpcode()))
        Modified |= replaceNonCallUsesWith(Fn, Cast, ThunkFn, Restart);
      if (Restart)
        return Modified;
      continue;
    }
    if (auto *CI = dyn_cast<CallInst>(U.getUser())) {
      if (U.getOperandNo() == CI->Op<-1>().getOperandNo())
        continue;
    }
    if (auto *II = dyn_cast<InvokeInst>(U.getUser())) {
      if (U.getOperandNo() == II->Op<-3>().getOperandNo())
        continue;
    }

    Modified = true;
    Value *New = getReplacementValue(Old, Fn, ThunkFn);
    // Must handle Constants specially: we cannot call replaceUsesOfWith on a
    // constant because they are uniqued.
    if (auto *C = dyn_cast<Constant>(U.getUser())) {
      if (!isa<GlobalValue>(C)) {
        C->handleOperandChange(Old, New);
        // This can result in the constant itself being RAUW'd, which can
        // modify the use list out from under us. In that case, we need to
        // start over from the beginning. (RAUW itself doesn't have this
        // problem, since it always takes the first use until there are no
        // more uses.)
        Restart = true;
        return Modified;
      }
    }
    U.set(New);
  }
  Restart = false;
  return Modified;
}

bool X866432ThunkReferencePass::runOnModule(Module &M) {
  bool Modified = false;

  for (auto &Fn : M) {
    // We only care about 32-bit callable functions.
    CallingConv::ID CC = Fn.getCallingConv();
    if (CC != CallingConv::X86_64_C32 && CC != CallingConv::X86_StdCall &&
        CC != CallingConv::X86_FastCall && CC != CallingConv::X86_ThisCall)
      continue;

    // Get the corresponding thunk function.
    StringRef Prefix = "__i386_on_x86_64";
    if (Fn.hasFnAttribute("thunk-prefix"))
      Prefix = Fn.getFnAttribute("thunk-prefix").getValueAsString();
    char GlobalPrefix = M.getDataLayout().getGlobalPrefix();
    StringRef FnName = Fn.getName();
    bool StartsWithEscape = FnName.startswith("\01");
    bool HasGlobalPrefix = false;
    if (StartsWithEscape)
    {
      FnName = FnName.drop_front(1);
      HasGlobalPrefix = GlobalPrefix && FnName.startswith(StringRef(&GlobalPrefix, 1));
      if (HasGlobalPrefix)
        FnName = FnName.drop_front(1);
    }
    std::string ThunkName = (Prefix + "_thunk_" + FnName).str();
    if (StartsWithEscape)
    {
      if (HasGlobalPrefix)
        ThunkName.insert(0, 1, GlobalPrefix);
      ThunkName.insert(0, "\01");
    }
    Function *ThunkFn = M.getFunction(ThunkName);
    if (!ThunkFn) {
      ThunkFn = Function::Create(
          Fn.getFunctionType(), Fn.getLinkage(), 32,
          ThunkName);
      M.getFunctionList().push_back(ThunkFn);
    }

    // Now replace all uses that aren't direct calls with the thunk.
    // This is... unfortunately not quite the same as replaceAllUsesWith().
    bool Restart = false;
    do {
      Modified |= replaceNonCallUsesWith(Fn, &Fn, *ThunkFn, Restart);
    } while (Restart);
  }

  return Modified;
}
