package com.unciv.models.ruleset.tile

import com.unciv.Constants
import com.unciv.UncivGame
import com.unciv.logic.civilization.Civilization
import com.unciv.logic.map.mapunit.MapUnit
import com.unciv.logic.map.tile.RoadStatus
import com.unciv.models.ruleset.Belief
import com.unciv.models.ruleset.Ruleset
import com.unciv.models.ruleset.RulesetStatsObject
import com.unciv.models.ruleset.unique.StateForConditionals
import com.unciv.models.ruleset.unique.UniqueTarget
import com.unciv.models.ruleset.unique.UniqueType
import com.unciv.models.ruleset.unit.BaseUnit
import com.unciv.models.translations.tr
import com.unciv.ui.screens.civilopediascreen.CivilopediaScreen.Companion.showReligionInCivilopedia
import com.unciv.ui.screens.civilopediascreen.FormattedLine
import com.unciv.ui.components.extensions.toPercent
import com.unciv.ui.screens.worldscreen.unit.actions.UnitActions
import kotlin.math.roundToInt

class TileImprovement : RulesetStatsObject() {

    var terrainsCanBeBuiltOn: Collection<String> = ArrayList()
    var techRequired: String? = null
    var uniqueTo:String? = null
    override fun getUniqueTarget() = UniqueTarget.Improvement
    val shortcutKey: Char? = null
    // This is the base cost. A cost of 0 means created instead of buildable.
    val turnsToBuild: Int = 0


    fun getTurnsToBuild(civInfo: Civilization, unit: MapUnit): Int {
        val state = StateForConditionals(civInfo, unit = unit)
        return unit.getMatchingUniques(UniqueType.TileImprovementTime, state, checkCivInfoUniques = true)
            .fold(turnsToBuild.toFloat() * civInfo.gameInfo.speed.improvementBuildLengthModifier) { it, unique ->
                it * unique.params[0].toPercent()
            }.roundToInt()
            .coerceAtLeast(1)
        // In some weird cases it was possible for something to take 0 turns, leading to it instead never finishing
    }

    fun getDescription(ruleset: Ruleset): String {
        val lines = ArrayList<String>()

        val statsDesc = cloneStats().toString()
        if (statsDesc.isNotEmpty()) lines += statsDesc
        if (!terrainsCanBeBuiltOn.isEmpty()) {
            val terrainsCanBeBuiltOnString: ArrayList<String> = arrayListOf()
            for (i in terrainsCanBeBuiltOn) {
                terrainsCanBeBuiltOnString.add(i.tr())
            }
            lines += "Can be built on".tr() + terrainsCanBeBuiltOnString.joinToString(", ", " ") //language can be changed when setting changes.
        }
        for (resource: TileResource in ruleset.tileResources.values.filter { it.isImprovedBy(name) }) {
            if (resource.improvementStats == null) continue
            val statsString = resource.improvementStats.toString()
            lines += "[${statsString}] <in [${resource.name}] tiles>".tr()
        }
        if (techRequired != null) lines += "Required tech: [$techRequired]".tr()

        for (unique in uniques)
            lines += unique.tr()

        return lines.joinToString("\n")
    }

    fun isGreatImprovement() = hasUnique(UniqueType.GreatImprovement)
    fun isRoad() = RoadStatus.values().any { it != RoadStatus.None && it.name == this.name }
    fun isAncientRuinsEquivalent() = hasUnique(UniqueType.IsAncientRuinsEquivalent)

    fun canBeBuiltOn(terrain: String): Boolean {
        return terrain in terrainsCanBeBuiltOn
    }

    fun handleImprovementCompletion(builder: MapUnit) {
        val tile = builder.getTile()
        if (hasUnique(UniqueType.TakesOverAdjacentTiles))
            UnitActions.takeOverTilesAround(builder)
        if (tile.resource != null) {
            val city = builder.getTile().getCity()
            if (city != null) {
                city.cityStats.update()
                city.civ.cache.updateCivResources()
            }
        }
        if (hasUnique(UniqueType.RemovesFeaturesIfBuilt)) {
            // Remove terrainFeatures that a Worker can remove
            // and that aren't explicitly allowed under the improvement
            val removableTerrainFeatures = tile.terrainFeatures.filter { feature ->
                val removingAction = "${Constants.remove}$feature"

                removingAction in tile.ruleset.tileImprovements
                && !isAllowedOnFeature(feature)
                && tile.ruleset.tileImprovements[removingAction]!!.let {
                    it.techRequired == null || builder.civ.tech.isResearched(it.techRequired!!)
                }
            }

            tile.setTerrainFeatures(tile.terrainFeatures.filterNot { it in removableTerrainFeatures })
        }
    }

    /**
     * Check: Is this improvement allowed on a [given][name] terrain feature?
     *
     * Background: This not used for e.g. a lumbermill - it derives the right to be placed on forest
     * from [terrainsCanBeBuiltOn]. Other improvements may be candidates without fulfilling the
     * [terrainsCanBeBuiltOn] check - e.g. they are listed by a resource as 'their' improvement.
     * I such cases, the 'unbuildable' property of the Terrain feature might prevent the improvement,
     * so this check is done in conjunction - for the user, success means he does not need to remove
     * a terrain feature, thus the unique name.
     */
    fun isAllowedOnFeature(name: String) = terrainsCanBeBuiltOn.contains(name) || getMatchingUniques(UniqueType.NoFeatureRemovalNeeded).any { it.params[0] == name }

    /** Implements [UniqueParameterType.ImprovementFilter][com.unciv.models.ruleset.unique.UniqueParameterType.ImprovementFilter] */
    fun matchesFilter(filter: String): Boolean {
        return when (filter) {
            name -> true
            "All" -> true
            "All Road" -> isRoad()
            "Great Improvement", "Great" -> isGreatImprovement()
            in uniques -> true
            else -> false
        }
    }

    override fun makeLink() = "Improvement/$name"

    override fun getCivilopediaTextLines(ruleset: Ruleset): List<FormattedLine> {
        val textList = ArrayList<FormattedLine>()

        val statsDesc = cloneStats().toString()
        if (statsDesc.isNotEmpty()) textList += FormattedLine(statsDesc)

        if (uniqueTo != null) {
            textList += FormattedLine()
            textList += FormattedLine("Unique to [$uniqueTo]", link="Nation/$uniqueTo")
        }

        val constructorUnits = getConstructorUnits(ruleset)
        val creatingUnits = getCreatingUnits(ruleset)
        val creatorExists = constructorUnits.isNotEmpty() || creatingUnits.isNotEmpty()

        if (creatorExists && terrainsCanBeBuiltOn.isNotEmpty()) {
            textList += FormattedLine()
            if (terrainsCanBeBuiltOn.size == 1) {
                with (terrainsCanBeBuiltOn.first()) {
                    textList += FormattedLine("{Can be built on} {$this}", link="Terrain/$this")
                }
            } else {
                textList += FormattedLine("{Can be built on}:")
                terrainsCanBeBuiltOn.forEach {
                    textList += FormattedLine(it, link="Terrain/$it", indent=1)
                }
            }
        }

        var addedLineBeforeResourceBonus = false
        for (resource in ruleset.tileResources.values) {
            if (resource.improvementStats == null || !resource.isImprovedBy(name)) continue
            if (!addedLineBeforeResourceBonus) {
                addedLineBeforeResourceBonus = true
                textList += FormattedLine()
            }
            val statsString = resource.improvementStats.toString()
            // Line intentionally modeled as UniqueType.Stats + ConditionalInTiles
            textList += FormattedLine("[${statsString}] <in [${resource.name}] tiles>", link = resource.makeLink())
        }

        if (techRequired != null) {
            textList += FormattedLine()
            textList += FormattedLine("Required tech: [$techRequired]", link="Technology/$techRequired")
        }

        if (uniques.isNotEmpty()) {
            textList += FormattedLine()
            for (unique in uniqueObjects)
                textList += FormattedLine(unique)
        }

        // Be clearer when one needs to chop down a Forest first... A "Can be built on Plains" is clear enough,
        // but a "Can be built on Land" is not - how is the user to know Forest is _not_ Land?
        if (creatorExists &&
                !isEmpty() && // Has any Stats
                !hasUnique(UniqueType.NoFeatureRemovalNeeded) &&
                !hasUnique(UniqueType.RemovesFeaturesIfBuilt) &&
                terrainsCanBeBuiltOn.none { it in ruleset.terrains }
        )
            textList += FormattedLine("Needs removal of terrain features to be built")

        if (isAncientRuinsEquivalent() && ruleset.ruinRewards.isNotEmpty()) {
            val difficulty = if (!UncivGame.isCurrentInitialized() || UncivGame.Current.gameInfo == null)
                    "Prince"  // most factors == 1
                else UncivGame.Current.gameInfo!!.gameParameters.difficulty
            val religionEnabled = showReligionInCivilopedia(ruleset)
            textList += FormattedLine()
            textList += FormattedLine("The possible rewards are:")
            ruleset.ruinRewards.values.asSequence()
                .filter { reward ->
                    difficulty !in reward.excludedDifficulties &&
                    (religionEnabled || !reward.hasUnique(UniqueType.HiddenWithoutReligion))
                }
                .forEach { reward ->
                    textList += FormattedLine(reward.name, starred = true, color = reward.color)
                    textList += reward.civilopediaText
                }
        }

        if (creatorExists)
            textList += FormattedLine()
        for (unit in constructorUnits)
            textList += FormattedLine("{Can be constructed by} {$unit}", unit.makeLink())
        for (unit in creatingUnits)
            textList += FormattedLine("{Can be created instantly by} {$unit}", unit.makeLink())

        textList += Belief.getCivilopediaTextMatching(name, ruleset)

        return textList
    }

    private fun getConstructorUnits(ruleset: Ruleset): List<BaseUnit> {
        //todo Why does this have to be so complicated? A unit's "Can build [Land] improvements on tiles"
        //     creates the _justified_ expectation that an improvement it can build _will_ have
        //     `matchesFilter("Land")==true` - but that's not the case.
        //     A kludge, but for display purposes the test below is meaningful enough.
        if (hasUnique(UniqueType.Unbuildable)) return emptyList()

        val canOnlyFilters = getMatchingUniques(UniqueType.CanOnlyBeBuiltOnTile)
            .map { it.params[0].run { if (this == "Coastal") "Land" else this } }.toSet()
        val cannotFilters = getMatchingUniques(UniqueType.CannotBuildOnTile).map { it.params[0] }.toSet()
        val resourcesImprovedByThis = ruleset.tileResources.values.filter { it.isImprovedBy(name) }

        val expandedCanBeBuiltOn = sequence {
            yieldAll(terrainsCanBeBuiltOn)
            yieldAll(terrainsCanBeBuiltOn.asSequence().mapNotNull { ruleset.terrains[it] }.flatMap { it.occursOn.asSequence() })
            if (hasUnique(UniqueType.CanOnlyImproveResource))
                yieldAll(resourcesImprovedByThis.asSequence().flatMap { it.terrainsCanBeFoundOn })
            if (name.startsWith(Constants.remove)) name.removePrefix(Constants.remove).apply {
                yield(this)
                ruleset.terrains[this]?.occursOn?.let { yieldAll(it) }
                ruleset.tileImprovements[this]?.terrainsCanBeBuiltOn?.let { yieldAll(it) }
            }
        }.filter { it !in cannotFilters }.toMutableSet()

        val terrainsCanBeBuiltOnTypes = sequence {
            yieldAll(expandedCanBeBuiltOn.asSequence()
                .mapNotNull { ruleset.terrains[it]?.type })
            yieldAll(TerrainType.values().asSequence()
                .filter { it.name in expandedCanBeBuiltOn })
        }.filter { it.name !in cannotFilters }.toMutableSet()

        if (canOnlyFilters.isNotEmpty() && canOnlyFilters.intersect(expandedCanBeBuiltOn).isEmpty()) {
            expandedCanBeBuiltOn.clear()
            if (terrainsCanBeBuiltOnTypes.none { it.name in canOnlyFilters })
                terrainsCanBeBuiltOnTypes.clear()
        }

        fun matchesBuildImprovementsFilter(filter: String) =
            matchesFilter(filter) ||
            filter in expandedCanBeBuiltOn ||
            terrainsCanBeBuiltOnTypes.any { it.name == filter }

        return ruleset.units.values.asSequence()
            .filter { unit ->
                turnsToBuild != 0
                    && unit.getMatchingUniques(UniqueType.BuildImprovements, StateForConditionals.IgnoreConditionals)
                        .any { matchesBuildImprovementsFilter(it.params[0]) }
                || unit.hasUnique(UniqueType.CreateWaterImprovements)
                    && terrainsCanBeBuiltOnTypes.contains(TerrainType.Water)
            }.toList()
    }

    private fun getCreatingUnits(ruleset: Ruleset): List<BaseUnit> {
        return ruleset.units.values.asSequence()
            .filter { unit ->
                unit.getMatchingUniques(UniqueType.ConstructImprovementConsumingUnit, StateForConditionals.IgnoreConditionals)
                    .any { it.params[0] == name }
            }.toList()
    }
}
