import os, tempfile
from nose.tools import *
from mercurial import (
    config,
    error,
    pycompat,
)
import pytest

from tortoisehg.util import (
    hglib,
    wconfig,
)

import helpers

def setup_module():
    global _tempdir
    _tempdir = helpers.mktmpdir(__name__)


def newrconfig(vals=None):
    if vals is None:
        vals = {}
    c = config.config()
    for k, v in isinstance(vals, dict) and vals.items() or vals:
        sec, it = k.split(b'.', 1)
        c.set(sec, it, v)
    return c

def newwconfig(vals=None):
    if vals is None:
        vals = {}
    return wconfig.config(newrconfig(vals))

# see wconfig.writefile()
if pycompat.ispy3:
    def written(c):
        dest = pycompat.io.StringIO()
        c.write(dest)
        return hglib.fromunicode(dest.getvalue())
else:
    def written(c):
        dest = pycompat.bytesio()
        c.write(dest)
        return dest.getvalue()

def writetempfile(s):
    fd, path = tempfile.mkstemp(dir=_tempdir)
    os.write(fd, s)
    os.close(fd)
    return path


def with_rconfig(f):
    f.__test__ = True
    return pytest.mark.usefixtures('rconfig')(f)

def with_wconfig(f):
    f.__test__ = True
    return pytest.mark.usefixtures('wconfig')(f)

def with_both(f):
    f.__test__ = True
    return pytest.mark.usefixtures('both')(f)


@pytest.fixture(name='wconfig')
def fixture_wconfig():
    if wconfig._hasiniparse:
        yield
    else:
        pytest.skip()

@pytest.fixture(name='rconfig')
def fixture_rconfig():
    orighasiniparse = wconfig._hasiniparse
    wconfig._hasiniparse = False
    try:
        yield
    finally:
        wconfig._hasiniparse = orighasiniparse

@pytest.fixture(name='both', params=['wconfig', 'rconfig'])
def fixture_both(request):
    request.getfixturevalue(request.param)


@with_both
def check_copy():
    c = newwconfig({b'foo.bar': b'baz'})
    assert_equals(c.__class__, c.copy().__class__)
    assert_equals(b'baz', c.copy().get(b'foo', b'bar'))

@with_both
def check_contains():
    c = newwconfig({b'foo.bar': b'baz'})
    assert b'foo' in c, c
    assert b'bar' not in c, c

@with_both
def check_getitem():
    c = newwconfig({b'foo.bar': b'x', b'foo.baz': b'y'})
    assert_equals({b'bar': b'x', b'baz': b'y'}, dict(c[b'foo']))
    assert_equals({}, dict(c[b'unknown']))

@with_both
def check_getitem_empty_then_set_no_effect():
    c = newwconfig()
    c[b'unknown'][b'bar'] = b'baz'
    assert not c.get(b'unknown', b'bar'), c.get(b'unknown', b'bar')

@with_both
def check_set_followed_by_getitem_empty():
    c = newwconfig()
    c[b'unknown']
    c.set(b'unknown', b'foo', b'bar')
    assert_equals(b'bar', c.get(b'unknown', b'foo'))
    assert_equals(b'bar', c[b'unknown'][b'foo'])

@with_both
def check_dict_contains():
    c = newwconfig({b'foo.bar': b'x'})
    assert b'bar' in c[b'foo'], c[b'foo']
    assert b'baz' not in c[b'foo'], c[b'foo']

@with_both
def check_dict_getitem():
    c = newwconfig({b'foo.bar': b'x'})
    assert_equals(b'x', c[b'foo'][b'bar'])
    assert_raises(KeyError, lambda: c[b'foo'][b'baz'])

@with_both
def check_dict_setitem():
    c = newwconfig({b'foo.bar': b'x'})
    c[b'foo'][b'bar'] = b'y'
    c[b'foo'][b'baz'] = b'z'
    assert_equals(b'y', c[b'foo'][b'bar'])
    assert_equals(b'z', c[b'foo'][b'baz'])

@with_wconfig  # original config doesn't preserve the order
def check_dict_setitem_preserve_order():
    c = newwconfig([(b'foo.bar', b'x'), (b'foo.baz', b'y')])
    assert_equals([b'bar', b'baz'], list(c[b'foo']))
    c[b'foo'][b'bar'] = b'z'
    assert_equals([b'bar', b'baz'], list(c[b'foo']))

@with_both
def check_dict_iter():
    c = newwconfig({b'foo.bar': b'x', b'foo.baz': b'y'})
    assert_equals({b'bar', b'baz'}, set(c[b'foo']))

@with_both
def check_dict_len():
    c = newwconfig({b'foo.bar': b'x'})
    assert_equals(1, len(c[b'foo']))

@with_both
def check_dict_update():
    c = newwconfig({b'foo.bar': b'x', b'foo.baz': b'y'})
    c[b'foo'].update(newwconfig({b'foo.bar': b'z', b'foo.baz': b'w'})[b'foo'])
    assert_equals(b'z', c[b'foo'][b'bar'])
    assert_equals(b'w', c[b'foo'][b'baz'])

@with_both
def check_dict_delitem():
    c = newwconfig({b'foo.bar': b'x'})
    del c[b'foo'][b'bar']
    assert b'bar' not in c[b'foo'], c[b'foo']

@with_both
def check_iter():
    c = newwconfig({b'foo.bar': b'x', b'baz.bax': b'y'})
    assert_equals({b'foo', b'baz'}, set(c))

@with_both
def check_update():
    c0 = newwconfig({b'foo.bar': b'x', b'foo.blah': b'w'})
    c1 = newwconfig({b'foo.bar': b'y', b'baz.bax': b'z'})
    c0.update(c1)
    assert_equals(b'y', c0.get(b'foo', b'bar'))
    assert_equals(b'z', c0.get(b'baz', b'bax'))
    assert_equals(b'w', c0.get(b'foo', b'blah'))

@with_both
def check_get():
    c = newwconfig({b'foo.bar': b'baz'})
    assert_equals(b'baz', c.get(b'foo', b'bar'))
    assert_equals(None, c.get(b'foo', b'baz'))
    assert_equals(b'x', c.get(b'foo', b'baz', b'x'))

@with_both
def check_source():
    c = newwconfig()
    c.set(b'foo', b'bar', b'baz', source=b'blah')
    assert_equals(b'blah', c.source(b'foo', b'bar'))

@with_both
def check_sections():
    c = newwconfig({b'foo.bar': b'x', b'baz.bax': b'y'})
    assert_equals([b'baz', b'foo'], c.sections())

@with_both
def check_items():
    c = newwconfig({b'foo.bar': b'x', b'foo.baz': b'y'})
    assert_equals({b'bar': b'x', b'baz': b'y'}, dict(c.items(b'foo')))

@with_both
def check_set():
    c = newwconfig({b'foo.bar': b'x'})
    c.set(b'foo', b'baz', b'y')
    c.set(b'foo', b'bar', b'w')
    c.set(b'newsection', b'bax', b'z')
    assert_equals(b'y', c.get(b'foo', b'baz'))
    assert_equals(b'w', c.get(b'foo', b'bar'))
    assert_equals(b'z', c.get(b'newsection', b'bax'))

@with_wconfig  # original config doesn't preserve the order
def check_set_preserve_order():
    c = newwconfig([(b'foo.bar', b'x'), (b'foo.baz', b'y')])
    assert_equals([b'bar', b'baz'], list(c[b'foo']))
    c.set(b'foo', b'bar', b'z')
    assert_equals([b'bar', b'baz'], list(c[b'foo']))

# TODO: test_parse
# TODO: test_read

@with_wconfig
def check_write_after_set():
    c = newwconfig()
    c.set(b'foo', b'bar', b'baz')
    assert_equals(b'[foo]\nbar = baz', written(c).rstrip())

@with_wconfig
def check_write_empty():
    c = newwconfig()
    assert_equals(b'', written(c).rstrip())

@with_wconfig
def check_write_after_update():
    c = newwconfig()
    c.update(newwconfig({b'foo.bar': b'baz'}))
    assert_equals(b'[foo]\nbar = baz', written(c).rstrip())

@with_wconfig
def check_read_write():
    c = newwconfig()
    s = b'[foo]\nbar = baz'
    c.read(path=b'foo', fp=pycompat.bytesio(s))
    assert_equals(s, written(c).rstrip())

@with_wconfig
@raises(error.ParseError)
def check_read_write_missing_section_header_error():
    c = newwconfig()
    s = b'bar = baz'  # missing header
    c.read(path=b'foo', fp=pycompat.bytesio(s))
    c.write(pycompat.bytesio())

@with_wconfig
@raises(error.ParseError)
def check_read_write_parsing_error():
    c = newwconfig()
    s = b'[foo]\n:bar = baz'  # Mercurial can parse it but INIConfig can't
    c.read(path=b'foo', fp=pycompat.bytesio(s))
    c.write(pycompat.bytesio())

@with_wconfig
def check_write_after_dict_setitem():
    c = newwconfig({b'foo.bar': b'x'})
    c[b'foo'][b'bar'] = b'y'
    assert_equals(b'[foo]\nbar = y', written(c).rstrip())

@with_wconfig
def check_write_after_dict_update():
    c = newwconfig({b'foo.bar': b'x'})
    c[b'foo'].update({b'bar': b'y'})
    assert_equals(b'[foo]\nbar = y', written(c).rstrip())

@with_wconfig
def check_write_after_dict_delitem():
    c = newwconfig({b'foo.bar': b'x', b'foo.baz': b'y'})
    del c[b'foo'][b'bar']
    assert_equals(b'[foo]\nbaz = y', written(c).rstrip())

@with_wconfig
def check_read_write_rem():
    c = newwconfig()
    s = b'[foo]\nrem = x'
    c.read(path=b'foo', fp=pycompat.bytesio(s))
    c.set(b'foo', b'rem', b'y')
    assert_equals(b'[foo]\nrem = y', written(c).rstrip())

@with_wconfig
def check_read_write_suboption():
    c = newwconfig()
    s = b'[foo]\nbar:baz = x'
    c.read(path=b'foo', fp=pycompat.bytesio(s))
    c.set(b'foo', b'bar:baz', b'y')
    assert_equals(b'[foo]\nbar:baz = y', written(c).rstrip())

@with_wconfig
def check_read_write_suboption_removal():
    c = newwconfig()
    s = b'[foo]\nbar:baz = x\nbar = y'
    c.read(path=b'foo', fp=pycompat.bytesio(s))
    del c[b'foo'][b'bar:baz']
    assert_equals(b'[foo]\nbar = y', written(c).rstrip())


@with_wconfig
def check_write_conflict_set_set():
    fname = writetempfile(b'[foo]\nbar = x')
    c0 = wconfig.readfile(fname)
    c1 = wconfig.readfile(fname)
    c1.set(b'foo', b'bar', b'y')
    wconfig.writefile(c1, fname)
    c0.set(b'foo', b'bar', b'z')
    wconfig.writefile(c0, fname)

    cr = wconfig.readfile(fname)
    assert_equals(b'z', cr.get(b'foo', b'bar'))

@with_wconfig
def check_write_conflict_del_set():
    fname = writetempfile(b'[foo]\nbar = x')
    c0 = wconfig.readfile(fname)
    c1 = wconfig.readfile(fname)
    del c1[b'foo'][b'bar']
    wconfig.writefile(c1, fname)
    c0.set(b'foo', b'bar', b'z')
    wconfig.writefile(c0, fname)

    cr = wconfig.readfile(fname)
    assert_equals(b'z', cr.get(b'foo', b'bar'))

@with_wconfig
def check_write_conflict_set_del():
    fname = writetempfile(b'[foo]\nbar = x')
    c0 = wconfig.readfile(fname)
    c1 = wconfig.readfile(fname)
    c1.set(b'foo', b'bar', b'y')
    wconfig.writefile(c1, fname)
    del c0[b'foo'][b'bar']
    wconfig.writefile(c0, fname)

    cr = wconfig.readfile(fname)
    assert not cr.get(b'foo', b'bar'), cr.get(b'foo', b'bar')

@with_wconfig
def check_write_conflict_del_del():
    fname = writetempfile(b'[foo]\nbar = x')
    c0 = wconfig.readfile(fname)
    c1 = wconfig.readfile(fname)
    del c1[b'foo'][b'bar']
    wconfig.writefile(c1, fname)
    del c0[b'foo'][b'bar']
    wconfig.writefile(c0, fname)  # shouldn't raise KeyError

    cr = wconfig.readfile(fname)
    assert not cr.get(b'foo', b'bar'), cr.get(b'foo', b'bar')

@with_wconfig
def check_write_noconflict_set_set():
    fname = writetempfile(b'[foo]\nbar = x')
    c0 = wconfig.readfile(fname)
    c1 = wconfig.readfile(fname)
    c1.set(b'foo', b'baz', b'y')
    wconfig.writefile(c1, fname)
    c0.set(b'foo', b'bar', b'z')
    wconfig.writefile(c0, fname)  # should not override foo.baz = y

    cr = wconfig.readfile(fname)
    assert_equals(b'z', cr.get(b'foo', b'bar'))
    assert_equals(b'y', cr.get(b'foo', b'baz'))
    # don't reload c1's change implicitly
    assert not c0.get(b'foo', b'baz'), c0.get(b'foo', b'baz')

@with_wconfig
def check_write_noconflict_del():
    fname = writetempfile(b'[foo]\nbar = x')
    c0 = wconfig.readfile(fname)
    c1 = wconfig.readfile(fname)
    del c1[b'foo'][b'bar']
    wconfig.writefile(c1, fname)
    wconfig.writefile(c0, fname)  # shouldn't override del foo.bar

    cr = wconfig.readfile(fname)
    assert not cr.get(b'foo', b'bar'), cr.get(b'foo', b'bar')
    # don't reload c1's change implicitly
    assert c0.get(b'foo', b'bar'), c0.get(b'foo', b'bar')


@with_wconfig
def check_write_copied():
    fname = writetempfile(b'[foo]\nbar = x')
    c0 = wconfig.readfile(fname)
    c1 = c0.copy()
    c1.set(b'foo', b'baz', b'y')
    wconfig.writefile(c1, fname)

    cr = wconfig.readfile(fname)
    assert_equals(b'x', cr.get(b'foo', b'bar'))
    assert_equals(b'y', cr.get(b'foo', b'baz'))

@with_wconfig
def check_write_copied_conflict():
    fname = writetempfile(b'[foo]\nbar = x')
    c0 = wconfig.readfile(fname)
    c1 = c0.copy()
    c0.set(b'foo', b'bar', b'y')
    wconfig.writefile(c0, fname)
    wconfig.writefile(c1, fname)  # shouldn't override foo.bar = y

    cr = wconfig.readfile(fname)
    assert_equals(b'y', cr.get(b'foo', b'bar'))

@with_wconfig
def check_write_copied_rconfig():
    c0 = newrconfig({b'foo.bar': b'x'})
    c1 = wconfig.config(c0)
    assert_equals(b'[foo]\nbar = x', written(c1).rstrip())

@with_both
def check_readfile():
    fname = writetempfile(b'[foo]\nbar = baz')
    c = wconfig.readfile(fname)
    assert_equals(b'baz', c.get(b'foo', b'bar'))

@with_wconfig
def check_writefile():
    c = newwconfig({b'foo.bar': b'baz'})
    fname = writetempfile(b'')
    wconfig.writefile(c, fname)
    assert_equals(b'[foo]\nbar = baz', open(fname, 'rb').read().rstrip())
