from __future__ import absolute_import

from xml.dom import minidom
from nose.tools import *

from tortoisehg.hgqt.qtcore import (
    QByteArray,
    QBuffer,
    QIODevice,
)

from mercurial import node

from tortoisehg.hgqt import (
    repotreemodel,
    repotreeitem,
)

def prettyxml(s):
    dom = minidom.parseString(s)
    return dom.toprettyxml(indent='  ').splitlines()

def with_qbuffer(data=b'', mode=QIODevice.ReadOnly):
    def decorate(func):
        def newfunc():
            ba = QByteArray(data)  # keep reference to avoid GC
            f = QBuffer(ba)
            f.open(mode)
            try:
                func(f)
            finally:
                f.close()
        return make_decorator(func)(newfunc)
    return decorate

full_data = br'''<?xml version="1.0" encoding="UTF-8"?>
<reporegistry>
  <treeitem>
    <allgroup name="default">
      <repo root="/thg" shortname="thg"
            basenode="bac32db38e52fd49acb62b94730a55f4f4b0cdee"/>
      <repo root="/mercurial" shortname="hg"
            basenode="9117c6561b0bd7792fa13b50d28239d51b78e51f"
            sharedpath="/mirror/mercurial/.hg"/>
      <subrepo root="/subroot/sub" shortname="sub"
               basenode="2f425e331c8cdffa5103f3b181358092245bdc10"/>
    </allgroup>
    <group name="bar">
      <group name="baz">
        <repo root="/qux" shortname="qux"
              basenode="6c30f00cc82daff63b1260eec198256a9c8e5a56"/>
      </group>
      <repo root="/subroot" shortname="subroot"
            basenode="b986218ba1c9b0d6a259fac9b050b1724ed8e545">
        <subrepo root="/subroot/svnsub" repotype="svn"/>
        <subrepo root="/subroot/sub" shortname="sub"
                 basenode="2f425e331c8cdffa5103f3b181358092245bdc10"/>
      </repo>
    </group>
  </treeitem>
</reporegistry>
'''

full_data_standalone_repos = ['/thg', '/mercurial', '/subroot/sub',
                              '/qux', '/subroot']
full_data_all_repos = (full_data_standalone_repos
                       + ['/subroot/svnsub', '/subroot/sub'])

@with_qbuffer(full_data, QIODevice.ReadWrite)
def test_readwritexml(f):
    root = repotreemodel.readXml(f, 'reporegistry')
    f.buffer().clear()
    f.reset()
    repotreemodel.writeXml(f, root, 'reporegistry')
    assert_equals(prettyxml(full_data), prettyxml(bytes(f.data())))

@with_qbuffer(full_data)
def test_iterrepoitemfromxml(f):
    repos = list(repotreemodel.iterRepoItemFromXml(f))
    assert_equals(full_data_standalone_repos,
                  [e.rootpath() for e in repos])
    assert_equals('thg', repos[0].shortname())
    assert_equals(b'bac32db38e52fd49acb62b94730a55f4f4b0cdee',
                  node.hex(repos[0].basenode()))

@with_qbuffer(full_data)
def test_getrepoitemlist_all(f):
    root = repotreemodel.readXml(f, 'reporegistry')
    items = repotreemodel.getRepoItemList(root)
    assert_equals(full_data_all_repos,
                  [e.rootpath() for e in items])

@with_qbuffer(full_data)
def test_getrepoitemlist_standalone(f):
    root = repotreemodel.readXml(f, 'reporegistry')
    items = repotreemodel.getRepoItemList(root, standalone=True)
    assert_equals(full_data_standalone_repos,
                  [e.rootpath() for e in items])

subrepos_data = br'''<?xml version="1.0" encoding="UTF-8"?>
<reporegistry>
  <treeitem>
    <allgroup name="default">
      <subrepo root="/subroot/sub" shortname="sub"
               basenode="2f425e331c8cdffa5103f3b181358092245bdc10"/>
      <repo root="/subroot" shortname="subroot"
            basenode="b986218ba1c9b0d6a259fac9b050b1724ed8e545">
        <subrepo root="/subroot/svnsub" repotype="svn"/>
        <subrepo root="/subroot/sub" shortname="sub"
                 basenode="2f425e331c8cdffa5103f3b181358092245bdc10"/>
      </repo>
    </allgroup>
  </treeitem>
</reporegistry>
'''

@with_qbuffer(subrepos_data)
def test_undumpsubrepos(f):
    """<subrepo> element should be mapped to different classes"""
    root = repotreemodel.readXml(f, 'reporegistry')
    allgroup = root.child(0)
    assert type(allgroup.child(0)) is repotreeitem.StandaloneSubrepoItem, \
        repr(allgroup.child(0))
    subroot = allgroup.child(1)
    assert type(subroot) is repotreeitem.RepoItem, repr(subroot)
    assert type(subroot.child(0)) is repotreeitem.AlienSubrepoItem, \
        repr(subroot.child(0))
    assert type(subroot.child(1)) is repotreeitem.SubrepoItem, \
        repr(subroot.child(1))
