<?php
/**
 * Zend Framework (http://framework.zend.com/)
 *
 * @link      http://github.com/zendframework/zf2 for the canonical source repository
 * @copyright Copyright (c) 2005-2012 Zend Technologies USA Inc. (http://www.zend.com)
 * @license   http://framework.zend.com/license/new-bsd New BSD License
 * @package   ZendGData
 */

namespace ZendGData\GApps;

use ZendGData\App\InvalidArgumentException;
use ZendGData\GApps;

/**
 * Assists in constructing queries for Google Apps owner entries.
 * Instances of this class can be provided in many places where a URL is
 * required.
 *
 * For information on submitting queries to a server, see the Google Apps
 * service class, ZendGData\GApps.
 *
 * @category   Zend
 * @package    ZendGData
 * @subpackage GApps
 */
class OwnerQuery extends AbstractQuery
{

    /**
     * Group owner is refering to
     *
     * @var string
     */
    protected $_groupId = null;

    /**
     * The email of the owner
     *
     * @var string
     */
    protected $_ownerEmail = null;

    /**
     * Create a new instance.
     *
     * @param string $domain (optional) The Google Apps-hosted domain to use
     *          when constructing query URIs.
     * @param string $groupId (optional) Value for the groupId property.
     * @param string $ownerEmail (optional) Value for the OwnerEmail property.
     */
    public function __construct($domain = null, $groupId = null, $ownerEmail = null)
    {
        parent::__construct($domain);
        $this->setGroupId($groupId);
        $this->setOwnerEmail($ownerEmail);
    }

    /**
     * Set the group id to query for.
     *
     * @see getGroupId
     * @param string $value
     */
    public function setGroupId($value)
    {
        $this->_groupId = $value;
    }

    /**
     * Get the group id to query for.
     *
     * @return string
     *
     */
    public function getGroupId()
    {
        return $this->_groupId;
    }

    /**
     * Set the owner email to query for.
     *
     * @see getOwnerEmail
     * @param string $value
     */
    public function setOwnerEmail($value)
    {
        $this->_ownerEmail = $value;
    }

    /**
     * Get the owner email to query for.
     *
     * @return string
     *
     */
    public function getOwnerEmail()
    {
        return $this->_ownerEmail;
    }

    /**
     * Returns the query URL generated by this query instance.
     *
     * @return string The query URL for this instance.
     */
    public function getQueryUrl()
    {
        if ($this->_groupId === null) {
            throw new InvalidArgumentException('groupId must not be null');
        }

        $uri = GApps::APPS_BASE_FEED_URI . GApps::APPS_GROUP_PATH . '/' . $this->_domain . '/' . $this->_groupId . '/owner';

        if ($this->_ownerEmail !== null) {
            $uri .= '/' . $this->_ownerEmail;
        }

        $uri .= $this->getQueryString();
        return $uri;
    }

}
