/**
 * @file src/capstone2llvmir/x86/x86_init.cpp
 * @brief Initializations for X86 implementation of @c Capstone2LlvmIrTranslator.
 * @copyright (c) 2017 Avast Software, licensed under the MIT license
 */

#include "capstone2llvmir/x86/x86_impl.h"

namespace retdec {
namespace capstone2llvmir {

//
//==============================================================================
// Pure virtual methods from Capstone2LlvmIrTranslator_impl
//==============================================================================
//

void Capstone2LlvmIrTranslatorX86_impl::initializeArchSpecific()
{
	initializeRegistersParentMap();
}

void Capstone2LlvmIrTranslatorX86_impl::initializeRegNameMap()
{
	std::map<uint32_t, std::string> r2n =
	{
			// x86_reg_rflags
			//
			{X86_REG_CF, "cf"},
			{X86_REG_PF, "pf"},
			{X86_REG_AF, "az"},
			{X86_REG_ZF, "zf"},
			{X86_REG_SF, "sf"},
			{X86_REG_TF, "tf"},
			{X86_REG_IF, "if"},
			{X86_REG_DF, "df"},
			{X86_REG_OF, "of"},
			{X86_REG_IOPL, "iopl"},
			{X86_REG_NT, "nt"},
			{X86_REG_RF, "rf"},
			{X86_REG_VM, "vm"},
			{X86_REG_AC, "ac"},
			{X86_REG_VIF, "vif"},
			{X86_REG_VIP, "vip"},
			{X86_REG_ID, "id"},

			// x87_reg_status
			//
			{X87_REG_IE, "fpu_stat_IE"},
			{X87_REG_DE, "fpu_stat_DE"},
			{X87_REG_ZE, "fpu_stat_ZE"},
			{X87_REG_OE, "fpu_stat_OE"},
			{X87_REG_UE, "fpu_stat_UE"},
			{X87_REG_PE, "fpu_stat_PE"},
			{X87_REG_SF, "fpu_stat_SF"},
			{X87_REG_ES, "fpu_stat_ES"},
			{X87_REG_C0, "fpu_stat_C0"},
			{X87_REG_C1, "fpu_stat_C1"},
			{X87_REG_C2, "fpu_stat_C2"},
			{X87_REG_C3, "fpu_stat_C3"},
			{X87_REG_TOP, "fpu_stat_TOP"},
			{X87_REG_B, "fpu_stat_B"},

			// x87_reg_control
			//
			{X87_REG_IM, "fpu_control_IM"},
			{X87_REG_DM, "fpu_control_DM"},
			{X87_REG_ZM, "fpu_control_ZM"},
			{X87_REG_OM, "fpu_control_OM"},
			{X87_REG_UM, "fpu_control_UM"},
			{X87_REG_PM, "fpu_control_PM"},
			{X87_REG_PC, "fpu_control_PC"},
			{X87_REG_RC, "fpu_control_RC"},
			{X87_REG_X, "fpu_control_X"},

			// FPU data registers
			// They are named as ST(X) in Capstone, which is not good for us.
			//
			{X86_REG_ST0, "st0"},
			{X86_REG_ST1, "st1"},
			{X86_REG_ST2, "st2"},
			{X86_REG_ST3, "st3"},
			{X86_REG_ST4, "st4"},
			{X86_REG_ST5, "st5"},
			{X86_REG_ST6, "st6"},
			{X86_REG_ST7, "st7"},
	};

	_reg2name = std::move(r2n);
}

void Capstone2LlvmIrTranslatorX86_impl::initializeRegTypeMap()
{
	auto* i1 = llvm::IntegerType::getInt1Ty(_module->getContext());
	auto* i2 = llvm::IntegerType::getIntNTy(_module->getContext(), 2);
	auto* i3 = llvm::IntegerType::getIntNTy(_module->getContext(), 3);
	auto* i8 = llvm::IntegerType::getInt8Ty(_module->getContext());
	auto* i16 = llvm::IntegerType::getInt16Ty(_module->getContext());
	auto* i32 = llvm::IntegerType::getInt32Ty(_module->getContext());
	auto* i64 = llvm::IntegerType::getInt64Ty(_module->getContext());
	auto* i128 = llvm::IntegerType::getInt128Ty(_module->getContext());
	auto* i256 = llvm::IntegerType::getIntNTy(_module->getContext(), 256);
	auto* i512 = llvm::IntegerType::getIntNTy(_module->getContext(), 512);
	auto* fp64 = llvm::IntegerType::getDoubleTy(_module->getContext());
	auto* fp80 = llvm::IntegerType::getX86_FP80Ty(_module->getContext());

	auto* defTy = _origBasicMode == CS_MODE_64 ? i64 : i32;

	std::map<uint32_t, llvm::Type*> r2t =
	{
			// x86_reg
			//
			{X86_REG_AH, i8},
			{X86_REG_AL, i8},
			{X86_REG_CH, i8},
			{X86_REG_CL, i8},
			{X86_REG_DH, i8},
			{X86_REG_DL, i8},
			{X86_REG_BH, i8},
			{X86_REG_BL, i8},
			{X86_REG_SPL, i8},
			{X86_REG_BPL, i8},
			{X86_REG_DIL, i8},
			{X86_REG_SIL, i8},
			{X86_REG_R8B, i8},
			{X86_REG_R9B, i8},
			{X86_REG_R10B, i8},
			{X86_REG_R11B, i8},
			{X86_REG_R12B, i8},
			{X86_REG_R13B, i8},
			{X86_REG_R14B, i8},
			{X86_REG_R15B, i8},

			{X86_REG_AX, i16},
			{X86_REG_CX, i16},
			{X86_REG_DX, i16},
			{X86_REG_BP, i16},
			{X86_REG_BX, i16},
			{X86_REG_DI, i16},
			{X86_REG_SP, i16},
			{X86_REG_SI, i16},
			{X86_REG_SS, i16},
			{X86_REG_CS, i16},
			{X86_REG_DS, i16},
			{X86_REG_ES, i16},
			{X86_REG_FS, i16},
			{X86_REG_GS, i16},
			{X86_REG_R8W, i16},
			{X86_REG_R9W, i16},
			{X86_REG_R10W, i16},
			{X86_REG_R11W, i16},
			{X86_REG_R12W, i16},
			{X86_REG_R13W, i16},
			{X86_REG_R14W, i16},
			{X86_REG_R15W, i16},
			{X86_REG_IP, i16},

			{X86_REG_EAX, i32},
			{X86_REG_EBP, i32},
			{X86_REG_EBX, i32},
			{X86_REG_ECX, i32},
			{X86_REG_EDI, i32},
			{X86_REG_EDX, i32},
			{X86_REG_ESI, i32},
			{X86_REG_ESP, i32},
			{X86_REG_R8D, i32},
			{X86_REG_R9D, i32},
			{X86_REG_R10D, i32},
			{X86_REG_R11D, i32},
			{X86_REG_R12D, i32},
			{X86_REG_R13D, i32},
			{X86_REG_R14D, i32},
			{X86_REG_R15D, i32},
			{X86_REG_EIP, i32},
			{X86_REG_EIZ, i32},

			{X86_REG_RAX, i64},
			{X86_REG_RBP, i64},
			{X86_REG_RBX, i64},
			{X86_REG_RCX, i64},
			{X86_REG_RDI, i64},
			{X86_REG_RDX, i64},
			{X86_REG_RIP, i64},
			{X86_REG_RIZ, i64},
			{X86_REG_RSI, i64},
			{X86_REG_RSP, i64},
			{X86_REG_R8, i64},
			{X86_REG_R9, i64},
			{X86_REG_R10, i64},
			{X86_REG_R11, i64},
			{X86_REG_R12, i64},
			{X86_REG_R13, i64},
			{X86_REG_R14, i64},
			{X86_REG_R15, i64},

			{X86_REG_ST0, fp80},
			{X86_REG_ST1, fp80},
			{X86_REG_ST2, fp80},
			{X86_REG_ST3, fp80},
			{X86_REG_ST4, fp80},
			{X86_REG_ST5, fp80},
			{X86_REG_ST6, fp80},
			{X86_REG_ST7, fp80},

			{X86_REG_FP0, fp64},
			{X86_REG_FP1, fp64},
			{X86_REG_FP2, fp64},
			{X86_REG_FP3, fp64},
			{X86_REG_FP4, fp64},
			{X86_REG_FP5, fp64},
			{X86_REG_FP6, fp64},
			{X86_REG_FP7, fp64},

			{X86_REG_EFLAGS, defTy},
			{X86_REG_DR0, defTy},
			{X86_REG_DR1, defTy},
			{X86_REG_DR2, defTy},
			{X86_REG_DR3, defTy},
			{X86_REG_DR4, defTy},
			{X86_REG_DR5, defTy},
			{X86_REG_DR6, defTy},
			{X86_REG_DR7, defTy},
			{X86_REG_DR8, defTy},
			{X86_REG_DR9, defTy},
			{X86_REG_DR10, defTy},
			{X86_REG_DR11, defTy},
			{X86_REG_DR12, defTy},
			{X86_REG_DR13, defTy},
			{X86_REG_DR14, defTy},
			{X86_REG_DR15, defTy},

			{X86_REG_CR0, defTy},
			{X86_REG_CR1, defTy},
			{X86_REG_CR2, defTy},
			{X86_REG_CR3, defTy},
			{X86_REG_CR4, defTy},
			{X86_REG_CR5, defTy},
			{X86_REG_CR6, defTy},
			{X86_REG_CR7, defTy},
			{X86_REG_CR8, defTy},
			{X86_REG_CR9, defTy},
			{X86_REG_CR10, defTy},
			{X86_REG_CR11, defTy},
			{X86_REG_CR12, defTy},
			{X86_REG_CR13, defTy},
			{X86_REG_CR14, defTy},
			{X86_REG_CR15, defTy},

			{X86_REG_FPSW, defTy},

			// opmask registers (AVX-512)
			{X86_REG_K0, i64},
			{X86_REG_K1, i64},
			{X86_REG_K2, i64},
			{X86_REG_K3, i64},
			{X86_REG_K4, i64},
			{X86_REG_K5, i64},
			{X86_REG_K6, i64},
			{X86_REG_K7, i64},

			// MMX
			{X86_REG_MM0, i64},
			{X86_REG_MM1, i64},
			{X86_REG_MM2, i64},
			{X86_REG_MM3, i64},
			{X86_REG_MM4, i64},
			{X86_REG_MM5, i64},
			{X86_REG_MM6, i64},
			{X86_REG_MM7, i64},

			// XMM
			{X86_REG_XMM0, i128},
			{X86_REG_XMM1, i128},
			{X86_REG_XMM2, i128},
			{X86_REG_XMM3, i128},
			{X86_REG_XMM4, i128},
			{X86_REG_XMM5, i128},
			{X86_REG_XMM6, i128},
			{X86_REG_XMM7, i128},
			{X86_REG_XMM8, i128},
			{X86_REG_XMM9, i128},
			{X86_REG_XMM10, i128},
			{X86_REG_XMM11, i128},
			{X86_REG_XMM12, i128},
			{X86_REG_XMM13, i128},
			{X86_REG_XMM14, i128},
			{X86_REG_XMM15, i128},
			{X86_REG_XMM16, i128},
			{X86_REG_XMM17, i128},
			{X86_REG_XMM18, i128},
			{X86_REG_XMM19, i128},
			{X86_REG_XMM20, i128},
			{X86_REG_XMM21, i128},
			{X86_REG_XMM22, i128},
			{X86_REG_XMM23, i128},
			{X86_REG_XMM24, i128},
			{X86_REG_XMM25, i128},
			{X86_REG_XMM26, i128},
			{X86_REG_XMM27, i128},
			{X86_REG_XMM28, i128},
			{X86_REG_XMM29, i128},
			{X86_REG_XMM30, i128},
			{X86_REG_XMM31, i128},

			// YMM
			{X86_REG_YMM0, i256},
			{X86_REG_YMM1, i256},
			{X86_REG_YMM2, i256},
			{X86_REG_YMM3, i256},
			{X86_REG_YMM4, i256},
			{X86_REG_YMM5, i256},
			{X86_REG_YMM6, i256},
			{X86_REG_YMM7, i256},
			{X86_REG_YMM8, i256},
			{X86_REG_YMM9, i256},
			{X86_REG_YMM10, i256},
			{X86_REG_YMM11, i256},
			{X86_REG_YMM12, i256},
			{X86_REG_YMM13, i256},
			{X86_REG_YMM14, i256},
			{X86_REG_YMM15, i256},
			{X86_REG_YMM16, i256},
			{X86_REG_YMM17, i256},
			{X86_REG_YMM18, i256},
			{X86_REG_YMM19, i256},
			{X86_REG_YMM20, i256},
			{X86_REG_YMM21, i256},
			{X86_REG_YMM22, i256},
			{X86_REG_YMM23, i256},
			{X86_REG_YMM24, i256},
			{X86_REG_YMM25, i256},
			{X86_REG_YMM26, i256},
			{X86_REG_YMM27, i256},
			{X86_REG_YMM28, i256},
			{X86_REG_YMM29, i256},
			{X86_REG_YMM30, i256},
			{X86_REG_YMM31, i256},

			// ZMM
			{X86_REG_ZMM0, i512},
			{X86_REG_ZMM1, i512},
			{X86_REG_ZMM2, i512},
			{X86_REG_ZMM3, i512},
			{X86_REG_ZMM4, i512},
			{X86_REG_ZMM5, i512},
			{X86_REG_ZMM6, i512},
			{X86_REG_ZMM7, i512},
			{X86_REG_ZMM8, i512},
			{X86_REG_ZMM9, i512},
			{X86_REG_ZMM10, i512},
			{X86_REG_ZMM11, i512},
			{X86_REG_ZMM12, i512},
			{X86_REG_ZMM13, i512},
			{X86_REG_ZMM14, i512},
			{X86_REG_ZMM15, i512},
			{X86_REG_ZMM16, i512},
			{X86_REG_ZMM17, i512},
			{X86_REG_ZMM18, i512},
			{X86_REG_ZMM19, i512},
			{X86_REG_ZMM20, i512},
			{X86_REG_ZMM21, i512},
			{X86_REG_ZMM22, i512},
			{X86_REG_ZMM23, i512},
			{X86_REG_ZMM24, i512},
			{X86_REG_ZMM25, i512},
			{X86_REG_ZMM26, i512},
			{X86_REG_ZMM27, i512},
			{X86_REG_ZMM28, i512},
			{X86_REG_ZMM29, i512},
			{X86_REG_ZMM30, i512},
			{X86_REG_ZMM31, i512},

			// x86_reg_rflags
			//
			{X86_REG_CF, i1},
			{X86_REG_PF, i1},
			{X86_REG_AF, i1},
			{X86_REG_ZF, i1},
			{X86_REG_SF, i1},
			{X86_REG_TF, i1},
			{X86_REG_IF, i1},
			{X86_REG_DF, i1},
			{X86_REG_OF, i1},
			{X86_REG_IOPL, i2},
			{X86_REG_NT, i1},
			{X86_REG_RF, i1},
			{X86_REG_VM, i1},
			{X86_REG_AC, i1},
			{X86_REG_VIF, i1},
			{X86_REG_VIP, i1},
			{X86_REG_ID, i1},

			// x87_reg_status
			//
			{X87_REG_IE, i1},
			{X87_REG_DE, i1},
			{X87_REG_ZE, i1},
			{X87_REG_OE, i1},
			{X87_REG_UE, i1},
			{X87_REG_PE, i1},
			{X87_REG_SF, i1},
			{X87_REG_ES, i1},
			{X87_REG_C0, i1},
			{X87_REG_C1, i1},
			{X87_REG_C2, i1},
			{X87_REG_C3, i1},
			{X87_REG_TOP, i3},
			{X87_REG_B, i1},

			// x87_reg_control
			//
			{X87_REG_IM, i1},
			{X87_REG_DM, i1},
			{X87_REG_ZM, i1},
			{X87_REG_OM, i1},
			{X87_REG_UM, i1},
			{X87_REG_PM, i1},
			{X87_REG_PC, i2},
			{X87_REG_RC, i2},
			{X87_REG_X, i1},
	};

	_reg2type = std::move(r2t);
}

void Capstone2LlvmIrTranslatorX86_impl::initializePseudoCallInstructionIDs()
{
	_callInsnIds =
	{
			X86_INS_CALL,
			X86_INS_LCALL,
	};

	_returnInsnIds =
	{
			X86_INS_RET,
			X86_INS_RETF,
			X86_INS_RETFQ
	};

	_branchInsnIds =
	{
			X86_INS_JMP,
			X86_INS_LJMP,
	};

	_condBranchInsnIds =
	{
			X86_INS_JCXZ,
			X86_INS_JECXZ,
			X86_INS_JRCXZ,
			//
			X86_INS_LOOP,
			X86_INS_LOOPE,
			X86_INS_LOOPNE,
			//
			X86_INS_JAE,
			X86_INS_JA,
			X86_INS_JBE,
			X86_INS_JB,
			X86_INS_JE,
			X86_INS_JGE,
			X86_INS_JG,
			X86_INS_JLE,
			X86_INS_JL,
			X86_INS_JNE,
			X86_INS_JNO,
			X86_INS_JNP,
			X86_INS_JNS,
			X86_INS_JO,
			X86_INS_JP,
			X86_INS_JS,
	};

	_controlFlowInsnIds =
	{
			// Currently, all instructions can be categorized based on their
			// IDs alone.
	};
}

//
//==============================================================================
// x86-specific methods.
//==============================================================================
//

void Capstone2LlvmIrTranslatorX86_impl::initializeRegistersParentMapToOther(
		const std::vector<x86_reg>& rs,
		x86_reg other)
{
	for (auto r : rs)
	{
		if (r >= _reg2parentMap.size())
		{
			throw GenericError("Register out of range.");
		}
		_reg2parentMap[r] = other;
	}
}

void Capstone2LlvmIrTranslatorX86_impl::initializeRegistersParentMap()
{
	switch (_origBasicMode)
	{
		case CS_MODE_16: initializeRegistersParentMap16(); break;
		case CS_MODE_32: initializeRegistersParentMap32(); break;
		case CS_MODE_64: initializeRegistersParentMap64(); break;
		default:
		{
			throw GenericError("Unhandled mode in "
					"initializeRegistersParentMap().");
			break;
		}
	}
}

void Capstone2LlvmIrTranslatorX86_impl::initializeRegistersParentMap16()
{
	// Last element in vector is its own parent.
	std::vector<std::vector<x86_reg>> rss =
	{
			{X86_REG_AH, X86_REG_AL, X86_REG_AX},
			{X86_REG_CH, X86_REG_CL, X86_REG_CX},
			{X86_REG_DH, X86_REG_DL, X86_REG_DX},
			{X86_REG_BH, X86_REG_BL, X86_REG_BX},
			{X86_REG_SPL, X86_REG_SP},
			{X86_REG_BPL, X86_REG_BP},
			{X86_REG_SIL, X86_REG_SI},
			{X86_REG_DIL, X86_REG_DI},
			{X86_REG_IP},
	};

	for (std::vector<x86_reg>& rs : rss)
	{
		initializeRegistersParentMapToOther(rs, rs.back());
	}
}

void Capstone2LlvmIrTranslatorX86_impl::initializeRegistersParentMap32()
{
	// Last element in vector is its own parent.
	std::vector<std::vector<x86_reg>> rss =
	{
			{X86_REG_AH, X86_REG_AL, X86_REG_AX, X86_REG_EAX},
			{X86_REG_CH, X86_REG_CL, X86_REG_CX, X86_REG_ECX},
			{X86_REG_DH, X86_REG_DL, X86_REG_DX, X86_REG_EDX},
			{X86_REG_BH, X86_REG_BL, X86_REG_BX, X86_REG_EBX},
			{X86_REG_SPL, X86_REG_SP, X86_REG_ESP},
			{X86_REG_BPL, X86_REG_BP, X86_REG_EBP},
			{X86_REG_SIL, X86_REG_SI, X86_REG_ESI},
			{X86_REG_DIL, X86_REG_DI, X86_REG_EDI},
			{X86_REG_IP, X86_REG_EIP},
			{X86_REG_EIZ},
	};

	for (std::vector<x86_reg>& rs : rss)
	{
		initializeRegistersParentMapToOther(rs, rs.back());
	}
}

void Capstone2LlvmIrTranslatorX86_impl::initializeRegistersParentMap64()
{
	// Last element in vector is its own parent.
	std::vector<std::vector<x86_reg>> rss =
	{
			{X86_REG_AH, X86_REG_AL, X86_REG_AX, X86_REG_EAX, X86_REG_RAX},
			{X86_REG_CH, X86_REG_CL, X86_REG_CX, X86_REG_ECX, X86_REG_RCX},
			{X86_REG_DH, X86_REG_DL, X86_REG_DX, X86_REG_EDX, X86_REG_RDX},
			{X86_REG_BH, X86_REG_BL, X86_REG_BX, X86_REG_EBX, X86_REG_RBX},
			{X86_REG_SPL, X86_REG_SP, X86_REG_ESP, X86_REG_RSP},
			{X86_REG_BPL, X86_REG_BP, X86_REG_EBP, X86_REG_RBP},
			{X86_REG_SIL, X86_REG_SI, X86_REG_ESI, X86_REG_RSI},
			{X86_REG_DIL, X86_REG_DI, X86_REG_EDI, X86_REG_RDI},
			{X86_REG_IP, X86_REG_EIP, X86_REG_RIP},
			{X86_REG_EIZ, X86_REG_RIZ},
			{X86_REG_R8B, X86_REG_R8W, X86_REG_R8D, X86_REG_R8},
			{X86_REG_R9B, X86_REG_R9W, X86_REG_R9D, X86_REG_R9},
			{X86_REG_R10B, X86_REG_R10W, X86_REG_R10D, X86_REG_R10},
			{X86_REG_R11B, X86_REG_R11W, X86_REG_R11D, X86_REG_R11},
			{X86_REG_R12B, X86_REG_R12W, X86_REG_R12D, X86_REG_R12},
			{X86_REG_R13B, X86_REG_R13W, X86_REG_R13D, X86_REG_R13},
			{X86_REG_R14B, X86_REG_R14W, X86_REG_R14D, X86_REG_R14},
			{X86_REG_R15B, X86_REG_R15W, X86_REG_R15D, X86_REG_R15}
	};

	for (std::vector<x86_reg>& rs : rss)
	{
		initializeRegistersParentMapToOther(rs, rs.back());
	}
}

//
//==============================================================================
// Instruction translation map initialization.
//==============================================================================
//

std::map<
	std::size_t,
	void (Capstone2LlvmIrTranslatorX86_impl::*)(
			cs_insn* i,
			cs_x86*,
			llvm::IRBuilder<>&)>
Capstone2LlvmIrTranslatorX86_impl::_i2fm =
{
		{X86_INS_INVALID, nullptr},

		{X86_INS_AAA, &Capstone2LlvmIrTranslatorX86_impl::translateAaa},
		{X86_INS_AAD, &Capstone2LlvmIrTranslatorX86_impl::translateAad},
		{X86_INS_AAM, &Capstone2LlvmIrTranslatorX86_impl::translateAam},
		{X86_INS_AAS, &Capstone2LlvmIrTranslatorX86_impl::translateAaa},
		{X86_INS_FABS, &Capstone2LlvmIrTranslatorX86_impl::translateFabs},
		{X86_INS_ADC, &Capstone2LlvmIrTranslatorX86_impl::translateAdc},
		{X86_INS_ADCX, &Capstone2LlvmIrTranslatorX86_impl::translateAdc},
		{X86_INS_ADD, &Capstone2LlvmIrTranslatorX86_impl::translateAdd},
		{X86_INS_ADDPD, nullptr},
		{X86_INS_ADDPS, nullptr},
		{X86_INS_ADDSD, nullptr},
		{X86_INS_ADDSS, nullptr},
		{X86_INS_ADDSUBPD, nullptr},
		{X86_INS_ADDSUBPS, nullptr},
		{X86_INS_FADD, &Capstone2LlvmIrTranslatorX86_impl::translateFadd},
		{X86_INS_FIADD, &Capstone2LlvmIrTranslatorX86_impl::translateFadd},
		{X86_INS_FADDP, &Capstone2LlvmIrTranslatorX86_impl::translateFadd},
		{X86_INS_ADOX, &Capstone2LlvmIrTranslatorX86_impl::translateAdc},
		{X86_INS_AESDECLAST, nullptr},
		{X86_INS_AESDEC, nullptr},
		{X86_INS_AESENCLAST, nullptr},
		{X86_INS_AESENC, nullptr},
		{X86_INS_AESIMC, nullptr},
		{X86_INS_AESKEYGENASSIST, nullptr},
		{X86_INS_AND, &Capstone2LlvmIrTranslatorX86_impl::translateAnd},
		{X86_INS_ANDN, nullptr},
		{X86_INS_ANDNPD, nullptr},
		{X86_INS_ANDNPS, nullptr},
		{X86_INS_ANDPD, nullptr},
		{X86_INS_ANDPS, nullptr},
		{X86_INS_ARPL, nullptr},
		{X86_INS_BEXTR, nullptr},
		{X86_INS_BLCFILL, nullptr},
		{X86_INS_BLCI, nullptr},
		{X86_INS_BLCIC, nullptr},
		{X86_INS_BLCMSK, nullptr},
		{X86_INS_BLCS, nullptr},
		{X86_INS_BLENDPD, nullptr},
		{X86_INS_BLENDPS, nullptr},
		{X86_INS_BLENDVPD, nullptr},
		{X86_INS_BLENDVPS, nullptr},
		{X86_INS_BLSFILL, nullptr},
		{X86_INS_BLSI, nullptr},
		{X86_INS_BLSIC, nullptr},
		{X86_INS_BLSMSK, nullptr},
		{X86_INS_BLSR, nullptr},
		{X86_INS_BOUND, nullptr},
		{X86_INS_BSF, &Capstone2LlvmIrTranslatorX86_impl::translateBsf},
		{X86_INS_BSR, &Capstone2LlvmIrTranslatorX86_impl::translateBsf},
		{X86_INS_BSWAP, &Capstone2LlvmIrTranslatorX86_impl::translateBswap},
		{X86_INS_BT, &Capstone2LlvmIrTranslatorX86_impl::translateBt},
		{X86_INS_BTC, &Capstone2LlvmIrTranslatorX86_impl::translateBtc},
		{X86_INS_BTR, &Capstone2LlvmIrTranslatorX86_impl::translateBtr},
		{X86_INS_BTS, &Capstone2LlvmIrTranslatorX86_impl::translateBts},
		{X86_INS_BZHI, nullptr},
		{X86_INS_CALL, &Capstone2LlvmIrTranslatorX86_impl::translateCall},
		{X86_INS_CBW, &Capstone2LlvmIrTranslatorX86_impl::translateCbw},
		{X86_INS_CDQ, &Capstone2LlvmIrTranslatorX86_impl::translateCdq},
		{X86_INS_CDQE, &Capstone2LlvmIrTranslatorX86_impl::translateCdqe},
		{X86_INS_FCHS, &Capstone2LlvmIrTranslatorX86_impl::translateFchs},
		{X86_INS_CLAC, nullptr},
		{X86_INS_CLC, &Capstone2LlvmIrTranslatorX86_impl::translateClc},
		{X86_INS_CLD, &Capstone2LlvmIrTranslatorX86_impl::translateCld},
		{X86_INS_CLFLUSH, nullptr},
		{X86_INS_CLFLUSHOPT, nullptr},
		{X86_INS_CLGI, nullptr},
		{X86_INS_CLI, &Capstone2LlvmIrTranslatorX86_impl::translateCli},
		{X86_INS_CLTS, nullptr},
		{X86_INS_CLWB, nullptr},
		{X86_INS_CMC, &Capstone2LlvmIrTranslatorX86_impl::translateCmc},
		{X86_INS_CMOVA, &Capstone2LlvmIrTranslatorX86_impl::translateCMovCc},
		{X86_INS_CMOVAE, &Capstone2LlvmIrTranslatorX86_impl::translateCMovCc},
		{X86_INS_CMOVB, &Capstone2LlvmIrTranslatorX86_impl::translateCMovCc},
		{X86_INS_CMOVBE, &Capstone2LlvmIrTranslatorX86_impl::translateCMovCc},
		{X86_INS_FCMOVBE, &Capstone2LlvmIrTranslatorX86_impl::translateFCMovCc},
		{X86_INS_FCMOVB, &Capstone2LlvmIrTranslatorX86_impl::translateFCMovCc},
		{X86_INS_CMOVE, &Capstone2LlvmIrTranslatorX86_impl::translateCMovCc},
		{X86_INS_FCMOVE, &Capstone2LlvmIrTranslatorX86_impl::translateFCMovCc},
		{X86_INS_CMOVG, &Capstone2LlvmIrTranslatorX86_impl::translateCMovCc},
		{X86_INS_CMOVGE, &Capstone2LlvmIrTranslatorX86_impl::translateCMovCc},
		{X86_INS_CMOVL, &Capstone2LlvmIrTranslatorX86_impl::translateCMovCc},
		{X86_INS_CMOVLE, &Capstone2LlvmIrTranslatorX86_impl::translateCMovCc},
		{X86_INS_FCMOVNBE, &Capstone2LlvmIrTranslatorX86_impl::translateFCMovCc},
		{X86_INS_FCMOVNB, &Capstone2LlvmIrTranslatorX86_impl::translateFCMovCc},
		{X86_INS_CMOVNE, &Capstone2LlvmIrTranslatorX86_impl::translateCMovCc},
		{X86_INS_FCMOVNE, &Capstone2LlvmIrTranslatorX86_impl::translateFCMovCc},
		{X86_INS_CMOVNO, &Capstone2LlvmIrTranslatorX86_impl::translateCMovCc},
		{X86_INS_CMOVNP, &Capstone2LlvmIrTranslatorX86_impl::translateCMovCc},
		{X86_INS_FCMOVNU, &Capstone2LlvmIrTranslatorX86_impl::translateFCMovCc},
		{X86_INS_CMOVNS, &Capstone2LlvmIrTranslatorX86_impl::translateCMovCc},
		{X86_INS_CMOVO, &Capstone2LlvmIrTranslatorX86_impl::translateCMovCc},
		{X86_INS_CMOVP, &Capstone2LlvmIrTranslatorX86_impl::translateCMovCc},
		{X86_INS_FCMOVU, &Capstone2LlvmIrTranslatorX86_impl::translateFCMovCc},
		{X86_INS_CMOVS, &Capstone2LlvmIrTranslatorX86_impl::translateCMovCc},
		{X86_INS_CMP, &Capstone2LlvmIrTranslatorX86_impl::translateSub},
		{X86_INS_CMPSB, &Capstone2LlvmIrTranslatorX86_impl::translateCompareString},
		{X86_INS_CMPSQ, &Capstone2LlvmIrTranslatorX86_impl::translateCompareString},
		{X86_INS_CMPSW, &Capstone2LlvmIrTranslatorX86_impl::translateCompareString},
		{X86_INS_CMPXCHG16B, &Capstone2LlvmIrTranslatorX86_impl::translateCmpxchg16b},
		{X86_INS_CMPXCHG, &Capstone2LlvmIrTranslatorX86_impl::translateCmpxchg},
		{X86_INS_CMPXCHG8B, &Capstone2LlvmIrTranslatorX86_impl::translateCmpxchg8b},
		{X86_INS_COMISD, nullptr},
		{X86_INS_COMISS, nullptr},
		{X86_INS_FCOMP, &Capstone2LlvmIrTranslatorX86_impl::translateFucomPop},
		{X86_INS_FCOMIP, &Capstone2LlvmIrTranslatorX86_impl::translateFucomPop},
		{X86_INS_FCOMI, &Capstone2LlvmIrTranslatorX86_impl::translateFucomPop},
		{X86_INS_FCOM, &Capstone2LlvmIrTranslatorX86_impl::translateFucomPop},
		{X86_INS_FCOS, &Capstone2LlvmIrTranslatorX86_impl::translateFcos},
		{X86_INS_CPUID, &Capstone2LlvmIrTranslatorX86_impl::translateCpuid},
		{X86_INS_CQO, &Capstone2LlvmIrTranslatorX86_impl::translateCqo},
		{X86_INS_CRC32, nullptr},
		{X86_INS_CVTDQ2PD, nullptr},
		{X86_INS_CVTDQ2PS, nullptr},
		{X86_INS_CVTPD2DQ, nullptr},
		{X86_INS_CVTPD2PS, nullptr},
		{X86_INS_CVTPS2DQ, nullptr},
		{X86_INS_CVTPS2PD, nullptr},
		{X86_INS_CVTSD2SI, nullptr},
		{X86_INS_CVTSD2SS, nullptr},
		{X86_INS_CVTSI2SD, nullptr},
		{X86_INS_CVTSI2SS, nullptr},
		{X86_INS_CVTSS2SD, nullptr},
		{X86_INS_CVTSS2SI, nullptr},
		{X86_INS_CVTTPD2DQ, nullptr},
		{X86_INS_CVTTPS2DQ, nullptr},
		{X86_INS_CVTTSD2SI, nullptr},
		{X86_INS_CVTTSS2SI, nullptr},
		{X86_INS_CWD, &Capstone2LlvmIrTranslatorX86_impl::translateCwd},
		{X86_INS_CWDE, &Capstone2LlvmIrTranslatorX86_impl::translateCwde},
		{X86_INS_DAA, &Capstone2LlvmIrTranslatorX86_impl::translateDaaDas},
		{X86_INS_DAS, &Capstone2LlvmIrTranslatorX86_impl::translateDaaDas},
		{X86_INS_DATA16, nullptr},
		{X86_INS_DEC, &Capstone2LlvmIrTranslatorX86_impl::translateDec},
		{X86_INS_DIV, &Capstone2LlvmIrTranslatorX86_impl::translateDiv},
		{X86_INS_DIVPD, nullptr},
		{X86_INS_DIVPS, nullptr},
		{X86_INS_FDIVR, &Capstone2LlvmIrTranslatorX86_impl::translateFdivr},
		{X86_INS_FIDIVR, &Capstone2LlvmIrTranslatorX86_impl::translateFdivr},
		{X86_INS_FDIVRP, &Capstone2LlvmIrTranslatorX86_impl::translateFdivr},
		{X86_INS_DIVSD, nullptr},
		{X86_INS_DIVSS, nullptr},
		{X86_INS_FDIV, &Capstone2LlvmIrTranslatorX86_impl::translateFdiv},
		{X86_INS_FIDIV, &Capstone2LlvmIrTranslatorX86_impl::translateFdiv},
		{X86_INS_FDIVP, &Capstone2LlvmIrTranslatorX86_impl::translateFdiv},
		{X86_INS_DPPD, nullptr},
		{X86_INS_DPPS, nullptr},
		{X86_INS_RET, &Capstone2LlvmIrTranslatorX86_impl::translateRet},
		{X86_INS_ENCLS, nullptr},
		{X86_INS_ENCLU, nullptr},
		{X86_INS_ENTER, &Capstone2LlvmIrTranslatorX86_impl::translateEnter},
		{X86_INS_EXTRACTPS, nullptr},
		{X86_INS_EXTRQ, nullptr},
		{X86_INS_F2XM1, &Capstone2LlvmIrTranslatorX86_impl::translateF2xm1},
		{X86_INS_LCALL, &Capstone2LlvmIrTranslatorX86_impl::translateLcall},
		{X86_INS_LJMP, &Capstone2LlvmIrTranslatorX86_impl::translateLjmp},
		{X86_INS_FBLD, &Capstone2LlvmIrTranslatorX86_impl::translateFbld},
		{X86_INS_FBSTP, &Capstone2LlvmIrTranslatorX86_impl::translateFbstp},
		{X86_INS_FCOMPP, &Capstone2LlvmIrTranslatorX86_impl::translateFucomPop},
		{X86_INS_FDECSTP, &Capstone2LlvmIrTranslatorX86_impl::translateFdecstp},
		{X86_INS_FEMMS, nullptr},
		{X86_INS_FFREE, &Capstone2LlvmIrTranslatorX86_impl::translateFfree},
		{X86_INS_FICOM, &Capstone2LlvmIrTranslatorX86_impl::translateFucomPop},
		{X86_INS_FICOMP, &Capstone2LlvmIrTranslatorX86_impl::translateFucomPop},
		{X86_INS_FINCSTP, &Capstone2LlvmIrTranslatorX86_impl::translateFincstp},
		{X86_INS_FLDCW, &Capstone2LlvmIrTranslatorX86_impl::translateNop},
		{X86_INS_FLDENV, &Capstone2LlvmIrTranslatorX86_impl::translatePseudoAsmFncOp0},
		{X86_INS_FLDL2E, &Capstone2LlvmIrTranslatorX86_impl::translateFloadConstant},
		{X86_INS_FLDL2T, &Capstone2LlvmIrTranslatorX86_impl::translateFloadConstant},
		{X86_INS_FLDLG2, &Capstone2LlvmIrTranslatorX86_impl::translateFloadConstant},
		{X86_INS_FLDLN2, &Capstone2LlvmIrTranslatorX86_impl::translateFloadConstant},
		{X86_INS_FLDPI, &Capstone2LlvmIrTranslatorX86_impl::translateFloadConstant},
		{X86_INS_FNCLEX, &Capstone2LlvmIrTranslatorX86_impl::translateFnclex},
		{X86_INS_FNINIT, &Capstone2LlvmIrTranslatorX86_impl::translateFninit},
		{X86_INS_FNOP, &Capstone2LlvmIrTranslatorX86_impl::translateNop},
		{X86_INS_FNSTCW, &Capstone2LlvmIrTranslatorX86_impl::translateNop},
		{X86_INS_FNSTSW, &Capstone2LlvmIrTranslatorX86_impl::translateFnstsw},
		{X86_INS_FPATAN, &Capstone2LlvmIrTranslatorX86_impl::translateFatan},
		{X86_INS_FPREM, &Capstone2LlvmIrTranslatorX86_impl::translateFprem},
		{X86_INS_FPREM1, &Capstone2LlvmIrTranslatorX86_impl::translateFprem},
		{X86_INS_FPTAN, &Capstone2LlvmIrTranslatorX86_impl::translateFtan},
		{X86_INS_FFREEP, nullptr},
		{X86_INS_FRNDINT, &Capstone2LlvmIrTranslatorX86_impl::translateFrndint},
		{X86_INS_FRSTOR, &Capstone2LlvmIrTranslatorX86_impl::translateFrstor},
		{X86_INS_FNSAVE, &Capstone2LlvmIrTranslatorX86_impl::translateFnsave},
		{X86_INS_FSCALE, &Capstone2LlvmIrTranslatorX86_impl::translateFscale},
		{X86_INS_FSETPM, nullptr},
		{X86_INS_FSINCOS, &Capstone2LlvmIrTranslatorX86_impl::translateFsincos},
		{X86_INS_FNSTENV, &Capstone2LlvmIrTranslatorX86_impl::translateFnstenv},
		{X86_INS_FXAM, &Capstone2LlvmIrTranslatorX86_impl::translateFxam},
		{X86_INS_FXRSTOR, &Capstone2LlvmIrTranslatorX86_impl::translateFxstor},
		{X86_INS_FXRSTOR64, &Capstone2LlvmIrTranslatorX86_impl::translateFxstor},
		{X86_INS_FXSAVE, &Capstone2LlvmIrTranslatorX86_impl::translateFxsave},
		{X86_INS_FXSAVE64, &Capstone2LlvmIrTranslatorX86_impl::translateFxsave},
		{X86_INS_FXTRACT, &Capstone2LlvmIrTranslatorX86_impl::translateFxtract},
		{X86_INS_FYL2X, &Capstone2LlvmIrTranslatorX86_impl::translateFyl2x},
		{X86_INS_FYL2XP1, &Capstone2LlvmIrTranslatorX86_impl::translateFyl2x},
		{X86_INS_MOVAPD, nullptr},
		{X86_INS_MOVAPS, nullptr},
		{X86_INS_ORPD, nullptr},
		{X86_INS_ORPS, nullptr},
		{X86_INS_VMOVAPD, nullptr},
		{X86_INS_VMOVAPS, nullptr},
		{X86_INS_XORPD, nullptr},
		{X86_INS_XORPS, nullptr},
		{X86_INS_GETSEC, nullptr},
		{X86_INS_HADDPD, nullptr},
		{X86_INS_HADDPS, nullptr},
		{X86_INS_HLT, nullptr},
		{X86_INS_HSUBPD, nullptr},
		{X86_INS_HSUBPS, nullptr},
		{X86_INS_IDIV, &Capstone2LlvmIrTranslatorX86_impl::translateDiv},
		{X86_INS_FILD, &Capstone2LlvmIrTranslatorX86_impl::translateFld},
		{X86_INS_IMUL, &Capstone2LlvmIrTranslatorX86_impl::translateImul},
		{X86_INS_IN, nullptr},
		{X86_INS_INC, &Capstone2LlvmIrTranslatorX86_impl::translateInc},
		{X86_INS_INSB, &Capstone2LlvmIrTranslatorX86_impl::translateIns},
		{X86_INS_INSERTPS, nullptr},
		{X86_INS_INSERTQ, nullptr},
		{X86_INS_INSD, &Capstone2LlvmIrTranslatorX86_impl::translateIns},
		{X86_INS_INSW, &Capstone2LlvmIrTranslatorX86_impl::translateIns},
		{X86_INS_INT, nullptr},
		{X86_INS_INT1, nullptr},
		{X86_INS_INT3, nullptr},
		{X86_INS_INTO, nullptr},
		{X86_INS_INVD, nullptr},
		{X86_INS_INVEPT, nullptr},
		{X86_INS_INVLPG, nullptr},
		{X86_INS_INVLPGA, nullptr},
		{X86_INS_INVPCID, nullptr},
		{X86_INS_INVVPID, nullptr},
		{X86_INS_IRET, nullptr},
		{X86_INS_IRETD, nullptr},
		{X86_INS_IRETQ, nullptr},
		{X86_INS_FISTTP, &Capstone2LlvmIrTranslatorX86_impl::translateFist},
		{X86_INS_FIST, &Capstone2LlvmIrTranslatorX86_impl::translateFist},
		{X86_INS_FISTP, &Capstone2LlvmIrTranslatorX86_impl::translateFist},
		{X86_INS_UCOMISD, nullptr},
		{X86_INS_UCOMISS, nullptr},
		{X86_INS_VCOMISD, nullptr},
		{X86_INS_VCOMISS, nullptr},
		{X86_INS_VCVTSD2SS, nullptr},
		{X86_INS_VCVTSI2SD, nullptr},
		{X86_INS_VCVTSI2SS, nullptr},
		{X86_INS_VCVTSS2SD, nullptr},
		{X86_INS_VCVTTSD2SI, nullptr},
		{X86_INS_VCVTTSD2USI, nullptr},
		{X86_INS_VCVTTSS2SI, nullptr},
		{X86_INS_VCVTTSS2USI, nullptr},
		{X86_INS_VCVTUSI2SD, nullptr},
		{X86_INS_VCVTUSI2SS, nullptr},
		{X86_INS_VUCOMISD, nullptr},
		{X86_INS_VUCOMISS, nullptr},
		{X86_INS_JAE, &Capstone2LlvmIrTranslatorX86_impl::translateJCc},
		{X86_INS_JA, &Capstone2LlvmIrTranslatorX86_impl::translateJCc},
		{X86_INS_JBE, &Capstone2LlvmIrTranslatorX86_impl::translateJCc},
		{X86_INS_JB, &Capstone2LlvmIrTranslatorX86_impl::translateJCc},
		{X86_INS_JCXZ, &Capstone2LlvmIrTranslatorX86_impl::translateJecxz},
		{X86_INS_JECXZ, &Capstone2LlvmIrTranslatorX86_impl::translateJecxz},
		{X86_INS_JE, &Capstone2LlvmIrTranslatorX86_impl::translateJCc},
		{X86_INS_JGE, &Capstone2LlvmIrTranslatorX86_impl::translateJCc},
		{X86_INS_JG, &Capstone2LlvmIrTranslatorX86_impl::translateJCc},
		{X86_INS_JLE, &Capstone2LlvmIrTranslatorX86_impl::translateJCc},
		{X86_INS_JL, &Capstone2LlvmIrTranslatorX86_impl::translateJCc},
		{X86_INS_JMP, &Capstone2LlvmIrTranslatorX86_impl::translateJmp},
		{X86_INS_JNE, &Capstone2LlvmIrTranslatorX86_impl::translateJCc},
		{X86_INS_JNO, &Capstone2LlvmIrTranslatorX86_impl::translateJCc},
		{X86_INS_JNP, &Capstone2LlvmIrTranslatorX86_impl::translateJCc},
		{X86_INS_JNS, &Capstone2LlvmIrTranslatorX86_impl::translateJCc},
		{X86_INS_JO, &Capstone2LlvmIrTranslatorX86_impl::translateJCc},
		{X86_INS_JP, &Capstone2LlvmIrTranslatorX86_impl::translateJCc},
		{X86_INS_JRCXZ, &Capstone2LlvmIrTranslatorX86_impl::translateJecxz},
		{X86_INS_JS, &Capstone2LlvmIrTranslatorX86_impl::translateJCc},
		{X86_INS_KANDB, nullptr},
		{X86_INS_KANDD, nullptr},
		{X86_INS_KANDNB, nullptr},
		{X86_INS_KANDND, nullptr},
		{X86_INS_KANDNQ, nullptr},
		{X86_INS_KANDNW, nullptr},
		{X86_INS_KANDQ, nullptr},
		{X86_INS_KANDW, nullptr},
		{X86_INS_KMOVB, nullptr},
		{X86_INS_KMOVD, nullptr},
		{X86_INS_KMOVQ, nullptr},
		{X86_INS_KMOVW, nullptr},
		{X86_INS_KNOTB, nullptr},
		{X86_INS_KNOTD, nullptr},
		{X86_INS_KNOTQ, nullptr},
		{X86_INS_KNOTW, nullptr},
		{X86_INS_KORB, nullptr},
		{X86_INS_KORD, nullptr},
		{X86_INS_KORQ, nullptr},
		{X86_INS_KORTESTB, nullptr},
		{X86_INS_KORTESTD, nullptr},
		{X86_INS_KORTESTQ, nullptr},
		{X86_INS_KORTESTW, nullptr},
		{X86_INS_KORW, nullptr},
		{X86_INS_KSHIFTLB, nullptr},
		{X86_INS_KSHIFTLD, nullptr},
		{X86_INS_KSHIFTLQ, nullptr},
		{X86_INS_KSHIFTLW, nullptr},
		{X86_INS_KSHIFTRB, nullptr},
		{X86_INS_KSHIFTRD, nullptr},
		{X86_INS_KSHIFTRQ, nullptr},
		{X86_INS_KSHIFTRW, nullptr},
		{X86_INS_KUNPCKBW, nullptr},
		{X86_INS_KXNORB, nullptr},
		{X86_INS_KXNORD, nullptr},
		{X86_INS_KXNORQ, nullptr},
		{X86_INS_KXNORW, nullptr},
		{X86_INS_KXORB, nullptr},
		{X86_INS_KXORD, nullptr},
		{X86_INS_KXORQ, nullptr},
		{X86_INS_KXORW, nullptr},
		{X86_INS_LAHF, &Capstone2LlvmIrTranslatorX86_impl::translateLahf},
		{X86_INS_LAR, nullptr},
		{X86_INS_LDDQU, nullptr},
		{X86_INS_LDMXCSR, nullptr},
		{X86_INS_LDS, &Capstone2LlvmIrTranslatorX86_impl::translateLoadFarPtr},
		{X86_INS_FLDZ, &Capstone2LlvmIrTranslatorX86_impl::translateFloadConstant},
		{X86_INS_FLD1, &Capstone2LlvmIrTranslatorX86_impl::translateFloadConstant},
		{X86_INS_FLD, &Capstone2LlvmIrTranslatorX86_impl::translateFld},
		{X86_INS_LEA, &Capstone2LlvmIrTranslatorX86_impl::translateLea},
		{X86_INS_LEAVE, &Capstone2LlvmIrTranslatorX86_impl::translateLeave},
		{X86_INS_LES, &Capstone2LlvmIrTranslatorX86_impl::translateLoadFarPtr},
		{X86_INS_LFENCE, nullptr},
		{X86_INS_LFS, &Capstone2LlvmIrTranslatorX86_impl::translateLoadFarPtr},
		{X86_INS_LGDT, nullptr},
		{X86_INS_LGS, &Capstone2LlvmIrTranslatorX86_impl::translateLoadFarPtr},
		{X86_INS_LIDT, nullptr},
		{X86_INS_LLDT, nullptr},
		{X86_INS_LMSW, nullptr},
		{X86_INS_OR, &Capstone2LlvmIrTranslatorX86_impl::translateOr},
		{X86_INS_SUB, &Capstone2LlvmIrTranslatorX86_impl::translateSub},
		{X86_INS_XOR, &Capstone2LlvmIrTranslatorX86_impl::translateXor},
		{X86_INS_LODSB, &Capstone2LlvmIrTranslatorX86_impl::translateLoadString},
		{X86_INS_LODSD, &Capstone2LlvmIrTranslatorX86_impl::translateLoadString},
		{X86_INS_LODSQ, &Capstone2LlvmIrTranslatorX86_impl::translateLoadString},
		{X86_INS_LODSW, &Capstone2LlvmIrTranslatorX86_impl::translateLoadString},
		{X86_INS_LOOP, &Capstone2LlvmIrTranslatorX86_impl::translateLoop},
		{X86_INS_LOOPE, &Capstone2LlvmIrTranslatorX86_impl::translateLoop},
		{X86_INS_LOOPNE, &Capstone2LlvmIrTranslatorX86_impl::translateLoop},
		{X86_INS_RETF, &Capstone2LlvmIrTranslatorX86_impl::translateRet},
		{X86_INS_RETFQ, &Capstone2LlvmIrTranslatorX86_impl::translateRet},
		{X86_INS_LSL, nullptr},
		{X86_INS_LSS, &Capstone2LlvmIrTranslatorX86_impl::translateLoadFarPtr},
		{X86_INS_LTR, nullptr},
		{X86_INS_XADD, &Capstone2LlvmIrTranslatorX86_impl::translateAdd},
		{X86_INS_LZCNT, nullptr},
		{X86_INS_MASKMOVDQU, nullptr},
		{X86_INS_MAXPD, nullptr},
		{X86_INS_MAXPS, nullptr},
		{X86_INS_MAXSD, nullptr},
		{X86_INS_MAXSS, nullptr},
		{X86_INS_MFENCE, nullptr},
		{X86_INS_MINPD, nullptr},
		{X86_INS_MINPS, nullptr},
		{X86_INS_MINSD, nullptr},
		{X86_INS_MINSS, nullptr},
		{X86_INS_CVTPD2PI, nullptr},
		{X86_INS_CVTPI2PD, nullptr},
		{X86_INS_CVTPI2PS, nullptr},
		{X86_INS_CVTPS2PI, nullptr},
		{X86_INS_CVTTPD2PI, nullptr},
		{X86_INS_CVTTPS2PI, nullptr},
		{X86_INS_EMMS, nullptr},
		{X86_INS_MASKMOVQ, nullptr},
		{X86_INS_MOVD, nullptr},
		{X86_INS_MOVDQ2Q, nullptr},
		{X86_INS_MOVNTQ, nullptr},
		{X86_INS_MOVQ2DQ, nullptr},
		{X86_INS_MOVQ, nullptr},
		{X86_INS_PABSB, nullptr},
		{X86_INS_PABSD, nullptr},
		{X86_INS_PABSW, nullptr},
		{X86_INS_PACKSSDW, nullptr},
		{X86_INS_PACKSSWB, nullptr},
		{X86_INS_PACKUSWB, nullptr},
		{X86_INS_PADDB, nullptr},
		{X86_INS_PADDD, nullptr},
		{X86_INS_PADDQ, nullptr},
		{X86_INS_PADDSB, nullptr},
		{X86_INS_PADDSW, nullptr},
		{X86_INS_PADDUSB, nullptr},
		{X86_INS_PADDUSW, nullptr},
		{X86_INS_PADDW, nullptr},
		{X86_INS_PALIGNR, nullptr},
		{X86_INS_PANDN, nullptr},
		{X86_INS_PAND, nullptr},
		{X86_INS_PAVGB, nullptr},
		{X86_INS_PAVGW, nullptr},
		{X86_INS_PCMPEQB, nullptr},
		{X86_INS_PCMPEQD, nullptr},
		{X86_INS_PCMPEQW, nullptr},
		{X86_INS_PCMPGTB, nullptr},
		{X86_INS_PCMPGTD, nullptr},
		{X86_INS_PCMPGTW, nullptr},
		{X86_INS_PEXTRW, nullptr},
		{X86_INS_PHADDSW, nullptr},
		{X86_INS_PHADDW, nullptr},
		{X86_INS_PHADDD, nullptr},
		{X86_INS_PHSUBD, nullptr},
		{X86_INS_PHSUBSW, nullptr},
		{X86_INS_PHSUBW, nullptr},
		{X86_INS_PINSRW, nullptr},
		{X86_INS_PMADDUBSW, nullptr},
		{X86_INS_PMADDWD, nullptr},
		{X86_INS_PMAXSW, nullptr},
		{X86_INS_PMAXUB, nullptr},
		{X86_INS_PMINSW, nullptr},
		{X86_INS_PMINUB, nullptr},
		{X86_INS_PMOVMSKB, nullptr},
		{X86_INS_PMULHRSW, nullptr},
		{X86_INS_PMULHUW, nullptr},
		{X86_INS_PMULHW, nullptr},
		{X86_INS_PMULLW, nullptr},
		{X86_INS_PMULUDQ, nullptr},
		{X86_INS_POR, nullptr},
		{X86_INS_PSADBW, nullptr},
		{X86_INS_PSHUFB, nullptr},
		{X86_INS_PSHUFW, nullptr},
		{X86_INS_PSIGNB, nullptr},
		{X86_INS_PSIGND, nullptr},
		{X86_INS_PSIGNW, nullptr},
		{X86_INS_PSLLD, nullptr},
		{X86_INS_PSLLQ, nullptr},
		{X86_INS_PSLLW, nullptr},
		{X86_INS_PSRAD, nullptr},
		{X86_INS_PSRAW, nullptr},
		{X86_INS_PSRLD, nullptr},
		{X86_INS_PSRLQ, nullptr},
		{X86_INS_PSRLW, nullptr},
		{X86_INS_PSUBB, nullptr},
		{X86_INS_PSUBD, nullptr},
		{X86_INS_PSUBQ, nullptr},
		{X86_INS_PSUBSB, nullptr},
		{X86_INS_PSUBSW, nullptr},
		{X86_INS_PSUBUSB, nullptr},
		{X86_INS_PSUBUSW, nullptr},
		{X86_INS_PSUBW, nullptr},
		{X86_INS_PUNPCKHBW, nullptr},
		{X86_INS_PUNPCKHDQ, nullptr},
		{X86_INS_PUNPCKHWD, nullptr},
		{X86_INS_PUNPCKLBW, nullptr},
		{X86_INS_PUNPCKLDQ, nullptr},
		{X86_INS_PUNPCKLWD, nullptr},
		{X86_INS_PXOR, nullptr},
		{X86_INS_MONITOR, nullptr},
		{X86_INS_MONTMUL, nullptr},
		{X86_INS_MOV, &Capstone2LlvmIrTranslatorX86_impl::translateMov},
		{X86_INS_MOVABS, &Capstone2LlvmIrTranslatorX86_impl::translateMov},
		{X86_INS_MOVBE, nullptr},
		{X86_INS_MOVDDUP, nullptr},
		{X86_INS_MOVDQA, nullptr},
		{X86_INS_MOVDQU, nullptr},
		{X86_INS_MOVHLPS, nullptr},
		{X86_INS_MOVHPD, nullptr},
		{X86_INS_MOVHPS, nullptr},
		{X86_INS_MOVLHPS, nullptr},
		{X86_INS_MOVLPD, nullptr},
		{X86_INS_MOVLPS, nullptr},
		{X86_INS_MOVMSKPD, nullptr},
		{X86_INS_MOVMSKPS, nullptr},
		{X86_INS_MOVNTDQA, nullptr},
		{X86_INS_MOVNTDQ, nullptr},
		{X86_INS_MOVNTI, nullptr},
		{X86_INS_MOVNTPD, nullptr},
		{X86_INS_MOVNTPS, nullptr},
		{X86_INS_MOVNTSD, nullptr},
		{X86_INS_MOVNTSS, nullptr},
		{X86_INS_MOVSB, &Capstone2LlvmIrTranslatorX86_impl::translateMoveString},
		{X86_INS_MOVSD, &Capstone2LlvmIrTranslatorX86_impl::translateMoveString},
		{X86_INS_MOVSHDUP, nullptr},
		{X86_INS_MOVSLDUP, nullptr},
		{X86_INS_MOVSQ, &Capstone2LlvmIrTranslatorX86_impl::translateMoveString},
		{X86_INS_MOVSS, nullptr},
		{X86_INS_MOVSW, &Capstone2LlvmIrTranslatorX86_impl::translateMoveString},
		{X86_INS_MOVSX, &Capstone2LlvmIrTranslatorX86_impl::translateMov},
		{X86_INS_MOVSXD, &Capstone2LlvmIrTranslatorX86_impl::translateMov},
		{X86_INS_MOVUPD, nullptr},
		{X86_INS_MOVUPS, nullptr},
		{X86_INS_MOVZX, &Capstone2LlvmIrTranslatorX86_impl::translateMov},
		{X86_INS_MPSADBW, nullptr},
		{X86_INS_MUL, &Capstone2LlvmIrTranslatorX86_impl::translateMul},
		{X86_INS_MULPD, nullptr},
		{X86_INS_MULPS, nullptr},
		{X86_INS_MULSD, nullptr},
		{X86_INS_MULSS, nullptr},
		{X86_INS_MULX, nullptr},
		{X86_INS_FMUL, &Capstone2LlvmIrTranslatorX86_impl::translateFmul},
		{X86_INS_FIMUL, &Capstone2LlvmIrTranslatorX86_impl::translateFmul},
		{X86_INS_FMULP, &Capstone2LlvmIrTranslatorX86_impl::translateFmul},
		{X86_INS_MWAIT, nullptr},
		{X86_INS_NEG, &Capstone2LlvmIrTranslatorX86_impl::translateNeg},
		{X86_INS_NOP, &Capstone2LlvmIrTranslatorX86_impl::translateNop},
		{X86_INS_NOT, &Capstone2LlvmIrTranslatorX86_impl::translateNot},
		{X86_INS_OUT, nullptr},
		{X86_INS_OUTSB, &Capstone2LlvmIrTranslatorX86_impl::translateOuts},
		{X86_INS_OUTSD, &Capstone2LlvmIrTranslatorX86_impl::translateOuts},
		{X86_INS_OUTSW, &Capstone2LlvmIrTranslatorX86_impl::translateOuts},
		{X86_INS_PACKUSDW, nullptr},
		{X86_INS_PAUSE, nullptr},
		{X86_INS_PAVGUSB, nullptr},
		{X86_INS_PBLENDVB, nullptr},
		{X86_INS_PBLENDW, nullptr},
		{X86_INS_PCLMULQDQ, nullptr},
		{X86_INS_PCMPEQQ, nullptr},
		{X86_INS_PCMPESTRI, nullptr},
		{X86_INS_PCMPESTRM, nullptr},
		{X86_INS_PCMPGTQ, nullptr},
		{X86_INS_PCMPISTRI, nullptr},
		{X86_INS_PCMPISTRM, nullptr},
		{X86_INS_PCOMMIT, nullptr},
		{X86_INS_PDEP, nullptr},
		{X86_INS_PEXT, nullptr},
		{X86_INS_PEXTRB, nullptr},
		{X86_INS_PEXTRD, nullptr},
		{X86_INS_PEXTRQ, nullptr},
		{X86_INS_PF2ID, nullptr},
		{X86_INS_PF2IW, nullptr},
		{X86_INS_PFACC, nullptr},
		{X86_INS_PFADD, nullptr},
		{X86_INS_PFCMPEQ, nullptr},
		{X86_INS_PFCMPGE, nullptr},
		{X86_INS_PFCMPGT, nullptr},
		{X86_INS_PFMAX, nullptr},
		{X86_INS_PFMIN, nullptr},
		{X86_INS_PFMUL, nullptr},
		{X86_INS_PFNACC, nullptr},
		{X86_INS_PFPNACC, nullptr},
		{X86_INS_PFRCPIT1, nullptr},
		{X86_INS_PFRCPIT2, nullptr},
		{X86_INS_PFRCP, nullptr},
		{X86_INS_PFRSQIT1, nullptr},
		{X86_INS_PFRSQRT, nullptr},
		{X86_INS_PFSUBR, nullptr},
		{X86_INS_PFSUB, nullptr},
		{X86_INS_PHMINPOSUW, nullptr},
		{X86_INS_PI2FD, nullptr},
		{X86_INS_PI2FW, nullptr},
		{X86_INS_PINSRB, nullptr},
		{X86_INS_PINSRD, nullptr},
		{X86_INS_PINSRQ, nullptr},
		{X86_INS_PMAXSB, nullptr},
		{X86_INS_PMAXSD, nullptr},
		{X86_INS_PMAXUD, nullptr},
		{X86_INS_PMAXUW, nullptr},
		{X86_INS_PMINSB, nullptr},
		{X86_INS_PMINSD, nullptr},
		{X86_INS_PMINUD, nullptr},
		{X86_INS_PMINUW, nullptr},
		{X86_INS_PMOVSXBD, nullptr},
		{X86_INS_PMOVSXBQ, nullptr},
		{X86_INS_PMOVSXBW, nullptr},
		{X86_INS_PMOVSXDQ, nullptr},
		{X86_INS_PMOVSXWD, nullptr},
		{X86_INS_PMOVSXWQ, nullptr},
		{X86_INS_PMOVZXBD, nullptr},
		{X86_INS_PMOVZXBQ, nullptr},
		{X86_INS_PMOVZXBW, nullptr},
		{X86_INS_PMOVZXDQ, nullptr},
		{X86_INS_PMOVZXWD, nullptr},
		{X86_INS_PMOVZXWQ, nullptr},
		{X86_INS_PMULDQ, nullptr},
		{X86_INS_PMULHRW, nullptr},
		{X86_INS_PMULLD, nullptr},
		{X86_INS_POP, &Capstone2LlvmIrTranslatorX86_impl::translatePop},
		{X86_INS_POPAW, &Capstone2LlvmIrTranslatorX86_impl::translatePopa}, // X86_INS_POPAW == POPA
		{X86_INS_POPAL, &Capstone2LlvmIrTranslatorX86_impl::translatePopa}, // X86_INS_POPAL == POPAD
		{X86_INS_POPCNT, nullptr},
		{X86_INS_POPF, &Capstone2LlvmIrTranslatorX86_impl::translatePopEflags},
		{X86_INS_POPFD, &Capstone2LlvmIrTranslatorX86_impl::translatePopEflags},
		{X86_INS_POPFQ, &Capstone2LlvmIrTranslatorX86_impl::translatePopEflags},
		{X86_INS_PREFETCH, nullptr},
		{X86_INS_PREFETCHNTA, nullptr},
		{X86_INS_PREFETCHT0, nullptr},
		{X86_INS_PREFETCHT1, nullptr},
		{X86_INS_PREFETCHT2, nullptr},
		{X86_INS_PREFETCHW, nullptr},
		{X86_INS_PSHUFD, nullptr},
		{X86_INS_PSHUFHW, nullptr},
		{X86_INS_PSHUFLW, nullptr},
		{X86_INS_PSLLDQ, nullptr},
		{X86_INS_PSRLDQ, nullptr},
		{X86_INS_PSWAPD, nullptr},
		{X86_INS_PTEST, nullptr},
		{X86_INS_PUNPCKHQDQ, nullptr},
		{X86_INS_PUNPCKLQDQ, nullptr},
		{X86_INS_PUSH, &Capstone2LlvmIrTranslatorX86_impl::translatePush},
		{X86_INS_PUSHAW, &Capstone2LlvmIrTranslatorX86_impl::translatePusha}, // X86_INS_PUSHAW = PUSHA
		{X86_INS_PUSHAL, &Capstone2LlvmIrTranslatorX86_impl::translatePusha}, // X86_INS_PUSHAL = PUSHAD
		{X86_INS_PUSHF, &Capstone2LlvmIrTranslatorX86_impl::translatePushEflags},
		{X86_INS_PUSHFD, &Capstone2LlvmIrTranslatorX86_impl::translatePushEflags},
		{X86_INS_PUSHFQ, &Capstone2LlvmIrTranslatorX86_impl::translatePushEflags},
		{X86_INS_RCL, &Capstone2LlvmIrTranslatorX86_impl::translateRcl},
		{X86_INS_RCPPS, nullptr},
		{X86_INS_RCPSS, nullptr},
		{X86_INS_RCR, &Capstone2LlvmIrTranslatorX86_impl::translateRcr},
		{X86_INS_RDFSBASE, nullptr},
		{X86_INS_RDGSBASE, nullptr},
		{X86_INS_RDMSR, nullptr},
		{X86_INS_RDPMC, nullptr},
		{X86_INS_RDRAND, nullptr},
		{X86_INS_RDSEED, nullptr},
		{X86_INS_RDTSC, &Capstone2LlvmIrTranslatorX86_impl::translateRdtsc},
		{X86_INS_RDTSCP, &Capstone2LlvmIrTranslatorX86_impl::translateRdtscp},
		{X86_INS_ROL, &Capstone2LlvmIrTranslatorX86_impl::translateRol},
		{X86_INS_ROR, &Capstone2LlvmIrTranslatorX86_impl::translateRor},
		{X86_INS_RORX, nullptr},
		{X86_INS_ROUNDPD, nullptr},
		{X86_INS_ROUNDPS, nullptr},
		{X86_INS_ROUNDSD, nullptr},
		{X86_INS_ROUNDSS, nullptr},
		{X86_INS_RSM, nullptr},
		{X86_INS_RSQRTPS, nullptr},
		{X86_INS_RSQRTSS, nullptr},
		{X86_INS_SAHF, &Capstone2LlvmIrTranslatorX86_impl::translateSahf},
		{X86_INS_SAL, &Capstone2LlvmIrTranslatorX86_impl::translateShiftLeft},
		{X86_INS_SALC, &Capstone2LlvmIrTranslatorX86_impl::translateSalc},
		{X86_INS_SAR, &Capstone2LlvmIrTranslatorX86_impl::translateShiftRight},
		{X86_INS_SARX, nullptr},
		{X86_INS_SBB, &Capstone2LlvmIrTranslatorX86_impl::translateSbb},
		{X86_INS_SCASB, &Capstone2LlvmIrTranslatorX86_impl::translateScanString},
		{X86_INS_SCASD, &Capstone2LlvmIrTranslatorX86_impl::translateScanString},
		{X86_INS_SCASQ, &Capstone2LlvmIrTranslatorX86_impl::translateScanString},
		{X86_INS_SCASW, &Capstone2LlvmIrTranslatorX86_impl::translateScanString},
		{X86_INS_SETAE, &Capstone2LlvmIrTranslatorX86_impl::translateSetCc},
		{X86_INS_SETA, &Capstone2LlvmIrTranslatorX86_impl::translateSetCc},
		{X86_INS_SETBE, &Capstone2LlvmIrTranslatorX86_impl::translateSetCc},
		{X86_INS_SETB, &Capstone2LlvmIrTranslatorX86_impl::translateSetCc},
		{X86_INS_SETE, &Capstone2LlvmIrTranslatorX86_impl::translateSetCc},
		{X86_INS_SETGE, &Capstone2LlvmIrTranslatorX86_impl::translateSetCc},
		{X86_INS_SETG, &Capstone2LlvmIrTranslatorX86_impl::translateSetCc},
		{X86_INS_SETLE, &Capstone2LlvmIrTranslatorX86_impl::translateSetCc},
		{X86_INS_SETL, &Capstone2LlvmIrTranslatorX86_impl::translateSetCc},
		{X86_INS_SETNE, &Capstone2LlvmIrTranslatorX86_impl::translateSetCc},
		{X86_INS_SETNO, &Capstone2LlvmIrTranslatorX86_impl::translateSetCc},
		{X86_INS_SETNP, &Capstone2LlvmIrTranslatorX86_impl::translateSetCc},
		{X86_INS_SETNS, &Capstone2LlvmIrTranslatorX86_impl::translateSetCc},
		{X86_INS_SETO, &Capstone2LlvmIrTranslatorX86_impl::translateSetCc},
		{X86_INS_SETP, &Capstone2LlvmIrTranslatorX86_impl::translateSetCc},
		{X86_INS_SETS, &Capstone2LlvmIrTranslatorX86_impl::translateSetCc},
		{X86_INS_SFENCE, nullptr},
		{X86_INS_SGDT, nullptr},
		{X86_INS_SHA1MSG1, nullptr},
		{X86_INS_SHA1MSG2, nullptr},
		{X86_INS_SHA1NEXTE, nullptr},
		{X86_INS_SHA1RNDS4, nullptr},
		{X86_INS_SHA256MSG1, nullptr},
		{X86_INS_SHA256MSG2, nullptr},
		{X86_INS_SHA256RNDS2, nullptr},
		{X86_INS_SHL, &Capstone2LlvmIrTranslatorX86_impl::translateShiftLeft},
		{X86_INS_SHLD, &Capstone2LlvmIrTranslatorX86_impl::translateShld},
		{X86_INS_SHLX, nullptr},
		{X86_INS_SHR, &Capstone2LlvmIrTranslatorX86_impl::translateShiftRight},
		{X86_INS_SHRD, &Capstone2LlvmIrTranslatorX86_impl::translateShrd},
		{X86_INS_SHRX, nullptr},
		{X86_INS_SHUFPD, nullptr},
		{X86_INS_SHUFPS, nullptr},
		{X86_INS_SIDT, nullptr},
		{X86_INS_FSIN, &Capstone2LlvmIrTranslatorX86_impl::translateFsin},
		{X86_INS_SKINIT, nullptr},
		{X86_INS_SLDT, nullptr},
		{X86_INS_SMSW, nullptr},
		{X86_INS_SQRTPD, nullptr},
		{X86_INS_SQRTPS, nullptr},
		{X86_INS_SQRTSD, nullptr},
		{X86_INS_SQRTSS, nullptr},
		{X86_INS_FSQRT, &Capstone2LlvmIrTranslatorX86_impl::translateFsqrt},
		{X86_INS_STAC, nullptr},
		{X86_INS_STC, &Capstone2LlvmIrTranslatorX86_impl::translateStc},
		{X86_INS_STD, &Capstone2LlvmIrTranslatorX86_impl::translateStd},
		{X86_INS_STGI, nullptr},
		{X86_INS_STI, nullptr},
		{X86_INS_STMXCSR, nullptr},
		{X86_INS_STOSB, &Capstone2LlvmIrTranslatorX86_impl::translateStoreString},
		{X86_INS_STOSD, &Capstone2LlvmIrTranslatorX86_impl::translateStoreString},
		{X86_INS_STOSQ, &Capstone2LlvmIrTranslatorX86_impl::translateStoreString},
		{X86_INS_STOSW, &Capstone2LlvmIrTranslatorX86_impl::translateStoreString},
		{X86_INS_STR, nullptr},
		{X86_INS_FST, &Capstone2LlvmIrTranslatorX86_impl::translateFst},
		{X86_INS_FSTP, &Capstone2LlvmIrTranslatorX86_impl::translateFst},
		{X86_INS_FSTPNCE, nullptr},
		{X86_INS_FXCH, &Capstone2LlvmIrTranslatorX86_impl::translateFxch},
		{X86_INS_SUBPD, nullptr},
		{X86_INS_SUBPS, nullptr},
		{X86_INS_FSUBR, &Capstone2LlvmIrTranslatorX86_impl::translateFsubr},
		{X86_INS_FISUBR, &Capstone2LlvmIrTranslatorX86_impl::translateFsubr},
		{X86_INS_FSUBRP, &Capstone2LlvmIrTranslatorX86_impl::translateFsubr},
		{X86_INS_SUBSD, nullptr},
		{X86_INS_SUBSS, nullptr},
		{X86_INS_FSUB, &Capstone2LlvmIrTranslatorX86_impl::translateFsub},
		{X86_INS_FISUB, &Capstone2LlvmIrTranslatorX86_impl::translateFsub},
		{X86_INS_FSUBP, &Capstone2LlvmIrTranslatorX86_impl::translateFsub},
		{X86_INS_SWAPGS, nullptr},
		{X86_INS_SYSCALL, nullptr},
		{X86_INS_SYSENTER, nullptr},
		{X86_INS_SYSEXIT, nullptr},
		{X86_INS_SYSRET, nullptr},
		{X86_INS_T1MSKC, nullptr},
		{X86_INS_TEST, &Capstone2LlvmIrTranslatorX86_impl::translateAnd},
		{X86_INS_UD2, &Capstone2LlvmIrTranslatorX86_impl::translateNop},
		{X86_INS_FTST, &Capstone2LlvmIrTranslatorX86_impl::translateFucomPop},
		{X86_INS_TZCNT, nullptr},
		{X86_INS_TZMSK, nullptr},
		{X86_INS_FUCOMIP, &Capstone2LlvmIrTranslatorX86_impl::translateFucomPop},
		{X86_INS_FUCOMI, &Capstone2LlvmIrTranslatorX86_impl::translateFucomPop},
		{X86_INS_FUCOMPP, &Capstone2LlvmIrTranslatorX86_impl::translateFucomPop},
		{X86_INS_FUCOMP, &Capstone2LlvmIrTranslatorX86_impl::translateFucomPop},
		{X86_INS_FUCOM, &Capstone2LlvmIrTranslatorX86_impl::translateFucomPop},
		{X86_INS_UD2B, &Capstone2LlvmIrTranslatorX86_impl::translateNop},
		{X86_INS_UNPCKHPD, nullptr},
		{X86_INS_UNPCKHPS, nullptr},
		{X86_INS_UNPCKLPD, nullptr},
		{X86_INS_UNPCKLPS, nullptr},
		{X86_INS_VADDPD, nullptr},
		{X86_INS_VADDPS, nullptr},
		{X86_INS_VADDSD, nullptr},
		{X86_INS_VADDSS, nullptr},
		{X86_INS_VADDSUBPD, nullptr},
		{X86_INS_VADDSUBPS, nullptr},
		{X86_INS_VAESDECLAST, nullptr},
		{X86_INS_VAESDEC, nullptr},
		{X86_INS_VAESENCLAST, nullptr},
		{X86_INS_VAESENC, nullptr},
		{X86_INS_VAESIMC, nullptr},
		{X86_INS_VAESKEYGENASSIST, nullptr},
		{X86_INS_VALIGND, nullptr},
		{X86_INS_VALIGNQ, nullptr},
		{X86_INS_VANDNPD, nullptr},
		{X86_INS_VANDNPS, nullptr},
		{X86_INS_VANDPD, nullptr},
		{X86_INS_VANDPS, nullptr},
		{X86_INS_VBLENDMPD, nullptr},
		{X86_INS_VBLENDMPS, nullptr},
		{X86_INS_VBLENDPD, nullptr},
		{X86_INS_VBLENDPS, nullptr},
		{X86_INS_VBLENDVPD, nullptr},
		{X86_INS_VBLENDVPS, nullptr},
		{X86_INS_VBROADCASTF128, nullptr},
		{X86_INS_VBROADCASTI32X4, nullptr},
		{X86_INS_VBROADCASTI64X4, nullptr},
		{X86_INS_VBROADCASTSD, nullptr},
		{X86_INS_VBROADCASTSS, nullptr},
		{X86_INS_VCOMPRESSPD, nullptr},
		{X86_INS_VCOMPRESSPS, nullptr},
		{X86_INS_VCVTDQ2PD, nullptr},
		{X86_INS_VCVTDQ2PS, nullptr},
		{X86_INS_VCVTPD2DQX, nullptr},
		{X86_INS_VCVTPD2DQ, nullptr},
		{X86_INS_VCVTPD2PSX, nullptr},
		{X86_INS_VCVTPD2PS, nullptr},
		{X86_INS_VCVTPD2UDQ, nullptr},
		{X86_INS_VCVTPH2PS, nullptr},
		{X86_INS_VCVTPS2DQ, nullptr},
		{X86_INS_VCVTPS2PD, nullptr},
		{X86_INS_VCVTPS2PH, nullptr},
		{X86_INS_VCVTPS2UDQ, nullptr},
		{X86_INS_VCVTSD2SI, nullptr},
		{X86_INS_VCVTSD2USI, nullptr},
		{X86_INS_VCVTSS2SI, nullptr},
		{X86_INS_VCVTSS2USI, nullptr},
		{X86_INS_VCVTTPD2DQX, nullptr},
		{X86_INS_VCVTTPD2DQ, nullptr},
		{X86_INS_VCVTTPD2UDQ, nullptr},
		{X86_INS_VCVTTPS2DQ, nullptr},
		{X86_INS_VCVTTPS2UDQ, nullptr},
		{X86_INS_VCVTUDQ2PD, nullptr},
		{X86_INS_VCVTUDQ2PS, nullptr},
		{X86_INS_VDIVPD, nullptr},
		{X86_INS_VDIVPS, nullptr},
		{X86_INS_VDIVSD, nullptr},
		{X86_INS_VDIVSS, nullptr},
		{X86_INS_VDPPD, nullptr},
		{X86_INS_VDPPS, nullptr},
		{X86_INS_VERR, nullptr},
		{X86_INS_VERW, nullptr},
		{X86_INS_VEXP2PD, nullptr},
		{X86_INS_VEXP2PS, nullptr},
		{X86_INS_VEXPANDPD, nullptr},
		{X86_INS_VEXPANDPS, nullptr},
		{X86_INS_VEXTRACTF128, nullptr},
		{X86_INS_VEXTRACTF32X4, nullptr},
		{X86_INS_VEXTRACTF64X4, nullptr},
		{X86_INS_VEXTRACTI128, nullptr},
		{X86_INS_VEXTRACTI32X4, nullptr},
		{X86_INS_VEXTRACTI64X4, nullptr},
		{X86_INS_VEXTRACTPS, nullptr},
		{X86_INS_VFMADD132PD, nullptr},
		{X86_INS_VFMADD132PS, nullptr},
		{X86_INS_VFMADDPD, nullptr},
		{X86_INS_VFMADD213PD, nullptr},
		{X86_INS_VFMADD231PD, nullptr},
		{X86_INS_VFMADDPS, nullptr},
		{X86_INS_VFMADD213PS, nullptr},
		{X86_INS_VFMADD231PS, nullptr},
		{X86_INS_VFMADDSD, nullptr},
		{X86_INS_VFMADD213SD, nullptr},
		{X86_INS_VFMADD132SD, nullptr},
		{X86_INS_VFMADD231SD, nullptr},
		{X86_INS_VFMADDSS, nullptr},
		{X86_INS_VFMADD213SS, nullptr},
		{X86_INS_VFMADD132SS, nullptr},
		{X86_INS_VFMADD231SS, nullptr},
		{X86_INS_VFMADDSUB132PD, nullptr},
		{X86_INS_VFMADDSUB132PS, nullptr},
		{X86_INS_VFMADDSUBPD, nullptr},
		{X86_INS_VFMADDSUB213PD, nullptr},
		{X86_INS_VFMADDSUB231PD, nullptr},
		{X86_INS_VFMADDSUBPS, nullptr},
		{X86_INS_VFMADDSUB213PS, nullptr},
		{X86_INS_VFMADDSUB231PS, nullptr},
		{X86_INS_VFMSUB132PD, nullptr},
		{X86_INS_VFMSUB132PS, nullptr},
		{X86_INS_VFMSUBADD132PD, nullptr},
		{X86_INS_VFMSUBADD132PS, nullptr},
		{X86_INS_VFMSUBADDPD, nullptr},
		{X86_INS_VFMSUBADD213PD, nullptr},
		{X86_INS_VFMSUBADD231PD, nullptr},
		{X86_INS_VFMSUBADDPS, nullptr},
		{X86_INS_VFMSUBADD213PS, nullptr},
		{X86_INS_VFMSUBADD231PS, nullptr},
		{X86_INS_VFMSUBPD, nullptr},
		{X86_INS_VFMSUB213PD, nullptr},
		{X86_INS_VFMSUB231PD, nullptr},
		{X86_INS_VFMSUBPS, nullptr},
		{X86_INS_VFMSUB213PS, nullptr},
		{X86_INS_VFMSUB231PS, nullptr},
		{X86_INS_VFMSUBSD, nullptr},
		{X86_INS_VFMSUB213SD, nullptr},
		{X86_INS_VFMSUB132SD, nullptr},
		{X86_INS_VFMSUB231SD, nullptr},
		{X86_INS_VFMSUBSS, nullptr},
		{X86_INS_VFMSUB213SS, nullptr},
		{X86_INS_VFMSUB132SS, nullptr},
		{X86_INS_VFMSUB231SS, nullptr},
		{X86_INS_VFNMADD132PD, nullptr},
		{X86_INS_VFNMADD132PS, nullptr},
		{X86_INS_VFNMADDPD, nullptr},
		{X86_INS_VFNMADD213PD, nullptr},
		{X86_INS_VFNMADD231PD, nullptr},
		{X86_INS_VFNMADDPS, nullptr},
		{X86_INS_VFNMADD213PS, nullptr},
		{X86_INS_VFNMADD231PS, nullptr},
		{X86_INS_VFNMADDSD, nullptr},
		{X86_INS_VFNMADD213SD, nullptr},
		{X86_INS_VFNMADD132SD, nullptr},
		{X86_INS_VFNMADD231SD, nullptr},
		{X86_INS_VFNMADDSS, nullptr},
		{X86_INS_VFNMADD213SS, nullptr},
		{X86_INS_VFNMADD132SS, nullptr},
		{X86_INS_VFNMADD231SS, nullptr},
		{X86_INS_VFNMSUB132PD, nullptr},
		{X86_INS_VFNMSUB132PS, nullptr},
		{X86_INS_VFNMSUBPD, nullptr},
		{X86_INS_VFNMSUB213PD, nullptr},
		{X86_INS_VFNMSUB231PD, nullptr},
		{X86_INS_VFNMSUBPS, nullptr},
		{X86_INS_VFNMSUB213PS, nullptr},
		{X86_INS_VFNMSUB231PS, nullptr},
		{X86_INS_VFNMSUBSD, nullptr},
		{X86_INS_VFNMSUB213SD, nullptr},
		{X86_INS_VFNMSUB132SD, nullptr},
		{X86_INS_VFNMSUB231SD, nullptr},
		{X86_INS_VFNMSUBSS, nullptr},
		{X86_INS_VFNMSUB213SS, nullptr},
		{X86_INS_VFNMSUB132SS, nullptr},
		{X86_INS_VFNMSUB231SS, nullptr},
		{X86_INS_VFRCZPD, nullptr},
		{X86_INS_VFRCZPS, nullptr},
		{X86_INS_VFRCZSD, nullptr},
		{X86_INS_VFRCZSS, nullptr},
		{X86_INS_VORPD, nullptr},
		{X86_INS_VORPS, nullptr},
		{X86_INS_VXORPD, nullptr},
		{X86_INS_VXORPS, nullptr},
		{X86_INS_VGATHERDPD, nullptr},
		{X86_INS_VGATHERDPS, nullptr},
		{X86_INS_VGATHERPF0DPD, nullptr},
		{X86_INS_VGATHERPF0DPS, nullptr},
		{X86_INS_VGATHERPF0QPD, nullptr},
		{X86_INS_VGATHERPF0QPS, nullptr},
		{X86_INS_VGATHERPF1DPD, nullptr},
		{X86_INS_VGATHERPF1DPS, nullptr},
		{X86_INS_VGATHERPF1QPD, nullptr},
		{X86_INS_VGATHERPF1QPS, nullptr},
		{X86_INS_VGATHERQPD, nullptr},
		{X86_INS_VGATHERQPS, nullptr},
		{X86_INS_VHADDPD, nullptr},
		{X86_INS_VHADDPS, nullptr},
		{X86_INS_VHSUBPD, nullptr},
		{X86_INS_VHSUBPS, nullptr},
		{X86_INS_VINSERTF128, nullptr},
		{X86_INS_VINSERTF32X4, nullptr},
		{X86_INS_VINSERTF32X8, nullptr},
		{X86_INS_VINSERTF64X2, nullptr},
		{X86_INS_VINSERTF64X4, nullptr},
		{X86_INS_VINSERTI128, nullptr},
		{X86_INS_VINSERTI32X4, nullptr},
		{X86_INS_VINSERTI32X8, nullptr},
		{X86_INS_VINSERTI64X2, nullptr},
		{X86_INS_VINSERTI64X4, nullptr},
		{X86_INS_VINSERTPS, nullptr},
		{X86_INS_VLDDQU, nullptr},
		{X86_INS_VLDMXCSR, nullptr},
		{X86_INS_VMASKMOVDQU, nullptr},
		{X86_INS_VMASKMOVPD, nullptr},
		{X86_INS_VMASKMOVPS, nullptr},
		{X86_INS_VMAXPD, nullptr},
		{X86_INS_VMAXPS, nullptr},
		{X86_INS_VMAXSD, nullptr},
		{X86_INS_VMAXSS, nullptr},
		{X86_INS_VMCALL, nullptr},
		{X86_INS_VMCLEAR, nullptr},
		{X86_INS_VMFUNC, nullptr},
		{X86_INS_VMINPD, nullptr},
		{X86_INS_VMINPS, nullptr},
		{X86_INS_VMINSD, nullptr},
		{X86_INS_VMINSS, nullptr},
		{X86_INS_VMLAUNCH, nullptr},
		{X86_INS_VMLOAD, nullptr},
		{X86_INS_VMMCALL, nullptr},
		{X86_INS_VMOVQ, nullptr},
		{X86_INS_VMOVDDUP, nullptr},
		{X86_INS_VMOVD, nullptr},
		{X86_INS_VMOVDQA32, nullptr},
		{X86_INS_VMOVDQA64, nullptr},
		{X86_INS_VMOVDQA, nullptr},
		{X86_INS_VMOVDQU16, nullptr},
		{X86_INS_VMOVDQU32, nullptr},
		{X86_INS_VMOVDQU64, nullptr},
		{X86_INS_VMOVDQU8, nullptr},
		{X86_INS_VMOVDQU, nullptr},
		{X86_INS_VMOVHLPS, nullptr},
		{X86_INS_VMOVHPD, nullptr},
		{X86_INS_VMOVHPS, nullptr},
		{X86_INS_VMOVLHPS, nullptr},
		{X86_INS_VMOVLPD, nullptr},
		{X86_INS_VMOVLPS, nullptr},
		{X86_INS_VMOVMSKPD, nullptr},
		{X86_INS_VMOVMSKPS, nullptr},
		{X86_INS_VMOVNTDQA, nullptr},
		{X86_INS_VMOVNTDQ, nullptr},
		{X86_INS_VMOVNTPD, nullptr},
		{X86_INS_VMOVNTPS, nullptr},
		{X86_INS_VMOVSD, nullptr},
		{X86_INS_VMOVSHDUP, nullptr},
		{X86_INS_VMOVSLDUP, nullptr},
		{X86_INS_VMOVSS, nullptr},
		{X86_INS_VMOVUPD, nullptr},
		{X86_INS_VMOVUPS, nullptr},
		{X86_INS_VMPSADBW, nullptr},
		{X86_INS_VMPTRLD, nullptr},
		{X86_INS_VMPTRST, nullptr},
		{X86_INS_VMREAD, nullptr},
		{X86_INS_VMRESUME, nullptr},
		{X86_INS_VMRUN, nullptr},
		{X86_INS_VMSAVE, nullptr},
		{X86_INS_VMULPD, nullptr},
		{X86_INS_VMULPS, nullptr},
		{X86_INS_VMULSD, nullptr},
		{X86_INS_VMULSS, nullptr},
		{X86_INS_VMWRITE, nullptr},
		{X86_INS_VMXOFF, nullptr},
		{X86_INS_VMXON, nullptr},
		{X86_INS_VPABSB, nullptr},
		{X86_INS_VPABSD, nullptr},
		{X86_INS_VPABSQ, nullptr},
		{X86_INS_VPABSW, nullptr},
		{X86_INS_VPACKSSDW, nullptr},
		{X86_INS_VPACKSSWB, nullptr},
		{X86_INS_VPACKUSDW, nullptr},
		{X86_INS_VPACKUSWB, nullptr},
		{X86_INS_VPADDB, nullptr},
		{X86_INS_VPADDD, nullptr},
		{X86_INS_VPADDQ, nullptr},
		{X86_INS_VPADDSB, nullptr},
		{X86_INS_VPADDSW, nullptr},
		{X86_INS_VPADDUSB, nullptr},
		{X86_INS_VPADDUSW, nullptr},
		{X86_INS_VPADDW, nullptr},
		{X86_INS_VPALIGNR, nullptr},
		{X86_INS_VPANDD, nullptr},
		{X86_INS_VPANDND, nullptr},
		{X86_INS_VPANDNQ, nullptr},
		{X86_INS_VPANDN, nullptr},
		{X86_INS_VPANDQ, nullptr},
		{X86_INS_VPAND, nullptr},
		{X86_INS_VPAVGB, nullptr},
		{X86_INS_VPAVGW, nullptr},
		{X86_INS_VPBLENDD, nullptr},
		{X86_INS_VPBLENDMB, nullptr},
		{X86_INS_VPBLENDMD, nullptr},
		{X86_INS_VPBLENDMQ, nullptr},
		{X86_INS_VPBLENDMW, nullptr},
		{X86_INS_VPBLENDVB, nullptr},
		{X86_INS_VPBLENDW, nullptr},
		{X86_INS_VPBROADCASTB, nullptr},
		{X86_INS_VPBROADCASTD, nullptr},
		{X86_INS_VPBROADCASTMB2Q, nullptr},
		{X86_INS_VPBROADCASTMW2D, nullptr},
		{X86_INS_VPBROADCASTQ, nullptr},
		{X86_INS_VPBROADCASTW, nullptr},
		{X86_INS_VPCLMULQDQ, nullptr},
		{X86_INS_VPCMOV, nullptr},
		{X86_INS_VPCMPB, nullptr},
		{X86_INS_VPCMPD, nullptr},
		{X86_INS_VPCMPEQB, nullptr},
		{X86_INS_VPCMPEQD, nullptr},
		{X86_INS_VPCMPEQQ, nullptr},
		{X86_INS_VPCMPEQW, nullptr},
		{X86_INS_VPCMPESTRI, nullptr},
		{X86_INS_VPCMPESTRM, nullptr},
		{X86_INS_VPCMPGTB, nullptr},
		{X86_INS_VPCMPGTD, nullptr},
		{X86_INS_VPCMPGTQ, nullptr},
		{X86_INS_VPCMPGTW, nullptr},
		{X86_INS_VPCMPISTRI, nullptr},
		{X86_INS_VPCMPISTRM, nullptr},
		{X86_INS_VPCMPQ, nullptr},
		{X86_INS_VPCMPUB, nullptr},
		{X86_INS_VPCMPUD, nullptr},
		{X86_INS_VPCMPUQ, nullptr},
		{X86_INS_VPCMPUW, nullptr},
		{X86_INS_VPCMPW, nullptr},
		{X86_INS_VPCOMB, nullptr},
		{X86_INS_VPCOMD, nullptr},
		{X86_INS_VPCOMPRESSD, nullptr},
		{X86_INS_VPCOMPRESSQ, nullptr},
		{X86_INS_VPCOMQ, nullptr},
		{X86_INS_VPCOMUB, nullptr},
		{X86_INS_VPCOMUD, nullptr},
		{X86_INS_VPCOMUQ, nullptr},
		{X86_INS_VPCOMUW, nullptr},
		{X86_INS_VPCOMW, nullptr},
		{X86_INS_VPCONFLICTD, nullptr},
		{X86_INS_VPCONFLICTQ, nullptr},
		{X86_INS_VPERM2F128, nullptr},
		{X86_INS_VPERM2I128, nullptr},
		{X86_INS_VPERMD, nullptr},
		{X86_INS_VPERMI2D, nullptr},
		{X86_INS_VPERMI2PD, nullptr},
		{X86_INS_VPERMI2PS, nullptr},
		{X86_INS_VPERMI2Q, nullptr},
		{X86_INS_VPERMIL2PD, nullptr},
		{X86_INS_VPERMIL2PS, nullptr},
		{X86_INS_VPERMILPD, nullptr},
		{X86_INS_VPERMILPS, nullptr},
		{X86_INS_VPERMPD, nullptr},
		{X86_INS_VPERMPS, nullptr},
		{X86_INS_VPERMQ, nullptr},
		{X86_INS_VPERMT2D, nullptr},
		{X86_INS_VPERMT2PD, nullptr},
		{X86_INS_VPERMT2PS, nullptr},
		{X86_INS_VPERMT2Q, nullptr},
		{X86_INS_VPEXPANDD, nullptr},
		{X86_INS_VPEXPANDQ, nullptr},
		{X86_INS_VPEXTRB, nullptr},
		{X86_INS_VPEXTRD, nullptr},
		{X86_INS_VPEXTRQ, nullptr},
		{X86_INS_VPEXTRW, nullptr},
		{X86_INS_VPGATHERDD, nullptr},
		{X86_INS_VPGATHERDQ, nullptr},
		{X86_INS_VPGATHERQD, nullptr},
		{X86_INS_VPGATHERQQ, nullptr},
		{X86_INS_VPHADDBD, nullptr},
		{X86_INS_VPHADDBQ, nullptr},
		{X86_INS_VPHADDBW, nullptr},
		{X86_INS_VPHADDDQ, nullptr},
		{X86_INS_VPHADDD, nullptr},
		{X86_INS_VPHADDSW, nullptr},
		{X86_INS_VPHADDUBD, nullptr},
		{X86_INS_VPHADDUBQ, nullptr},
		{X86_INS_VPHADDUBW, nullptr},
		{X86_INS_VPHADDUDQ, nullptr},
		{X86_INS_VPHADDUWD, nullptr},
		{X86_INS_VPHADDUWQ, nullptr},
		{X86_INS_VPHADDWD, nullptr},
		{X86_INS_VPHADDWQ, nullptr},
		{X86_INS_VPHADDW, nullptr},
		{X86_INS_VPHMINPOSUW, nullptr},
		{X86_INS_VPHSUBBW, nullptr},
		{X86_INS_VPHSUBDQ, nullptr},
		{X86_INS_VPHSUBD, nullptr},
		{X86_INS_VPHSUBSW, nullptr},
		{X86_INS_VPHSUBWD, nullptr},
		{X86_INS_VPHSUBW, nullptr},
		{X86_INS_VPINSRB, nullptr},
		{X86_INS_VPINSRD, nullptr},
		{X86_INS_VPINSRQ, nullptr},
		{X86_INS_VPINSRW, nullptr},
		{X86_INS_VPLZCNTD, nullptr},
		{X86_INS_VPLZCNTQ, nullptr},
		{X86_INS_VPMACSDD, nullptr},
		{X86_INS_VPMACSDQH, nullptr},
		{X86_INS_VPMACSDQL, nullptr},
		{X86_INS_VPMACSSDD, nullptr},
		{X86_INS_VPMACSSDQH, nullptr},
		{X86_INS_VPMACSSDQL, nullptr},
		{X86_INS_VPMACSSWD, nullptr},
		{X86_INS_VPMACSSWW, nullptr},
		{X86_INS_VPMACSWD, nullptr},
		{X86_INS_VPMACSWW, nullptr},
		{X86_INS_VPMADCSSWD, nullptr},
		{X86_INS_VPMADCSWD, nullptr},
		{X86_INS_VPMADDUBSW, nullptr},
		{X86_INS_VPMADDWD, nullptr},
		{X86_INS_VPMASKMOVD, nullptr},
		{X86_INS_VPMASKMOVQ, nullptr},
		{X86_INS_VPMAXSB, nullptr},
		{X86_INS_VPMAXSD, nullptr},
		{X86_INS_VPMAXSQ, nullptr},
		{X86_INS_VPMAXSW, nullptr},
		{X86_INS_VPMAXUB, nullptr},
		{X86_INS_VPMAXUD, nullptr},
		{X86_INS_VPMAXUQ, nullptr},
		{X86_INS_VPMAXUW, nullptr},
		{X86_INS_VPMINSB, nullptr},
		{X86_INS_VPMINSD, nullptr},
		{X86_INS_VPMINSQ, nullptr},
		{X86_INS_VPMINSW, nullptr},
		{X86_INS_VPMINUB, nullptr},
		{X86_INS_VPMINUD, nullptr},
		{X86_INS_VPMINUQ, nullptr},
		{X86_INS_VPMINUW, nullptr},
		{X86_INS_VPMOVDB, nullptr},
		{X86_INS_VPMOVDW, nullptr},
		{X86_INS_VPMOVM2B, nullptr},
		{X86_INS_VPMOVM2D, nullptr},
		{X86_INS_VPMOVM2Q, nullptr},
		{X86_INS_VPMOVM2W, nullptr},
		{X86_INS_VPMOVMSKB, nullptr},
		{X86_INS_VPMOVQB, nullptr},
		{X86_INS_VPMOVQD, nullptr},
		{X86_INS_VPMOVQW, nullptr},
		{X86_INS_VPMOVSDB, nullptr},
		{X86_INS_VPMOVSDW, nullptr},
		{X86_INS_VPMOVSQB, nullptr},
		{X86_INS_VPMOVSQD, nullptr},
		{X86_INS_VPMOVSQW, nullptr},
		{X86_INS_VPMOVSXBD, nullptr},
		{X86_INS_VPMOVSXBQ, nullptr},
		{X86_INS_VPMOVSXBW, nullptr},
		{X86_INS_VPMOVSXDQ, nullptr},
		{X86_INS_VPMOVSXWD, nullptr},
		{X86_INS_VPMOVSXWQ, nullptr},
		{X86_INS_VPMOVUSDB, nullptr},
		{X86_INS_VPMOVUSDW, nullptr},
		{X86_INS_VPMOVUSQB, nullptr},
		{X86_INS_VPMOVUSQD, nullptr},
		{X86_INS_VPMOVUSQW, nullptr},
		{X86_INS_VPMOVZXBD, nullptr},
		{X86_INS_VPMOVZXBQ, nullptr},
		{X86_INS_VPMOVZXBW, nullptr},
		{X86_INS_VPMOVZXDQ, nullptr},
		{X86_INS_VPMOVZXWD, nullptr},
		{X86_INS_VPMOVZXWQ, nullptr},
		{X86_INS_VPMULDQ, nullptr},
		{X86_INS_VPMULHRSW, nullptr},
		{X86_INS_VPMULHUW, nullptr},
		{X86_INS_VPMULHW, nullptr},
		{X86_INS_VPMULLD, nullptr},
		{X86_INS_VPMULLQ, nullptr},
		{X86_INS_VPMULLW, nullptr},
		{X86_INS_VPMULUDQ, nullptr},
		{X86_INS_VPORD, nullptr},
		{X86_INS_VPORQ, nullptr},
		{X86_INS_VPOR, nullptr},
		{X86_INS_VPPERM, nullptr},
		{X86_INS_VPROTB, nullptr},
		{X86_INS_VPROTD, nullptr},
		{X86_INS_VPROTQ, nullptr},
		{X86_INS_VPROTW, nullptr},
		{X86_INS_VPSADBW, nullptr},
		{X86_INS_VPSCATTERDD, nullptr},
		{X86_INS_VPSCATTERDQ, nullptr},
		{X86_INS_VPSCATTERQD, nullptr},
		{X86_INS_VPSCATTERQQ, nullptr},
		{X86_INS_VPSHAB, nullptr},
		{X86_INS_VPSHAD, nullptr},
		{X86_INS_VPSHAQ, nullptr},
		{X86_INS_VPSHAW, nullptr},
		{X86_INS_VPSHLB, nullptr},
		{X86_INS_VPSHLD, nullptr},
		{X86_INS_VPSHLQ, nullptr},
		{X86_INS_VPSHLW, nullptr},
		{X86_INS_VPSHUFB, nullptr},
		{X86_INS_VPSHUFD, nullptr},
		{X86_INS_VPSHUFHW, nullptr},
		{X86_INS_VPSHUFLW, nullptr},
		{X86_INS_VPSIGNB, nullptr},
		{X86_INS_VPSIGND, nullptr},
		{X86_INS_VPSIGNW, nullptr},
		{X86_INS_VPSLLDQ, nullptr},
		{X86_INS_VPSLLD, nullptr},
		{X86_INS_VPSLLQ, nullptr},
		{X86_INS_VPSLLVD, nullptr},
		{X86_INS_VPSLLVQ, nullptr},
		{X86_INS_VPSLLW, nullptr},
		{X86_INS_VPSRAD, nullptr},
		{X86_INS_VPSRAQ, nullptr},
		{X86_INS_VPSRAVD, nullptr},
		{X86_INS_VPSRAVQ, nullptr},
		{X86_INS_VPSRAW, nullptr},
		{X86_INS_VPSRLDQ, nullptr},
		{X86_INS_VPSRLD, nullptr},
		{X86_INS_VPSRLQ, nullptr},
		{X86_INS_VPSRLVD, nullptr},
		{X86_INS_VPSRLVQ, nullptr},
		{X86_INS_VPSRLW, nullptr},
		{X86_INS_VPSUBB, nullptr},
		{X86_INS_VPSUBD, nullptr},
		{X86_INS_VPSUBQ, nullptr},
		{X86_INS_VPSUBSB, nullptr},
		{X86_INS_VPSUBSW, nullptr},
		{X86_INS_VPSUBUSB, nullptr},
		{X86_INS_VPSUBUSW, nullptr},
		{X86_INS_VPSUBW, nullptr},
		{X86_INS_VPTESTMD, nullptr},
		{X86_INS_VPTESTMQ, nullptr},
		{X86_INS_VPTESTNMD, nullptr},
		{X86_INS_VPTESTNMQ, nullptr},
		{X86_INS_VPTEST, nullptr},
		{X86_INS_VPUNPCKHBW, nullptr},
		{X86_INS_VPUNPCKHDQ, nullptr},
		{X86_INS_VPUNPCKHQDQ, nullptr},
		{X86_INS_VPUNPCKHWD, nullptr},
		{X86_INS_VPUNPCKLBW, nullptr},
		{X86_INS_VPUNPCKLDQ, nullptr},
		{X86_INS_VPUNPCKLQDQ, nullptr},
		{X86_INS_VPUNPCKLWD, nullptr},
		{X86_INS_VPXORD, nullptr},
		{X86_INS_VPXORQ, nullptr},
		{X86_INS_VPXOR, nullptr},
		{X86_INS_VRCP14PD, nullptr},
		{X86_INS_VRCP14PS, nullptr},
		{X86_INS_VRCP14SD, nullptr},
		{X86_INS_VRCP14SS, nullptr},
		{X86_INS_VRCP28PD, nullptr},
		{X86_INS_VRCP28PS, nullptr},
		{X86_INS_VRCP28SD, nullptr},
		{X86_INS_VRCP28SS, nullptr},
		{X86_INS_VRCPPS, nullptr},
		{X86_INS_VRCPSS, nullptr},
		{X86_INS_VRNDSCALEPD, nullptr},
		{X86_INS_VRNDSCALEPS, nullptr},
		{X86_INS_VRNDSCALESD, nullptr},
		{X86_INS_VRNDSCALESS, nullptr},
		{X86_INS_VROUNDPD, nullptr},
		{X86_INS_VROUNDPS, nullptr},
		{X86_INS_VROUNDSD, nullptr},
		{X86_INS_VROUNDSS, nullptr},
		{X86_INS_VRSQRT14PD, nullptr},
		{X86_INS_VRSQRT14PS, nullptr},
		{X86_INS_VRSQRT14SD, nullptr},
		{X86_INS_VRSQRT14SS, nullptr},
		{X86_INS_VRSQRT28PD, nullptr},
		{X86_INS_VRSQRT28PS, nullptr},
		{X86_INS_VRSQRT28SD, nullptr},
		{X86_INS_VRSQRT28SS, nullptr},
		{X86_INS_VRSQRTPS, nullptr},
		{X86_INS_VRSQRTSS, nullptr},
		{X86_INS_VSCATTERDPD, nullptr},
		{X86_INS_VSCATTERDPS, nullptr},
		{X86_INS_VSCATTERPF0DPD, nullptr},
		{X86_INS_VSCATTERPF0DPS, nullptr},
		{X86_INS_VSCATTERPF0QPD, nullptr},
		{X86_INS_VSCATTERPF0QPS, nullptr},
		{X86_INS_VSCATTERPF1DPD, nullptr},
		{X86_INS_VSCATTERPF1DPS, nullptr},
		{X86_INS_VSCATTERPF1QPD, nullptr},
		{X86_INS_VSCATTERPF1QPS, nullptr},
		{X86_INS_VSCATTERQPD, nullptr},
		{X86_INS_VSCATTERQPS, nullptr},
		{X86_INS_VSHUFPD, nullptr},
		{X86_INS_VSHUFPS, nullptr},
		{X86_INS_VSQRTPD, nullptr},
		{X86_INS_VSQRTPS, nullptr},
		{X86_INS_VSQRTSD, nullptr},
		{X86_INS_VSQRTSS, nullptr},
		{X86_INS_VSTMXCSR, nullptr},
		{X86_INS_VSUBPD, nullptr},
		{X86_INS_VSUBPS, nullptr},
		{X86_INS_VSUBSD, nullptr},
		{X86_INS_VSUBSS, nullptr},
		{X86_INS_VTESTPD, nullptr},
		{X86_INS_VTESTPS, nullptr},
		{X86_INS_VUNPCKHPD, nullptr},
		{X86_INS_VUNPCKHPS, nullptr},
		{X86_INS_VUNPCKLPD, nullptr},
		{X86_INS_VUNPCKLPS, nullptr},
		{X86_INS_VZEROALL, nullptr},
		{X86_INS_VZEROUPPER, nullptr},
		{X86_INS_WAIT, nullptr},
		{X86_INS_WBINVD, nullptr},
		{X86_INS_WRFSBASE, nullptr},
		{X86_INS_WRGSBASE, nullptr},
		{X86_INS_WRMSR, nullptr},
		{X86_INS_XABORT, nullptr},
		{X86_INS_XACQUIRE, nullptr},
		{X86_INS_XBEGIN, nullptr},
		{X86_INS_XCHG, &Capstone2LlvmIrTranslatorX86_impl::translateXchg},
		{X86_INS_XCRYPTCBC, nullptr},
		{X86_INS_XCRYPTCFB, nullptr},
		{X86_INS_XCRYPTCTR, nullptr},
		{X86_INS_XCRYPTECB, nullptr},
		{X86_INS_XCRYPTOFB, nullptr},
		{X86_INS_XEND, nullptr},
		{X86_INS_XGETBV, nullptr},
		{X86_INS_XLATB, &Capstone2LlvmIrTranslatorX86_impl::translateXlatb},
		{X86_INS_XRELEASE, nullptr},
		{X86_INS_XRSTOR, nullptr},
		{X86_INS_XRSTOR64, nullptr},
		{X86_INS_XRSTORS, nullptr},
		{X86_INS_XRSTORS64, nullptr},
		{X86_INS_XSAVE, nullptr},
		{X86_INS_XSAVE64, nullptr},
		{X86_INS_XSAVEC, nullptr},
		{X86_INS_XSAVEC64, nullptr},
		{X86_INS_XSAVEOPT, nullptr},
		{X86_INS_XSAVEOPT64, nullptr},
		{X86_INS_XSAVES, nullptr},
		{X86_INS_XSAVES64, nullptr},
		{X86_INS_XSETBV, nullptr},
		{X86_INS_XSHA1, nullptr},
		{X86_INS_XSHA256, nullptr},
		{X86_INS_XSTORE, nullptr},
		{X86_INS_XTEST, nullptr},
		{X86_INS_FDISI8087_NOP, &Capstone2LlvmIrTranslatorX86_impl::translateNop},
		{X86_INS_FENI8087_NOP, &Capstone2LlvmIrTranslatorX86_impl::translateNop},

		// pseudo instructions
		{X86_INS_CMPSS, nullptr},
		{X86_INS_CMPEQSS, nullptr},
		{X86_INS_CMPLTSS, nullptr},
		{X86_INS_CMPLESS, nullptr},
		{X86_INS_CMPUNORDSS, nullptr},
		{X86_INS_CMPNEQSS, nullptr},
		{X86_INS_CMPNLTSS, nullptr},
		{X86_INS_CMPNLESS, nullptr},
		{X86_INS_CMPORDSS, nullptr},

		{X86_INS_CMPSD, &Capstone2LlvmIrTranslatorX86_impl::translateCompareString},
		{X86_INS_CMPEQSD, nullptr},
		{X86_INS_CMPLTSD, nullptr},
		{X86_INS_CMPLESD, nullptr},
		{X86_INS_CMPUNORDSD, nullptr},
		{X86_INS_CMPNEQSD, nullptr},
		{X86_INS_CMPNLTSD, nullptr},
		{X86_INS_CMPNLESD, nullptr},
		{X86_INS_CMPORDSD, nullptr},

		{X86_INS_CMPPS, nullptr},
		{X86_INS_CMPEQPS, nullptr},
		{X86_INS_CMPLTPS, nullptr},
		{X86_INS_CMPLEPS, nullptr},
		{X86_INS_CMPUNORDPS, nullptr},
		{X86_INS_CMPNEQPS, nullptr},
		{X86_INS_CMPNLTPS, nullptr},
		{X86_INS_CMPNLEPS, nullptr},
		{X86_INS_CMPORDPS, nullptr},

		{X86_INS_CMPPD, nullptr},
		{X86_INS_CMPEQPD, nullptr},
		{X86_INS_CMPLTPD, nullptr},
		{X86_INS_CMPLEPD, nullptr},
		{X86_INS_CMPUNORDPD, nullptr},
		{X86_INS_CMPNEQPD, nullptr},
		{X86_INS_CMPNLTPD, nullptr},
		{X86_INS_CMPNLEPD, nullptr},
		{X86_INS_CMPORDPD, nullptr},

		{X86_INS_VCMPSS, nullptr},
		{X86_INS_VCMPEQSS, nullptr},
		{X86_INS_VCMPLTSS, nullptr},
		{X86_INS_VCMPLESS, nullptr},
		{X86_INS_VCMPUNORDSS, nullptr},
		{X86_INS_VCMPNEQSS, nullptr},
		{X86_INS_VCMPNLTSS, nullptr},
		{X86_INS_VCMPNLESS, nullptr},
		{X86_INS_VCMPORDSS, nullptr},
		{X86_INS_VCMPEQ_UQSS, nullptr},
		{X86_INS_VCMPNGESS, nullptr},
		{X86_INS_VCMPNGTSS, nullptr},
		{X86_INS_VCMPFALSESS, nullptr},
		{X86_INS_VCMPNEQ_OQSS, nullptr},
		{X86_INS_VCMPGESS, nullptr},
		{X86_INS_VCMPGTSS, nullptr},
		{X86_INS_VCMPTRUESS, nullptr},
		{X86_INS_VCMPEQ_OSSS, nullptr},
		{X86_INS_VCMPLT_OQSS, nullptr},
		{X86_INS_VCMPLE_OQSS, nullptr},
		{X86_INS_VCMPUNORD_SSS, nullptr},
		{X86_INS_VCMPNEQ_USSS, nullptr},
		{X86_INS_VCMPNLT_UQSS, nullptr},
		{X86_INS_VCMPNLE_UQSS, nullptr},
		{X86_INS_VCMPORD_SSS, nullptr},
		{X86_INS_VCMPEQ_USSS, nullptr},
		{X86_INS_VCMPNGE_UQSS, nullptr},
		{X86_INS_VCMPNGT_UQSS, nullptr},
		{X86_INS_VCMPFALSE_OSSS, nullptr},
		{X86_INS_VCMPNEQ_OSSS, nullptr},
		{X86_INS_VCMPGE_OQSS, nullptr},
		{X86_INS_VCMPGT_OQSS, nullptr},
		{X86_INS_VCMPTRUE_USSS, nullptr},

		{X86_INS_VCMPSD, nullptr},
		{X86_INS_VCMPEQSD, nullptr},
		{X86_INS_VCMPLTSD, nullptr},
		{X86_INS_VCMPLESD, nullptr},
		{X86_INS_VCMPUNORDSD, nullptr},
		{X86_INS_VCMPNEQSD, nullptr},
		{X86_INS_VCMPNLTSD, nullptr},
		{X86_INS_VCMPNLESD, nullptr},
		{X86_INS_VCMPORDSD, nullptr},
		{X86_INS_VCMPEQ_UQSD, nullptr},
		{X86_INS_VCMPNGESD, nullptr},
		{X86_INS_VCMPNGTSD, nullptr},
		{X86_INS_VCMPFALSESD, nullptr},
		{X86_INS_VCMPNEQ_OQSD, nullptr},
		{X86_INS_VCMPGESD, nullptr},
		{X86_INS_VCMPGTSD, nullptr},
		{X86_INS_VCMPTRUESD, nullptr},
		{X86_INS_VCMPEQ_OSSD, nullptr},
		{X86_INS_VCMPLT_OQSD, nullptr},
		{X86_INS_VCMPLE_OQSD, nullptr},
		{X86_INS_VCMPUNORD_SSD, nullptr},
		{X86_INS_VCMPNEQ_USSD, nullptr},
		{X86_INS_VCMPNLT_UQSD, nullptr},
		{X86_INS_VCMPNLE_UQSD, nullptr},
		{X86_INS_VCMPORD_SSD, nullptr},
		{X86_INS_VCMPEQ_USSD, nullptr},
		{X86_INS_VCMPNGE_UQSD, nullptr},
		{X86_INS_VCMPNGT_UQSD, nullptr},
		{X86_INS_VCMPFALSE_OSSD, nullptr},
		{X86_INS_VCMPNEQ_OSSD, nullptr},
		{X86_INS_VCMPGE_OQSD, nullptr},
		{X86_INS_VCMPGT_OQSD, nullptr},
		{X86_INS_VCMPTRUE_USSD, nullptr},

		{X86_INS_VCMPPS, nullptr},
		{X86_INS_VCMPEQPS, nullptr},
		{X86_INS_VCMPLTPS, nullptr},
		{X86_INS_VCMPLEPS, nullptr},
		{X86_INS_VCMPUNORDPS, nullptr},
		{X86_INS_VCMPNEQPS, nullptr},
		{X86_INS_VCMPNLTPS, nullptr},
		{X86_INS_VCMPNLEPS, nullptr},
		{X86_INS_VCMPORDPS, nullptr},
		{X86_INS_VCMPEQ_UQPS, nullptr},
		{X86_INS_VCMPNGEPS, nullptr},
		{X86_INS_VCMPNGTPS, nullptr},
		{X86_INS_VCMPFALSEPS, nullptr},
		{X86_INS_VCMPNEQ_OQPS, nullptr},
		{X86_INS_VCMPGEPS, nullptr},
		{X86_INS_VCMPGTPS, nullptr},
		{X86_INS_VCMPTRUEPS, nullptr},
		{X86_INS_VCMPEQ_OSPS, nullptr},
		{X86_INS_VCMPLT_OQPS, nullptr},
		{X86_INS_VCMPLE_OQPS, nullptr},
		{X86_INS_VCMPUNORD_SPS, nullptr},
		{X86_INS_VCMPNEQ_USPS, nullptr},
		{X86_INS_VCMPNLT_UQPS, nullptr},
		{X86_INS_VCMPNLE_UQPS, nullptr},
		{X86_INS_VCMPORD_SPS, nullptr},
		{X86_INS_VCMPEQ_USPS, nullptr},
		{X86_INS_VCMPNGE_UQPS, nullptr},
		{X86_INS_VCMPNGT_UQPS, nullptr},
		{X86_INS_VCMPFALSE_OSPS, nullptr},
		{X86_INS_VCMPNEQ_OSPS, nullptr},
		{X86_INS_VCMPGE_OQPS, nullptr},
		{X86_INS_VCMPGT_OQPS, nullptr},
		{X86_INS_VCMPTRUE_USPS, nullptr},

		{X86_INS_VCMPPD, nullptr},
		{X86_INS_VCMPEQPD, nullptr},
		{X86_INS_VCMPLTPD, nullptr},
		{X86_INS_VCMPLEPD, nullptr},
		{X86_INS_VCMPUNORDPD, nullptr},
		{X86_INS_VCMPNEQPD, nullptr},
		{X86_INS_VCMPNLTPD, nullptr},
		{X86_INS_VCMPNLEPD, nullptr},
		{X86_INS_VCMPORDPD, nullptr},
		{X86_INS_VCMPEQ_UQPD, nullptr},
		{X86_INS_VCMPNGEPD, nullptr},
		{X86_INS_VCMPNGTPD, nullptr},
		{X86_INS_VCMPFALSEPD, nullptr},
		{X86_INS_VCMPNEQ_OQPD, nullptr},
		{X86_INS_VCMPGEPD, nullptr},
		{X86_INS_VCMPGTPD, nullptr},
		{X86_INS_VCMPTRUEPD, nullptr},
		{X86_INS_VCMPEQ_OSPD, nullptr},
		{X86_INS_VCMPLT_OQPD, nullptr},
		{X86_INS_VCMPLE_OQPD, nullptr},
		{X86_INS_VCMPUNORD_SPD, nullptr},
		{X86_INS_VCMPNEQ_USPD, nullptr},
		{X86_INS_VCMPNLT_UQPD, nullptr},
		{X86_INS_VCMPNLE_UQPD, nullptr},
		{X86_INS_VCMPORD_SPD, nullptr},
		{X86_INS_VCMPEQ_USPD, nullptr},
		{X86_INS_VCMPNGE_UQPD, nullptr},
		{X86_INS_VCMPNGT_UQPD, nullptr},
		{X86_INS_VCMPFALSE_OSPD, nullptr},
		{X86_INS_VCMPNEQ_OSPD, nullptr},
		{X86_INS_VCMPGE_OQPD, nullptr},
		{X86_INS_VCMPGT_OQPD, nullptr},
		{X86_INS_VCMPTRUE_USPD, nullptr},

		{X86_INS_ENDING, nullptr}, // mark the end of the list of insn
};

} // namespace capstone2llvmir
} // namespace retdec
