/***************************************************************************
     testqgsmaprendererjob.cpp
     --------------------------------------
    Date                 : Sun Sep 16 12:22:54 AKDT 2007
    Copyright            : (C) 2007 by Tim Sutton
    Email                : tim @ linfiniti.com
 ***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "qgstest.h"
#include <QObject>
#include <QString>
#include <QStringList>
#include <QPainter>
#include <QTime>
#include <QApplication>
#include <QDesktopServices>

#include "qgsvectorlayer.h"
#include "qgsvectorfilewriter.h"
#include "qgsfeature.h"
#include "qgsgeometry.h"
#include "qgscoordinatereferencesystem.h"
#include "qgsapplication.h"
#include "qgsfield.h"
#include "qgis.h"
#include "qgsmaprenderersequentialjob.h"
#include "qgsmaprenderercustompainterjob.h"
#include "qgsnullpainterdevice.h"
#include "qgsmaplayer.h"
#include "qgsreadwritecontext.h"
#include "qgsproject.h"
#include "qgsrenderedfeaturehandlerinterface.h"
#include "qgsmaprendererstagedrenderjob.h"
#include "qgspallabeling.h"
#include "qgsvectorlayerlabeling.h"
#include "qgsfontutils.h"
#include "qgsrasterlayer.h"
#include "qgssinglesymbolrenderer.h"
#include "qgsrasterlayertemporalproperties.h"
#include "qgslinesymbol.h"
#include "qgslabelsink.h"
#include "qgspointcloudlayer.h"
#include "qgspointcloudattributebyramprenderer.h"
#include "qgsmeshlayer.h"
#include "qgsfillsymbol.h"
#include "qgsrasterlayerelevationproperties.h"

/**
 * \ingroup UnitTests
 * This is a unit test for the QgsMapRendererJob class.
 * It will do some performance testing too
 *
 */
class TestQgsMapRendererJob : public QgsTest
{
    Q_OBJECT

  public:
    TestQgsMapRendererJob()
      : QgsTest( QStringLiteral( "Map Renderer Job Tests" ), QStringLiteral( "map_renderer" ) ) {}

    ~TestQgsMapRendererJob() override
    {
      delete mMapSettings;
    }

  private slots:
    void initTestCase();    // will be called before the first testfunction is executed.
    void cleanupTestCase(); // will be called after the last testfunction was executed.

    //! This method tests render performance
    void performanceTest();

    /**
     * This unit test checks if rendering of adjacent tiles (e.g. to render images for tile caches)
     * does not result in border effects
     */
    void testFourAdjacentTiles_data();
    void testFourAdjacentTiles();

    void testRenderedFeatureHandlers();
    void stagedRenderer();
    void stagedRendererWithStagedLabeling();

    void vectorLayerBoundsWithReprojection();

    void temporalRender();

    void labelSink();
    void skipSymbolRendering();

    void customNullPainterJob();

    void testMapShading();

  private:
    QString mEncoding;
    QgsVectorFileWriter::WriterError mError = QgsVectorFileWriter::NoError;
    QgsCoordinateReferenceSystem mCRS;
    QgsFields mFields;
    QgsMapSettings *mMapSettings = nullptr;
    QgsMapLayer *mpPolysLayer = nullptr;
};


void TestQgsMapRendererJob::initTestCase()
{
  //
  // Runs once before any tests are run
  //
  QgsApplication::init();
  QgsApplication::initQgis();
  QgsApplication::showSettings();

  mMapSettings = new QgsMapSettings();

  //create some objects that will be used in all tests...
  mEncoding = QStringLiteral( "UTF-8" );
  QgsField myField1( QStringLiteral( "Value" ), QMetaType::Type::Int, QStringLiteral( "int" ), 10, 0, QStringLiteral( "Value on lon" ) );
  mFields.append( myField1 );
  mCRS = QgsCoordinateReferenceSystem( geoWkt() );
  //
  // Create the test dataset if it doesn't exist
  //
  QString myDataDir( TEST_DATA_DIR ); //defined in CmakeLists.txt
  QString myTestDataDir = myDataDir + '/';
  QString myTmpDir = QDir::tempPath() + '/';
  QString myFileName = myTmpDir + "maprender_testdata.gpkg";
  //copy over the default qml for our generated layer
  QString myQmlFileName = myTestDataDir + "maprender_testdata.qml";
  QFile::remove( myTmpDir + "maprender_testdata.qml" );
  QVERIFY( QFile::copy( myQmlFileName, myTmpDir + "maprender_testdata.qml" ) );
  qDebug( "Checking test dataset exists...\n%s", myFileName.toLocal8Bit().constData() );
  if ( !QFile::exists( myFileName ) )
  {
    qDebug( "Creating test dataset: " );

    QgsVectorFileWriter::SaveVectorOptions saveOptions;
    saveOptions.fileEncoding = mEncoding;
    std::unique_ptr<QgsVectorFileWriter> writer( QgsVectorFileWriter::create( myFileName, mFields, Qgis::WkbType::Polygon, mCRS, QgsCoordinateTransformContext(), saveOptions ) );
    double myInterval = 0.5;
    for ( double i = -180.0; i <= 180.0; i += myInterval )
    {
      for ( double j = -90.0; j <= 90.0; j += myInterval )
      {
        //
        // Create a polygon feature
        //
        QgsPolylineXY myPolyline;
        QgsPointXY myPoint1 = QgsPointXY( i, j );
        QgsPointXY myPoint2 = QgsPointXY( i + myInterval, j );
        QgsPointXY myPoint3 = QgsPointXY( i + myInterval, j + myInterval );
        QgsPointXY myPoint4 = QgsPointXY( i, j + myInterval );
        myPolyline << myPoint1 << myPoint2 << myPoint3 << myPoint4 << myPoint1;
        QgsPolygonXY myPolygon;
        myPolygon << myPolyline;
        //polygon: first item of the list is outer ring,
        // inner rings (if any) start from second item
        //
        // NOTE: don't delete this pointer again -
        // ownership is passed to the feature which will
        // delete it in its dtor!
        QgsGeometry mypPolygonGeometry = QgsGeometry::fromPolygonXY( myPolygon );
        QgsFeature myFeature;
        myFeature.setGeometry( mypPolygonGeometry );
        myFeature.initAttributes( 1 );
        myFeature.setAttribute( 0, i );
        //
        // Write the feature to the filewriter
        // and check for errors
        //
        QVERIFY( writer->addFeature( myFeature ) );
        mError = writer->hasError();
        if ( mError == QgsVectorFileWriter::ErrDriverNotFound )
        {
          std::cout << "Driver not found error" << std::endl;
        }
        else if ( mError == QgsVectorFileWriter::ErrCreateDataSource )
        {
          std::cout << "Create data source error" << std::endl;
        }
        else if ( mError == QgsVectorFileWriter::ErrCreateLayer )
        {
          std::cout << "Create layer error" << std::endl;
        }
        QVERIFY( mError == QgsVectorFileWriter::NoError );
      }
    }
  } //file exists
  //
  //create a poly layer that will be used in all tests...
  //
  QFileInfo myPolyFileInfo( myFileName );
  mpPolysLayer = new QgsVectorLayer( myPolyFileInfo.filePath(), myPolyFileInfo.completeBaseName(), QStringLiteral( "ogr" ) );
  QVERIFY( mpPolysLayer->isValid() );
  // Register the layer with the registry
  QgsProject::instance()->addMapLayers( QList<QgsMapLayer *>() << mpPolysLayer );
  // add the test layer to the maprender
  mMapSettings->setLayers( QList<QgsMapLayer *>() << mpPolysLayer );
}

void TestQgsMapRendererJob::cleanupTestCase()
{
  QgsApplication::exitQgis();
}

void TestQgsMapRendererJob::performanceTest()
{
  mMapSettings->setExtent( mpPolysLayer->extent() );
  mMapSettings->setFlag( Qgis::MapSettingsFlag::Antialiasing );
  QGSVERIFYRENDERMAPSETTINGSCHECK( QStringLiteral( "maprender" ), QStringLiteral( "maprender" ), *mMapSettings, 0, 5 );
}

void TestQgsMapRendererJob::testFourAdjacentTiles_data()
{
  QTest::addColumn<QStringList>( "bboxList" );
  QTest::addColumn<QString>( "controlName" );
  QTest::addColumn<QString>( "shapeFile" );
  QTest::addColumn<QString>( "qmlFile" );

  QString shapeFile = TEST_DATA_DIR + QStringLiteral( "/france_parts.shp" );
  QString qmlFile = TEST_DATA_DIR + QStringLiteral( "/adjacent_tiles/line_pattern_30_degree.qml" );
  QString controlName = QStringLiteral( "adjacent_line_fill" );

  QStringList bboxList1;
  bboxList1 << QStringLiteral( "-1.5,48,-0.5,49" );
  bboxList1 << QStringLiteral( "-0.5,48,0.5,49" );
  bboxList1 << QStringLiteral( "-1.5,47,-0.5,48" );
  bboxList1 << QStringLiteral( "-0.5,47,0.5,48" );

  QTest::newRow( "adjacent_line_fill" ) << bboxList1 << controlName << shapeFile << qmlFile;

  qmlFile = TEST_DATA_DIR + QStringLiteral( "/adjacent_tiles/point_pattern_simple_marker.qml" );
  controlName = QStringLiteral( "adjacent_marker_fill" );

  QTest::newRow( "adjacent_marker_fill" ) << bboxList1 << controlName << shapeFile << qmlFile;

  shapeFile = TEST_DATA_DIR + QStringLiteral( "/lines.shp" );
  qmlFile = TEST_DATA_DIR + QStringLiteral( "/adjacent_tiles/simple_line_dashed.qml" );
  controlName = QStringLiteral( "adjacent_dashed_line" );

  QStringList bboxList2;
  bboxList2 << QStringLiteral( "-105,35,-95,45" );
  bboxList2 << QStringLiteral( "-95,35,-85,45" );
  bboxList2 << QStringLiteral( "-105,25,-95,35" );
  bboxList2 << QStringLiteral( "-95,25,-85,35" );

  QTest::newRow( "adjacent_dashed_line" ) << bboxList2 << controlName << shapeFile << qmlFile;
}

void TestQgsMapRendererJob::testFourAdjacentTiles()
{
  QFETCH( QStringList, bboxList );
  QFETCH( QString, controlName );
  QFETCH( QString, shapeFile );
  QFETCH( QString, qmlFile );

  QVERIFY( bboxList.size() == 4 );

  //create maplayer, set QML and add to maplayer registry
  QgsVectorLayer *vectorLayer = new QgsVectorLayer( shapeFile, QStringLiteral( "testshape" ), QStringLiteral( "ogr" ) );

  //todo: read QML
  QFile symbologyFile( qmlFile );
  if ( !symbologyFile.open( QIODevice::ReadOnly ) )
  {
    QFAIL( "Open symbology file failed" );
  }

  QDomDocument qmlDoc;
  if ( !qmlDoc.setContent( &symbologyFile ) )
  {
    QFAIL( "QML file not valid" );
  }

  QString errorMsg;
  QgsReadWriteContext context = QgsReadWriteContext();
  if ( !vectorLayer->readSymbology( qmlDoc.documentElement(), errorMsg, context ) )
  {
    QFAIL( errorMsg.toLocal8Bit().data() );
  }

  QgsProject::instance()->addMapLayers( QList<QgsMapLayer *>() << vectorLayer );

  QImage globalImage( 512, 512, QImage::Format_ARGB32_Premultiplied );
  globalImage.fill( Qt::white );
  QPainter globalPainter( &globalImage );

  for ( int i = 0; i < 4; ++i )
  {
    QgsMapSettings mapSettings;

    //extent
    QStringList rectCoords = bboxList.at( i ).split( ',' );
    if ( rectCoords.size() != 4 )
    {
      QFAIL( "bbox string invalid" );
    }
    QgsRectangle rect( rectCoords[0].toDouble(), rectCoords[1].toDouble(), rectCoords[2].toDouble(), rectCoords[3].toDouble() );
    mapSettings.setExtent( rect );
    mapSettings.setOutputSize( QSize( 256, 256 ) );
    mapSettings.setLayers( QList<QgsMapLayer *>() << vectorLayer );
    mapSettings.setFlags( Qgis::MapSettingsFlag::RenderMapTile );
    mapSettings.setOutputDpi( 96 );

    QgsMapRendererSequentialJob renderJob( mapSettings );
    renderJob.start();
    renderJob.waitForFinished();
    QImage img = renderJob.renderedImage();
    int globalImageX = ( i % 2 ) * 256;
    int globalImageY = ( i < 2 ) ? 0 : 256;
    globalPainter.drawImage( globalImageX, globalImageY, img );
  }

  QgsProject::instance()->removeMapLayers( QStringList() << vectorLayer->id() );

  QGSVERIFYIMAGECHECK( QTest::currentDataTag(), controlName, globalImage, QString(), 100 );
}

class TestHandler : public QgsRenderedFeatureHandlerInterface
{
  public:
    TestHandler( QList<QgsFeature> &features, QList<QgsGeometry> &geometries, bool allAttributes = false )
      : features( features )
      , geometries( geometries )
      , mAllAttributes( allAttributes )
    {}

    void handleRenderedFeature( const QgsFeature &feature, const QgsGeometry &geom, const QgsRenderedFeatureHandlerInterface::RenderedFeatureContext & ) override
    {
      features.append( feature );
      geometries.append( geom );
    }

    QSet<QString> usedAttributes( QgsVectorLayer *, const QgsRenderContext & ) const override
    {
      if ( !mAllAttributes )
        return QSet<QString>();
      else
        return QSet<QString>() << QgsFeatureRequest::ALL_ATTRIBUTES;
    }

    QList<QgsFeature> &features;
    QList<QgsGeometry> &geometries;

    bool mAllAttributes = false;
};


void TestQgsMapRendererJob::testRenderedFeatureHandlers()
{
  auto pointsLayer = std::make_unique<QgsVectorLayer>( TEST_DATA_DIR + QStringLiteral( "/points.shp" ), QStringLiteral( "points" ), QStringLiteral( "ogr" ) );
  QVERIFY( pointsLayer->isValid() );
  auto linesLayer = std::make_unique<QgsVectorLayer>( TEST_DATA_DIR + QStringLiteral( "/lines.shp" ), QStringLiteral( "lines" ), QStringLiteral( "ogr" ) );
  QVERIFY( linesLayer->isValid() );
  auto polygonsLayer = std::make_unique<QgsVectorLayer>( TEST_DATA_DIR + QStringLiteral( "/polys.shp" ), QStringLiteral( "polys" ), QStringLiteral( "ogr" ) );
  QVERIFY( polygonsLayer->isValid() );

  QgsMapSettings mapSettings;
  mapSettings.setExtent( linesLayer->extent() );
  mapSettings.setDestinationCrs( linesLayer->crs() );
  mapSettings.setOutputSize( QSize( 256, 256 ) );
  mapSettings.setLayers( QList<QgsMapLayer *>() << pointsLayer.get() << linesLayer.get() << polygonsLayer.get() );
  mapSettings.setFlags( Qgis::MapSettingsFlag::RenderMapTile );
  mapSettings.setOutputDpi( 96 );

  QList<QgsFeature> features1;
  QList<QgsGeometry> geometries1;
  TestHandler handler1( features1, geometries1 );
  QList<QgsFeature> features2;
  QList<QgsGeometry> geometries2;
  TestHandler handler2( features2, geometries2 );
  mapSettings.addRenderedFeatureHandler( &handler1 );
  mapSettings.addRenderedFeatureHandler( &handler2 );

  QgsMapRendererSequentialJob renderJob( mapSettings );
  renderJob.start();
  renderJob.waitForFinished();

  QCOMPARE( features1.count(), 33 );
  QCOMPARE( geometries1.count(), 33 );
  QCOMPARE( features2.count(), 33 );
  QCOMPARE( geometries2.count(), 33 );
  features1.clear();
  geometries1.clear();
  features2.clear();
  geometries2.clear();

  polygonsLayer->setSubsetString( QStringLiteral( "value=13" ) );
  pointsLayer->setSubsetString( QStringLiteral( "importance<=3 and \"Cabin Crew\">=1 and \"Cabin Crew\"<=2 and \"heading\" > 90 and \"heading\" < 290" ) );
  linesLayer->setSubsetString( QStringLiteral( "name='Highway'" ) );

  QgsMapRendererSequentialJob renderJob2( mapSettings );
  renderJob2.start();
  renderJob2.waitForFinished();

  QCOMPARE( features1.count(), 5 );
  QCOMPARE( geometries1.count(), 5 );
  QCOMPARE( features2.count(), 5 );
  QCOMPARE( geometries2.count(), 5 );

  QStringList attributes;
  for ( const QgsFeature &f : std::as_const( features1 ) )
  {
    QStringList bits;
    for ( const QVariant &v : f.attributes() )
      bits << v.toString();
    attributes << bits.join( ',' );
  }
  attributes.sort();
  QCOMPARE( attributes.at( 0 ), QStringLiteral( "Biplane,,,,," ) );
  QCOMPARE( attributes.at( 1 ), QStringLiteral( "Dam," ) );
  QCOMPARE( attributes.at( 2 ), QStringLiteral( "Highway," ) );
  QCOMPARE( attributes.at( 3 ), QStringLiteral( "Highway," ) );
  QCOMPARE( attributes.at( 4 ), QStringLiteral( "Jet,,,,," ) );

  QStringList wkts;
  for ( const QgsGeometry &g : std::as_const( geometries1 ) )
  {
    QgsDebugMsgLevel( g.asWkt( 1 ), 1 );
    wkts << g.asWkt( 1 );
  }
  wkts.sort();
  QCOMPARE( wkts.at( 0 ), QStringLiteral( "LineString (0 124.4, 0.3 124.4, 4 123.4, 5.5 121.9, 7.3 119.5, 8.2 118.9, 10.6 116.4, 11.2 115.8, 15.8 113.1, 20.4 111, 24 110.1, 30.7 108.5, 32.8 108.5, 36.5 108.5, 42.6 109.1, 44.4 109.5, 47.4 110.4, 49.9 110.4, 54.1 111.9, 55 112.5, 57.2 113.1, 58.4 113.7, 60.8 114.9, 62 115.5, 63.8 116.4, 64.8 116.8, 66.3 117.4, 72.4 119.2, 80 119.5, 83.9 120.7, 85.7 121.3, 87.3 121.6, 89.4 121.6, 91.8 121.6, 93.6 121.9, 95.5 122.2, 99.7 122.2, 100.3 122.2, 103.7 122.8, 106.4 122.8, 109.8 122.5, 112.8 122.5, 117.1 122.5, 121 121, 125.6 119.5, 127.4 118.3, 134.7 118, 144.7 120.4, 148.4 121.3, 156 121.9, 159.6 121.9, 168.4 119.2, 173.6 116.4, 179.1 112.5, 183.9 110.1, 188.5 107.6, 197.3 104.9, 197.3 104.9, 198.5 104.6, 204 105.2, 206.4 106.4, 211.3 107.6, 216.2 109.5, 218 110.1, 221 111, 229.5 112.2, 230.8 112.2, 243.2 111.6, 243.5 111, 251.4 107.3, 253.3 106.7)" ) );
  QCOMPARE( wkts.at( 1 ), QStringLiteral( "LineString (93.3 45.3, 94.6 48.3, 95.2 49.6, 95.8 50.5, 97.9 52.3, 104 55.3, 110.4 58.1, 112.2 59.6, 115.2 62.6, 115.5 63.2, 116.1 66.9, 116.4 68.7, 117.4 71.4, 117.7 72.7, 118 75.7, 118.6 76.6, 119.5 78.4, 120.4 80.3, 122.5 82.7, 122.8 83.3, 123.1 85.1, 123.1 87.9, 123.1 89.1, 123.7 91.5, 124.4 92.7, 126.8 94.6, 128.9 95.8, 131 97, 133.2 97.9, 133.5 99.7, 133.5 100.9, 132.3 103.1, 131.6 103.7, 130.7 105.5, 129.8 106.7, 128.9 107.6, 128.6 108.2, 128.3 109.8, 128 111, 128 112.2, 128.9 114.3, 129.5 115.5, 129.5 115.5, 132.6 122.2, 131 125.3, 129.8 126.5, 127.7 130.1, 127.7 132.6, 125 135, 124 136.5, 122.5 138.9, 122.5 140.2, 122.5 141.7, 122.5 142.9, 122.5 144.4, 123.4 145.6, 123.7 147.2, 124.4 148.4, 125.6 149.6, 126.8 151.4, 128.6 153.5, 131.6 159.3, 132 159.6, 134.1 161.7, 135 162.7, 137.1 165.7, 138.6 166.9, 141.4 168.7, 143.2 170.3, 145.3 172.4, 146.9 173.3, 149 175.1, 151.7 176.6, 153.8 178.2, 157.2 180.6, 158.4 181.8, 159.3 183.9, 159.3 185.8, 159.3 187.6, 159 188.2, 157.8 190, 156 192.8, 154.8 194, 154.5 195.2, 154.5 196.7, 154.8 197.9, 155.7 200.4, 156 201.6, 157.8 203.7, 158.7 204.6, 159.9 206.4, 161.7 209.2, 166.6 211.3)" ) );
  QCOMPARE( wkts.at( 2 ), QStringLiteral( "Polygon ((122.4 105.6, 165.2 105.6, 165.2 148.4, 122.4 148.4, 122.4 105.6))" ) );
  QCOMPARE( wkts.at( 3 ), QStringLiteral( "Polygon ((26.5 151.7, 31.3 153.8, 32.2 155.7, 32.8 156.3, 38 156.6, 41.3 156.6, 43.8 156.6, 48.3 157.5, 50.8 158.7, 52.9 160.8, 55 165.1, 55 165.1, 52.6 180.6, 52.6 182.7, 52.3 186.4, 50.2 190.3, 47.7 192.8, 45.9 194, 45 194.3, 30.4 193.7, 25.8 192.8, 20.4 190.9, 15.8 190, 13.4 188.8, 12.8 188.5, 11.6 187.9, 9.7 186.1, 8.8 184.9, 7.9 183.6, 7 181.8, 6.7 180.3, 7 177.9, 7.6 176.3, 8.5 173.9, 8.5 172.4, 7.9 170.6, 6.7 170, 4 168.4, -0.3 165.7, -1.8 163, -1.8 159.6, -1.5 156.6, 0 152.3, 7.6 147.5, 18.9 148.4, 25.2 150.5, 26.5 151.7))" ) );
  QCOMPARE( wkts.at( 4 ), QStringLiteral( "Polygon ((4.2 59.5, 73.5 59.5, 73.5 128.8, 4.2 128.8, 4.2 59.5))" ) );

  // now, use a handler which requires all attributes to be fetched
  QList<QgsFeature> features3;
  QList<QgsGeometry> geometries3;
  TestHandler handler3( features3, geometries3, true );
  mapSettings.addRenderedFeatureHandler( &handler3 );

  QgsMapRendererSequentialJob renderJob3( mapSettings );
  renderJob3.start();
  renderJob3.waitForFinished();

  QCOMPARE( features3.count(), 5 );
  QCOMPARE( geometries3.count(), 5 );
  attributes.clear();
  for ( const QgsFeature &f : std::as_const( features3 ) )
  {
    QStringList bits;
    for ( const QVariant &v : f.attributes() )
      bits << v.toString();
    attributes << bits.join( ',' );
  }
  attributes.sort();
  QCOMPARE( attributes.at( 0 ), QStringLiteral( "Biplane,240,1,3,2,5" ) );
  QCOMPARE( attributes.at( 1 ), QStringLiteral( "Dam,13" ) );
  QCOMPARE( attributes.at( 2 ), QStringLiteral( "Highway,1" ) );
  QCOMPARE( attributes.at( 3 ), QStringLiteral( "Highway,1" ) );
  QCOMPARE( attributes.at( 4 ), QStringLiteral( "Jet,95,3,1,1,2" ) );
}

void TestQgsMapRendererJob::stagedRenderer()
{
  // test the staged map renderer job subclass

  auto pointsLayer = std::make_unique<QgsVectorLayer>( TEST_DATA_DIR + QStringLiteral( "/points.shp" ), QStringLiteral( "points" ), QStringLiteral( "ogr" ) );
  QVERIFY( pointsLayer->isValid() );
  auto linesLayer = std::make_unique<QgsVectorLayer>( TEST_DATA_DIR + QStringLiteral( "/lines.shp" ), QStringLiteral( "lines" ), QStringLiteral( "ogr" ) );
  QVERIFY( linesLayer->isValid() );
  auto polygonsLayer = std::make_unique<QgsVectorLayer>( TEST_DATA_DIR + QStringLiteral( "/polys.shp" ), QStringLiteral( "polys" ), QStringLiteral( "ogr" ) );
  QVERIFY( polygonsLayer->isValid() );
  auto rasterLayer = std::make_unique<QgsRasterLayer>( TEST_DATA_DIR + QStringLiteral( "/raster_layer.tiff" ), QStringLiteral( "raster" ), QStringLiteral( "gdal" ) );
  QVERIFY( rasterLayer->isValid() );

  QgsMapSettings mapSettings;
  mapSettings.setExtent( linesLayer->extent() );
  mapSettings.setDestinationCrs( linesLayer->crs() );
  mapSettings.setOutputSize( QSize( 512, 512 ) );
  mapSettings.setFlag( Qgis::MapSettingsFlag::DrawLabeling, false );
  mapSettings.setOutputDpi( 96 );

  auto job = std::make_unique<QgsMapRendererStagedRenderJob>( mapSettings );
  job->start();
  // nothing to render
  QVERIFY( job->isFinished() );
  QVERIFY( !job->renderCurrentPart( nullptr ) );
  QVERIFY( job->currentLayerId().isEmpty() );
  QCOMPARE( job->currentStage(), QgsMapRendererStagedRenderJob::Finished );

  // with layers
  mapSettings.setLayers( QList<QgsMapLayer *>() << pointsLayer.get() << linesLayer.get() << rasterLayer.get() << polygonsLayer.get() );
  job = std::make_unique<QgsMapRendererStagedRenderJob>( mapSettings );
  job->start();
  QVERIFY( !job->isFinished() );
  QCOMPARE( job->currentLayerId(), polygonsLayer->id() );
  QCOMPARE( job->currentStage(), QgsMapRendererStagedRenderJob::Symbology );

  mapSettings.setBackgroundColor( QColor( 255, 255, 0 ) ); // should be ignored in this job
  QImage im( 512, 512, QImage::Format_ARGB32_Premultiplied );
  im.fill( Qt::transparent );
  QPainter painter( &im );
  QVERIFY( job->renderCurrentPart( &painter ) );
  painter.end();
  QGSVERIFYIMAGECHECK( QStringLiteral( "staged_render1" ), QStringLiteral( "staged_render1" ), im, QString(), 0, QSize(), 2 );
  QVERIFY( !job->isFinished() );
  QVERIFY( job->nextPart() );
  QCOMPARE( job->currentLayerId(), rasterLayer->id() );
  QCOMPARE( job->currentStage(), QgsMapRendererStagedRenderJob::Symbology );

  // second layer
  im = QImage( 512, 512, QImage::Format_ARGB32_Premultiplied );
  im.fill( Qt::transparent );
  painter.begin( &im );
  QVERIFY( job->renderCurrentPart( &painter ) );
  painter.end();
  QGSVERIFYIMAGECHECK( QStringLiteral( "staged_render_raster" ), QStringLiteral( "staged_render_raster" ), im, QString(), 0, QSize(), 2 );
  QVERIFY( !job->isFinished() );
  QVERIFY( job->nextPart() );
  QCOMPARE( job->currentLayerId(), linesLayer->id() );
  QCOMPARE( job->currentStage(), QgsMapRendererStagedRenderJob::Symbology );

  // third layer
  im = QImage( 512, 512, QImage::Format_ARGB32_Premultiplied );
  im.fill( Qt::transparent );
  painter.begin( &im );
  QVERIFY( job->renderCurrentPart( &painter ) );
  painter.end();
  QGSVERIFYIMAGECHECK( QStringLiteral( "staged_render2" ), QStringLiteral( "staged_render2" ), im, QString(), 0, QSize(), 2 );
  QVERIFY( !job->isFinished() );
  QVERIFY( job->nextPart() );
  QCOMPARE( job->currentLayerId(), pointsLayer->id() );
  QCOMPARE( job->currentStage(), QgsMapRendererStagedRenderJob::Symbology );

  // fourth layer
  im = QImage( 512, 512, QImage::Format_ARGB32_Premultiplied );
  im.fill( Qt::transparent );
  painter.begin( &im );
  QVERIFY( job->renderCurrentPart( &painter ) );
  painter.end();
  QGSVERIFYIMAGECHECK( QStringLiteral( "staged_render3" ), QStringLiteral( "staged_render3" ), im, QString(), 0, QSize(), 2 );

  // nothing left!
  QVERIFY( !job->nextPart() );
  QVERIFY( job->currentLayerId().isEmpty() );
  QCOMPARE( job->currentStage(), QgsMapRendererStagedRenderJob::Finished );
  QVERIFY( job->isFinished() );
  QVERIFY( !job->renderCurrentPart( &painter ) );
  // double check...
  QVERIFY( !job->renderCurrentPart( &painter ) );

  // job with labels
  mapSettings.setFlag( Qgis::MapSettingsFlag::DrawLabeling, true );

  QgsPalLayerSettings settings;
  settings.fieldName = QStringLiteral( "Class" );
  QgsTextFormat format;
  format.setFont( QgsFontUtils::getStandardTestFont( QStringLiteral( "Bold" ) ).family() );
  format.setSize( 12 );
  format.setNamedStyle( QStringLiteral( "Bold" ) );
  format.setColor( QColor( 200, 0, 200 ) );
  settings.setFormat( format );

  pointsLayer->setLabeling( new QgsVectorLayerSimpleLabeling( settings ) );
  pointsLayer->setLabelsEnabled( true );

  job = std::make_unique<QgsMapRendererStagedRenderJob>( mapSettings );
  job->start();
  QVERIFY( !job->isFinished() );
  QCOMPARE( job->currentLayerId(), polygonsLayer->id() );
  QCOMPARE( job->currentStage(), QgsMapRendererStagedRenderJob::Symbology );

  mapSettings.setBackgroundColor( QColor( 255, 255, 0 ) ); // should be ignored in this job
  im = QImage( 512, 512, QImage::Format_ARGB32_Premultiplied );
  im.fill( Qt::transparent );
  painter.begin( &im );
  QVERIFY( job->renderCurrentPart( &painter ) );
  painter.end();
  QGSVERIFYIMAGECHECK( QStringLiteral( "staged_render1" ), QStringLiteral( "staged_render1" ), im, QString(), 0, QSize(), 2 );
  QVERIFY( job->nextPart() );
  QVERIFY( !job->isFinished() );
  QCOMPARE( job->currentLayerId(), rasterLayer->id() );
  QCOMPARE( job->currentStage(), QgsMapRendererStagedRenderJob::Symbology );

  // second layer
  im = QImage( 512, 512, QImage::Format_ARGB32_Premultiplied );
  im.fill( Qt::transparent );
  painter.begin( &im );
  QVERIFY( job->renderCurrentPart( &painter ) );
  painter.end();
  QGSVERIFYIMAGECHECK( QStringLiteral( "staged_render_raster" ), QStringLiteral( "staged_render_raster" ), im, QString(), 0, QSize(), 2 );
  QVERIFY( job->nextPart() );
  QVERIFY( !job->isFinished() );
  QCOMPARE( job->currentLayerId(), linesLayer->id() );
  QCOMPARE( job->currentStage(), QgsMapRendererStagedRenderJob::Symbology );

  // third layer
  im = QImage( 512, 512, QImage::Format_ARGB32_Premultiplied );
  im.fill( Qt::transparent );
  painter.begin( &im );
  QVERIFY( job->renderCurrentPart( &painter ) );
  painter.end();
  QGSVERIFYIMAGECHECK( QStringLiteral( "staged_render2" ), QStringLiteral( "staged_render2" ), im, QString(), 0, QSize(), 2 );
  QVERIFY( job->nextPart() );
  QVERIFY( !job->isFinished() );
  QCOMPARE( job->currentLayerId(), pointsLayer->id() );
  QCOMPARE( job->currentStage(), QgsMapRendererStagedRenderJob::Symbology );

  // fourth layer
  im = QImage( 512, 512, QImage::Format_ARGB32_Premultiplied );
  im.fill( Qt::transparent );
  painter.begin( &im );
  QVERIFY( job->renderCurrentPart( &painter ) );
  painter.end();
  QGSVERIFYIMAGECHECK( QStringLiteral( "staged_render3" ), QStringLiteral( "staged_render3" ), im, QString(), 0, QSize(), 2 );
  QVERIFY( job->nextPart() );
  QVERIFY( !job->isFinished() );
  QVERIFY( job->currentLayerId().isEmpty() );
  QCOMPARE( job->currentStage(), QgsMapRendererStagedRenderJob::Labels );
  // labels
  im = QImage( 512, 512, QImage::Format_ARGB32_Premultiplied );
  im.fill( Qt::transparent );
  painter.begin( &im );
  QVERIFY( job->renderCurrentPart( &painter ) );
  painter.end();
  QGSVERIFYIMAGECHECK( QStringLiteral( "staged_render_points_labels" ), QStringLiteral( "staged_render_points_labels" ), im, QString(), 0, QSize(), 2 );

  // nothing left!
  QVERIFY( !job->nextPart() );
  QVERIFY( job->currentLayerId().isEmpty() );
  QCOMPARE( job->currentStage(), QgsMapRendererStagedRenderJob::Finished );
  QVERIFY( job->isFinished() );
  QVERIFY( !job->renderCurrentPart( &painter ) );
  // double check...
  QVERIFY( !job->renderCurrentPart( &painter ) );
}

void TestQgsMapRendererJob::stagedRendererWithStagedLabeling()
{
  // test the staged map renderer job subclass, when using staged labeling

  auto pointsLayer = std::make_unique<QgsVectorLayer>( TEST_DATA_DIR + QStringLiteral( "/points.shp" ), QStringLiteral( "points" ), QStringLiteral( "ogr" ) );
  QVERIFY( pointsLayer->isValid() );
  auto linesLayer = std::make_unique<QgsVectorLayer>( TEST_DATA_DIR + QStringLiteral( "/lines.shp" ), QStringLiteral( "lines" ), QStringLiteral( "ogr" ) );
  QVERIFY( linesLayer->isValid() );
  auto polygonsLayer = std::make_unique<QgsVectorLayer>( TEST_DATA_DIR + QStringLiteral( "/polys.shp" ), QStringLiteral( "polys" ), QStringLiteral( "ogr" ) );
  QVERIFY( polygonsLayer->isValid() );

  QgsMapSettings mapSettings;
  mapSettings.setExtent( linesLayer->extent() );
  mapSettings.setDestinationCrs( linesLayer->crs() );
  mapSettings.setOutputSize( QSize( 512, 512 ) );
  mapSettings.setFlag( Qgis::MapSettingsFlag::DrawLabeling, false );
  mapSettings.setOutputDpi( 96 );

  auto job = std::make_unique<QgsMapRendererStagedRenderJob>( mapSettings, QgsMapRendererStagedRenderJob::RenderLabelsByMapLayer );
  job->start();
  // nothing to render
  QVERIFY( job->isFinished() );
  QVERIFY( !job->renderCurrentPart( nullptr ) );
  QVERIFY( job->currentLayerId().isEmpty() );
  QCOMPARE( job->currentStage(), QgsMapRendererStagedRenderJob::Finished );

  // with layers
  mapSettings.setLayers( QList<QgsMapLayer *>() << pointsLayer.get() << linesLayer.get() << polygonsLayer.get() );
  job = std::make_unique<QgsMapRendererStagedRenderJob>( mapSettings, QgsMapRendererStagedRenderJob::RenderLabelsByMapLayer );
  job->start();
  QVERIFY( !job->isFinished() );
  QCOMPARE( job->currentLayerId(), polygonsLayer->id() );
  QCOMPARE( job->currentStage(), QgsMapRendererStagedRenderJob::Symbology );

  mapSettings.setBackgroundColor( QColor( 255, 255, 0 ) ); // should be ignored in this job
  QImage im( 512, 512, QImage::Format_ARGB32_Premultiplied );
  im.fill( Qt::transparent );
  QPainter painter( &im );
  QVERIFY( job->renderCurrentPart( &painter ) );
  painter.end();
  QGSVERIFYIMAGECHECK( QStringLiteral( "staged_render1" ), QStringLiteral( "staged_render1" ), im, QString(), 0, QSize(), 2 );
  QVERIFY( !job->isFinished() );
  QVERIFY( job->nextPart() );
  QCOMPARE( job->currentLayerId(), linesLayer->id() );
  QCOMPARE( job->currentStage(), QgsMapRendererStagedRenderJob::Symbology );

  // second layer
  im = QImage( 512, 512, QImage::Format_ARGB32_Premultiplied );
  im.fill( Qt::transparent );
  painter.begin( &im );
  QVERIFY( job->renderCurrentPart( &painter ) );
  painter.end();
  QGSVERIFYIMAGECHECK( QStringLiteral( "staged_render2" ), QStringLiteral( "staged_render2" ), im, QString(), 0, QSize(), 2 );
  QVERIFY( !job->isFinished() );
  QVERIFY( job->nextPart() );
  QCOMPARE( job->currentLayerId(), pointsLayer->id() );
  QCOMPARE( job->currentStage(), QgsMapRendererStagedRenderJob::Symbology );

  // third layer
  im = QImage( 512, 512, QImage::Format_ARGB32_Premultiplied );
  im.fill( Qt::transparent );
  painter.begin( &im );
  QVERIFY( job->renderCurrentPart( &painter ) );
  painter.end();
  QGSVERIFYIMAGECHECK( QStringLiteral( "staged_render3" ), QStringLiteral( "staged_render3" ), im, QString(), 0, QSize(), 2 );

  // nothing left!
  QVERIFY( !job->nextPart() );
  QVERIFY( job->currentLayerId().isEmpty() );
  QCOMPARE( job->currentStage(), QgsMapRendererStagedRenderJob::Finished );
  QVERIFY( job->isFinished() );
  QVERIFY( !job->renderCurrentPart( &painter ) );
  // double check...
  QVERIFY( !job->renderCurrentPart( &painter ) );

  // job with labels
  mapSettings.setFlag( Qgis::MapSettingsFlag::DrawLabeling, true );

  QgsPalLayerSettings settings;
  settings.fieldName = QStringLiteral( "Class" );
  QgsTextFormat format;
  format.setFont( QgsFontUtils::getStandardTestFont( QStringLiteral( "Bold" ) ).family() );
  format.setSize( 12 );
  format.setNamedStyle( QStringLiteral( "Bold" ) );
  format.setColor( QColor( 200, 0, 200 ) );
  settings.setFormat( format );
  settings.zIndex = 1;

  pointsLayer->setLabeling( new QgsVectorLayerSimpleLabeling( settings ) );
  pointsLayer->setLabelsEnabled( true );

  settings.fieldName = QStringLiteral( "Name" );
  settings.placement = Qgis::LabelPlacement::Line;
  settings.zIndex = 3;
  linesLayer->setLabeling( new QgsVectorLayerSimpleLabeling( settings ) );
  linesLayer->setLabelsEnabled( true );

  settings.fieldName = QStringLiteral( "Name" );
  settings.placement = Qgis::LabelPlacement::OverPoint;
  settings.zIndex = 2;
  settings.obstacleSettings().setType( QgsLabelObstacleSettings::ObstacleType::PolygonInterior );
  polygonsLayer->setLabeling( new QgsVectorLayerSimpleLabeling( settings ) );
  polygonsLayer->setLabelsEnabled( true );

  job = std::make_unique<QgsMapRendererStagedRenderJob>( mapSettings, QgsMapRendererStagedRenderJob::RenderLabelsByMapLayer );
  job->start();
  QVERIFY( !job->isFinished() );
  QCOMPARE( job->currentLayerId(), polygonsLayer->id() );
  QCOMPARE( job->currentStage(), QgsMapRendererStagedRenderJob::Symbology );

  mapSettings.setBackgroundColor( QColor( 255, 255, 0 ) ); // should be ignored in this job
  im = QImage( 512, 512, QImage::Format_ARGB32_Premultiplied );
  im.fill( Qt::transparent );
  painter.begin( &im );
  QVERIFY( job->renderCurrentPart( &painter ) );
  painter.end();
  QGSVERIFYIMAGECHECK( QStringLiteral( "staged_render1" ), QStringLiteral( "staged_render1" ), im, QString(), 0, QSize(), 2 );
  QVERIFY( job->nextPart() );
  QVERIFY( !job->isFinished() );
  QCOMPARE( job->currentLayerId(), linesLayer->id() );
  QCOMPARE( job->currentStage(), QgsMapRendererStagedRenderJob::Symbology );

  // second layer
  im = QImage( 512, 512, QImage::Format_ARGB32_Premultiplied );
  im.fill( Qt::transparent );
  painter.begin( &im );
  QVERIFY( job->renderCurrentPart( &painter ) );
  painter.end();
  QGSVERIFYIMAGECHECK( QStringLiteral( "staged_render2" ), QStringLiteral( "staged_render2" ), im, QString(), 0, QSize(), 2 );
  QVERIFY( job->nextPart() );
  QVERIFY( !job->isFinished() );
  QCOMPARE( job->currentLayerId(), pointsLayer->id() );
  QCOMPARE( job->currentStage(), QgsMapRendererStagedRenderJob::Symbology );

  // third layer
  im = QImage( 512, 512, QImage::Format_ARGB32_Premultiplied );
  im.fill( Qt::transparent );
  painter.begin( &im );
  QVERIFY( job->renderCurrentPart( &painter ) );
  painter.end();
  QGSVERIFYIMAGECHECK( QStringLiteral( "staged_render3" ), QStringLiteral( "staged_render3" ), im, QString(), 0, QSize(), 2 );

  // points labels (these must be in z-order!)
  QVERIFY( job->nextPart() );
  QVERIFY( !job->isFinished() );
  QCOMPARE( job->currentLayerId(), pointsLayer->id() );
  QCOMPARE( job->currentStage(), QgsMapRendererStagedRenderJob::Labels );
  im = QImage( 512, 512, QImage::Format_ARGB32_Premultiplied );
  im.fill( Qt::transparent );
  painter.begin( &im );
  QVERIFY( job->renderCurrentPart( &painter ) );
  painter.end();
  QGSVERIFYIMAGECHECK( QStringLiteral( "staged_render_points_staged_labels" ), QStringLiteral( "staged_render_points_staged_labels" ), im, QString(), 0, QSize(), 2 );

  // polygon labels
  QVERIFY( job->nextPart() );
  QVERIFY( !job->isFinished() );
  QCOMPARE( job->currentLayerId(), polygonsLayer->id() );
  QCOMPARE( job->currentStage(), QgsMapRendererStagedRenderJob::Labels );
  // labels
  im = QImage( 512, 512, QImage::Format_ARGB32_Premultiplied );
  im.fill( Qt::transparent );
  painter.begin( &im );
  QVERIFY( job->renderCurrentPart( &painter ) );
  painter.end();
  QGSVERIFYIMAGECHECK( QStringLiteral( "staged_render_polygons_staged_labels" ), QStringLiteral( "staged_render_polygons_staged_labels" ), im, QString(), 0, QSize(), 2 );

  // line labels
  QVERIFY( job->nextPart() );
  QVERIFY( !job->isFinished() );
  QCOMPARE( job->currentLayerId(), linesLayer->id() );
  QCOMPARE( job->currentStage(), QgsMapRendererStagedRenderJob::Labels );
  // labels
  im = QImage( 512, 512, QImage::Format_ARGB32_Premultiplied );
  im.fill( Qt::transparent );
  painter.begin( &im );
  QVERIFY( job->renderCurrentPart( &painter ) );
  painter.end();
  QGSVERIFYIMAGECHECK( QStringLiteral( "staged_render_lines_staged_labels" ), QStringLiteral( "staged_render_lines_staged_labels" ), im, QString(), 0, QSize(), 2 );

  // nothing left!
  QVERIFY( !job->nextPart() );
  QVERIFY( job->currentLayerId().isEmpty() );
  QCOMPARE( job->currentStage(), QgsMapRendererStagedRenderJob::Finished );
  QVERIFY( job->isFinished() );
  QVERIFY( !job->renderCurrentPart( &painter ) );
  // double check...
  QVERIFY( !job->renderCurrentPart( &painter ) );
}

void TestQgsMapRendererJob::vectorLayerBoundsWithReprojection()
{
  auto gridLayer = std::make_unique<QgsVectorLayer>( TEST_DATA_DIR + QStringLiteral( "/grid_4326.geojson" ), QStringLiteral( "grid" ), QStringLiteral( "ogr" ) );
  QVERIFY( gridLayer->isValid() );

  auto symbol = std::make_unique<QgsLineSymbol>();
  symbol->setColor( QColor( 255, 0, 255 ) );
  symbol->setWidth( 2 );
  auto renderer = std::make_unique<QgsSingleSymbolRenderer>( symbol.release() );
  gridLayer->setRenderer( renderer.release() );

  QgsMapSettings mapSettings;

  mapSettings.setDestinationCrs( QgsCoordinateReferenceSystem( QStringLiteral( "EPSG:3857" ) ) );
  mapSettings.setExtent( QgsRectangle( -37000835.1, -20182273.7, 37000835.1, 20182273.7 ) );
  mapSettings.setOutputSize( QSize( 512, 512 ) );
  mapSettings.setFlag( Qgis::MapSettingsFlag::DrawLabeling, false );
  mapSettings.setOutputDpi( 96 );
  mapSettings.setLayers( QList<QgsMapLayer *>() << gridLayer.get() );

  QgsMapRendererSequentialJob renderJob( mapSettings );
  renderJob.start();
  renderJob.waitForFinished();
  QImage img = renderJob.renderedImage();
  QGSVERIFYIMAGECHECK( QStringLiteral( "vector_layer_bounds_with_reprojection" ), QStringLiteral( "vector_layer_bounds_with_reprojection" ), img, QString(), 0, QSize(), 2 );
}

void TestQgsMapRendererJob::temporalRender()
{
  auto rasterLayer = std::make_unique<QgsRasterLayer>( TEST_DATA_DIR + QStringLiteral( "/raster_layer.tiff" ), QStringLiteral( "raster" ), QStringLiteral( "gdal" ) );
  QVERIFY( rasterLayer->isValid() );

  QgsMapSettings mapSettings;
  mapSettings.setExtent( rasterLayer->extent() );
  mapSettings.setDestinationCrs( rasterLayer->crs() );
  mapSettings.setOutputSize( QSize( 512, 512 ) );
  mapSettings.setFlag( Qgis::MapSettingsFlag::DrawLabeling, false );
  mapSettings.setOutputDpi( 96 );
  mapSettings.setLayers( QList<QgsMapLayer *>() << rasterLayer.get() );

  QgsMapRendererSequentialJob renderJob( mapSettings );
  renderJob.start();
  renderJob.waitForFinished();
  QImage img = renderJob.renderedImage();
  QGSVERIFYIMAGECHECK( QStringLiteral( "temporal_render_visible" ), QStringLiteral( "temporal_render_visible" ), img, QString(), 0, QSize(), 2 );

  // set temporal properties for layer
  QgsRasterLayerTemporalProperties *temporalProperties = qobject_cast<QgsRasterLayerTemporalProperties *>( rasterLayer->temporalProperties() );
  temporalProperties->setIsActive( true );
  temporalProperties->setMode( Qgis::RasterTemporalMode::FixedTemporalRange );
  temporalProperties->setFixedTemporalRange( QgsDateTimeRange( QDateTime( QDate( 2020, 1, 1 ), QTime( 0, 0, 0 ) ), QDateTime( QDate( 2020, 1, 5 ), QTime( 0, 0, 0 ) ) ) );

  // should still be visible -- map render job isn't temporal
  QgsMapRendererSequentialJob renderJob2( mapSettings );
  renderJob2.start();
  renderJob2.waitForFinished();
  img = renderJob2.renderedImage();
  QGSVERIFYIMAGECHECK( QStringLiteral( "temporal_render_visible" ), QStringLiteral( "temporal_render_visible" ), img, QString(), 0, QSize(), 2 );

  // make render job temporal, outside of layer's fixed range
  mapSettings.setIsTemporal( true );
  mapSettings.setTemporalRange( QgsDateTimeRange( QDateTime( QDate( 2021, 1, 1 ), QTime( 0, 0, 0 ) ), QDateTime( QDate( 2021, 1, 5 ), QTime( 0, 0, 0 ) ) ) );
  // should no longer be visible
  QgsMapRendererSequentialJob renderJob3( mapSettings );
  renderJob3.start();
  renderJob3.waitForFinished();
  img = renderJob3.renderedImage();
  QGSVERIFYIMAGECHECK( QStringLiteral( "temporal_render_invisible" ), QStringLiteral( "temporal_render_invisible" ), img, QString(), 0, QSize(), 2 );

  // temporal range ok for layer
  mapSettings.setTemporalRange( QgsDateTimeRange( QDateTime( QDate( 2020, 1, 2 ), QTime( 0, 0, 0 ) ), QDateTime( QDate( 2020, 1, 3 ), QTime( 0, 0, 0 ) ) ) );
  // should be visible
  QgsMapRendererSequentialJob renderJob4( mapSettings );
  renderJob4.start();
  renderJob4.waitForFinished();
  img = renderJob4.renderedImage();
  QGSVERIFYIMAGECHECK( QStringLiteral( "temporal_render_visible" ), QStringLiteral( "temporal_render_visible" ), img, QString(), 0, QSize(), 2 );
}

class TestLabelSink : public QgsLabelSink
{
  public:
    TestLabelSink() {};

    void drawLabel( const QString &layerId, QgsRenderContext &context, pal::LabelPosition *label, const QgsPalLayerSettings &settings ) override
    {
      Q_UNUSED( layerId )
      Q_UNUSED( context )
      Q_UNUSED( label )
      Q_UNUSED( settings )
      drawnCount++;
    };

    int drawnCount = 0;
};

void TestQgsMapRendererJob::labelSink()
{
  auto pointsLayer = std::make_unique<QgsVectorLayer>( TEST_DATA_DIR + QStringLiteral( "/points.shp" ), QStringLiteral( "points" ), QStringLiteral( "ogr" ) );
  QVERIFY( pointsLayer->isValid() );

  QgsPalLayerSettings settings;
  settings.fieldName = QStringLiteral( "Class" );
  QgsTextFormat format;
  format.setFont( QgsFontUtils::getStandardTestFont( QStringLiteral( "Bold" ) ).family() );
  format.setSize( 12 );
  format.setNamedStyle( QStringLiteral( "Bold" ) );
  format.setColor( QColor( 200, 0, 200 ) );
  settings.setFormat( format );
  settings.zIndex = 1;

  pointsLayer->setLabeling( new QgsVectorLayerSimpleLabeling( settings ) );
  pointsLayer->setLabelsEnabled( true );

  QgsMapSettings mapSettings;
  mapSettings.setDestinationCrs( pointsLayer->crs() );
  mapSettings.setExtent( pointsLayer->extent() );
  mapSettings.setOutputSize( QSize( 512, 512 ) );
  mapSettings.setFlag( Qgis::MapSettingsFlag::DrawLabeling, true );
  mapSettings.setOutputDpi( 96 );
  mapSettings.setLayers( QList<QgsMapLayer *>() << pointsLayer.get() );


  QgsMapRendererSequentialJob renderJob( mapSettings );

  auto labelSink = std::make_unique<TestLabelSink>();
  renderJob.setLabelSink( labelSink.get() );
  renderJob.start();
  renderJob.waitForFinished();
  QImage img = renderJob.renderedImage();
  QGSVERIFYIMAGECHECK( QStringLiteral( "label_sink" ), QStringLiteral( "label_sink" ), img, QString(), 0, QSize(), 2 );
  QCOMPARE( labelSink->drawnCount, 17 );
}

void TestQgsMapRendererJob::skipSymbolRendering()
{
  auto pointsLayer = std::make_unique<QgsVectorLayer>( TEST_DATA_DIR + QStringLiteral( "/points.shp" ), QStringLiteral( "points" ), QStringLiteral( "ogr" ) );
  QVERIFY( pointsLayer->isValid() );

  QgsPalLayerSettings settings;
  settings.fieldName = QStringLiteral( "Class" );
  QgsTextFormat format;
  format.setFont( QgsFontUtils::getStandardTestFont( QStringLiteral( "Bold" ) ).family() );
  format.setSize( 12 );
  format.setNamedStyle( QStringLiteral( "Bold" ) );
  format.setColor( QColor( 200, 0, 200 ) );
  settings.setFormat( format );
  settings.zIndex = 1;

  pointsLayer->setLabeling( new QgsVectorLayerSimpleLabeling( settings ) );
  pointsLayer->setLabelsEnabled( true );

  QgsMapSettings mapSettings;
  mapSettings.setDestinationCrs( pointsLayer->crs() );
  mapSettings.setExtent( pointsLayer->extent() );
  mapSettings.setOutputSize( QSize( 512, 512 ) );
  mapSettings.setFlag( Qgis::MapSettingsFlag::DrawLabeling, true );
  mapSettings.setFlag( Qgis::MapSettingsFlag::SkipSymbolRendering, true );
  mapSettings.setOutputDpi( 96 );
  mapSettings.setLayers( QList<QgsMapLayer *>() << pointsLayer.get() );

  QgsMapRendererSequentialJob renderJob( mapSettings );
  renderJob.start();
  renderJob.waitForFinished();
  QImage img = renderJob.renderedImage();
  QGSVERIFYIMAGECHECK( QStringLiteral( "skip_symbol_rendering" ), QStringLiteral( "skip_symbol_rendering" ), img, QString(), 0, QSize(), 2 );
}

void TestQgsMapRendererJob::customNullPainterJob()
{
  auto pointsLayer = std::make_unique<QgsVectorLayer>( TEST_DATA_DIR + QStringLiteral( "/points.shp" ), QStringLiteral( "points" ), QStringLiteral( "ogr" ) );
  QVERIFY( pointsLayer->isValid() );

  QgsPalLayerSettings settings;
  settings.fieldName = QStringLiteral( "Class" );
  QgsTextFormat format;
  format.setFont( QgsFontUtils::getStandardTestFont( QStringLiteral( "Bold" ) ).family() );
  format.setSize( 12 );
  format.setNamedStyle( QStringLiteral( "Bold" ) );
  format.setColor( QColor( 200, 0, 200 ) );
  settings.setFormat( format );
  settings.zIndex = 1;

  pointsLayer->setLabeling( new QgsVectorLayerSimpleLabeling( settings ) );
  pointsLayer->setLabelsEnabled( true );

  QgsMapSettings mapSettings;
  mapSettings.setDestinationCrs( pointsLayer->crs() );
  mapSettings.setExtent( pointsLayer->extent() );
  mapSettings.setOutputSize( QSize( 512, 512 ) );
  mapSettings.setFlag( Qgis::MapSettingsFlag::DrawLabeling, true );
  mapSettings.setOutputDpi( 96 );
  mapSettings.setLayers( QList<QgsMapLayer *>() << pointsLayer.get() );

  auto nullPaintDevice = std::make_unique<QgsNullPaintDevice>();
  nullPaintDevice->setOutputSize( QSize( 512, 512 ) );
  nullPaintDevice->setOutputDpi( 96 );
  auto painter = std::make_unique<QPainter>( nullPaintDevice.get() );

  QgsMapRendererCustomPainterJob renderJob( mapSettings, painter.get() );

  auto labelSink = std::make_unique<TestLabelSink>();
  renderJob.setLabelSink( labelSink.get() );

  renderJob.start();
  renderJob.waitForFinished();

  QCOMPARE( labelSink->drawnCount, 17 );
}

void TestQgsMapRendererJob::testMapShading()
{
  auto pointCloudLayer = std::make_unique<QgsPointCloudLayer>(
    TEST_DATA_DIR + QStringLiteral( "/point_clouds/ept/lone-star-laszip/ept.json" ),
    QStringLiteral( "point-cloud" ),
    QStringLiteral( "ept" )
  );
  QVERIFY( pointCloudLayer->isValid() );

  std::unique_ptr<QgsPointCloudAttributeByRampRenderer> pointCloudRenderer( new QgsPointCloudAttributeByRampRenderer );
  pointCloudRenderer->setDrawOrder2d( Qgis::PointCloudDrawOrder::BottomToTop );
  pointCloudLayer->setRenderer( pointCloudRenderer.release() );

  auto rasterLayer = std::make_unique<QgsRasterLayer>(
    TEST_DATA_DIR + QStringLiteral( "/raster/raster_shading.tif" ),
    QStringLiteral( "raster" ),
    QStringLiteral( "gdal" )
  );
  QVERIFY( rasterLayer->isValid() );
  static_cast<QgsRasterLayerElevationProperties *>( rasterLayer->elevationProperties() )->setEnabled( true );
  rasterLayer->dataProvider()->enableProviderResampling( true );
  rasterLayer->dataProvider()->setZoomedOutResamplingMethod( Qgis::RasterResamplingMethod::Cubic );

  auto meshLayer = std::make_unique<QgsMeshLayer>(
    TEST_DATA_DIR + QStringLiteral( "/mesh/mesh_shading.nc" ),
    QStringLiteral( "mesh" ),
    QStringLiteral( "mdal" )
  );
  QVERIFY( meshLayer->isValid() );

  auto vectorLayer = std::make_unique<QgsVectorLayer>(
    QStringLiteral( "Polygon?crs=%1&field=id:integer&field=name:string(20)&index=no" )
      .arg( pointCloudLayer->crs().toWkt( Qgis::CrsWktVariant::Preferred ) ),
    QStringLiteral( "vector-layer" ),
    QStringLiteral( "memory" )
  );
  QVERIFY( vectorLayer->isValid() );

  QgsFeature ft0( vectorLayer->fields() );
  QgsGeometry geom = QgsGeometry::fromWkt( QStringLiteral( "Polygon ((515336.29208192101214081 4918401.81226893234997988, 515336.29208192101214081 4918321.69825699832290411, 515439.75482995307538658 4918321.26983982231467962, 515441.89691583369858563 4918402.02647752035409212, 515336.29208192101214081 4918401.81226893234997988))" ) );
  ft0.setGeometry( geom );
  vectorLayer->startEditing();
  vectorLayer->addFeature( ft0 );
  vectorLayer->commitChanges();
  QVERIFY( vectorLayer->featureCount() == 1 );
  std::unique_ptr<QgsFillSymbol> fill( static_cast<QgsFillSymbol *>( QgsSymbol::defaultSymbol( Qgis::GeometryType::Polygon ) ) );
  fill->setColor( QColor( 255, 0, 255 ) );
  vectorLayer->setRenderer( new QgsSingleSymbolRenderer( fill.release() ) );

  QgsMapSettings mapSettings;
  mapSettings.setDestinationCrs( pointCloudLayer->crs() );
  mapSettings.setExtent( pointCloudLayer->extent() );
  mapSettings.setOutputSize( QSize( 512, 512 ) );
  mapSettings.setOutputDpi( 96 );
  mapSettings.setLayers( QList<QgsMapLayer *>() << pointCloudLayer.get() << rasterLayer.get() << vectorLayer.get() << meshLayer.get() );

  QgsElevationShadingRenderer shadingRenderer;
  shadingRenderer.setActive( true );
  shadingRenderer.setActiveHillshading( true );
  shadingRenderer.setActiveEyeDomeLighting( false );
  mapSettings.setElevationShadingRenderer( shadingRenderer );
  auto renderJob = std::make_unique<QgsMapRendererSequentialJob>( mapSettings );
  renderJob->start();
  renderJob->waitForFinished();
  QImage img = renderJob->renderedImage();
  QGSVERIFYIMAGECHECK( QStringLiteral( "render_shading_1" ), QStringLiteral( "render_shading_1" ), img, QString(), 0, QSize(), 2 );

  shadingRenderer.setLightAltitude( 20 );
  shadingRenderer.setLightAzimuth( 60 );
  mapSettings.setElevationShadingRenderer( shadingRenderer );
  renderJob.reset( new QgsMapRendererSequentialJob( mapSettings ) );
  renderJob->start();
  renderJob->waitForFinished();
  img = renderJob->renderedImage();
  QGSVERIFYIMAGECHECK( QStringLiteral( "render_shading_2" ), QStringLiteral( "render_shading_2" ), img, QString(), 0, QSize(), 2 );

  shadingRenderer.setHillshadingMultidirectional( true );
  shadingRenderer.setHillshadingZFactor( 5 );
  mapSettings.setElevationShadingRenderer( shadingRenderer );
  renderJob.reset( new QgsMapRendererSequentialJob( mapSettings ) );
  renderJob->start();
  renderJob->waitForFinished();
  img = renderJob->renderedImage();
  QGSVERIFYIMAGECHECK( QStringLiteral( "render_shading_3" ), QStringLiteral( "render_shading_3" ), img, QString(), 0, QSize(), 2 );

  shadingRenderer.setCombinedElevationMethod( Qgis::ElevationMapCombineMethod::NewerElevation );
  shadingRenderer.setActiveHillshading( false );
  shadingRenderer.setActiveEyeDomeLighting( true );
  mapSettings.setElevationShadingRenderer( shadingRenderer );
  renderJob.reset( new QgsMapRendererSequentialJob( mapSettings ) );
  renderJob->start();
  renderJob->waitForFinished();
  img = renderJob->renderedImage();
  QGSVERIFYIMAGECHECK( QStringLiteral( "render_shading_4" ), QStringLiteral( "render_shading_4" ), img, QString(), 0, QSize(), 2 );

  shadingRenderer.setEyeDomeLightingDistance( 10 );
  shadingRenderer.setEyeDomeLightingStrength( 4000 );
  mapSettings.setElevationShadingRenderer( shadingRenderer );
  renderJob.reset( new QgsMapRendererSequentialJob( mapSettings ) );
  renderJob->start();
  renderJob->waitForFinished();
  img = renderJob->renderedImage();
  QGSVERIFYIMAGECHECK( QStringLiteral( "render_shading_5" ), QStringLiteral( "render_shading_5" ), img, QString(), 0, QSize(), 2 );

  // test elevation map when rendering point cloud with triangulation
  QgsElevationShadingRenderer shadingRenderer2;
  shadingRenderer2.setActive( true );
  shadingRenderer2.setActiveHillshading( true );
  shadingRenderer2.setActiveEyeDomeLighting( false );
  pointCloudLayer->renderer()->setRenderAsTriangles( true );
  mapSettings.setLayers( QList<QgsMapLayer *>() << pointCloudLayer.get() );
  mapSettings.setElevationShadingRenderer( shadingRenderer2 );
  renderJob.reset( new QgsMapRendererSequentialJob( mapSettings ) );
  renderJob->start();
  renderJob->waitForFinished();
  img = renderJob->renderedImage();
  QGSVERIFYIMAGECHECK( QStringLiteral( "render_shading_point_cloud_triangles" ), QStringLiteral( "render_shading_point_cloud_triangles" ), img, QString(), 0, QSize(), 2 );
}

QGSTEST_MAIN( TestQgsMapRendererJob )
#include "testqgsmaprendererjob.moc"
