/* PSPP - a program for statistical analysis.
   Copyright (C) 2009, 2010, 2011, 2018 Free Sonftware Foundation, Inc.

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>. */

#ifndef OUTPUT_PAGE_SETUP_ITEM_H
#define OUTPUT_PAGE_SETUP_ITEM_H 1

/* Page setup items.

   A page setup item configures the paper size, margins, header and footer,
   and other attributes used for printing. */

#include <stdbool.h>
#include "output/output-item.h"
#include "table.h"

enum page_orientation
  {
    PAGE_PORTRAIT,
    PAGE_LANDSCAPE
  };

enum page_chart_size
  {
    PAGE_CHART_AS_IS,
    PAGE_CHART_FULL_HEIGHT,
    PAGE_CHART_HALF_HEIGHT,
    PAGE_CHART_QUARTER_HEIGHT,
  };

struct page_paragraph
  {
    char *markup;
    int halign;                 /* TAB_LEFT, TAB_CENTER, TAB_RIGHT. */
  };

struct page_heading
  {
    struct page_paragraph *paragraphs;
    size_t n;
  };

void page_heading_copy (struct page_heading *, const struct page_heading *);
void page_heading_uninit (struct page_heading *);

struct page_setup
  {
    int initial_page_number;
    double paper[TABLE_N_AXES];         /* Paper size in inches. */
    double margins[TABLE_N_AXES][2];    /* In inches. */
    enum page_orientation orientation;
    double object_spacing;      /* Space between objects, in inches. */
    enum page_chart_size chart_size;
    struct page_heading headings[2]; /* Header and footer. */
    char *file_name;
  };

#define PAGE_SETUP_INITIALIZER                                          \
    {                                                                   \
        .initial_page_number = 1,                                       \
        .paper = { [TABLE_HORZ] = 8.5, [TABLE_VERT] = 11.0 },           \
        .margins = { { 0.5, 0.5 }, { 0.5, 0.5 } },                      \
        .orientation = PAGE_PORTRAIT,                                   \
        .object_spacing = 12.0 / 72.0,                                  \
        .chart_size = PAGE_CHART_AS_IS,                                 \
    }

struct page_setup *page_setup_clone (const struct page_setup *);
void page_setup_destroy (struct page_setup *);

/* A page setup item. */
struct page_setup_item
  {
    struct output_item output_item;
    struct page_setup *page_setup;
  };

struct page_setup_item *page_setup_item_create (const struct page_setup *);

/* This boilerplate for page_setup_item, a subclass of output_item, was
   autogenerated by mk-class-boilerplate. */

#include <assert.h>
#include "libpspp/cast.h"

extern const struct output_item_class page_setup_item_class;

/* Returns true if SUPER is a page_setup_item, otherwise false. */
static inline bool
is_page_setup_item (const struct output_item *super)
{
  return super->class == &page_setup_item_class;
}

/* Returns SUPER converted to page_setup_item.  SUPER must be a page_setup_item, as
   reported by is_page_setup_item. */
static inline struct page_setup_item *
to_page_setup_item (const struct output_item *super)
{
  assert (is_page_setup_item (super));
  return UP_CAST (super, struct page_setup_item, output_item);
}

/* Returns INSTANCE converted to output_item. */
static inline struct output_item *
page_setup_item_super (const struct page_setup_item *instance)
{
  return CONST_CAST (struct output_item *, &instance->output_item);
}

/* Increments INSTANCE's reference count and returns INSTANCE. */
static inline struct page_setup_item *
page_setup_item_ref (const struct page_setup_item *instance)
{
  return to_page_setup_item (output_item_ref (&instance->output_item));
}

/* Decrements INSTANCE's reference count, then destroys INSTANCE if
   the reference count is now zero. */
static inline void
page_setup_item_unref (struct page_setup_item *instance)
{
  output_item_unref (&instance->output_item);
}

/* Returns true if INSTANCE's reference count is greater than 1,
   false otherwise. */
static inline bool
page_setup_item_is_shared (const struct page_setup_item *instance)
{
  return output_item_is_shared (&instance->output_item);
}

void page_setup_item_submit (struct page_setup_item *);

#endif /* output/page-setup-item.h */
