/* PgSqlClient - ADO.NET Data Provider for PostgreSQL 7.4+
 * Copyright (c) 2003-2004 Carlos Guzman Alvarez
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

using System;
using System.Data;
using System.Drawing;
using System.Collections;
using System.Collections.Specialized;
using System.ComponentModel;
using System.Text;
using System.Text.RegularExpressions;

using PostgreSql.Data.NPgClient;

namespace PostgreSql.Data.PgSqlClient
{
	[ToolboxItem(true),
	ToolboxBitmap(typeof(PgCommand), "Resources.ToolBox.PgCommand.bmp")]
	public sealed class PgCommand : Component, IDbCommand, ICloneable
	{				
		#region Fields
		
		private PgConnection			connection;
		private PgTransaction			transaction;				
		private PgParameterCollection	parameters;
		private UpdateRowSource			updatedRowSource;
		private CommandBehavior			commandBehavior;
		private bool					disposed;
		private int						actualCommand;
		private string[]				commands;
		private string					commandText;
		private CommandType				commandType;
		private int						commandTimeout;
		private bool					designTimeVisible;
		private StringCollection		namedParameters;
		private int						matchIndex;
		private Hashtable				matchReplaces;
		private PgStatement				statement;
		
		#endregion

		#region Properties

		[Category("Data"),
		DefaultValue(""),
		RefreshProperties(RefreshProperties.All)]
		public string CommandText
		{
			get { return this.commandText; }
			set 
			{
				if (this.statement != null && this.commandText != value && 
					this.commandText != null && this.commandText.Length != 0)
				{
					this.InternalClose();
				}

				this.commandText	= value;
				this.actualCommand	= 0;
				this.commands		= null;
			}
		}

		[Category("Data"),
		DefaultValue(CommandType.Text),
		RefreshProperties(RefreshProperties.All)]		
		public CommandType CommandType
		{
			get { return this.commandType; }
			set { this.commandType = value; }
		}
		
		[ReadOnly(true),
		DefaultValue(30)]
		public int CommandTimeout
		{
			get { return this.commandTimeout; }			
			set
			{
				if (value < 0) 
				{
					throw new ArgumentException("The property value assigned is less than 0.");
				}
				else
				{					
					throw new NotSupportedException();
				}
			}
		}

		IDbConnection IDbCommand.Connection
		{
			get { return this.Connection; }
			set { this.Connection = (PgConnection)value; }
		}

		[Category("Behavior"), DefaultValue(null)]
		public PgConnection Connection
		{
			get { return this.connection; }
			set
			{
				if (this.connection != null && 
					this.connection.DataReader != null)
				{
					throw new InvalidOperationException("There is already an open DataReader associated with this Connection which must be closed first.");
				}
				if (this.transaction != null && !this.transaction.IsUpdated)
				{
					throw new InvalidOperationException("The Connection property was changed while a transaction was in progress.");
				}
				if (this.connection != value)
				{										
					if (this.transaction != null)
					{
						this.transaction = null;
					}

					this.InternalClose();
				}

				this.connection = value;
			}
		}

		[Browsable(false),
		DesignOnly(true),
		DefaultValue(true)]
		public bool DesignTimeVisible
		{
			get { return designTimeVisible; }
			set { designTimeVisible = value; }
		}

		IDataParameterCollection IDbCommand.Parameters
		{
			get { return this.Parameters; }
		}

		[Category("Data"),
		DesignerSerializationVisibility(DesignerSerializationVisibility.Content)]		
		public PgParameterCollection Parameters
		{
			get { return this.parameters; }
		}

		IDbTransaction IDbCommand.Transaction
		{
			get { return this.Transaction; }
			set { this.Transaction = (PgTransaction)value; }
		}

		[Browsable(false),
		DataSysDescription("Tansaction context used by the command."),
		DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]		
		public PgTransaction Transaction
		{
			get { return this.transaction; }
			set
			{
				if (this.connection != null && 
					this.connection.DataReader != null)
				{
					throw new InvalidOperationException("There is already an open DataReader associated with this Connection which must be closed first.");
				}

				this.transaction = value; 
			}
		}
		
		[Category("Behavior"), DefaultValue(UpdateRowSource.Both)]
		public UpdateRowSource UpdatedRowSource
		{
			get { return this.updatedRowSource; }
			set { this.updatedRowSource = value; }
		}

		#endregion

		#region Internal Properties

		internal CommandBehavior CommandBehavior
		{
			get { return this.commandBehavior; }
		}

		internal PgStatement Statement
		{
			get { return this.statement; }
		}

		internal int RecordsAffected
		{
			get 
			{ 
				if (this.statement != null)
				{
					return this.statement.RecordsAffected; 
				}
				return -1;
			}
		}

		internal bool IsDisposed
		{
			get { return this.disposed; }
		}
		
		#endregion

		#region Constructors

		public PgCommand() : base()
		{
			this.commandText		= String.Empty;
			this.commandType		= CommandType.Text;
			this.commandTimeout		= 30;
			this.actualCommand		= -1;
			this.updatedRowSource	= UpdateRowSource.Both;
			this.commandBehavior	= CommandBehavior.Default;
			this.designTimeVisible	= true;
			this.parameters			= new PgParameterCollection();
			this.namedParameters	= new StringCollection();
			this.matchReplaces		= new Hashtable();

			GC.SuppressFinalize(this);
		}

		public PgCommand(string cmdText) : this(cmdText, null, null)
		{
		}
		
		public PgCommand(string cmdText, PgConnection connection) : this(cmdText, connection, null)
		{
		}
		
		public PgCommand(string cmdText, PgConnection connection, PgTransaction transaction) : this()
		{
			this.CommandText = cmdText;
			this.connection  = connection;
			this.transaction = transaction;
		}				 

		#endregion

		#region IDisposable Methods
		
		protected override void Dispose(bool disposing)
		{
			if (!this.disposed)
			{
				try
				{
					if (disposing)
					{
						// release any managed resources

						if (this.connection != null &&
							this.connection.ActiveCommands != null)
						{
							this.connection.ActiveCommands.Remove(this);
						}
						
						this.InternalClose();

						this.commandText	= String.Empty;
						this.actualCommand	= -1;
						this.commands		= null;

						this.matchReplaces.Clear();
						this.matchReplaces = null;

						this.namedParameters.Clear();
						this.namedParameters = null;
					}
					
					// release any unmanaged resources
					
					disposed = true;
				}
				finally 
				{
					base.Dispose(disposing);
				}
			}
		}

		#endregion

		#region ICloneable Methods

		object ICloneable.Clone()
		{
			PgCommand command	= new PgCommand(
				this.commandText, 
				this.connection, 
				this.transaction);

			command.CommandType	= this.commandType;

			return command;
		}

		#endregion

		#region Methods
						
		public void Cancel()
		{			
			throw new NotSupportedException();
		}
		
		IDbDataParameter IDbCommand.CreateParameter()
		{
			return CreateParameter();
		}

		public PgParameter CreateParameter()
		{
			return new PgParameter();
		}

		public int ExecuteNonQuery()
		{
			this.checkCommand();

			this.splitBatchCommands(false);

			if (this.connection.DbConnection.Settings.SimpleQueryMode)
			{
				this.InternalQuery();
			}
			else
			{
				this.InternalPrepare();
				this.InternalExecute();
			}

			this.InternalSetOutputParameters();
			
			return this.statement.RecordsAffected;
		}
				
		IDataReader IDbCommand.ExecuteReader()
		{	
			return ExecuteReader();
		}

		IDataReader IDbCommand.ExecuteReader(CommandBehavior behavior)
		{
			return ExecuteReader(behavior);
		}

		public PgDataReader ExecuteReader()
		{	
			return ExecuteReader(CommandBehavior.Default);			
		}
		
		public PgDataReader ExecuteReader(CommandBehavior behavior)
		{
			this.checkCommand();

			commandBehavior = behavior;

			this.splitBatchCommands(true);

			this.InternalPrepare();

			if ((commandBehavior & System.Data.CommandBehavior.SequentialAccess) == System.Data.CommandBehavior.SequentialAccess ||
				(commandBehavior & System.Data.CommandBehavior.SingleResult) == System.Data.CommandBehavior.SingleResult ||
				(commandBehavior & System.Data.CommandBehavior.SingleRow) == System.Data.CommandBehavior.SingleRow ||
				(commandBehavior & System.Data.CommandBehavior.CloseConnection) == System.Data.CommandBehavior.CloseConnection ||
				commandBehavior == System.Data.CommandBehavior.Default)				
			{
				this.InternalExecute();
			}

			return new PgDataReader(this, this.connection);
		}

		public object ExecuteScalar()
		{
			this.checkCommand();

			object returnValue = null;

			this.splitBatchCommands(false);

			if (this.connection.DbConnection.Settings.SimpleQueryMode)
			{
				this.InternalQuery();
			}
			else
			{
				this.InternalPrepare();
				this.InternalExecute();
			}

			if (this.statement != null && this.statement.HasRows)
			{
				returnValue = ((object[])this.statement.Rows[0])[0];
			}

			return returnValue;
		}

		public void Prepare()
		{
			this.checkCommand();

			this.splitBatchCommands(false);
			this.InternalPrepare();
			this.connection.ActiveCommands.Add(this);
		}

		public string GetCommandPlan(bool verbose)
		{
			string plan;

			this.checkCommand();

			try
			{
				if (this.statement == null)
				{
					this.statement = this.connection.DbConnection.DB.CreateStatement(commandText);
					plan = this.statement.GetPlan(verbose);
					this.statement = null;
				}
				else
				{
					plan = this.statement.GetPlan(verbose);
				}
			}
			catch (PgClientException ex)
			{
				throw new PgException(ex.Message, ex);
			}

			return plan;
		}

		#endregion

		#region Internal Methods

		internal void InternalPrepare()
		{
			if (commands == null)
			{
				splitBatchCommands(false);
			}

			try
			{
				if (this.statement == null || 
					this.statement.Status == PgStatementStatus.Initial || 
					this.statement.Status == PgStatementStatus.Error)
				{
					if (commandType == CommandType.StoredProcedure)
					{
						commands[actualCommand] = parseSPCommandText();
					}
														
					string prepareName	= "PS" + getStmtName();
					string portalName	= "PR" + getStmtName();
				
					this.statement = this.connection.DbConnection.DB.CreateStatement(
						prepareName, 
						portalName, 
						this.parseParameterNames());

					// Parse statement
					this.statement.Parse();

					// Describe statement
					this.statement.Describe();
				}
				else
				{
					// Close existent portal
					this.statement.ClosePortal();
				}
			}
			catch (PgClientException ex)
			{
				throw new PgException(ex.Message, ex);
			}
		}

		internal void InternalExecute()
		{
			try
			{
				if (parameters.Count != 0)
				{
					// Set parameter values
					setParameterValues();
				}

				// Bind Statement
				this.statement.Bind();

				// Execute Statement
				this.statement.Execute();
			}
			catch (PgClientException ex)
			{
				throw new PgException(ex.Message, ex);
			}
		}

		internal void InternalQuery()
		{		
			if (this.commands == null)
			{
				this.splitBatchCommands(false);
			}

			// Add this command to the active command list
			if (this.connection.ActiveCommands != null)
			{
				if (!this.connection.ActiveCommands.Contains(this))
				{
					this.connection.ActiveCommands.Add(this);
				}
			}

			try
			{
				string commandText = String.Empty;

				commandText = null;

				if (this.commandType == CommandType.StoredProcedure)
				{
					this.commands[actualCommand] = this.parseSPCommandText();
				}

				this.statement = this.connection.DbConnection.DB.CreateStatement(
					String.Empty, 
					String.Empty, 
					this.parseParameterNames());

				this.statement.Query();
			}
			catch (PgClientException ex)
			{
				throw new PgException(ex.Message, ex);
			}
		}

		internal void InternalClose()
		{
			if (this.statement != null)
			{
				try
				{
					// Closing the prepared statement closes all his portals too.
					this.statement.Close();
					this.statement = null;
				}
				catch (PgClientException ex)
				{
					throw new PgException(ex.Message, ex);
				}
			}
		}
	
		internal bool NextResult()
		{
			bool returnValue = false;

			this.statement.Close();
			this.statement.ClosePortal();
			this.statement = null;

			if ((this.commandBehavior & CommandBehavior.SingleResult) == CommandBehavior.SingleResult ||
				this.commandBehavior == System.Data.CommandBehavior.Default)
			{
				this.actualCommand++;

				if (actualCommand >= commands.Length)
				{
					this.actualCommand--;
				}
				else
				{
					string commandText = this.commands[actualCommand];

					if (commandText != null && commandText.Trim().Length > 0)
					{
						this.InternalPrepare();
						this.InternalExecute();

						returnValue = true;
					}
				}
			}		

			return returnValue;
		}
	
		internal void InternalSetOutputParameters()
		{
			if (this.CommandType == CommandType.StoredProcedure && 
				parameters.Count > 0)
			{
				IEnumerator paramEnumerator = Parameters.GetEnumerator();
				int i = 0;

				object[] values = (object[])this.statement.Rows[0];

				while (paramEnumerator.MoveNext())
				{
					PgParameter parameter = ((PgParameter)paramEnumerator.Current);

					if (parameter.Direction == ParameterDirection.Output ||
						parameter.Direction == ParameterDirection.InputOutput ||
						parameter.Direction == ParameterDirection.ReturnValue)
					{
						parameter.Value = values[i];
						i++;
					}					
				}
			}			
		}

		#endregion

		#region Private Methods

		private void checkCommand()
		{
			if (this.transaction != null &&
				this.transaction.IsUpdated)
			{
				this.transaction = null;
			}
			if (this.connection == null || 
				this.connection.State != ConnectionState.Open)
			{
				throw new InvalidOperationException("Connection must valid and open");
			}
			if (this.connection.DataReader != null)
			{
				throw new InvalidOperationException("There is already an open DataReader associated with this Connection which must be closed first.");
			}
			if (this.connection.ActiveTransaction != null &&
				!this.connection.ActiveTransaction.IsUpdated &&
				this.Transaction == null)
			{
				throw new InvalidOperationException("Execute requires the Command object to have a Transaction object when the Connection object assigned to the command is in a pending local transaction.  The Transaction property of the Command has not been initialized.");
			}
			if (this.transaction != null && 
				!this.connection.Equals(Transaction.Connection))
			{
				throw new InvalidOperationException("Command Connection is not equal to Transaction Connection");
			}
			if (this.commandText == String.Empty || this.commandText == null)
			{
				throw new InvalidOperationException ("The command text for this Command has not been set.");
			}
		}

		private string parseSPCommandText()
		{
			string	result = CommandText;

			if (!commandText.Trim().ToLower().StartsWith("select "))
			{
				StringBuilder paramsText = new StringBuilder();

				// Append the stored proc parameter name
				paramsText.Append(CommandText);
				paramsText.Append("(");
				for (int i = 0; i < parameters.Count; i++)
				{
					if (parameters[i].Direction == ParameterDirection.Input ||
						parameters[i].Direction == ParameterDirection.InputOutput)
					{
						// Append parameter name to parameter list
						paramsText.Append(parameters[i].ParameterName);
						if (i != parameters.Count - 1)
						{
							paramsText = paramsText.Append(",");
						}
					}
				}
				paramsText.Append(")");
				paramsText.Replace(",)", ")");
				
				result = "select * from "  + paramsText.ToString();
			}

			return result;
		}

		private string getStmtName()
		{
			return GetHashCode().ToString() + 
				this.connection.GetHashCode().ToString() +
				DateTime.Now.Ticks;
		}

		private string parseParameterNames()
		{
			string sql = this.commands[actualCommand];
			
			this.namedParameters.Clear();

			if (sql.IndexOf("@") != -1)
			{
				this.matchReplaces.Clear();
				this.matchIndex = 0;

				string pattern = @"(('[^']*?\@[^']*')*[^'@]*?)*(?<param>@\w+)*([^'@]*?('[^']*?\@*[^']*'))*";

				Regex r = new Regex(pattern, RegexOptions.ExplicitCapture);

				MatchEvaluator me = new MatchEvaluator(matchEvaluator);

				sql = r.Replace(sql, me);

				this.matchReplaces.Clear();
			}			

			return sql;
		}

		private string matchEvaluator(Match match)
		{
			string input	= match.Value;
			string replace	= String.Empty;

			if (match.Groups["param"].Success)
			{	
				Group g = match.Groups["param"];

				if (!this.matchReplaces.ContainsKey(g.Value))
				{
					this.namedParameters.Add(g.Value);

					if (this.connection.DbConnection.Settings.SimpleQueryMode)
					{
						replace = this.parameters[matchIndex++].ConvertToPgString();
					}
					else
					{
						replace = "$" + ((this.matchIndex++) + 1).ToString();
					}

					this.matchReplaces.Add(g.Value, replace);
				}
				else
				{
					replace = this.matchReplaces[g.Value].ToString();
				}
												
				return Regex.Replace(input, g.Value, replace);
			}
			else
			{
				return match.Value;
			}
		}

		private void setParameterValues()
		{
			if (parameters.Count != 0)
			{
				for (int i = 0; i < this.statement.Parameters.Length; i++)
				{
					string parameterName = parameters[i].ParameterName;
					if (namedParameters.Count != 0)
					{
						try
						{
							parameterName = namedParameters[i].Trim();
						}
						catch
						{
							parameterName = parameters[i].ParameterName;
						}
					}

					int index = parameters.IndexOf(parameterName);

					if (parameters[index].Direction == ParameterDirection.Input ||
						parameters[index].Direction == ParameterDirection.InputOutput)
					{
						if (parameters[index].Value == System.DBNull.Value)
						{
							this.statement.Parameters[i].Value = null;
						}
						else
						{
							this.statement.Parameters[i].Value = parameters[index].Value;
						}
					}
				}
			}
		}

		private void splitBatchCommands(bool batchAllowed)
		{
			if (this.commands == null)
			{
				if (batchAllowed)
				{
					MatchCollection matches = Regex.Matches(
						this.commandText,
						"([^';]+('[^']*'))*[^';]*(?=;*)");

					this.commands = new string[matches.Count/2];
					int count = 0;
					for (int i = 0; i < matches.Count; i++)
					{
						if (matches[i].Value.Trim() != String.Empty)
						{
							this.commands[count] = matches[i].Value.Trim();
							count++;
						}
					}
				}
				else
				{
					this.commands = new string[]{this.commandText};
				}
			}
		}

		#endregion
	}
}