/* PgSqlClient - ADO.NET Data Provider for PostgreSQL 7.4+
 * Copyright (c) 2003-2004 Carlos Guzman Alvarez
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

using System;
using System.Data;
using System.Text;

namespace PostgreSql.Data.PgSqlClient.DbSchema
{
	internal class PgTablesSchema : PgAbstractDbSchema
	{
		#region Constructors

		public PgTablesSchema() : base("Tables")
		{
		}

		#endregion

		#region Add Methods

		public override void AddTables()
		{
			AddTable("pg_class");
		}

		public override void AddRestrictionColumns()
		{
			AddRestrictionColumn("pg_namespace.nspname"	, "TABLE_SCHEMA", null);
			AddRestrictionColumn("pg_class.relname"		, "TABLE_NAME", null);
			AddRestrictionColumn(getTableTypeExpression("pg_class.relkind"), "TABLE_TYPE", "pg_class.relkind");
		}

		public override void AddDataColumns()
		{
			AddDataColumn("pg_class.relhasindex"	, "HAS_INDEXES");
			AddDataColumn("pg_class.relisshared"	, "IS_SHARED");
			AddDataColumn("pg_class.relchecks"		, "CONSTRAINT_COUNT");
			AddDataColumn("pg_class.reltriggers"	, "TRIGGER_COUNT");
			AddDataColumn("pg_class.relhaspkey"		, "HAS_PRIMARY");
			AddDataColumn("pg_class.relhasrules"	, "HAS_RULES");
			AddDataColumn("pg_class.relhassubclass"	, "HAS_SUBCLASS");
			AddDataColumn("pg_description.description", "DESCRIPTION");
		}

		public override void AddJoins()
		{
			AddJoin("left join", "pg_namespace"		, "pg_class.relnamespace = pg_namespace.oid");
			AddJoin("left join", "pg_description"	, "pg_class.oid = pg_description.objoid");
		}

		public override void AddOrderByColumns()
		{
			AddOrderBy("pg_class.relkind");
			AddOrderBy("pg_namespace.nspname");
			AddOrderBy("pg_class.relname");
		}

		public override void AddWhereFilters()
		{
			AddWhereFilter("pg_class.relkind = 'r'");
		}

		#endregion

		#region Parse Methods

		public override object[] ParseRestrictions(object[] restrictions)
		{
			object[] parsed = restrictions;

			if (parsed != null)
			{
				if (parsed.Length == 4 && parsed[3] != null)
				{
					switch (parsed[3].ToString().ToUpper())
					{
						case "TABLE":
							parsed[3] = "r";
							break;

						case "INDEX":
							parsed[3] = "i";
							break;

						case "SEQUENCE":
							parsed[3] = "S";
							break;

						case "VIEW":
							parsed[3] = "v";
							break;

						case "COMPOSITE":
							parsed[3] = "c";
							break;

						case "SPECIAL":
							parsed[3] = "s";
							break;

						case "TOAST":
							parsed[3] = "t";
							break;
					}
				}
			}

			return parsed;
		}

		#endregion

		#region Private Methods

		private string getTableTypeExpression(string fieldName)
		{
			StringBuilder expression = new StringBuilder();

			expression.AppendFormat(" case {0} ", fieldName);
			expression.Append(" when 'r' THEN 'TABLE'");
			expression.Append(" when 'i' THEN 'INDEX'");
			expression.Append(" when 'S' THEN 'SEQUENCE'");
			expression.Append(" when 'v' THEN 'VIEW'");
			expression.Append(" when 'c' THEN 'COMPOSITE'");
			expression.Append(" when 's' THEN 'SPECIAL'");
			expression.Append(" when 't' THEN 'TOAST'");
			expression.Append(" END ");

			return expression.ToString();
		}

		#endregion
	}
}