/* -*- pftp-c -*- */
#ifdef HAVE_CONFIG_H
# include "config.h"
#endif
#if HAVE_STDLIB_H
# include <stdlib.h>
#endif
#if HAVE_STDIO_H
# include <stdio.h>
#endif
#if HAVE_UNISTD_H
# include <unistd.h>
#endif
#if HAVE_SIGNAL_H
# include <signal.h>
#endif
#if HAVE_MATH_H
# include <math.h>
#endif
#if HAVE_STDARG_H
# include <stdarg.h>
#endif
#if HAVE_INTTYPES_H
# include <inttypes.h>
#endif
#if HAVE_PWD_H
# include <pwd.h>
#endif
#if HAVE_TERMIOS_H
# include <termios.h>
#endif
#if HAVE_ERRNO_H
# include <errno.h>
#endif

#include <readline/readline.h>
#include <readline/history.h>

#if HAVE_GETOPT_LONG || HAVE_GETOPT
#include <getopt.h>
#endif

#if !HAVE_GETLINE
# include <getline.h>
#endif

#ifdef DEBUG
# include <assert.h>
#else
# define assert(x) //
#endif

#ifdef WITH_DMALLOC
# include <dmalloc.h>
#endif

#include <str.h>
#include <pftp.h>
#include <pftputil.h>
#include <pftp_client.h>
#include <pftputil_settings.h>
#include "ftp_basic_cli.h"
#include "tab-comp.h"
#include "nice_list.h"
#include "cli_comm.h"

#ifdef OLD_READLINE
#define rl_get_screen_size(x, y) (*x) = (*y) = 0
#endif /* OLD_READLINE */

#ifdef WIN32
# define WIN32_LEAN_AND_MEAN
# include <windows.h>
#endif

#if !HAVE_RINTL

long double rintl(long double x) 
{
    long double b = (long double)((int64_t)x);
    if (fabs((float)(x - b)) <= 0.5f)
    return b;
    if (b > 0)
    return b + 1;
    else
    return b - 1;
}

#endif

/* Used by output status functions */

typedef enum { INVALID, NEXTNONFATAL, NEXTFATAL } ftp_output_action_t;
typedef struct ftp_output_status_s *ftp_output_status_t;

struct ftp_output_status_s {
    pftp_server_t server;
    ftp_output_action_t action;
    ftp_output_status_t next;
};

/* Command functions. */
static int ftp_help(char *args, const char *usage);
static int ftp_cd(char *args, const char *usage);
static int ftp_lcd(char *args, const char *usage);
static int ftp_rm(char *args, const char *usage);
static int ftp_rmdir(char *args, const char *usage);
static int ftp_mkdir(char *args, const char *usage);
static int ftp_ls(char *args, const char *usage);
static int ftp_lls(char *args, const char *usage);
static int ftp_get(char *args, const char *usage);
static int ftp_put(char *args, const char *usage);
static int ftp_fxp(char *args, const char *usage);
static int ftp_rename(char *args, const char *usage);
static int ftp_rawcmd(char *args, const char *usage);
static int ftp_pwd(char *args, const char *usage);
static int ftp_quit(char *args, const char *usage);
static int ftp_begin_trans(char *args, const char *usage);
static int ftp_retry_trans(char *args, const char *usage);
static int ftp_clear_trans(char *args, const char *usage);
static int ftp_open(char *args, const char *usage);
static int ftp_user(char *args, const char *usage);
static int ftp_current(char *args, const char *usage);
static int ftp_load(char *args, const char *usage);
static int ftp_save(char *args, const char *usage);
static int ftp_savedef(char *args, const char *usage);
static int ftp_close(char *args, const char *usage);
static int ftp_view_trans(char *args, const char *usage);
static int ftp_refresh(char *args, const char *usage);
static int ftp_site(char *args, const char *usage);
static int ftp_set(char *args, const char *usage);
static int ftp_setdef(char *args, const char *usage);
static int ftp_debug(char *args, const char *usage);
static int ftp_mirror(char *args, const char *usage);

const command_t commands[] = {
    { "!", NULL, "Usage: !<shell cmd>", 
      "Runs whatever you wish in the shell." },
    { "?", ftp_help, "Usage: ? <command>", 
      "Shows help for a given command." },
    { "cd", ftp_cd, "Usage: cd <dir>", 
      "Change the remote directory." },
    { "clear", ftp_clear_trans, "Usage: clear [-e]",
      "Clear transferring queue.\n\t-e Clear errorqueue instead." },
    { "close", ftp_close, "Usage: close [<sitenumber>|<sitename>]",
      "Closes the current site or the given site"
      " (numbers correspond to those of current)." },
    { "current", ftp_current, "Usage: current [<site>]",
      "When given no arguments current displays all the opened sites."
      " When given a site number or name, it activates that site." }, 
    { "debug", ftp_debug, "Usage: debug <debuglevel>",
      "Sets current debuglevel. Higher debuglevel means more output." },
    { "exit", ftp_quit, "Usage: exit", 
      "Exits pftp." },
    { "fxp", ftp_fxp, "Usage: fxp <destftp> <filemask1> [<filemask2> ...]"
      "\nUsage: fxp -m <destftp> <srcfile/srcdir> <destfile/destdir>",
      "Transfer everything complying to filemask from srcftp to destftp."
      " With -m it transfers srcfile/srcdir from srcftp to destfile/destdir" 
      " at destftp." }, 
    { "get", ftp_get, "Usage: get <filemask1> [<filemask2> ...]"
      "\nUsage: get -m <remotefile/remotedir> <localfile/localdir>", 
      "Downloads everything complying to filemask. With -m it gets"
      " remotefile/remotedir to localfile/localdir. Does not accept filemasks"
      " with -m." },
    { "help", ftp_help, "Usage: help <command>", 
      "Shows help for a given command." },
    { "mirror", ftp_mirror, "Usage: mirror [<sourceftp> [<targetftp>]]",
      "Mirrors current directory on sourceftp to current directory at "
      "targetftp, if sourceftp is omitted current open ftp will be used, if "
      "targetftp omitted current local directory will be used. Use -R flag "
      "if you wish to mirror from local to a ftp."
      "\n\t-e Delete files not present at source."
      "\n\t-t Ignore time when deciding whether to download."
      "\n\t-n Only download newer files."
      "\n\t-r Don\'t go into subdirectories."
      "\n\t-R Switch source and target." },
    { "mkdir", ftp_mkdir, "Usage: mkdir <directory>", 
      "Creates remote directory." },
    { "lcd", ftp_lcd, "Usage: lcd <dir>",
      "Change the local directory. Can not be done with !cd." },
    { "load", ftp_load, "Usage: load [<sitenumber>|<sitename>]", 
      "Without arguments displays all the saved sites and their corresponding"
      " number. Activates a saved site by giving its number or name as"
      " argument." },
    { "logout", ftp_close, "Usage: logout [<sitenumber>|<sitename>]",
      "Closes the current site or the given site (numbers correspond to those"
      " of current)." },
    { "lls", ftp_lls, "Usage: lls [-ahlrd] [<filemask>]",
      "Lists everything, locally, complying to the filemask. Behaves exactly"
      " like pftp ls. For your own ls, use !ls.\n\t-a Do not hide entries"
      " starting with `.'.\n\t-h Human readable.\n\t-l Verbose mode.\n"
      "\t-r Do not use cache (useless)."
      "\n\t-d List directory entries instead of contents" },
    { "ls", ftp_ls, "Usage: ls [-ahlrd] [<filemask>]",
      "Lists everything complying to the filemask.\n\t-a Do not hide entries"
      " starting with `.'.\n\t-h Human readable.\n\t-l Verbose mode."
      "\n\t-r Do not use cache."
      "\n\t-d List directory entries instead of contents" },
    { "open", ftp_open, "Usage: open [username@]hostname[:port] | sitename",
      "Opens a host with a username on a port, or a stored sitename." },
    { "put", ftp_put, "Usage: put <filemask1> [<filemask2> ...]"
      "\nUsage: put -m <localfile/localdir> <remotefile/remotedir>", 
      "Works exactly like get, but in reverse." },  
    { "pwd", ftp_pwd, "Usage: pwd", 
      "Prints working directory." }, 
    { "quit", ftp_quit, "Usage: quit", 
      "Quits pftp." },
    { "rawcmd", ftp_rawcmd, "Usage: rawcmd <cmd>",
      "Sends cmd as a FTP protocol call."},
    { "refresh", ftp_refresh, "Usage: refresh [-a]",
      "Clears the cache of current directory.\n\t-a Clears the entire cache."},
    { "rename", ftp_rename, "Usage: rename <from> <to>", 
      "Renames remote file." },
    { "retry", ftp_retry_trans, "Usage: retry",
      "Moves all queued items from errorqueue to transfer queue." },
    { "rm", ftp_rm, "Usage: rm [-fr] <filemask>",
      "Removes everything that applies to filemask."
      "\n\t-f Force remove.\n\t-r Recursive remove." },
    { "rmdir", ftp_rmdir, "Usage: rmdir <directory>", 
      "Removes remote directory." },
    { "save", ftp_save, "Usage: save [-X -Y] [<sitenumber>|<sitename>]",
      "Without arguments saves current site."
      "\nSaves all data to sitename|sitenumber."
      "\nIf an unknown sitename is given for a unsaved site, the site is"
      " saved as that."
      "\nIf no site is open, nothing happens."
      "\n\t-X Save password. (Really not recommended)."
      "\n\t-Y Clear any saved password for the site." },
    { "savedef", ftp_savedef, "Usage: savedef",
      "Saves current global options to file." },
    { "set", ftp_set, "Usage: set [<sitenumber>|<sitename>] [<variable>]"
      " [<value>]",
      "If value is given it sets the variable to that value."
      " Otherwise it prints the current value. If no variable is given,"
      " it prints all variables and values. If no sitenumber or sitename is"
      " given, current site is used." },
    { "setdef", ftp_setdef, "Usage: setdef [<variable>] [<value>]",
      "Same as set but for defaults instead of a site." },
    { "site", ftp_site, "Usage: site <site-cmd>",
      "Sends site command to FTP server."},
    { "trans", ftp_begin_trans, "Usage: trans", 
      "Begin transferring queue." },
    { "user", ftp_user, "Usage: user <username> [<account>]",
      "Login to an ftp server, when it is opened." },
    { "view", ftp_view_trans, "Usage: view [-e]",
      "View transferring queue.\n\t-e View errorqueue instead." },
    { (char *)NULL, (ftp_cmdfunc_t *)NULL, (char *)NULL, (char *)NULL }
};

const size_t no_commands = (sizeof(commands) / sizeof(command_t)) - 1;
static int TERM_width = 80, TERM_height = 24;
static unsigned long debuglevel = 0;
static ftp_output_status_t ftp_output_status = NULL;

static void clear_output_status(void);
static void remove_server_output_status(const char *name);
static void add_output_status_action(pftp_server_t ftp, 
				     ftp_output_action_t action);
static int get_output_status_action(pftp_server_t ftp, 
				    ftp_output_action_t *action);

static int confirm(const char *que, int yes_default);
static void signal_handler(int signum);
static void error(const char *format, ...);
static int command_central(pftp_msg_t msg, pftp_server_t ftp,
			   void *userdata,
			   pftp_param_t p1, pftp_param_t p2);
static ssize_t simple_getline(char **pass, size_t *len, FILE *fh, 
			      const char *start_val, int echo);
static int execute_line(char *line);
static const command_t *find_command(char *name);
static char *stripwhite(char *in);
static void updateCurrentFTP(void);
static pftp_server_t getCurFTP(void);
static void initialize_readline(void);
static int parseArgs(const char *args, const char *flags, char **set_flags, 
		     char ***argv, size_t *argc);
static void freeParseArgs(char **set_flags, char ***argv, size_t *argc);
static void print_ls(pftp_directory_t *filelist, unsigned short format);
static void getMeAPager(void);
static FILE *startPager(void);
static void endPager(FILE *fh);
static int shell_out(const char *line);
static unsigned int dialog(pftp_file_t existing,const  pftp_item_t transfering);
static int parseProgArgs(int argc, char **argv);

static void disableQueDisplay(void);
static void enableQueDisplay(void);

/* Global Vars */
static char *CurFtp;
static pftp_que_t fileque, errorque;
static int display_que_info = 1;

#ifdef WIN32
static HANDLE console_in = NULL;
static HANDLE console_out = NULL;
#endif

size_t open_list_len = 0;
struct open_list *open_list = NULL;

static char *prompt = NULL, *pager = NULL;

const unsigned short LS_LONGLIST = 0x0001;
const unsigned short LS_HUMAN = 0x0002;

/* The three first powers of 8. */
static const int EXP8[3] = {1, 8, 64};

int main(int argc, char **argv)
{
    char *command, *line;
    int done = 0;

#ifdef WIN32
    console_in = GetStdHandle(STD_INPUT_HANDLE); 
    console_out = GetStdHandle(STD_OUTPUT_HANDLE); 
    SetConsoleTitle("pftp " VERSION);
#endif

    if ((signal(SIGINT, signal_handler) == SIG_ERR) 
#ifndef WIN32
        || (signal(SIGWINCH, signal_handler) == SIG_ERR)
#endif
        ) {
	perror("Unable to setup signal handlers");
	return EXIT_FAILURE;
    }

#ifndef WIN32
    /* We like EPIPE better than "killed program" */
    signal(SIGPIPE, SIG_IGN);
#endif

    rl_initialize();
    initialize_readline();
    rl_get_screen_size(&TERM_height, &TERM_width);

    if (!TERM_width) TERM_width = 80;
    if (!TERM_height) TERM_height = 24;

    command_central(CLI_INIT, NULL, NULL, NULL, NULL);

    pftp_initUtil(command_central);
    getMeAPager();
    fileque = pftp_CreateQue();
    errorque = pftp_CreateQue();
    CurFtp = NULL;
    updateCurrentFTP();

    if (!parseProgArgs(argc, argv)) {
        while (done != -10000) {
	    line = readline(prompt);
	    if (!line)
		break;
	    command = stripwhite(line);
	    if (*command) {
		add_history(command);
		done = execute_line(command);
		if (done && done != -10000 && done != -500 && 
		    CurFtp && strlen(CurFtp))
		    puts("Error executing command.");
	    }
	    free(line);        
	}
	
	puts("");
    }
    
    if (CurFtp) free(CurFtp);
    if (prompt) free(prompt);
    
    if (open_list) {
	size_t o;
	for (o = 0; o < open_list_len; o++) {
	    if (open_list[o].name) free(open_list[o].name);
	    if (open_list[o].real_name) free(open_list[o].real_name);
	}
	free(open_list);
	open_list = NULL;
    }
    
    pftp_clear_que(fileque);
    pftp_clear_que(errorque);
    pftp_FreeQue(fileque);
    pftp_FreeQue(errorque);

    pftp_freeUtil();

    clear_output_status();    

    if (pager) 
	free(pager);

    command_central(CLI_DONE, NULL, NULL, NULL, NULL);
        
    return 0;
}

/* Execute a command line. */
int execute_line(char *line)
{
    int i;
    const command_t *command;
    char *word;
  
    /* Isolate the command word. */
    i = 0;
    while (line[i] && whitespace (line[i]))
	i++;
    word = line + i;
    
    if (word[0] == '!') {
	return shell_out(word + 1);
    }
    
    while (line[i] && !whitespace (line[i]))
	i++;
    
    if (line[i])
	line[i++] = '\0';
    
    command = find_command(word);
    
    if (!command){
	fprintf(stderr, "%s: No such command for pFTP.\n", word);
	return -500;
    }
    
    /* Get argument to command, if any. */
    while (whitespace (line[i]))
	i++;
    
    word = line + i;
    
    /* Call the function. */
    return ((*(command->func)) (word, command->usage));
}

const command_t *find_command(char *name)
{
    int i;
    for (i = 0; commands[i].name; i++)
	if (strcmp(name, commands[i].name) == 0)
	    return (&commands[i]);
    return ((const command_t *) NULL);
}

char *stripwhite(char *in)
{
    char *s, *t;
    for (s = in; whitespace(*s); s++);
    if (*s == 0)
	return s;
    t = s + strlen(s) - 1;
    while (t > s && whitespace(*t))
	t--;
    *++t = '\0';
    
    return s;
}

int ftp_quit(char *args, const char *usage)
{
    char **argv = NULL, *flags = NULL;
    size_t argc = 0;
    int ret = parseArgs(args, NULL, &flags, &argv, &argc);

    if (!ret && argc == 0) {
	ret = -10000;
    } else {
	puts(usage);
	ret = 0;
    }

    freeParseArgs(&flags, &argv, &argc);
  
    return ret;
}

void show_cmds(void)
{
    size_t l, x, y, lines, columns, *colwidth = NULL, *len = NULL;
    char *space = malloc(TERM_width);
    FILE *out = startPager();
    memset(space, ' ', TERM_width);
    len = malloc(no_commands * sizeof(size_t));
    
    for (l = 0; l < no_commands; l++)
	len[l] = strlen(commands[l].name);
    
    calc_columns(len, no_commands, TERM_width, 2, &columns, &colwidth, &lines);
    
    for (y = 0; y < lines; y++) {
	for (x = 0; x < columns; x++) {
	    if ((l = y+(x*lines)) >= no_commands) 
		break;
	    fwrite(commands[l].name, 1, len[l], out);
	    if (len[l] < colwidth[x])
		fwrite(space, 1, colwidth[x] - len[l], out);
	}
	fputs("\n", out);
    }
        
    free(space);
    free(colwidth);
    free(len);

    endPager(out);
}

void display_usage(const command_t *cmd)
{
    FILE *out = startPager();
    fputs(cmd->usage, out);
    fputs("\n", out);
    fputs(cmd->doc, out);
    fputs("\n", out);
    endPager(out);
}

int ftp_help(char *args, const char *usage)
{
    char **argv = NULL, *flags = NULL;
    size_t argc = 0;
    const command_t *cmd;
    int ret = parseArgs(args, NULL, &flags, &argv, &argc);
    
    if (ret || argc > 1) {
	puts(usage);
	ret = 0;
    } else if (argc == 0) {
	show_cmds();
    } else {
	if ((cmd = find_command(argv[0]))) {
	    display_usage(cmd);
	} else {
	    fprintf(stderr, "Unknown command: `%s'\n", argv[0]);
	}
    }
    
    freeParseArgs(&flags, &argv, &argc);

    return ret;
}

pftp_server_t getCurFTP(void)
{
    int was;
    pftp_server_t ret;
    
    if (!CurFtp || strlen(CurFtp) == 0) {
	fprintf(stderr, "No ftp site is currently opened.\n"
		"Use `open' or `load' to open a site.\n");
	return NULL;
    }
    
    was = pftp_getFTPConnected(CurFtp);
    ret = pftp_getFTPCtrl(CurFtp);
    
    if (!was) {
	if (ret)
	    updateCurrentFTP();
	else
	    fprintf(stderr, "Error: Unable to login in to site.\n");
    }
    
    return ret;
}

void closeCurrentFTP(void)
{
    if (CurFtp && CurFtp[0]) {
    size_t o;
    /* Remove from openlist */
    for (o = 0; o < open_list_len; o++) {
        if (strcmp(open_list[o].real_name, CurFtp) == 0) {
	    free(open_list[o].name);
	    free(open_list[o].real_name);
	    open_list_len--;
	    if (o < open_list_len)
		memmove(open_list + o, open_list + o + 1,
			(open_list_len - o) * sizeof(struct open_list));
	    break;
        }
    }
    
    free(CurFtp);

    while (o >= open_list_len && o > 0)
        o--;
    
    if (o < open_list_len)
        CurFtp = strdup(open_list[o].real_name);    
    else
        CurFtp = NULL;
    }    
    
    updateCurrentFTP();
}

void updateCurrentFTP(void)
{
    if (CurFtp && strlen(CurFtp) > 0) {
	/* OK, show info then */
	pftp_server_t con;
	
	if (CurFtp[0] == '\n') {
	    /* Non-saved ftp then */
	    pftp_settings_t ftp = pftp_getFTPSettings(CurFtp);
	    prompt = realloc_strcpy(prompt, ftp->username);
	    prompt = realloc_strcat(prompt, "@");
	    if (ftp->hosts) {
		prompt = realloc_strcat(prompt, ftp->hostname[0]);
	    } else {
		prompt = realloc_strcat(prompt, "what?");
	    }
	} else {
	    /* Saved ftp */
	    prompt = realloc_strcpy(prompt, CurFtp);        
	}
	
	if (pftp_getFTPConnected(CurFtp) && (con = pftp_getFTPCtrl(CurFtp))) {
	    char *dir = NULL;
	    
	    prompt = realloc_strcat(prompt, ":");
	    pftp_curdir(con, &dir, 0);
	    if (dir) {
		prompt = realloc_strcat(prompt, dir);           
		free(dir);
	    } else {
		prompt = realloc_strcat(prompt, "?");
	    }
	    
	    prompt = realloc_strcat(prompt, "> ");
	} else {
	    prompt = realloc_strcat(prompt, ":~> ");
	}
    } else {
	/* No site -> no info */
	prompt = realloc_strcpy(prompt, "> ");
    }
}

static 
#ifndef WIN32
const 
#endif
char *getPassword(const char *username)
{
    size_t len = (strlen(username) + 20) % (TERM_width / 2);
    char *prompt = malloc(len + 4), *ret = NULL;
    snprintf(prompt, len, "Password for %s", username);
    strcat(prompt, ": ");
#if WIN32
    {
        size_t len;
        ssize_t val;
        fputs(prompt, stdout);
        len = 0;
        val = simple_getline(&ret, &len, stdin, NULL, 0);
        if (val < 0) {
            if (ret) {
                free(ret);
                ret = NULL;
            }
        } else {
            if (ret && val)
                ret[val - 1] = '\0'; /* Remove last \n */
        }
        fputs("\n", stdout);
    }
#else
    ret = getpass(prompt);
#endif
    free(prompt);
    if (!ret) {
	ret = malloc(1);
	ret[0] = '\0';
    }
    return ret;
}

int ftp_open(char *args, const char *usage)
{
    char **argv = NULL, *flags = NULL;
    size_t argc = 0;
    int ret = parseArgs(args, NULL, &flags, &argv, &argc);
    int error = 0, added = 0;
    size_t id = 0, p;

    if (!ret && argc == 1) {
	if (pftp_getFTPSettings(argv[0])) {
	    int found = 0;
	    CurFtp = realloc_strcpy(CurFtp, argv[0]);
	
	    for (p = 0; p < open_list_len; p++) {
		if (strcmp(open_list[p].real_name, CurFtp) == 0) {
		    found = 1;
		    id = p;
		    break;
		}
	    }
	    
	    if (!found) {
		id = open_list_len++;
		open_list = realloc(open_list, open_list_len *
				    sizeof(struct open_list));
		open_list[id].name = alloc_strcpy(CurFtp);
		open_list[id].real_name = alloc_strcpy(CurFtp);
		
		added = 1;
	    }
	} else {        
	    char *start = argv[0];
	    char *username = NULL, *port = NULL, *hostname = NULL, 
		*password = NULL;
	    char *path = NULL;
	    unsigned long real_port = 0;
	    int ssl = 0, sftp = 0;
	    char *at, *semi1, *semi2, *prot, *paths;
	    
	    prot = strstr(start, "://");
	    if (prot)
		start = prot + 3;      
	    semi1 = strchr(start, ':');  
	    at = strrchr(start, '@');
	    if (at) {
		start = at + 1;
		if (semi1 >= at) 
		    semi1 = NULL;
	    } else {
		semi1 = NULL;
	    }
	    semi2 = strchr(start, ':');
	    paths = strchr(start, '/');
	    if (paths && semi2 && paths < semi2)
		semi2 = NULL;
	
	    if (prot) {
		if (strncmp("ftp", argv[0], prot - argv[0]) == 0) {
		    ssl = 0;
		} else if (strncmp("ftps", argv[0], prot - argv[0]) == 0) {
		    ssl = 1;
		} else if (strncmp("sftp", argv[0], prot - argv[0]) == 0) {
		    sftp = 1;		
		} else {
		    char *tmp = alloc_strncpy(argv[0], prot - argv[0]);
		    fprintf(stderr, "Unsupported protocol: `%s'\n", tmp);
		    free(tmp);
		    error = 1;
		}
		if (at) {
		    if (semi1) {
			username = alloc_strncpy(prot + 3, semi1 - (prot + 3));
			password = alloc_strncpy(semi1 + 1, at - (semi1 + 1));
		    } else {
			username = alloc_strncpy(prot + 3, at - (prot + 3));
		    }
		}
	    } else {
		if (at) {
		    if (semi1) {
			username = alloc_strncpy(argv[0], semi1 - argv[0]);
			password = alloc_strncpy(semi1 + 1, at - (semi1 + 1));
		    } else {
			username = alloc_strncpy(argv[0], at - argv[0]);
		    }
		}
	    }
	    
	    if (semi2) {
		hostname = alloc_strncpy(start, semi2 - start);
		if (paths) {
		    port = alloc_strncpy(semi2 + 1, paths - (semi2 + 1));
		    path = alloc_strcpy(paths);
		} else {
		    port = alloc_strcpy(semi2 + 1);
		}
	    } else {
		if (paths) {
		    hostname = alloc_strncpy(start, paths - start);
		    path = alloc_strcpy(paths);
		} else {
		    hostname = alloc_strcpy(start);
		}
	    }
	    
	    if (port) {
		real_port = strtoul(port, &semi2, 10);
		if (!semi2 || (!xisspace(semi2[0]) && semi2[0] != '\0') || 
		    real_port == 0 || real_port > 65535) {
		    fprintf(stderr, "Invalid port value: `%s'\n", port);
		    error = 1;            
		}
	    }
	    if (hostname && strlen(hostname)) {
		if (!pftp_valid_hostname(hostname)) {
		    fprintf(stderr, "Invalid hostname value: `%s'\n", 
			    hostname);
		    error = 1;
		}
	    } else {
		fprintf(stderr, "Missing hostname\n");
		error = 1;
	    }
	    
	    if (!error) {
		char *tmp = NULL;
		int found = 0;
		size_t len;
		pftp_server_t ftp;
		
		hostname = strtolower(hostname);
		
		if (!username || !strlen(username))
		    username = realloc_strcpy(username, "anonymous");
		if (real_port == 0) {
		    if (sftp)
			real_port = 22;
		    else
			real_port = 21;  
		}      
		
		len = strlen(username) + strlen(hostname) + 20;
		tmp = malloc(len + 1);
		snprintf(tmp, len, "%s@%s:%lu", username, hostname, 
			 real_port);
		
		for (p = 0; p < open_list_len; p++) {
		    if (strcmp(open_list[p].name, tmp) == 0) {
			found = 1;
			id = p;
			CurFtp = realloc_strcpy(CurFtp, open_list[p].real_name);
			break;
		    }
		}
		
		if (!found) {
		    if (password == NULL) {
#ifndef WIN32
			const
#endif
			    char *passwd = NULL;

			if (strcmp(username, "anonymous")) {
			    passwd = getPassword(username);
			    password = strdup(passwd);
			}
#ifdef WIN32
			if (passwd)
			    free(passwd);
#endif
		    }
		    
		    id = pftp_getFreeFTPID();
		    
		    CurFtp = realloc(CurFtp, 30);
		    snprintf(CurFtp, 30, "\n%lu", (unsigned long)id);
		    
		    pftp_createFTPSettings(CurFtp, username, password, 
					   hostname, (unsigned short) real_port, ssl, sftp);
		    
		    id = open_list_len++;
		    open_list = realloc(open_list, open_list_len *
					sizeof(struct open_list));
		    open_list[id].name = alloc_strcpy(tmp);
		    open_list[id].real_name = malloc(30);
		    memcpy(open_list[p].real_name, CurFtp, 30);
		    
		    added = 1;		   
		}
        
		if (path && (ftp = getCurFTP())) {
		    pftp_cd(ftp, path, NULL);
		    updateCurrentFTP();
		}
		
		if (tmp) free(tmp);
	    }
        
	    if (hostname) free(hostname);
	    if (port) free(port);
	    if (username) free(username);
	    if (password) free(password);
	    if (path) free(path);
	}
	
	if (!error) {
	    if (added) {
		printf("Added %s as #%lu\n", open_list[id].name, 
		       (unsigned long)(id + 1));
	    } else {
		printf("Loaded %s as #%lu\n", open_list[id].name, 
		       (unsigned long)(id + 1));
	    }
	}    
    } else {
	puts(usage);
    }

    freeParseArgs(&flags, &argv, &argc);
    updateCurrentFTP();
    
    return 0;
}

int ftp_user(char *args, const char *usage)
{
    char **argv = NULL, *flags = NULL;
    size_t argc = 0;
    pftp_settings_t settings;
    int ret = parseArgs(args, NULL, &flags, &argv, &argc);
    
    if (!ret && argc > 0 && argc < 3) {       
	if ((settings = pftp_getFTPSettings(CurFtp))) {
#ifndef WIN32
	    const
#endif
	    char *passwd;
	    passwd = getPassword(argv[0]);
	    settings->username = realloc_strcpy(settings->username, argv[0]);
	    settings->password = realloc_strcpy(settings->password, passwd);
	    if (argc == 2)
		settings->account = realloc_strcpy(settings->account, argv[1]);
	    
	    if (pftp_getFTPConnected(CurFtp)) {
		if (confirm("Do you wish to reconnect?", 0)) {
		    pftp_closeFTPCtrl(CurFtp);
		    remove_server_output_status(CurFtp);
		    getCurFTP();
		}
	    }       
	    
#ifdef WIN32
	    if (passwd)
		free(passwd);
#endif
	} else {
	    fprintf(stderr, "Error: You must issue an open command first.\n");
	}
    } else {
	puts(usage);
    }
    
    freeParseArgs(&flags, &argv, &argc);
    updateCurrentFTP();

    return 0;
}

int get_ftpsettings_for_load(char *site, pftp_settings_t *found, 
			     char **nice_name)
{
    size_t test, count = pftp_getNrOfSettings();
    char *end = NULL, *start = site;

    (*found) = NULL;
    
    if (start[0] == '#') start++;
    test = strtoul(start, &end, 10);
    
    if (test > 0 && end && !end[0] && test <= count) {
	/* OK, valid nr */
	(*found) = pftp_getFTPSettings2(test - 1);
    }
    
    if (!(*found)) {
	/* OK, search by name then */
	(*found) = pftp_getFTPSettings(site);
    }
    
    if ((*found)) {
	(*nice_name) = realloc_strcpy((*nice_name), (*found)->name);
	return 0;
    } else
	return -1;
}

int make_current_ftp(pftp_settings_t found, const char *nice_name)
{
    if (found) {
	size_t o;
	CurFtp = realloc_strcpy(CurFtp, found->name);
	/* add to open_list if not already there */
	for (o = 0; o < open_list_len; o++)
	    if (strcmp(open_list[o].real_name, found->name) == 0)
		return 0;
	open_list = realloc(open_list, (open_list_len + 1) * 
			    sizeof(struct open_list));
	open_list[open_list_len].name = alloc_strcpy(nice_name);
	open_list[open_list_len].real_name = alloc_strcpy(found->name);
	open_list_len++;
	return 0;
    }
    
    return -1;
}

int get_ftpsettings_for_current(char *site, pftp_settings_t *found,
				char **nice_name)
{
    /* Select that name if open, else open it (if saved) */
    /* Check if it's a number first */
    int try_load = 1;
    size_t test;
    char *end = NULL, *start = site;
    
    if (start[0] == '#') start++;
    test = strtoul(start, &end, 10);
    
    if (test > 0 && end && !end[0]) {
	if (test <= open_list_len) {
	    /* OK, valid nr */
	    (*found) = pftp_getFTPSettings(open_list[test - 1].real_name);
	    (*nice_name) = realloc_strcpy((*nice_name), 
					  open_list[test - 1].name);
	}
	try_load = 0;
    }
    
    if (!(*found)) {
	/* OK, search current open for full match of name */
	size_t o;
	for (o = 0; o < open_list_len; o++) {
	    if (strcmp(site, open_list[o].name) == 0) {
		(*found) = pftp_getFTPSettings(open_list[o].real_name);
		(*nice_name) = realloc_strcpy((*nice_name), open_list[o].name);
		break;
	    }
	}
    }
    
    if (!(*found)) {
	/* Search current open for hostname match */
	size_t o;
	for (o = 0; o < open_list_len; o++) {
	    if (!(start = strrchr(open_list[o].name, '@'))) 
		start = open_list[o].name;
	    if (!(end = strrchr(open_list[o].name, ':')))
		end = open_list[o].name + strlen(open_list[o].name);
	    if (strncmp(site, start, end - start) == 0) {
		(*found) = pftp_getFTPSettings(open_list[o].real_name);
		(*nice_name) = realloc_strcpy((*nice_name), open_list[o].name);
		break;
	    }
	}
    }
    
    if (!(*found) && try_load) {
	/* Search saved names */
	return get_ftpsettings_for_load(site, found, nice_name);
    }
    
    if ((*found))
	return 0;
    else
	return -1;
}

int ftp_begin_trans(char *args, const char *usage)
{
    char **argv = NULL, *flags = NULL;
    size_t argc = 0;
    int ret = parseArgs(args, NULL, &flags, &argv, &argc);
    
    if (ret || argc) {
	puts(usage);
	ret = 0;
    } else {
	ret = pftp_begin_trans(dialog, fileque, errorque);
    }
    
    freeParseArgs(&flags, &argv, &argc);

    return ret;
}

int ftp_close(char *args, const char *usage)
{
    char **argv = NULL, *flags = NULL;
    size_t argc = 0;
    int ret = parseArgs(args, NULL, &flags, &argv, &argc);

    if (ret || argc > 1) {
	puts(usage);
	ret = 0;
    } else if (argc == 1) {
	/* Close the name if it's open */
	int found = 0;
	size_t test;
	char *end = NULL, *ftp = NULL, *nice_name = NULL, *start = argv[0];
	
	if (start[0] == '#') start++;
	test = strtoul(start, &end, 10);
	
	if (test > 0 && end && !end[0]) {
	    if (test <= open_list_len) {
		/* OK, valid nr */
		found = 1;
		ftp = alloc_strcpy(open_list[test - 1].real_name);
		nice_name = alloc_strcpy(open_list[test - 1].name);
	    }
	}

	if (!found) {
	    /* OK, search current open for full match of name */
	    size_t o;
	    for (o = 0; o < open_list_len; o++) {
		if (strcmp(argv[0], open_list[o].name) == 0) {
		    found = 1;
		    ftp = alloc_strcpy(open_list[o].real_name);
		    nice_name = alloc_strcpy(open_list[o].name);
		    break;
		}
	    }
	}

	if (!found) {
	    /* Search current open for hostname match */
	    size_t o;
	    for (o = 0; o < open_list_len; o++) {
		if (!(start = strrchr(open_list[o].name, '@'))) 
		    start = open_list[o].name;
		if (!(end = strrchr(open_list[o].name, ':')))
		    end = open_list[o].name + strlen(open_list[o].name);
		if (strncmp(argv[0], start, end - start) == 0) {
		    found = 1;
		    ftp = alloc_strcpy(open_list[o].real_name);
		    nice_name = alloc_strcpy(open_list[o].name);
		    break;
		}
	    }
	}
	
	if (found) {
	    if (pftp_getFTPConnected(ftp)) {
		pftp_closeFTPCtrl(ftp);
		remove_server_output_status(ftp);
		printf("%s is now closed.\n", nice_name);
	    } else {
		printf("%s was already closed.\n", nice_name);
	    }
	    
	    if (CurFtp && strcmp(CurFtp, ftp) == 0)
		closeCurrentFTP();
	    else
		updateCurrentFTP();
	} else {
	    fprintf(stderr, "Error: %s is unknown to me.\n"
		    "Use `current' without any arguments to see list of open"
		    " ftps.\n", argv[0]);
	}
	
	if (nice_name) free(nice_name);
	if (ftp) free(ftp);
    } else {
	/* Close current if it's open */
	if (CurFtp) {
	    size_t o;
	    char *nice_name = alloc_strcpy("Unknown");
	    
	    for (o = 0; o < open_list_len; o++) {
		if (strcmp(open_list[o].real_name, CurFtp) == 0) {
		    nice_name = realloc_strcpy(nice_name, open_list[o].name);
		    break;
		}
	    }
	    
	    if (pftp_getFTPConnected(CurFtp)) {
		pftp_closeFTPCtrl(CurFtp);
		remove_server_output_status(CurFtp);
		printf("%s is now closed.\n", nice_name);
	    } else {
		printf("%s was already closed.\n", nice_name);
	    }
	    
	    closeCurrentFTP();
	    
	    free(nice_name);
	} else {
	    printf("Not connected to any servers at the moment.\n");
	}
    }

    return 0;
}

int ftp_current(char *args, const char *usage)
{
    char **argv = NULL, *flags = NULL;
    size_t argc = 0;
    int ret = parseArgs(args, NULL, &flags, &argv, &argc);
    
    if (ret || argc > 1) {
	puts(usage);
	ret = 0;
    } else if (argc == 1) {
	/* Select that name if open, else open it (if saved) */
	pftp_settings_t found = NULL;
	char *nice_name = NULL;
	
	if (!get_ftpsettings_for_current(argv[0], &found, &nice_name)) {
	    make_current_ftp(found, nice_name);
	    printf("%s is now the current site.\n", nice_name);
	    updateCurrentFTP();
	} else {
	    fprintf(stderr, "Error: %s is unknown to me.\n"
		    "Use `current' without any arguments to see list of open"
		    " ftps.\nUse `load' without any arguments to see a list of"
		    " saved sites.\n", argv[0]);
	}
	
	if (nice_name)
	    free(nice_name);
    } else {
	/* View opened ftp's */
	size_t o;
	if (open_list_len) {
	    printf("Current open connections:\n");
	    for (o = 0; o < open_list_len; o++) 
		printf("#%lu: %s\n", (unsigned long)(o + 1), 
		       open_list[o].name);
	} else {
	    printf("No currently opened connections.\n"
		   "Use `open' or `load' commands to open a connection.\n");
	}
    }
    
    freeParseArgs(&flags, &argv, &argc);
    
    return 0;
}

int ftp_load(char *args, const char *usage)
{
    char **argv = NULL, *flags = NULL;
    size_t argc = 0;
    int ret = parseArgs(args, NULL, &flags, &argv, &argc);
    
    if (ret || argc > 1) {
	puts(usage);
    } else if (argc == 1) {
	pftp_settings_t found = NULL;
	char *nice_name = NULL;
	
	if (!get_ftpsettings_for_load(argv[0], &found, &nice_name)) {
	    make_current_ftp(found, nice_name);
	    printf("%s is now the current site.\n", nice_name);
	    updateCurrentFTP();
	} else {
	    fprintf(stderr, "Error: %s is unknown to me.\n"
		    "Use `load' without any arguments to see list of site.\n", 
		    argv[0]);
	}
	
	if (nice_name)
	    free(nice_name);
    } else {
	/* View saved sites */
	size_t count = pftp_getNrOfSettings();
	size_t o;
	if (count) {
	    printf("Saved sites:\n");
	    for (o = 0; o < count; o++) 
		printf("#%lu: %s\n", (unsigned long)(o + 1), 
		       pftp_getFTPSettings2(o)->name);
	} else {
	    printf("No saved sites were found.\n"
		   "Use `open' to open a new connection.\n"
		   "Then use `save' to save the site for later use.\n");
	}
    }
    
    freeParseArgs(&flags, &argv, &argc);

    return 0;
}

int ftp_save(char *args, const char *usage)
{
    size_t argc = 0;
    char **argv = NULL, *flags = NULL;
    int ret, save_pass, clr_pass;

    ret = parseArgs(args, "XY", &flags, &argv, &argc);
    save_pass = 0;
    clr_pass = 0;

    if (strchr(flags, 'Y')) {
	clr_pass = 1;
    } else if (strchr(flags, 'X')) {
	save_pass = 1;		
    }
    
    if (ret || argc > 1) {
	puts(usage);
	ret = 0;
    } else {
	char *orgname = NULL, *target = NULL;
	
	if (argc == 0) {
	    if (CurFtp && CurFtp[0]) {
		/* We have a current site. */
		orgname = alloc_strcpy(CurFtp);
	    } else {
		printf("No current site to save.\n");
	    }
	} else {
	    pftp_settings_t found = NULL;
	    char *tmp = NULL;
	    if (get_ftpsettings_for_current(argv[0], &found, &tmp)) {
		if (CurFtp && CurFtp[0] == '\n' && argc == 1) {
		    orgname = strdup(CurFtp);
		    target = strdup(argv[0]);
		} else {
		    fprintf(stderr, "Error: %s is unknown to me.\n"
			    "Use `current' without any arguments to see list"
			    " of open ftps.\nUse `load' without any arguments"
			    " to see list of saved sites.\n", argv[0]);
		}
	    } else {
		free(tmp);
		orgname = alloc_strcpy(found->name);
	    }
	}    
	
	if (orgname) {
	    if (orgname[0] != '\n')
		target = alloc_strcpy(orgname);
	    
	    if (!target) {
		target = readline("Save as what? (empty to abort): ");
		if (target && !target[0]) {
		    free(target);
		    target = NULL;
		}
	    }
	    
	    if (target) {
		if (save_pass) {
		    fprintf(stdout, "Warning: Consider saved passwords as if they where saved in clear-text.\nYou can unset a password with -Y flag.\n"); 
		}

		if (!(ret = pftp_saveFTPSettings(orgname, target, 
						 save_pass, clr_pass)) && 
		    strcmp(orgname, target)) {
		    if (orgname[0] == '\n') {
			size_t p;
			for (p = 0; p < open_list_len; p++) {
			    if (strcmp(open_list[p].real_name, orgname) == 0) {
				open_list[p].real_name = 
				    realloc_strcpy(open_list[p].real_name, 
						   target);
				open_list[p].name = 
				    realloc_strcpy(open_list[p].name, target);
				break;
			    }
			}
		    }
		    
		    if (CurFtp && strcmp(CurFtp, orgname) == 0) {
			make_current_ftp(pftp_getFTPSettings(target), target);
			updateCurrentFTP();
		    }
		}
		free(target);
	    }
	    
	    free(orgname);
	}
    }
    
    freeParseArgs(&flags, &argv, &argc);
    
    return ret;
}

int ftp_savedef(char *args, const char *usage)
{
    size_t argc = 0;                                                     
    char **argv = NULL, *flags = NULL;                                    
    int ret = parseArgs(args, NULL, &flags, &argv, &argc);                
    
    if (ret || argc > 0) {   
	puts(usage);                                                    
	ret = 0;
    } else {
	ret = pftp_saveFTPSettings(NULL, NULL, 0, 0);
    }                                                   
    
    freeParseArgs(&flags, &argv, &argc);                    
    
    return ret; 
}

int ftp_setdef(char *args, const char *usage)
{
    size_t argc = 0;                                                     
    char **argv = NULL, *flags = NULL;                                    
    int ret = parseArgs(args, NULL, &flags, &argv, &argc);                
    
    if (ret || argc > 2) {
	puts(usage);
	ret = 0;
    } else {
	char *tmpstr = NULL;
	
	if (argc == 0) {
	    if (!getDefValue(pftp_getDefaultSettings(), NULL, &tmpstr)) {
		printf(tmpstr);
		free(tmpstr);
	    } else {
		ret = -1;
	    }
	} else if (argc == 1) {
	    if (getDefValue(pftp_getDefaultSettings(), argv[0], &tmpstr)) {
		fprintf(stderr, "Unknown variable: `%s'\n", argv[0]);
	    } else {
		printf(tmpstr);
		free(tmpstr);
	    }
	} else if (argc == 2) {
	    int ret = setDefValue(pftp_getDefaultSettings(), argv[0], argv[1]);
	    switch (ret) {
	    case 0:
		/* OK */
		break;
	    case -1:
		fprintf(stderr, "Invalid value for %s: `%s'\n", 
			argv[0], argv[1]);
		break;
	    default:
		fprintf(stderr, "Unknown variable: `%s'\n", argv[0]);
	    }
	}
    }
    
    freeParseArgs(&flags, &argv, &argc);
    return ret;
}

int ftp_set(char *args, const char *usage)
{
    char **argv = NULL, *flags = NULL;
    size_t argc = 0;
    int ret = parseArgs(args, NULL, &flags, &argv, &argc);
    
    if (!ret && argc > 3) {
	puts(usage);
	ret = 0;
    } else {
	pftp_settings_t found = NULL;
	int use_current = 0;
	char *tmp = NULL;
	
	if (argc > 0) {
	    if (get_ftpsettings_for_current(argv[0], &found, &tmp)) {
		use_current = 1;        
	    } else {
		free(tmp);
		tmp = NULL;
		
		if (argc == 1) {
		    if (!getSiteValue(found, NULL, &tmp)) {
			printf(tmp);
			free(tmp);
		    } else {
			ret = -1;
		    }
		} else if (argc == 2) {
		    if (getSiteValue(found, argv[1], &tmp)) {
			fprintf(stderr, "Unknown variable: `%s'\n", argv[1]);
		    } else {
			printf(tmp);
			free(tmp);
		    }
		} else if (argc == 3) {
		    int ret = setSiteValue(found, argv[1], argv[2]);
		    switch (ret) {
		    case 0:
			/* OK */
			break;
		    case -1:
			fprintf(stderr, "Invalid value for %s: `%s'\n", 
				argv[1], argv[2]);
			break;
		    default:
			fprintf(stderr, "Unknown variable: `%s'\n", argv[1]);
		    }
		}
	    }        
	} else {
	    use_current = 1;
	}
	
	if (use_current) {
	    if (!CurFtp || !CurFtp[0]) {
		printf("No ftp currently loaded or unknown sitename.\n");
	    } else {
		found = pftp_getFTPSettings(CurFtp);
		
		if (argc == 0) {
		    if (!getSiteValue(found, NULL, &tmp)) {
			printf(tmp);
			free(tmp);
		    } else {
			ret = -1;
		    }
		} else if (argc == 1) {
		    if (getSiteValue(found, argv[0], &tmp)) {
			fprintf(stderr, "Unknown variable: `%s'"
				" (or unknown sitename)\n", argv[0]);
		    } else {
			printf(tmp);
			free(tmp);
		    }
		} else if (argc == 2) {
		    int ret = setSiteValue(found, argv[0], argv[1]);
		    switch (ret) {
		    case 0:
			/* OK */
			break;
		    case -1:
			fprintf(stderr, "Invalid value for %s: `%s'\n",
				argv[0], argv[1]);
			break;
		    default:
			fprintf(stderr, "Unknown variable: `%s'"
				" (or unknown sitename)\n", argv[0]);
		    }
		}
	    }
	}
    }
    
    freeParseArgs(&flags, &argv, &argc);
    return ret;
}

int ftp_lcd(char *args, const char *usage)
{
    char **argv = NULL, *flags = NULL;
    size_t argc = 0;
    int ret = parseArgs(args, NULL, &flags, &argv, &argc);
    
    if (!ret && argc < 2) {
	if (argc == 1) {
	    expand_tilde(argv + 0);
	    ret = chdir(argv[0]);
	} else {
	    char *tmp = malloc(2);
	    strcpy(tmp, "~");
	    expand_tilde(&tmp);
	    ret = chdir(tmp);
	    free(tmp);
	}
	if (ret)
	    fprintf(stderr, "Unable to change to `%s': %s\n", 
		    argc == 1 ? argv[0] : "~", strerror(errno));
    } else {
	puts(usage);
	ret = 0;
    }
    
    freeParseArgs(&flags, &argv, &argc);
    return ret;
}

int ftp_cd(char *args, const char *usage)
{
    char **argv = NULL, *flags = NULL;
    size_t argc = 0;
    int ret = parseArgs(args, NULL, &flags, &argv, &argc);
    
    if (!ret && argc >= 1) {
	pftp_server_t tmp;
	size_t a;
	char *real_cd = alloc_strcpy(argv[0]);
	for (a = 1; a < argc; a++) {
	    real_cd = realloc_strcat(real_cd, " ");
	    real_cd = realloc_strcat(real_cd, argv[a]);
	}
	
	if ((tmp = getCurFTP())) {
	    ret = pftp_cd(tmp, real_cd, NULL);
	    updateCurrentFTP();
	}
	
	free(real_cd);
    } else {
	puts(usage);
	ret = 0;
    }
    
    freeParseArgs(&flags, &argv, &argc);

    return ret;
}

int ftp_debug(char *args, const char *usage)
{
    char **argv = NULL;
    size_t argc = 0;
    char *flags = NULL;
    int ret = parseArgs(args, "", &flags, &argv, &argc);
    
    if (!ret && argc == 1) {
	unsigned long tmp;
	char *end = NULL;
	errno = 0;
	tmp = strtoul(argv[0], &end, 10);
	if (errno || !end || end[0]) {
	    if (errno) {
		fprintf(stderr, "Error: `%s' is a invalid number: %s\n",
			argv[0], strerror(errno));
	    } else {
		fprintf(stderr, "Error: `%s' is a invalid number.\n",
			argv[0]);
	    }
	} else {
	    debuglevel = tmp;
	}
    } else {
	puts(usage);
    }

    freeParseArgs(&flags, &argv, &argc);
    
    return 0;
}

int ftp_rm(char *args, const char *usage)
{
    char **argv = NULL;
    size_t argc = 0;
    char *flags = NULL;
    int ret = parseArgs(args, "rf", &flags, &argv, &argc);
    
    if (!ret && argc > 0) {
	pftp_server_t ftp;
	
	if ((ftp = getCurFTP())) {
	    pftp_directory_t *ls;
	    char *path = NULL, *orgpath = NULL;
	    size_t f, c;
	    pftp_que_t rmque, rmerrque;
	    int force = (strchr(flags, 'f') != NULL);
	    pftp_curdir(ftp, &orgpath, 0);
	    
	    rmque = pftp_CreateQue();
	    rmerrque = pftp_CreateQue();
	    
	    for (c = 0; c < argc; c++) {    
		if ((ls = pftp_trans_ls2(ftp, argv[c], &path, 1, 0))) {
		    if (ls->length == 0) {
			fprintf(stderr, 
				"Error: `%s' doesn\'t match anything.\n",
				argv[c]);
		    }
		    
		    for (f = 0; f < ls->length; f++) {
			if (ls->files[f].type == pft_directory &&
			    strchr(flags, 'r') == NULL) {
			    fprintf(stderr, "Error: `%s' is a directory.\n",
				    ls->files[f].name);
			} else {
			    pftp_item_t item;
			    item = malloc(sizeof(struct pftp_item_s));
			    memset(item, 0, sizeof(struct pftp_item_s));
			    item->ftp_from = strdup(CurFtp);
			    item->dir = (ls->files[f].type == pft_directory);
			    item->srcpath = strdup(path);
			    item->filename = strdup(ls->files[f].name);
			    item->action = PFTP_DELETE_FROM;
			    item->preserve = PFTP_PRESERVE_NONE;
			    item->mode = PFTP_MODE_NORMAL;
			    pftp_que_add_last(item, rmque);            
			}
		    }
		    
		    pftp_free_fdt((*ls));
		    free(ls);
		} else {
		    if (!force) {
			fprintf(stderr, "Error: Could not match `%s'.\n",
				argv[c]);
		    }
		}
		
		if (path) {
		    free(path);
		    path = NULL;
		}
		
		if (pftp_cd(ftp, orgpath, NULL)) {
		    break; 
		}           
	    }
	    
	    free(orgpath);
	    
	    if (pftp_queLength(rmque) > 0) {
		disableQueDisplay();
		
		pftp_begin_trans(NULL, rmque, rmerrque);
		
		enableQueDisplay();
	    }
            
	    pftp_clear_que(rmque);
	    pftp_clear_que(rmerrque);
	    pftp_FreeQue(rmque);
	    pftp_FreeQue(rmerrque);        
	}
    } else {
	puts(usage);
    }
    
    freeParseArgs(&flags, &argv, &argc);

    return 0;
}

int ftp_rmdir(char *args, const char *usage)
{
    char **argv = NULL;
    size_t argc = 0;
    char *flags = NULL;
    int ret = parseArgs(args, NULL, &flags, &argv, &argc);
    
    if (!ret && argc > 0) {
	pftp_server_t ftp;
	size_t c;
	
	if ((ftp = getCurFTP())) {        
	    for (c = 0; c < argc; c++) {
		if (pftp_rmdir(ftp, argv[c]))
		    ret = 1;
	    }
	    pftp_clear_dir_cache(ftp, ".");
	}
    } else {
	puts(usage);
	ret = 0;
    }
    
    freeParseArgs(&flags, &argv, &argc);
    
    return ret;
}

int ftp_mkdir(char *args, const char *usage)
{
    char **argv = NULL;
    size_t argc = 0;
    char *flags = NULL;
    int ret = parseArgs(args, NULL, &flags, &argv, &argc);

    if (!ret && argc > 0) {
	pftp_server_t ftp;
	size_t c;
	if ((ftp = getCurFTP())) {
	    for (c = 0; c < argc; c++) {
		if (pftp_mkdir(ftp, argv[c]))
		    ret = 1;
	    }
	    pftp_clear_dir_cache(ftp, ".");
	}
    } else {
	puts(usage);
	ret = 0;
    }
    
    freeParseArgs(&flags, &argv, &argc);

    return ret;
}

int ftp_ls(char *args, const char *usage)
{
    char **argv = NULL, *flags = NULL;
    size_t argc = 0;
    int ret = parseArgs(args, "lhrad", &flags, &argv, &argc);    

    if (!ret && argc < 2) {
	pftp_directory_t *d = NULL;
	pftp_server_t ftp = NULL;
    
	if ((ftp = getCurFTP())) {
	    /* !!! FIX-ME: Only clears cache in current dir, so if mask is:
	       "../" it's useless */
	    if (strchr(flags, 'r')) 
		pftp_clear_dir_cache(ftp, ".");
	    
	    if ((d = pftp_trans_ls2(ftp, argc == 0 ? "*" : argv[0], NULL,
				    strchr(flags, 'a') != NULL, 1))) {
		
		if (argc 
		    && 
		    (d->length == 1)
		    && 
		    (d->files[0].type == pft_directory)
		    &&
		    (strchr(flags, 'd') == NULL)) {
		    /* List what's in that directory instead then */
		    char *mask = NULL;
		    pftp_directory_t *e = NULL;
		    if (argc) {
			char *pos = NULL;
			mask = alloc_strcpy(argv[0]);
			if ((pos = strrchr(mask, '/')))
			    pos[1] = '\0';
			else
			    mask = realloc_strcpy(mask, "./");
		    } else {
			mask = alloc_strcpy("./");
		    }
		    mask = realloc_strcat(mask, d->files[0].name);
		    mask = realloc_strcat(mask, "/*");
		    
		    if ((e = pftp_trans_ls2(ftp, mask, NULL,
					    strchr(flags, 'a') ? 1 : 0, 1))) {
			pftp_free_fdt((*d));
			free(d);
			d = e;
		    }
		    
		    free(mask);
		}
		
		print_ls(d, (strchr(flags, 'l') ? LS_LONGLIST : 0) |
			 (strchr(flags, 'h') ? LS_HUMAN : 0));
		pftp_free_fdt((*d));
		free(d);
	    } else {
		ret = -1;
	    }
	}
    } else {
	puts(usage);
	ret = 0;
    }
    
    freeParseArgs(&flags, &argv, &argc);

    return ret;
}

int ftp_lls(char *args, const char *usage)
{
    char **argv = NULL, *flags = NULL;
    size_t argc = 0;
    int ret = parseArgs(args, "lhrad", &flags, &argv, &argc);

    if (!ret && argc < 2) {
	pftp_directory_t *d = NULL;
	if (argc) expand_tilde(argv + 0);
	if ((d = pftp_trans_ls(NULL, argc ? argv[0] : "*", NULL,
			       strchr(flags, 'a') ? 1 : 0, 1))) {
	    if (argc 
		&& 
		(d->length == 1) 
		&& 
		(d->files[0].type == pft_directory) 
		&& 
		(strchr(flags, 'd') == NULL)) {
		/* List what's in that directory instead then */
		char *mask = NULL;
		pftp_directory_t *e = NULL;
		if (argc) {
		    char *pos = NULL;
		    mask = alloc_strcpy(argv[0]);
		    if ((pos = strrchr(mask, '/')))
			pos[1] = '\0';
		    else
			mask = realloc_strcpy(mask, "./");
		} else {
		    mask = alloc_strcpy("./");
		}
		mask = realloc_strcat(mask, d->files[0].name);
		mask = realloc_strcat(mask, "/*");
		
		if ((e = pftp_trans_ls(NULL, mask, NULL,
				       strchr(flags, 'a') ? 1 : 0, 1))) {
		    pftp_free_fdt((*d));
		    free(d);
		    d = e;
		}
		
		free(mask);
	    }
	    
	    print_ls(d, (strchr(flags, 'l') ? LS_LONGLIST : 0) |
		     (strchr(flags, 'h') ? LS_HUMAN : 0));
	    
	    pftp_free_fdt((*d));
	    free(d);
	}
    } else {
	puts(usage);
	ret = 0;
    }
    
    freeParseArgs(&flags, &argv, &argc);

    return ret;
}

int ftp_fxp(char *args, const char *usage)
{
    char **argv = NULL, *flags = NULL;
    size_t argc = 0;
    int ret = parseArgs(args, "m", &flags, &argv, &argc);
    int move = (flags && strchr(flags, 'm')), movedir = 0;  
    
    if (!ret && argc > 1 && (!move || argc == 3)) {
	pftp_server_t srcftp = NULL, destftp = NULL;
	pftp_settings_t destftpset = NULL;
	char *destname = NULL;
	
	if (!(srcftp = getCurFTP())) {
	    ret = -1;
	}
	if (get_ftpsettings_for_current(argv[0], &destftpset, &destname)) {
	    fprintf(stderr, "Error: `%s' is unknown to me.\n"
		    "Use current without arguments to list known ftps\n", 
		    argv[0]);
	    ret = -1;
	}
	
	if (!ret) {
	    if (!(destftp = pftp_getFTPCtrl(destftpset->name))) {
		fprintf(stderr, "Error: Unable to login in to %s.\n", 
			destname);
		ret = -1;
	    }
	}
	
	if (destname)
	    free(destname);
	
	if (!ret) {
	    if (move) {
		char *srcpath = NULL;
		pftp_directory_t *lsdata = pftp_trans_ls2(srcftp, argv[1], 
							  &srcpath, 1, 0);
		
		if (!lsdata) {
		    ret = -1;
		} else {
		    if (lsdata->length > 1) {
			puts("Only files or directories are permitted.");
			puts("No filemasks.");
			ret = -1;
		    } else {
			if (lsdata->files[0].type == pft_unknown) {
			    puts("Filelisting doesnt seem to work.");
			    puts("Use normal get (without the -m flag)");
			    ret = -1;
			} else {
			    movedir = (lsdata->files[0].type == pft_directory);
			}
		    }
		}
		
		if (!ret && (strchr(argv[2], '*') || strchr(argv[2], '?'))) {
		    puts("Invalid target name.");
		    ret = -1;
		}
		
		if (!ret) {
		    /* Copy file or directory to target name */
		    pftp_item_t que_item = malloc(sizeof(struct pftp_item_s));
		    memset(que_item, 0, sizeof(struct pftp_item_s));
		    que_item->ftp_from = alloc_strcpy(CurFtp);
		    que_item->ftp_to = alloc_strcpy(destftpset->name);
		    que_item->size = lsdata->files[0].size;
		    que_item->dir = movedir;
		    que_item->filename = alloc_strcpy(lsdata->files[0].name);
		    que_item->target = alloc_strcpy(argv[2]);
		    que_item->changed = alloc_strcpy(lsdata->files[0].changed);
		    que_item->srcpath = srcpath;
		    que_item->action = PFTP_COPY;
		    que_item->preserve = PFTP_PRESERVE_NONE;
		    que_item->mode = PFTP_MODE_NORMAL;
		    pftp_curdir(destftp, &que_item->destpath, 0);
		    if (!que_item->srcpath || !que_item->destpath) {
			ret = -1;
		    }
		    if (!ret) {
			pftp_que_add_last(que_item, fileque);
		    } else {
			pftp_free_queitem(&que_item);
		    }    
		} else {
		    if (srcpath)
			free(srcpath);
		}
	    } else {
		char **srcpath = malloc((argc - 1) * sizeof(char *));
		size_t a;    
		pftp_directory_t **lsdata = malloc(sizeof(pftp_directory_t *) 
						   * (argc - 1));
		memset(srcpath, 0, (argc - 1) * sizeof(char *));

		for (a = 0; a < argc - 1; a++) {
		    if (!(lsdata[a] = pftp_trans_ls2(srcftp, argv[a + 1], 
						     srcpath + a, 1, 0))) { 
			ret = -1;
			break;
		    }
		}
		
		if (!ret) {
		    for (a = 0; a < argc - 1; a++) {
			ret = pftp_get_files_ordered(CurFtp,
						     destftpset->name, 
						     srcpath[a],
						     lsdata[a], 
						     fileque, 1, 
						     PFTP_PRESERVE_NONE,
						     PFTP_COPY,
						     PFTP_MODE_NORMAL);
			pftp_free_fdt((*lsdata[a]));
			free(lsdata[a]);            
		    }
		} else {
		    while (a > 0) {
			pftp_free_fdt((*lsdata[a]));
			free(lsdata[a]);
		    }
		}
		
		for (a = 0; a < argc - 1; a++)
		    if (srcpath[a])
			free(srcpath[a]);
		
		free(srcpath);
		free(lsdata);
	    }
	}
    } else {
	puts(usage);
	ret = 0;
    }
    
    freeParseArgs(&flags, &argv, &argc);
    
    return ret;
}

int ftp_mirror(char *args, const char *usage)
{
    char **argv = NULL, *flags = NULL;
    size_t argc = 0;
    int ret = parseArgs(args, "etnrR", &flags, &argv, &argc);

    int flag_delete, flag_ign_time, flag_only_new, flag_recursive, flag_reverse;

    flag_delete = flag_ign_time = flag_only_new = flag_reverse = 0;
    flag_recursive = 1;
    
    if (flags) {
	if (strchr(flags, 'e'))
	    flag_delete = 1;
	if (strchr(flags, 't'))
	    flag_ign_time = 1;
	if (strchr(flags, 'n'))
	    flag_only_new = 1;
	if (strchr(flags, 'r'))
	    flag_recursive = 0;
	if (strchr(flags, 'R'))
	    flag_reverse = 1;
    }
    
    if (!ret && argc < 3) {
	pftp_server_t srcftp = NULL, destftp = NULL;
	char *destname = NULL, *srcname = NULL;
	pftp_settings_t srcftpset = NULL, destftpset = NULL;
	
	if (argc > 0) {
	    if (get_ftpsettings_for_current(argv[0], &srcftpset, &srcname)) {
		fprintf(stderr, "Error: `%s' is unknown to me.\n"
			"Use current without arguments to list known ftps\n", 
			argv[0]);
		ret = -1;
	    }
	} else {
	    srcftp = getCurFTP();
	    if (CurFtp)
		srcname = strdup(CurFtp);
	}
	
	if (argc > 1) {
	    if (get_ftpsettings_for_current(argv[1], &destftpset, 
					    &destname)) {
		fprintf(stderr, "Error: `%s' is unknown to me.\n"
			"Use current without arguments to list known ftps\n", 
			argv[1]);
		ret = -1;
	    }
	} else {
	    destftp = NULL; /* I.e. local */
	    destname = NULL;
	}
	
	if (!ret) {
	    if (!srcftp && srcname) {
		if (!(srcftp = pftp_getFTPCtrl(srcftpset->name))) {
		    fprintf(stderr, "Error: Unable to login in to %s.\n", 
			    srcname);
		    ret = -1;
		}
	    }
	    
	    if (!destftp && destname) {
		if (!(destftp = pftp_getFTPCtrl(destftpset->name))) {
		    fprintf(stderr, "Error: Unable to login in to %s.\n", 
			    destname);
		    ret = -1;
		}
	    }        
	}
	
	if (!ret) {
	    if (srcftp == destftp) {
		fprintf(stderr, "Error: Same source and target.\n");
		ret = -1;
	    }
	}
	
	if (flag_reverse) {
	    char *tmp;
	    tmp = srcname;
	    srcname = destname;
	    destname = tmp;
	}
	
	if (!ret) {
	    ret = pftp_mirror(srcname, NULL, destname, NULL, fileque, 
			      flag_delete, flag_recursive, flag_only_new, 
			      flag_ign_time);
	}
	
	if (destname)
	    free(destname);
	if (srcname)
	    free(srcname);       
    } else {
	puts(usage);
	ret = 0;
    }
    
    freeParseArgs(&flags, &argv, &argc);
    
    return ret;
}

int ftp_get(char *args, const char *usage)
{   
    char **argv = NULL, *flags = NULL;
    size_t argc = 0;
    int ret = parseArgs(args, "m", &flags, &argv, &argc);
    int move = (flags && strchr(flags, 'm'));  
    
    if (!ret && argc > 0 && (!move || argc == 2)) {
	pftp_server_t ftp;
	
	if ((ftp = getCurFTP())) {
	    if (move) {
		char *srcpath = NULL, *targetpath = NULL, *target = NULL;
		int singlefile = 0;
		pftp_directory_t *lsdata = pftp_trans_ls2(ftp, argv[0], 
							  &srcpath, 1, 0);
		
		expand_tilde(argv + 1);
		
		if (!lsdata) {
		    ret = -1;
		} else {
		    singlefile = (lsdata->length == 1 &&
				  lsdata->files[0].type != pft_directory);
		    
		    if ((strchr(argv[1], '*') || strchr(argv[1], '?'))) {
			puts("Invalid target name.");
			ret = -1;
		    } else {
			pftp_directory_t *dest;
			dest = pftp_trans_ls(NULL, argv[1], &targetpath,
					     1, 0);
			
			if (dest) {
			    if (dest->length == 0) {
				/* OK, if single file: 
				 *     create file with target name.
				 * if single directory:
				 *     create directory with target name.
				 * if multiple files, grr! 
				 */
				
				if (lsdata->length > 1) {
				    fprintf(stderr, "Target doesn\'t exist."
					    "\nCan only get single source"
					    " then.\n");
				    ret = -1;                    
				} else {
				    char *pos = strrchr(argv[1], '/');
				    if (pos) {
					target = strdup(pos + 1);
				    } else { 
					target = strdup(argv[1]);
				    }
				}
			    } else {
				if (dest->files[0].type == pft_directory) {
				    /* OK, everything will be put inside
				     * directory. */                
				    size_t len = strlen(targetpath);
				    if (targetpath[len - 1] != '/')
					targetpath = realloc_strcat(targetpath,
								    "/");
				    targetpath 
					= realloc_strcat(targetpath,
							 dest->files[0].name);
				} else {
				    /* If single file, overwrite then.
				       If directory, grr
				       If multiple files, grr
				    */
				    
				    if (singlefile) {
					target = strdup(dest->files[0].name);
				    } else {
					fprintf(stderr, "Target is a file."
						"\nCan only get a single file"
						" then.\n");
					ret = -1;
				    }
				}
			    }
			    
			    pftp_free_fdt((*dest));
			    free(dest);
			} else {
			    ret = -1;
			}
		    }
		}
		
		if (!ret) {
		    /* Copy file or directory to target name */
		    if (lsdata->length == 1) {
			pftp_item_t que_item = malloc(sizeof(struct pftp_item_s));
			memset(que_item, 0, sizeof(struct pftp_item_s));
			que_item->ftp_from = alloc_strcpy(CurFtp);
			que_item->ftp_to = NULL;
			que_item->size = lsdata->files[0].size;
			que_item->dir = singlefile ? 0 : 1;
			que_item->filename = strdup(lsdata->files[0].name);
			if (target)
			    que_item->target = target;
			else
			    que_item->target = strdup(lsdata->files[0].name);
			que_item->changed = strdup(lsdata->files[0].changed);
			que_item->srcpath = srcpath;
			que_item->destpath = targetpath;
			que_item->action = PFTP_COPY;
			que_item->mode = PFTP_MODE_NORMAL;
			que_item->preserve = PFTP_PRESERVE_NONE;
			
			pftp_que_add_last(que_item, fileque);
		    } else {
			char *oldpath = NULL;
			
			if (pftp_get_path_dir(NULL, &oldpath) ||
			    chdir(targetpath)) {
			    ret = -1;
			} else {
			    ret = pftp_get_files_ordered(CurFtp, NULL, srcpath,
							 lsdata, fileque, 1,
							 PFTP_PRESERVE_NONE,
							 PFTP_COPY,
							 PFTP_MODE_NORMAL);
			    chdir(oldpath);
			} 
			
			if (oldpath) free(oldpath);
			free(srcpath);
			free(targetpath);
		    }
		    
		    pftp_free_fdt((*lsdata));
		    free(lsdata);            
		} else {
		    if (target) free(target);
		    free(srcpath);
		    free(targetpath);
		}
	    } else {
		size_t a;           
		char **srcpath = malloc(argc * sizeof(char *));
		pftp_directory_t **lsdata = malloc(sizeof(pftp_directory_t *) 
						   * argc);
		memset(srcpath, 0, sizeof(char *) * argc);
		
		for (a = 0; a < argc; a++) {
		    if (!(lsdata[a] = pftp_trans_ls2(ftp, argv[a], srcpath + a,
						     1, 0))) {          
			ret = -1;
			break;
		    }
		    
		    if (lsdata[a]->length == 0 
			&& strchr(argv[a], '*') == NULL
			&& strchr(argv[a], '?') == NULL)
			printf("Warning: Mask `%s' doesn\'t match anything.\n",
			       argv[a]);
		}
		
		if (!ret) {
		    size_t oldlen;

		    for (a = 0; a < argc; a++) {
			oldlen = pftp_queLength(fileque);
			ret = pftp_get_files_ordered(CurFtp, NULL, srcpath[a],
						     lsdata[a], fileque, 1,
						     PFTP_PRESERVE_NONE,
						     PFTP_COPY,
						     PFTP_MODE_NORMAL);

			if ((oldlen == pftp_queLength(fileque))
			    &&
			    (lsdata[a]->length > 0)
			    && 
			    (strchr(argv[a], '*') == NULL)
			    && 
			    (strchr(argv[a], '?') == NULL))
			    printf("Warning: Mask `%s' doesn\'t match "
				   "anything.\n", argv[a]);

			pftp_free_fdt((*lsdata[a]));
			free(lsdata[a]);            
		    }
		} else {
		    while (a > 0) {
			pftp_free_fdt((*lsdata[a]));
			free(lsdata[a]);
		    }
		}
		
		for (a = 0; a < argc; a++)
		    if (srcpath[a])
			free(srcpath[a]);
		
		free(srcpath);
		free(lsdata);
	    }
	}
    } else {
	puts(usage);
	ret = 0;
    }
    
    freeParseArgs(&flags, &argv, &argc);
    
    return ret;
}

int ftp_put(char *args, const char *usage)
{
    char **argv = NULL, *flags = NULL;
    size_t argc = 0;
    int ret = parseArgs(args, "m", &flags, &argv, &argc);
    int move = (flags && strchr(flags, 'm')), movedir = 0;  
    
    if (!ret && argc > 0 && (!move || argc == 2)) {
	pftp_server_t ftp;
	
	if ((ftp = getCurFTP())) {
	    if (move) {
		pftp_directory_t *lsdata;
		char *srcpath = NULL;
		
		expand_tilde(argv + 0);
		lsdata = pftp_trans_ls(NULL, argv[0], &srcpath, 1, 0);
		
		if (!lsdata) {
		    ret = -1;
		} else {
		    if (lsdata->length > 1) {
			puts("Only files or directories are permitted.");
			puts("No filemasks.");
			ret = -1;
		    } else {
			if (lsdata->files[0].type == pft_unknown) {
			    puts("Filelisting doesnt seem to work.");
			    puts("Use normal put (without the -m flag)");
			    ret = -1;
			} else {
			    movedir = (lsdata->files[0].type == pft_directory);
			}
		    }
		}
		
		if (!ret && (strchr(argv[1], '*') || strchr(argv[1], '?'))) {
		    puts("Invalid target name.");
		    ret = -1;
		}
		
		if (!ret) {
		    /* Copy file or directory to target name */
		    pftp_item_t que_item = malloc(sizeof(struct pftp_item_s));
		    memset(que_item, 0, sizeof(struct pftp_item_s));
		    que_item->ftp_from = NULL;
		    que_item->ftp_to = alloc_strcpy(CurFtp);
		    que_item->size = lsdata->files[0].size;
		    que_item->dir = movedir;
		    que_item->filename = alloc_strcpy(lsdata->files[0].name);
		    que_item->target = alloc_strcpy(argv[1]);
		    que_item->changed = alloc_strcpy(lsdata->files[0].changed);
		    pftp_curdir(ftp, &que_item->destpath, 0);
		    que_item->srcpath = srcpath;
		    que_item->mode = PFTP_MODE_NORMAL;
		    que_item->action = PFTP_COPY;
		    que_item->preserve = PFTP_PRESERVE_NONE;
		    if (!que_item->srcpath || !que_item->destpath) {
			ret = -1;
		    }
		    if (!ret) {
			pftp_que_add_last(que_item, fileque);
		    } else {
			pftp_free_queitem(&que_item);
		    }
		} else {
		    if (srcpath)
			free(srcpath);           
		}
	    } else {
		size_t a;          
		char **srcpath = malloc(argc * sizeof(char *));
		pftp_directory_t **lsdata = malloc(sizeof(pftp_directory_t *) 
						   * argc);
		memset(srcpath, 0, sizeof(char *) * argc);
		
		for (a = 0; a < argc; a++) {
		    expand_tilde(argv + a);
		    if (!(lsdata[a] = pftp_trans_ls(NULL, argv[a], srcpath + a,
						    1, 0))) {
			ret = -1;
			break;
		    }
		    
		    if (lsdata[a]->length == 0 
			&& strchr(argv[a], '*') == NULL
			&& strchr(argv[a], '?') == NULL)
			printf("Warning: Mask `%s' doesn\'t match anything.\n",
			       argv[a]);
		}
		
		if (!ret) {
		    for (a = 0; a < argc; a++) {
			ret = pftp_get_files_ordered(NULL, CurFtp, srcpath[a],
						     lsdata[a], fileque, 1,
						     PFTP_PRESERVE_NONE,
						     PFTP_COPY,
						     PFTP_MODE_NORMAL);
			pftp_free_fdt((*lsdata[a]));
			free(lsdata[a]);        
		    }
		} else {
		    while (a > 0) {
			pftp_free_fdt((*lsdata[a]));
			free(lsdata[a]);
		    }
		}
		
		for (a = 0; a < argc; a++)
		    if (srcpath[a])
			free(srcpath[a]);
		
		free(srcpath);
		free(lsdata);
	    }
	}
    } else {
	puts(usage);
	ret = 0;
    }
    
    freeParseArgs(&flags, &argv, &argc);
    
    return ret;
}

int ftp_rename(char *args, const char *usage)
{
    char **argv = NULL, *flags = NULL;
    size_t argc = 0;
    int ret = parseArgs(args, NULL, &flags, &argv, &argc);
    
    if (!ret && argc == 2) {
	pftp_server_t ftp;
	if ((ftp = getCurFTP())) {
	    pftp_rename(ftp, argv[0], argv[1]);
	}
    } else {
	puts(usage);
	ret = 0;
    }

    freeParseArgs(&flags, &argv, &argc);

    return ret;
}

int ftp_rawcmd(char *args, const char *usage)
{
    char **argv = NULL, *flags = NULL;
    size_t argc = 0;
    int ret = parseArgs(args, NULL, &flags, &argv, &argc);
    
    if (!ret && argc == 0) {
	puts(usage);
	ret = 0;
    } else {
	pftp_server_t ftp;
	char **data = NULL;
	if ((ftp = getCurFTP())) {
	    add_output_status_action(ftp, NEXTFATAL);
	    ret = pftp_rawcmd(ftp, args, data);
	    
	    if (ret != -1)
		ret = 0;
	}
    }
    
    freeParseArgs(&flags, &argv, &argc);
    
    return ret;
}

int ftp_pwd(char *args, const char *usage)
{
    char **argv = NULL, *flags = NULL;
    size_t argc = 0;
    int ret = parseArgs(args, NULL, &flags, &argv, &argc);
    
    if (!ret && argc == 0) {
	char *dir = NULL;
	pftp_server_t ftp;
	if ((ftp = getCurFTP())) {
	    pftp_curdir(ftp, &dir, 1);
	    if (dir) {
		printf("%s\n", dir);
		updateCurrentFTP();
		free(dir);
	    }
	} else {
	    ret = -1;
	}
    } else {
	puts(usage);
	ret = 0;
    }

    freeParseArgs(&flags, &argv, &argc);

    return ret;
}

int confirm(const char *que, int yes_default)
{
    ssize_t ret;
    size_t len = 0;
    char *buf = NULL;
    
    for (;;) {
	printf("%s", que);
	if (yes_default)
	    printf(" (Y/n) ");
	else
	    printf(" (y/N) ");
	
	ret = getline(&buf, &len, stdin);
	
	if (ret == -1) 
	    return 0;
	
	len = 0;
	while (xisspace(buf[len])) len++;
	
	if (buf[len]) {
	    if (tolower(buf[len]) == 'y') {
		free(buf);
		return 1;
	    } else if (tolower(buf[len]) == 'n') {
		free(buf);
		return 0;
	    }
	    
	    printf("`%s' is not a valid answer.\n", buf + len);        
	} else {
	    free(buf);
	    return yes_default;
	}
    }
}

#define IFCOMPLETE_CMD(cmdstr) (strncmp(line, cmdstr, cmd - line) == 0)
#define COMPLETE_REMOTE_CMD(mask) ret = ((CurFtp && strlen(CurFtp) && (ftp = getCurFTP())) ? tabPressed(&line, &new_point, ftp, mask | TAB_REMOTE, TERM_width) : -1)
#define COMPLETE_CMD(mask)					\
    ret = tabPressed(&line, &new_point, ftp, mask, TERM_width)

/* Called by readline when [TAB] is pressed. Does the pretty stuff... */
char **pftp_completion(const char *text, int start, int end)
{
    /* Get copies of rl_line_buffer and rl_point */
    char *line = malloc(rl_end + 1);
    int ret = -1;
    size_t new_point = rl_point;
    char *cmd = NULL;
    pftp_server_t ftp = NULL;
    int noredraw = 1;
    
    memcpy(line, rl_line_buffer, rl_end);
    line[rl_end] = '\0';
    cmd = strchr(line, ' ');
    
    /* Stop readlines own completion */
    rl_attempted_completion_over = 1; 
    
    command_central(CLI_INITTABCOMP, NULL, NULL, NULL, NULL);
    
    if (cmd == NULL || start < (cmd - line)) {
	ret = tabPressed(&line, &new_point, NULL, TAB_COMMAND, TERM_width);
    } else {
	if (IFCOMPLETE_CMD("help")) COMPLETE_CMD(TAB_COMMAND);
	else if (IFCOMPLETE_CMD("?")) COMPLETE_CMD(TAB_COMMAND);
	else if (IFCOMPLETE_CMD("set")) COMPLETE_CMD(TAB_SITENAME);
	else if (IFCOMPLETE_CMD("save")) COMPLETE_CMD(TAB_SITENAME);
	else if (line[0] == '!')
	    COMPLETE_CMD(TAB_FILE | TAB_DIRECTORY | TAB_LOCAL);
	else if (IFCOMPLETE_CMD("open")) 
	    COMPLETE_CMD(TAB_SITENAME | TAB_LOADSITE);
	else if (IFCOMPLETE_CMD("current")) COMPLETE_CMD(TAB_SITENAME);
	else if (IFCOMPLETE_CMD("load")) 
	    COMPLETE_CMD(TAB_SITENAME | TAB_LOADSITE);
	else if (IFCOMPLETE_CMD("close")) COMPLETE_CMD(TAB_SITENAME);
	else if (IFCOMPLETE_CMD("logout")) COMPLETE_CMD(TAB_SITENAME);
	else if (IFCOMPLETE_CMD("cd")) COMPLETE_REMOTE_CMD(TAB_DIRECTORY);
	else if (IFCOMPLETE_CMD("lcd")) 
	    COMPLETE_CMD(TAB_DIRECTORY | TAB_LOCAL);
	else if (IFCOMPLETE_CMD("rm")) 
	    COMPLETE_REMOTE_CMD(TAB_DIRECTORY | TAB_FILE);
	else if (IFCOMPLETE_CMD("rmdir")) COMPLETE_REMOTE_CMD(TAB_DIRECTORY);
	else if (IFCOMPLETE_CMD("mkdir")) COMPLETE_REMOTE_CMD(TAB_DIRECTORY);
	else if (IFCOMPLETE_CMD("ls"))
	    COMPLETE_REMOTE_CMD(TAB_FILE | TAB_DIRECTORY);
	else if (IFCOMPLETE_CMD("lls"))
	    COMPLETE_CMD(TAB_FILE | TAB_DIRECTORY | TAB_LOCAL);
	else if (IFCOMPLETE_CMD("rename"))
	    COMPLETE_REMOTE_CMD(TAB_FILE | TAB_DIRECTORY);
	else if (IFCOMPLETE_CMD("mv"))
	    COMPLETE_REMOTE_CMD(TAB_FILE | TAB_DIRECTORY);
	else if (IFCOMPLETE_CMD("rawcmd"))
	    COMPLETE_REMOTE_CMD(TAB_FILE | TAB_DIRECTORY);
	else if (IFCOMPLETE_CMD("site"))
	    COMPLETE_REMOTE_CMD(TAB_FILE | TAB_DIRECTORY);
	else {
	    if (IFCOMPLETE_CMD("get") || IFCOMPLETE_CMD("put") ||
		IFCOMPLETE_CMD("fxp")) {
		char *flags = NULL, **argv = NULL;
		size_t argc;
		int ret2 = parseArgs(cmd + 1, "m", &flags, &argv, &argc);
		
		if (!ret2) {
		    if (strchr(flags, 'm')) {
			size_t curarg = 0;
			while (curarg < argc) {
			    if (strcmp(text, argv[curarg]) == 0)
				break;
			    else
				curarg++;
			}
			
			if (IFCOMPLETE_CMD("get")) {
			    if (curarg == 0)
				COMPLETE_REMOTE_CMD(TAB_FILE | TAB_DIRECTORY);
			    else
				COMPLETE_CMD(TAB_FILE|TAB_DIRECTORY|TAB_LOCAL);
			} else if (IFCOMPLETE_CMD("put")) {
			    if (curarg == 0)
				COMPLETE_CMD(TAB_FILE|TAB_DIRECTORY|TAB_LOCAL);
			    else
				COMPLETE_REMOTE_CMD(TAB_FILE | TAB_DIRECTORY);
			} else if (IFCOMPLETE_CMD("fxp")) {
			    if (curarg == 0) {
				COMPLETE_CMD(TAB_SITENAME);
			    } else {
				if (curarg == 1) {
				    ftp = getCurFTP();
				} else {
				    ftp = pftp_getFTPCtrl(argv[0]);
				}
				
				if (ftp)
				    COMPLETE_CMD(TAB_FILE | TAB_DIRECTORY |
						 TAB_REMOTE);
			    }        
			}
			
		    } else {
			if (IFCOMPLETE_CMD("get"))
			    COMPLETE_REMOTE_CMD(TAB_FILE | TAB_DIRECTORY);
			else if (IFCOMPLETE_CMD("put")) 
			    COMPLETE_CMD(TAB_FILE|TAB_DIRECTORY|TAB_LOCAL);
			else if (IFCOMPLETE_CMD("fxp")) {
			    size_t curarg = 0;
			    while (curarg < argc) {
				if (strcmp(text, argv[curarg]) == 0)
				    break;
				else
				    curarg++;
			    }
			    
			    if (curarg == 0) {
				COMPLETE_CMD(TAB_SITENAME);
			    } else {
				if ((ftp = getCurFTP())) {
				    COMPLETE_CMD(TAB_FILE | TAB_DIRECTORY |
						 TAB_REMOTE);
				}
			    }        
			}
		    }
		}
		
		freeParseArgs(&flags, &argv, &argc);
	    }
	}
    }
    
    noredraw = command_central(CLI_DONETABCOMP, NULL, NULL, NULL, NULL);
    
    if (ret == 1) {
	/* One completeion returned and line updates, 
	   so update rl_line_buffer */
	rl_begin_undo_group();
	rl_point = 0;
	rl_delete_text(0, rl_end);
	rl_insert_text(line);    
	rl_point = (int)new_point;
	rl_end_undo_group();
	if (!noredraw)
	    rl_on_new_line();
	else
	    rl_redisplay();
    } else if (ret == 0) {
	/* A list of completions returned, so cmd line */
	rl_on_new_line();
    } else {
	assert(ret == -1);
	/* Didn't find any matches, so dont update line if you dont have to */
	if (!noredraw)
	    rl_on_new_line();
    }
    
    free(line);
    
    return NULL;
}

void initialize_readline(void)
{
    /* Allow conditional parsing of the ~/.inputrc file. */
    rl_readline_name = "pftp";
    
    /* Tell the completer that we want a crack first. */
    rl_attempted_completion_function = pftp_completion;
}

static size_t argsplit(const char *str, char ***part)
{
    size_t parts = 0;
    char *cpy = strdup(str);
    size_t len = strlen(cpy);
    char *pos = cpy, *last = cpy, killer = '\0';
    int live = 1;

    while (pos[0]) {
	if (live) {
	    if (pos[0] == '\\') {
#ifdef WIN32
		pos[0] = '/';
#else
		memmove(pos, pos + 1, (len--) - (pos - cpy));
#endif
	    } else if (pos[0] == ' ') {
		if (pos == last) {
		    last++;
		} else {
		    (*part) = realloc((*part), (parts + 1) * sizeof(char *));
		    (*part)[parts] = malloc((pos - last) + 1);
		    memcpy((*part)[parts], last, pos - last);
		    (*part)[parts][pos - last] = '\0';
		    parts++;
		    last = pos + 1;
		}
	    } else if (pos[0] == '\'' || pos[0] == '\"') {
		killer = pos[0];
		memmove(pos, pos + 1, (len--) - (pos - cpy));
		live = 0;
		continue;
	    }
	} else {
	    if (pos[0] == killer) {
		live = 1;
		memmove(pos, pos + 1, (len--) - (pos - cpy));
		continue;
	    }
	}
	
	pos++;
    }
    
    if (pos > last) {
	(*part) = realloc((*part), (parts + 1) * sizeof(char *));
	(*part)[parts] = malloc((pos - last) + 1);
	memcpy((*part)[parts], last, pos - last);
	(*part)[parts][pos - last] = '\0';
	parts++;
    }
    
    free(cpy);
    
    return parts;
}

int parseArgs(const char *args, const char *flags, char **set_flags, 
	      char ***argv, size_t *argc)
{
    char **real_argv = NULL;
    size_t real_argc = argsplit(args, &real_argv);
    int ret = 0;
    size_t c, start = real_argc, p = 0;
    
    freeParseArgs(set_flags, argv, argc);    
    if (flags) {
	(*set_flags) = malloc(strlen(flags) + 1);
	memset((*set_flags), '\0', strlen(flags) + 1);
    } else {
	(*set_flags) = malloc(1);
	(*set_flags)[0] = '\0';
    }
    
    for (c = 0; c < real_argc && !ret; c++) {
	if (strcmp(real_argv[c], "--") == 0) {
	    start = c + 1;
	    break;
	} else if (real_argv[c][0] == '-' && flags) {
	    p = 1;
	    while (real_argv[c][p] && !ret) {
		ret = 1;
		if (strchr(flags, real_argv[c][p])) {
		    ret = 0;
		    if (!strchr((*set_flags), real_argv[c][p])) 
			(*set_flags)[strlen((*set_flags))] = real_argv[c][p];
		    
		}
		p++;
	    }
	} else {
	    start = c;
	    break;
	}
    }
    
    if (ret) {
	fprintf(stderr, "Invalid flag: `%s'\n", real_argv[c - c]);
    } else {
	for (c = start; c < real_argc; c++) {
	    /*if (real_argv[c][0] == '-')
	      continue;*/
	    
	    (*argv) = realloc((*argv), ((*argc) + 1) * sizeof(char *));
	    (*argv)[(*argc)++] = alloc_strcpy(real_argv[c]);
	}
    }
    
    free_split(&real_argv, &real_argc);   
    
    for (c = 0; c < (*argc); c++) {
	size_t len = strlen((*argv)[c]);       
/*
 *      argsplit now handles this.
 *
 if ((*argv)[c][0] == '\"'
 && (*argv)[c][len - 1] == '\"') {
 memmove((*argv)[c], (*argv)[c] + 1, len);
 (*argv)[c][len - 2] = '\0';
 len -= 2;
 } else if ((*argv)[c][0] == '\''
 && (*argv)[c][len - 1] == '\'') {
 memmove((*argv)[c], (*argv)[c] + 1, len);
 (*argv)[c][len - 2] = '\0';
 len -= 2;
 }
*/
	if (len > 1 && (*argv)[c][len - 1] == '/') {
	    (*argv)[c][len - 1] = '\0';
	    len--;
	}
    }
    
    return ret;
}

void freeParseArgs(char **set_flags, char ***argv, size_t *argc)
{
    if (set_flags && (*set_flags)) {
	free((*set_flags));
	(*set_flags) = NULL;
    }
    if (argv && (*argv)) {
	if (argc) {
	    size_t c;
	    for (c = 0; c < (*argc); c++)
		free((*argv)[c]);
	    (*argc) = 0;
	}
	
	free((*argv));
	(*argv) = NULL;
    } else if (argc) {
	(*argc) = 0;
    }
}

void print_filesize(int width, unsigned short format, uint64_t size, FILE *out)
{
    char sformat[12];

    if (width >= 0) {
	if (format & LS_HUMAN) {
	    if (size < 1024)
		snprintf(sformat, 10, "%%%dllu", width - 1);
	    else
		snprintf(sformat, 10, "%%%d.1f", width - 1);
	} else {
	    snprintf(sformat, 10, "%%%dllu", width);
	}
    } else {
	if (format & LS_HUMAN) {
	    if (size < 1024)
		strcpy(sformat, "%llu");
	    else
		strcpy(sformat, "%.1f");
	} else {
	    strcpy(sformat, "%llu");
	}
    }
    
    if (format & LS_HUMAN) {
	if (size < 1024) {
	    fprintf(out, sformat, (unsigned long long) size);
	    fputs(" ", out);
	} else if (size < 1024 * 1024) {
	    strcat(sformat, "k");
	    fprintf(out, sformat, (double)size / 1024.0);
	} else if (size < 1024 * 1024 * 1024) {        
	    strcat(sformat, "m");
	    fprintf(out, sformat, (double)size / (1024.0 * 1024.0));
	} else {
	    strcat(sformat, "g");
	    fprintf(out, sformat, (double)size / (1024.0 * 1024.0 * 1024.0));
	}
    } else {
	fprintf(out, sformat, (unsigned long long) size);
    }
}

void print_ls(pftp_directory_t *filelist, unsigned short format)
{
    FILE *out = startPager();
    
    if (filelist && filelist->length) {
	if (format & LS_LONGLIST) {
	    size_t f, i, maxuser = strlen(filelist->files[0].user), 
		maxgroup = strlen(filelist->files[0].group);
	    char *user = NULL, *group = NULL;       
	    uint64_t total_size = filelist->files[0].size;
	    
	    for (f = 1; f < filelist->length; f++) {
		if (strlen(filelist->files[f].user) > maxuser)
		    maxuser = strlen(filelist->files[f].user);
		if (strlen(filelist->files[f].group) > maxgroup)
		    maxgroup = strlen(filelist->files[f].group);
		total_size += filelist->files[f].size;
	    }
	    
	    user = malloc(maxuser + 1);
	    user[maxuser] = '\0';
	    group = malloc(maxgroup + 1);
	    group[maxgroup] = '\0';
	    
	    fputs("total ", out);
	    print_filesize(-1, format, total_size, out);
	    fputs("\n", out);
	    
	    for (f = 0; f < filelist->length; f++) {
		if (filelist->files[f].link) {
		    fputs("l", out);
		} else if (filelist->files[f].type == pft_directory) {
		    fputs("d", out);
		} else {
		    fputs("-", out);
		}

		for (i = 0; i < 3; i++) {
		    if (filelist->files[f].perm & (4 * EXP8[2-i])) {
			fputs("r", out);
		    } else {
			fputs("-", out);
		    }
		    if (filelist->files[f].perm & (2 * EXP8[2-i])) {
			fputs("w", out);
		    } else {
			fputs("-", out);
		    }
		    if (filelist->files[f].perm & (1 * EXP8[2-i])) {
			fputs("x", out);
		    } else {
			fputs("-", out);
		    }
		}

		fputs(" ", out);
		fprintf(out, "%4lu ", filelist->files[f].links);
		
		memset(user, ' ', maxuser);
		memset(group, ' ', maxgroup);
		memcpy(user, filelist->files[f].user, 
		       strlen(filelist->files[f].user));
		memcpy(group, filelist->files[f].group, 
		       strlen(filelist->files[f].group));
		fputs(user, out);
		fputs(" ", out);
		fputs(group, out);
		fputs(" ", out);
		print_filesize(9, format, filelist->files[f].size, out);
		fprintf(out, " %s ", filelist->files[f].changed);
		fprintf(out, "%s\n", filelist->files[f].name);
	    }
	    
	    free(user);
	    free(group);
	} else {       
	    size_t l, x, y, lines, columns, *colwidth = NULL, *len = NULL;
	    char *space = malloc(TERM_width);
	    memset(space, ' ', TERM_width);
	    len = malloc(filelist->length * sizeof(size_t));
	    
	    for (l = 0; l < filelist->length; l++)
		len[l] = strlen(filelist->files[l].name);
	    
	    calc_columns(len, filelist->length, TERM_width, 2,
			 &columns, &colwidth, &lines);
	    
	    for (y = 0; y < lines; y++) {
		for (x = 0; x < columns; x++) {
		    if ((l = y+(x*lines)) >= filelist->length) break;
		    fwrite(filelist->files[l].name, 1, len[l], out);
		    if (len[l] < colwidth[x])
			fwrite(space, 1, colwidth[x] - len[l], out);
		}
		fputs("\n", out);
	    }
	    
	    free(space);
	    free(colwidth);
	    free(len);
	}
    }
    
    endPager(out);
}

void getMeAPager(void)
{
    const char *test = NULL;
    char *cmd = NULL;
    size_t try;
    FILE *f;
    
    if (pager) {
	free(pager);
	pager = NULL;
    }   
    
    for (try = 0; try < 4 && !pager; try++) {
	switch (try) {
	case 0: {
	    pftp_conf_t conf = pftp_getConfig();
	    test = conf->pager;
	}; break;
	case 1:
#ifdef WIN32
	    test = "";
#else
	    test = "less -E -n -X";
#endif
	    break;
	case 2:
	    test = "more";
	    break;
	case 3:
	    test = "most";
	    break;
	}
	
	if (test && test[0]) {
#ifdef WIN32
	    cmd = realloc_strcpy(cmd, "echo \" \" | ");
	    cmd = realloc_strcat(cmd, test);
#else
	    cmd = realloc_strcpy(cmd, test);
	    cmd = realloc_strcat(cmd, " < /dev/null");
#endif
	    f = popen(cmd, "r");
	    
	    if (f && !pclose(f)) {
		pager = realloc_strcpy(pager, test);
	    }
	}
    }    
    
    if (cmd) free(cmd);
}

FILE *startPager(void)
{
    FILE *ret = NULL;

    if (pager) 
	ret = popen(pager, "w");
    
    if (!ret)
	ret = stdout;
    
    return ret;
}

void endPager(FILE *fh)
{
    if (fh != stdout)
	pclose(fh);
}

int shell_out(const char *line)
{
    if (strlen(line))
	return system(line);
    else
	return system("/bin/sh -i");
}

unsigned int dialog(pftp_file_t existing,const  pftp_item_t transfering)
{
    char *choice = NULL;
    size_t s = 1;
    unsigned int ret = 1, not_valid = 1;
    printf("The file %s already exist.\n", existing.name);
    printf("The existing filesize is ");
    print_filesize(6, LS_HUMAN, existing.size, stdout);
    printf(" and was changed the %s\n", existing.changed);
    if (transfering->ftp_from && !transfering->ftp_to) {
	printf("The size of the file you want to download is ");
    } else if (!transfering->ftp_from && transfering->ftp_to) {
	printf("The size of the file you want to upload is ");
    } else {
	printf("The size of the file you want to transfer is ");
    }
    print_filesize(6, LS_HUMAN, transfering->size, stdout);
    printf(" and was changed the %s\n", transfering->changed);
    while(not_valid) {
	printf("What do you want to do?\n");
	printf("[o]verwrite\n");
	printf("[O]verwrite All\n");
	if(existing.size < transfering->size){
	    printf("[r]esume\n");    
	    printf("[R]esume All\n");
	}
	printf("[s]kip\n");    
	printf("[S]kip All\n");    
	printf("[A]bort All Transfers\n");
	
	getline(&choice, &s, stdin);
	
	if (choice[0] == 'o'){
	    ret = PFTP_DLG_OVERWRITE;
	    not_valid = 0;
	} else if (choice[0] == 'O'){
	    ret = PFTP_DLG_OVERWRITE | PFTP_DLG_ALL;
	    not_valid = 0;
	} else if (choice[0] == 'r' && 
		   existing.size < transfering->size){
	    ret = PFTP_DLG_RESUME;
	    not_valid = 0;
	} else if (choice[0] == 'R' && 
		   existing.size < transfering->size){
	    ret = PFTP_DLG_RESUME | PFTP_DLG_ALL;
	    not_valid = 0;
	} else if (choice[0] == 's'){
	    ret = PFTP_DLG_SKIP;
	    not_valid = 0;
	} else if (choice[0] == 'S'){
	    ret = PFTP_DLG_SKIP | PFTP_DLG_ALL;
	    not_valid = 0;
	} else if (choice[0] == 'A'){
	    ret = PFTP_DLG_ABORT;
	    not_valid = 0;
	}
    }
    
    free(choice);
    return ret;
}

int ftp_clear_trans(char *args, const char *usage)
{
    char **argv = NULL, *flags = NULL;
    size_t argc = 0;
    int ret = parseArgs(args, "e", &flags, &argv, &argc);
    
    if (!ret && argc == 0) {
	if (strchr(flags, 'e'))
	    pftp_clear_que(errorque);
	else
	    pftp_clear_que(fileque);
    } else {
	puts(usage);
	ret = 0;
    }
    
    freeParseArgs(&flags, &argv, &argc);

    return ret;
}

int ftp_view_trans(char *args, const char *usage)
{
    char **argv = NULL, *flags = NULL;
    size_t argc = 0;
    int ret = parseArgs(args, "e", &flags, &argv, &argc);
    size_t pos = 0;
    pftp_item_t item;
    pftp_que_t q = fileque;
    
    if (!ret && argc == 0) {
	if (strchr(flags, 'e'))
	    q = errorque;
	
	if (!(item = pftp_que_peek(q, pos))) {
	    puts("Queue is empty");
	}
	
	while (item) {
	    pos++;
	    
	    switch (item->action) {
	    case PFTP_COPY:
		puts("** Action: Copy.");
		break;
	    case PFTP_MOVE:
		puts("** Action: Move.");
		break;
	    case PFTP_DELETE_FROM:
	    case PFTP_DELETE_FROM_DIR:
		puts("** Action: Delete.");
		break;
	    }
	    
	    if (item->action == PFTP_COPY || item->action == PFTP_MOVE) {
		printf("-- Source name: `%s' (%lluB)\n", item->filename, 
		       (unsigned long long) item->size);
		printf("   Target name: `%s'\n", 
		       item->target ? item->target : item->filename);
		printf("   Server: %s -> %s\n", 
		       item->ftp_from ? item->ftp_from : "localhost",
		       item->ftp_to ? item->ftp_to : "localhost");
		printf("   Path: %s -> %s\n", item->srcpath, item->destpath);
	    } else {
		printf("-- Source name: `%s' (%lluB)\n", item->filename, 
		       (unsigned long long) item->size);
		printf("   Server: %s\n", 
		       item->ftp_from ? item->ftp_from : "localhost");
		printf("   Path: %s\n", item->srcpath);
	    }
	    
	    item = pftp_que_peek(q, pos);
	}
    } else {
	puts(usage);
	ret = 0;
    }
    
    freeParseArgs(&flags, &argv, &argc);
    
    return ret;
}

void error(const char *format, ...)
{
    va_list list;
    va_start(list, format);
    
#if HAVE_VPRINTF
    vprintf(format, list);
#else
# if HAVE_DOPRNT
    _doprnt(format, list, stdout);
# else
#  error Need vprintf or _doprnt
# endif 
#endif
    
    puts("");
    va_end(list);
}

int ftp_refresh(char *args, const char *usage)
{
    char **argv = NULL, *flags = NULL;
    size_t argc = 0;
    int ret = parseArgs(args, "a", &flags, &argv, &argc);
    if (ret == 0) {
	if (strchr(flags, 'a' ))
	    pftp_clear_dir_cache(getCurFTP(), NULL);
	else
	    pftp_clear_dir_cache(getCurFTP(), ".");
    }
    freeParseArgs(&flags, &argv, &argc);
    return ret;
}

int ftp_site(char *args, const char *usage)
{
    char **argv = NULL, *flags = NULL;
    size_t argc = 0;
    int ret = parseArgs(args, NULL, &flags, &argv, &argc);
    
    if (ret || argc == 0) {
	puts(usage);
	ret = 0;
    } else {
	pftp_server_t ftp;
	char *line = malloc(strlen(args) + 6);
	strcpy(line, "site ");
	strcat(line, args);
	if ((ftp = getCurFTP())) {
	    add_output_status_action(ftp, NEXTFATAL);
	    ret = pftp_rawcmd(ftp, line, NULL);
	    
	    if (ret != -1)
		ret = 0;
	}
	free(line);
    }
    
    freeParseArgs(&flags, &argv, &argc);
    
    return ret;
}

void signal_handler(int signum)
{
    if (signum == SIGINT) {
	command_central(CLI_ABORT, NULL, NULL, 0, NULL);
#ifndef WIN32
    } else if (signum == SIGWINCH) {
	rl_get_screen_size(&TERM_height, &TERM_width);
	if (!TERM_width) TERM_width = 80;
	if (!TERM_height) TERM_height = 24;
#endif
    }
}

int ftp_retry_trans(char *args, const char *usage)
{
    char **argv = NULL, *flags = NULL;
    size_t argc = 0;
    int ret = parseArgs(args, NULL, &flags, &argv, &argc);   

    if (!ret && argc == 0) {
	pftp_item_t item;
	size_t items = 0;
	
	while ((item = pftp_get_next(errorque))) {
	    pftp_que_add_last(item, fileque);
	    items++;
	}
	
	printf("Moved %lu items from error que to transfer que.\n", 
	       (unsigned long)items);
    } else {
	puts(usage);
	ret = 0;
    }
    
    freeParseArgs(&flags, &argv, &argc);
    
    return ret;
}

int parseProgArgs(int argc, char **argv)
{
    int version = 0, usage = 0, start = 1, syntax_error = 0;
#if HAVE_GETOPT_LONG
    int c;

    for (;;) {
	static struct option long_options[] =
	    {
		{"version",  no_argument, 0, 'v'},
		{"help",     no_argument, 0, 'h'},
		{NULL,       0,           0, 0}
	    };
	
	int option_index = 0;
	
	c = getopt_long(argc, argv, "vh", long_options, &option_index);
	
	if (c == -1)
	    break;
	
	switch (c) {
	case 'v':
	    version = 1;
	    break;
	case 'h':
	    usage = 1;
	    break;
	case '?':
	    syntax_error = 1;
	    break;
	default:
	    abort();
	}
    }
    
    start = optind;
#else
# if HAVE_GETOPT
    int c;
    
    for (;;) {
	c = getopt(argc, argv, "vh");
	
	if (c == -1)
	    break;
	
	switch (c) {
	case 'v':
	    version = 1;
	    break;
	case 'h':
	    usage = 1;
	    break;
	case '?':
	    syntax_error = 1;
	    break;
	default:
	    abort();
	}
    }
    
    start = optind;
# else
    int c;
    
    for (c = 1; c < argc; c++) {
	if (argv[c][0] == '-') {
	    if (argv[c][1] == '-') {
		c++;
		break;
	    } else if (argv[c][1] == '\0') {
		break;
	    } else {
		size_t p = 1;
		while (argv[c][p]) {
		    switch (argv[c][p]) {
		    case 'h':
			usage = 1;
			break;
		    case 'v':
			version = 1;
			break;
		    case '\0':
			break;
		    default:
			fprintf(stderr, "%s: invalid option -- %c\n", argv[0], 
				argv[c][p]);
			syntax_error = 1;
			break;
		    }
		    
		    p++;
		}
	    }
	} else {
	    break;
	}
    }
    
    start = c;
# endif
#endif

    if (syntax_error) {
	printf("Use `%s --help' to see usage.\n", argv[0]);
	return 1;
    }
    
    if (usage) {
	printf("Usage: %s [options] [open]\n", argv[0]);
	puts("Options:");
#if HAVE_GETOPT_LONG
	puts("  -v, --version    Print version and exit.");
	puts("  -h, --help       Show this text and exit.");
#else
	puts("  -v               Print version and exit.");
	puts("  -h               Show this text and exit.");
#endif
	return 1;
    }
    
    if (version) {
	puts("PFTP " VERSION);
	return 1;
    }
    
    if (start < argc) {
	size_t c;
	
	if (start - argc > 1) {
	    fprintf(stderr, "Too many open arguments.\n");
	    fprintf(stderr, "Use `%s --help' to see usage.\n", argv[0]);
	    return 1;
	}
	
	for (c = 0; c < no_commands; c++) {
	    if (commands[c].func == ftp_open) {
		commands[c].func(argv[start], commands[c].usage);
		break;
	    }
	}
    }
    
    return 0;
}

void get_homedir_of_user(char **target, const char *username)
{
#ifdef WIN32
    char *path = getenv("HOMEPATH"), *drive = getenv("HOMEDRIVE");
    if (*target) 
        free(*target);
    if (path && drive) {
        *target = realloc_strcpy(*target, drive);
        *target = realloc_strcat(*target, path);
    } else {
        *target = strdup("C:\\");
    }
#else
    struct passwd *data = getpwnam(username);
    
    if (data && data->pw_dir) {
	size_t len = strlen(data->pw_dir) + 1;
	(*target) = realloc((*target), len);
	memcpy((*target), data->pw_dir, len);
    }
#endif
}

void expand_tilde(char **path)
{
    char *pos = strchr((*path), '~');
    char *expand_to = NULL;
    size_t start, end, len, path_len;
    int free_expand = 0;
    
    if (!pos)
	return;
    
    if (pos == (*path)) {
	/* Only care if it's starts the path */
	if (pos[1] == '/' || pos[1] == '\0') {
	    /* Expand to $HOME directory */
	    start = pos - (*path);
	    end = start + 1;
	    
	    if (!(expand_to = getenv("HOME"))) {
		get_homedir_of_user(&expand_to, getenv("USER"));
		free_expand = 1;
	    }
	    
	    if (expand_to && pos[1] == '\0') {
		size_t l = strlen(expand_to);
		if (free_expand) {
		    expand_to = realloc(expand_to, l + 2);
		} else {
		    char *tmp = malloc(l + 2);
		    memcpy(tmp, expand_to, l);
		    free_expand = 1;
		    expand_to = tmp;
		}
		
		expand_to[l] = '/';
		expand_to[l + 1] = '\0';
	    }
	} else {
	    /* Expand to ~USER's home directory */
	    char *pos2 = strchr(pos + 2, '/'), *username;
	    if (!pos2)
		pos2 = pos + strlen(pos);
	    
	    start = pos - (*path);
	    end = pos2 - (*path);
	    username = malloc(end - start);
	    memcpy(username, pos + 1, end - (start + 1));
	    username[end - (start + 1)] = '\0';
	    
	    get_homedir_of_user(&expand_to, username);
	    free(username);
	    free_expand = 1;
	}
    } else {
	/* Ignore it, it's part of a name */
	return;
    }
    
    if (!expand_to) {
	/* Unable to find anything to expand to */
	return;
    }
    
    len = strlen(expand_to);
    path_len = strlen((*path));
    
    if (len > end - start) {
	(*path) = realloc((*path), path_len + (len - (end - start)) + 1);
    } 
    
    memmove((*path) + start + len, (*path) + end, (path_len - end) + 1);    
    memcpy((*path) + start, expand_to, len);
    
    if (free_expand)
	free(expand_to);
}

ssize_t simple_getline(char **pass, size_t *len, FILE *fh, 
		       const char *start_val, int echo)
{
    ssize_t ret = -1;
#ifdef WIN32
    DWORD oldmode, setmode;
    
    if (!GetConsoleMode(console_in, &oldmode)) {
        fprintf(stderr, "Error: Unable to get console mode: `%s'.\n",
                strerror(errno));
        return 0;
    }
    
    if (echo)
        setmode = oldmode | ENABLE_ECHO_INPUT;
    else
        setmode = oldmode & ~ENABLE_ECHO_INPUT;
    
    if (!SetConsoleMode(console_in, setmode)) {
        fprintf(stderr, "Error: Unable to set console mode: `%s'.\n",
                strerror(errno));
        return 0;
    }
#else
    struct termios old, new;
    
    (*len) = 0;
    
    if (tcgetattr(fileno(stdin), &old)) {
	return 0;
    }
    
    new = old;
    if (echo)
        new.c_lflag |= ECHO;
    else
        new.c_lflag &= ~ECHO;
    
    if (tcsetattr(fileno(stdin), TCSAFLUSH, &new)) {
	return 0;
    }
#endif
    
    if (start_val) {
	size_t startlen;
	startlen = strlen(start_val);
	(*pass) = realloc((*pass), (*len) + startlen);
	memcpy((*pass) + (*len), start_val, startlen);
	(*len) += startlen;
	if (echo)
	    fwrite(start_val, 1, startlen, stdout);
    }
    
    for (;;) {
	int c;
	
#ifdef WIN32
	DWORD got;
	char buf[1];
	if (!ReadFile(console_in, buf, 1, &got, NULL)) {
	    c = EOF;
	} else {
	    if (got == 0)
		c = EOF;
	    else
		c = buf[0];
	}
#else
	c = getc(stdin);
#endif
	
#ifdef WIN32
	if (c == EOF || c == '\n') {
#else
	if (c == EOF || c == '\n' || c == '\r') {
#endif
	    (*pass) = realloc((*pass), (*len) + 2);
	    (*pass)[(*len)++] = '\n';
	    (*pass)[(*len)] = '\0';
	    ret = (*len);
	    break;
	} else if (c == 8) {
	    /* Backspace */
	    if ((*len) > 0)
		(*len)--;
	    
	    if (echo) {
		putc(c, stdout);
		putc(' ', stdout);
		putc(c, stdout);
	    }
	} else if (c > 31 && c < 255) {
	    (*pass) = realloc((*pass), (*len) + 1);
	    (*pass)[(*len)++] = c;
	    
	    if (echo)
		putc(c, stdout);
	}
    }
    
#ifdef WIN32
    if (!SetConsoleMode(console_in, setmode)) {
        fprintf(stderr, "Error: Unable to set console mode: `%s'.\n",
                strerror(errno));
        return 0;
    }
#else
    if (tcsetattr(fileno(stdin), TCSAFLUSH, &old)) {
	return 0;
    }
#endif
    
    return ret;
}

static int output_status_message(pftp_server_t server, 
                                 const char *ftpname, const char *msg) 
{
    typedef enum { DIR_IN, DIR_OUT, DIR_OTHER } dir_t;
    
    static unsigned long last_ftp_code = 0;
    static ftp_output_action_t last_action = INVALID;
    static int last_multiline = 0;
    dir_t dir = DIR_OTHER;
    int ftp = 0;
    unsigned long ftp_code = 0;
    
    if (strncmp(msg, "-> ", 3) == 0) {
	dir = DIR_OUT;
    } else if (strncmp(msg, "<- ", 3) == 0) {
	dir = DIR_IN;
    }
    
    if (dir == DIR_IN || dir == DIR_OUT) {
	ftp = 1;
	if (dir == DIR_IN) {
	    ftp_output_action_t action;
	    char *end = NULL;
	    int ok = 1;
	    ftp_code = strtoul(msg + 3, &end, 10);
	    
	    if (ftp_code > 0 && end && *end == '-') {
		last_multiline = 1;
	    }
	    
	    if (ftp_code == 0 && last_multiline) {
		ftp_code = last_ftp_code;
	    }
	    
	    if (ftp_code == last_ftp_code) {
		action = last_action;
		ok = 1;
	    } else if (get_output_status_action(server, &action)) {
		ok = 1;
	    } else {
		ok = 0;
	    }
	    
	    if (ok) {
		last_action = action;
		last_ftp_code = ftp_code;
		
		switch (action) {
		case NEXTNONFATAL:
		    if (debuglevel >= 2)
			return 1;
		    return 0;
		case NEXTFATAL:
		    return 1;
		case INVALID:
		    /* well... */
		    break;
		}
	    }
	} else {
	    last_ftp_code = 0;
	    last_action = INVALID;
	    ftp_code = 0;
	}
    }
    
    switch (debuglevel) {
    case 0:
	if (ftp && ftp_code >= 400)
	    return 1;
	
	if (!ftp)
	    return 1;
	break;
    case 1:
	if (ftp && ftp_code >= 200)
	    return 1;
	
	if (!ftp)
	    return 1;
	break;
    default:
    case 2:
	return 1;
    }
    
    return 0;
}

static int command_central(pftp_msg_t msg, pftp_server_t ftp, void *userdata, 
			   pftp_param_t p1, pftp_param_t p2)
{
    static size_t que_size = 0, que_pos = 0, que_error = 0, que_skipped = 0;
    static int abort_transfer = 0, abort_que = 0;
    static char *TERM_update = NULL;
//    static pftp_server_t fxp_src = NULL, fxp_dest = NULL;
    static int tab_comp_redraw = 0, tab_comp = 0;
    static int que_is_on = 0, file_is_on = 0, transfer_dir = 0; 
//, fxp_is_on = 0;
    static char speed_unit[3], trans_unit[3];
    static uint64_t file_start = 0, file_size = 0;
    
    switch (msg) {
    case CLI_INIT: {
	TERM_update = realloc(TERM_update, TERM_width + 1);
	return 0;
    }; break;
    case CLI_DONE: {
	if (TERM_update) {
	    free(TERM_update);
	    TERM_update = NULL;
	}
	return 0;
    }; break;
    case CLI_ABORT: {
	int what = ((intptr_t) p1);
	
	switch (what) {
	case 0:
	    abort_transfer = 1;
	    abort_que = 1;
	    break;
	case 1:
	    abort_que = 1;
	    break;
	}
	
	return 0;
    }; break;
    case CLI_INITTABCOMP: {
	tab_comp_redraw = 0;
	tab_comp = 1;
	
	return 0;
    }; break;
    case CLI_DONETABCOMP: {
	tab_comp = 0;
	
	return tab_comp_redraw ? 0 : -1;
    }; break;
    case PFTPUTIL_INITQUE: {
	que_skipped = 0;
	que_size = (size_t)p1;
	que_pos = 0;
	que_error = 0;
	abort_que = 0;
	que_is_on = 1;
	
	if (display_que_info)
	    printf("Transfer que.\n");
	
	return 0;
    }; break;
    case PFTPUTIL_UPDATEQUELENGTH: {
	que_size = (size_t)p1;
	
	return 0;
    }; break;
    case PFTPUTIL_UPDATEQUE: {
	que_pos = que_size - (size_t)p1;
	que_error = que_pos - (size_t)p2;
	
	return (abort_que ? 0 : -1);
    }; break;
    case PFTPUTIL_DONEQUE: {
	int error = (intptr_t) p1;
	
	que_skipped = (size_t)p2;
	que_error -= que_skipped;
	
	if (display_que_info) {
	    if (error) {
		if (que_skipped == 0) {
		    printf("Transfer que aborted. %lu of %lu was completed.\n",
			   (unsigned long)((que_pos - 1) - que_error), 
			   (unsigned long)que_size);
		} else {
		    printf("Transfer que aborted. %lu of %lu was completed"
			   " (%ld skipped).\n",
			   (unsigned long)((que_pos - 1) - (que_error + 
							    que_skipped)), 
			   (unsigned long)que_size,
			   (unsigned long)que_skipped);
		}
	    } else {
		if (que_skipped == 0) {
		    printf("Transfer que done. %lu of %lu was successfull.\n",
			   (unsigned long)(que_pos - que_error), 
			   (unsigned long)que_size);
		} else {
		    printf("Transfer que done. %lu of %lu was successfull"
			   " (%ld skipped).\n",
			   (unsigned long)(que_pos - (que_error + que_skipped)),
			   (unsigned long)que_size,
			   (unsigned long)que_skipped);
		}
	    }
	}
	
	que_error = 0;
	que_skipped = 0;
	que_size = 0;
	que_pos = 0;
	que_is_on = 0;
	
	return 0;
    }; break;
    case PFTPUTIL_NEEDCURFTP: {
	const char *ftp_name = (const char *)p1;
	pftp_server_t *ret = (pftp_server_t *)p2;
	
	(*ret) = NULL;
	
	if (CurFtp && ftp_name && strcmp(CurFtp, ftp_name) == 0) {
	    (*ret) = getCurFTP();
	} else {
	    (*ret) = pftp_getFTPCtrl(ftp_name);
	}
	
	return (*ret) ? 0 : -1;
    }; break;
    case PFTPUTIL_ERROR: {
	error("%s", (const char *)p1);
	return 0;
    }; break;
    case PFTP_STATUS: {
	const char *ftpname = (const char *)p1;
	const char *msg = (const char *)p2;

	if (output_status_message(ftp, ftpname, msg)) {
	    if (tab_comp && !tab_comp_redraw)
		puts(""); // Add a \n to leave readline line
	    
	    if (ftpname[0] == '\n')
		printf("%s: %s\n", ftpname + 1, msg);    
	    else
		printf("%s: %s\n", ftpname, msg);    
	    
	    tab_comp_redraw = 1;
	}
	
	return 0;
    }; break;
    case PFTP_NEXTSTATUS_IS_NOT_FATAL: {
	add_output_status_action(ftp, NEXTNONFATAL);
	return 0;
    }; break;    
    case PFTP_NEEDPASS: {
	char **username = (char **)p1;
	char **passwd = (char **)p2;
	int ret = 0;
	
	if (username) {
	    char *user = NULL;
	    
	    if (tab_comp) {
		size_t len = 0;
		ssize_t _ret;
		
		if (!tab_comp_redraw)
		    puts("");
		
		printf("Username: ");
		_ret = simple_getline(&user, &len, stdin, (*username), 1);
		if (_ret > 0 && user) {
		    puts("");
		    user[_ret-1] = '\0'; // Remove last '\n'        
		}
		tab_comp_redraw = 1;        
	    } else {
		if ((*username)) {
		    char *c;
		    c = *username;
		    while (*c) rl_stuff_char(*(c++));
		}
		user = readline("Username: ");    
	    }
	    
	    if (user) {
		if (strlen(user)) {
		    (*username) = realloc_strcpy((*username), user);
		    ret = 1;
		}
		free(user);
	    }
	}
	
	if ((ret == 1 || username == NULL) && passwd) {
	    if (tab_comp) {
		char *pass = NULL;
		size_t len = 0;
		ssize_t _ret;
		
		if (!tab_comp_redraw)
		    puts("");
		
		printf("Password: ");
		
		_ret = simple_getline(&pass, &len, stdin, NULL, 0);
		
		if (_ret > 0 && pass) {
		    pass[_ret-1] = '\0'; // Remove last '\n'
		    puts("");
		}
		
		tab_comp_redraw = 1;
		
		if (pass) {
		    if (strlen(pass)) {
			(*passwd) = realloc_strcpy((*passwd), pass);
			ret = 1;
		    }
		    
		    free(pass);
		}
	    } else {
#ifndef WIN32
		const char *pass = getpass("Password: ");
		if (pass) {
		    if (pass[0]) {
			(*passwd) = realloc_strcpy((*passwd), pass);
			ret = 1;
		    }
		}
#else
		char *pass = NULL;
		size_t pass_len = 0;
		ssize_t _ret;
		fputs("Password: ", stdout);
		_ret = simple_getline(&pass, &pass_len, stdin, NULL, 0);
		fputs("\n", stdout);
		if (_ret >= 0 && pass) {
		    if (pass[0]) {
			pass[_ret - 1] = '\0'; /* Remove last \n */
			(*passwd) = realloc_strcpy((*passwd), pass);
			ret = 1;
		    }            
		}
		if (pass)
		    free(pass);
#endif
	    }
	}
	
	return ret;
    }; break;
/*  case PFTP_NEEDPORT:   - Handled by libpftputil */
    case PFTP_INITTRANSFER: {
	/* 0 = timeout, 1 = download, 2 = upload */
	transfer_dir = (intptr_t) p1; 
	abort_transfer = 0;
	
	switch (transfer_dir) {
	case 0:
	    break;
	case 1:
	    memcpy(TERM_update, "<--", 3);
	    break;
	case 2:
	    memcpy(TERM_update, "-->", 3);
	    break;
	case 3:
	default:
	    memcpy(TERM_update, "***", 3);
	    break;
	}
	
	if (file_is_on && que_is_on) {
	    if (display_que_info) {
		if (tab_comp && !tab_comp_redraw)
		    puts("");
		
		printf("Que Progress [%lu of %lu]\n", 
		       (unsigned long)que_pos + 1, 
		       (unsigned long)que_size);
		
		tab_comp_redraw = 1;
	    }
	}
	
	return 0;
    }; break;
    case PFTP_FILETRANSFER: {
	file_start = (*((const uint64_t *)p1));
	file_size = (*((const uint64_t *)p2));
	
	file_is_on = 1;
	
	return 0;
    }; break;
    case PFTP_TRANSFER: {
	const double *speed = (const double *)p1;
	const float *percent = (const float *)p2;
	double fix_speed = 0.0, fix_trans = 0.0;
	
	if (!transfer_dir)
	    return abort_transfer ? 0 : -1;        
	
	if (speed) {
	    if ((*speed) < 1000.0) {
		fix_speed = (*speed);
		memcpy(speed_unit, "B", 2);
	    } else if ((*speed) < 1000.0 * 1000.0) {
		fix_speed = (*speed) / 1024.0;
		memcpy(speed_unit, "kB", 3);
	    } else if ((*speed) < 1000.0 * 1000.0 * 1000.0) {
		fix_speed = (*speed) / (1024.0 * 1024.0);
		memcpy(speed_unit, "MB", 3);    
	    } else {
		fix_speed = (*speed) / (1024.0 * 1024.0 * 1024.0);
		memcpy(speed_unit, "GB", 3);
	    }
	}
	
	memset(TERM_update + 3, ' ', TERM_width - 2);
	
	if (file_is_on && speed) {
	    if (percent) {
		uint64_t trans = (uint64_t)rintl(((float)file_size * (*percent)) / 100.0);
		trans -= file_start;
		
		if (trans < 1000) {
		    fix_trans = (double) trans;
		    memcpy(trans_unit, "B", 2);
		} else if (trans < 1000 * 1000) {
		    fix_trans = (double)trans / 1024.0;
		    memcpy(trans_unit, "kB", 3);
		} else if (trans < 1000 * 1000 * 1000) {
		    fix_trans = (double)trans / (1024.0 * 1024.0);
		    memcpy(trans_unit, "MB", 3);    
		} else {
		    fix_trans = (double)trans / (1024.0 * 1024.0 * 1024.0);
		    memcpy(trans_unit, "GB", 3);
		}        
		
		snprintf(TERM_update + 4, TERM_width - 4, 
			 "[Progress: %0.1f%%] [Speed: %0.1f%s/s] "
			 "[Transferred: %0.1f%s]", (*percent), fix_speed, 
			 speed_unit, fix_trans, trans_unit);
	    } else {
		snprintf(TERM_update + 4, TERM_width - 4, 
			 "[Speed: %0.1f%s/s]", fix_speed, speed_unit);
	    }
	} else {
	    if (speed) {
		snprintf(TERM_update + 4, TERM_width - 4, 
			 "[Speed: %0.1f%s/s] Transfer in progress...",
			 fix_speed, speed_unit);
	    } else {
		if (transfer_dir == 3)
		    strncpy(TERM_update + 4, "Waiting...", 
			    TERM_width - 4);
		else
		    strncpy(TERM_update + 4, "Transfer in progress...", 
			    TERM_width - 4);
	    }        
	}
	
	if (tab_comp && !tab_comp_redraw)
	    puts("");
	
#if WIN32
	{
	    COORD pos;
	    CONSOLE_SCREEN_BUFFER_INFO info;
	    if (GetConsoleScreenBufferInfo(console_out, &info)) {
		pos.X = 0;
		pos.Y = info.dwCursorPosition.Y - 1;
		SetConsoleCursorPosition(console_out, pos);
	    }
	}
#else
	fputs("\r", stdout);
#endif
	fwrite(TERM_update, 1, TERM_width + 1, stdout);
	tab_comp_redraw = 1;
	
	return abort_transfer ? 0 : -1;
    }; break;
    case PFTP_DONETRANSFER: {
//    int error = (int)p1;
	if (transfer_dir) {
	    if (tab_comp_redraw) {
		puts("");        
	    }
	}
	
	abort_transfer = 0;
	transfer_dir = 0;
	file_is_on = 0;
	
	return 0;
    }; break;
    default: {
	error("Unknown comm_central value.");
	return -1;
    }; break;
    }
}

void clear_output_status(void)
{
    ftp_output_status_t del;
    
    while (ftp_output_status) {
	del = ftp_output_status;
	ftp_output_status = del->next;
	free(del);
    }
}

void remove_server_output_status(const char *ftpname)
{
    ftp_output_status_t *cur;
    pftp_server_t ftp;
    
    ftp = pftp_getFTPCtrl(ftpname);
    
    if (!ftp)
	return;
    
    cur = &ftp_output_status;
    
    while (*cur) {
	if (ftp == (*cur)->server) {
	    ftp_output_status_t del;
	    
	    del = (*cur);
	    (*cur) = del->next;
	    
	    free(del);        
	} else {
	    cur = &(*cur)->next;
	}
    }
}

void add_output_status_action(pftp_server_t ftp, ftp_output_action_t action)
{
    ftp_output_status_t *cur;
    
    if (!ftp)
	return;
    
    cur = &ftp_output_status;
    
    while ((*cur))
	cur = &(*cur)->next;

    (*cur) = malloc(sizeof(struct ftp_output_status_s));
    memset((*cur), 0, sizeof(struct ftp_output_status_s));
    (*cur)->server = ftp;
    (*cur)->action = action;
    (*cur)->next = NULL;
}

int get_output_status_action(pftp_server_t ftp, ftp_output_action_t *action)
{
    ftp_output_status_t *cur;
    
    if (!ftp)
	return 0;
    
    cur = &ftp_output_status;
    
    while (*cur) {
	if ((*cur)->server == ftp) {
	    ftp_output_status_t del;
	    
	    del = (*cur);
	    (*cur) = del->next;
	    
	    *action = del->action;
	    
	    free(del);        
	    
	    return 1;
	} else {
	    cur = &(*cur)->next;
	}
    }   
    
    return 0;
}

void disableQueDisplay(void)
{
    display_que_info = 0;
}

void enableQueDisplay(void)
{
    display_que_info = 1;
}
