/*
 * palm-db-tools: PDB->CSV conversion tool
 * Copyright (C) 1998-2000 by Tom Dyas (tdyas@users.sourceforge.net)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <iostream>
#include <fstream>
#include <string>
#include <strstream>
#include <fstream>

#include <ctype.h>

#ifndef WIN32
# if TIME_WITH_SYS_TIME
#  include <sys/time.h>
#  include <time.h>
# else
#  if HAVE_SYS_TIME_H
#   include <sys/time.h>
#  else
#   include <time.h>
#  endif
# endif
#else
# include <time.h>
#endif

#include "pdbtools.h"

using namespace std;
using namespace PalmLib::FlatFile;

static std::string program;
static std::ostream* err = &std::cerr;

static void usage(void)
{
    std::cout << "Usage: " << program << " [-v | -h | -i FILE | [others options]] PDB_FILE [CSV_FILE] [INFO_FILE]\n";
	std::cout << "  Database description options:\n";
    std::cout << "  -i FILE, --info=FILE         Read database metadata commands from FILE\n";
    std::cout << "  -e, --extended               Use extended CSV mode.\n";
    std::cout << "  -s SEP, --separator=SEP      Change the default field separator.\n";
	std::cout << "                               (Default: \",\")\n";
    std::cout << "  -n FILE, --errors=FILE       Send all error messages to FILE.\n";
    std::cout << "  -d FORMAT, --date=FORMAT     Change the output format for date fields.\n";
    std::cout << "  -t FORMAT, --time=FORMAT     Change the output format for time fields.\n";
    std::cout << "  -D FORMAT, --datetime=FORMAT Change the output format for datetime fields.\n";
    std::cout << "Note: Info option can not be present with the others of this options\n";
	std::cout << "  Others options:\n";
    std::cout << "  -h, --help                   Display this help screen.\n";
    std::cout << "  -v, --version                Display the program version.\n";
    std::cout << "Note: The data are written on the standard output,\n";
	std::cout << "      if the CSV file name is not present\n";
    std::cout << "      and if info option doesn't give a file with the \"csvfile\" element.\n";
}

int main(int argc, char *argv[])
{
    static const CLP::option_t options[] = {
        { "info", 'i', "info", CLP::ARGUMENT_STRING, CLP::ACTION_MAP },
        { "extended", 'e', "extended", CLP::ARGUMENT_NONE, CLP::ACTION_MAP },
        { "sep", 's', "separator", CLP::ARGUMENT_STRING, CLP::ACTION_MAP },
        { "help", 'h', "help", CLP::ARGUMENT_NONE, CLP::ACTION_MAP },
        { "version", 'v', "version", CLP::ARGUMENT_NONE, CLP::ACTION_MAP },
        { "errors", 'n', "errors", CLP::ARGUMENT_STRING, CLP::ACTION_MAP },
        { "date", 'd', "date", CLP::ARGUMENT_STRING, CLP::ACTION_MAP },
        { "time", 't', "time", CLP::ARGUMENT_STRING, CLP::ACTION_MAP },
        { "datetime", 'D', "datetime", CLP::ARGUMENT_STRING, CLP::ACTION_MAP },
        { 0 },
    };
    CLP::args_t args;
    CLP::option_map_t opts;
    DataFile::CSVConfig state;

    program = argv[0];

    try {
        CLP::parse(argc - 1, argv + 1, options, opts, args);
    } catch (CLP::missing_value_error e) {
        *err << argv[0] << ": option `" << e.option_name()
             << "' takes an argument\n";
        usage();
        return 1;
    } catch (CLP::invalid_option_error e) {
        *err << argv[0] << ": unknown option `" << e.option_name()
             << "' was passed\n";
        usage();
        return 1;
    } catch (CLP::value_present_error e) {
        *err << argv[0] << ": option `" << e.option_name()
             << "' does not take an argument\n";
        usage();
        return 1;
    }

    // Display the usage screen if requested or no arguments present.
    if (argc == 1 || opts.find("help") != opts.end()) {
        usage();
        return 0;
    }

    // Display the program version if requested.
    if (opts.find("version") != opts.end()) {
        cout << "pdb2csv (" << PACKAGE_NAME << ' ' << PACKAGE_VERSION << ')' << endl;
        return 0;
    }

    // Redirect errors to a file. (Added to support a GUI wraparound on Win32.)
    if (opts.find("errors") != opts.end()) {
        std::ofstream* f = new std::ofstream(opts["errors"].c_str());
        if (!f) {
            *err << program << ": unable to open error file '"
                 << opts["errors"] << "'\n";
        } else {
            err = f;
        }
    }

    std::string l_PDBPath;
    if (opts.find("info") != opts.end())
    {
        if (opts.find("type") != opts.end() ||
            opts.find("extended") != opts.end() ||
            opts.find("title") != opts.end() ||
            opts.find("backup") != opts.end() ||
            opts.find("option") != opts.end() ||
            opts.find("field") != opts.end() ||
            opts.find("sep") != opts.end()) {
            *err << "-i option not supported with at least one of the others options.\n";
            return 1;
        }
        DataFile::InfoFile l_IfoFile(opts["info"]);
        l_IfoFile.read(state);
        l_PDBPath = l_IfoFile.readPDBPath();
    } else {

	    // Check to see if extended CSV mode should be enabled.
	    if (opts.find("extended") != opts.end())
	        state.extended_csv_mode = true;

	    // Change the field seperator if requested.
	    if (opts.find("sep") != opts.end())
	        state.field_sep = opts["sep"];

	    // Check to see if any of the date/time formats were changed.
	    if (opts.find("date") != opts.end())
	        state.format_date = opts["date"];
	    if (opts.find("time") != opts.end())
	        state.format_time = opts["time"];
	    if (opts.find("datetime") != opts.end())
	        state.format_datetime = opts["datetime"];
	}

    //Initialize the library
    PDBTools::setConfigLib(err, l_PDBPath, state);

    // Ensure that all 3 arguements were specified.
    std::string pdb_fname;
    std::string csv_fname;
    std::string info_fname;
    switch (args.size()) {
    case 3:
        // Assign better names to the arguements.
        pdb_fname = std::string(args[0]);
        state.csv_fname = std::string(args[1]);
        info_fname = std::string(args[2]);
    break;
    case 2:
        // Assign better names to the arguements.
        pdb_fname = std::string(args[0]);
        info_fname = std::string(args[1]);
    break;
    case 1:
        pdb_fname = std::string(args[0]);
    break;
    default:
        *err << program << ": 1, 2 or 3 arguments were expected\n";
        usage();
        return 1;
    }

    // Read the PDB file into memory.
    PalmLib::File pdb(pdb_fname);
    try {
        pdb.load();
    } catch (const PalmLib::error& e) {
        *err << pdb_fname << ": " << e.what() << endl;
        return 1;
    }

    // Instantiate an object based on parameters.
    PalmLib::FlatFile::Factory factory;
    PalmLib::FlatFile::Database* db = 0;
    try {
        db = factory.makeDatabase(pdb);
        if (!db)
            throw PalmLib::error("unable to determine database type");
    } catch (const PalmLib::error& e) {
        *err << pdb_fname << ": " << e.what() << endl;
        return 1;
    }

    // Output the metadata file.
    try {
	    if (!info_fname.empty())
	    {
	        DataFile::InfoFile l_IfoFile(info_fname);        
	        l_IfoFile.write( *db, state);
	    }
    } catch (const PalmLib::error& e) {
        *err << info_fname << ": " << e.what() << endl;
        return 1;
    }		

    if (!state.csv_fname.empty() && state.csv_fname != std::string("stdIO")) {
        try {
            DataFile::CSVFile csvfile(state.csv_fname);
            // Output all of the records.
            csvfile.write(*db, state);
        } catch (const PalmLib::error& e) {
            *err << state.csv_fname << ": " << e.what() << endl;
            return 1;
        }		
    } else {
        try {
            DataFile::CSVFile csvfile("stdIO");
            // Output all of the records.
            csvfile.write(*db, state);
        } catch (const PalmLib::error& e) {
            *err << "std Output: " << e.what() << endl;
            return 1;
        }		
    }

    return 0;
}
