/*
 * Generic GPIO led
 *
 * Copyright (C) 2019 - 2020 Andy Green <andy@warmcat.com>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */
#include "private-lib-core.h"

static const lws_led_intensity_t sineq16[] = {

	/*
	 * Quadrant at sin(270) in 16 samples, normalized so
	 * -1 == 0 and 0 == 32767
	 */

	    0,   158,   630,  1411,  2494,  3869,  5522,  7437,
	 9597, 11980, 14562, 17321, 20228, 23225, 26374, 29555,
	32767 /* to interpolate against */
};

/*
 * Elaborate the 90 degree phase table to 360 degrees and offset to +32768,
 * notice for the last sample we have to interpolate against a 17th sample
 * reflecting full scale to avoid clipping due to interpolation against the
 * 16th sample again
 */

static lws_led_intensity_t
sine_lu(int n, int next)
{
        switch ((n >> 4) & 3) {
        case 1:
        	/* forwards */
                return 32768 + sineq16[(n & 15) + next];
        case 2:
        	/* scan it backwards */
                return 32768 + sineq16[15 - (n & 15) + (!next)];
        case 3:
        	/* forwards */
                return 32768 - sineq16[(n & 15) + next];
        default:
        	/* scan it backwards */
                return 32768 - sineq16[15 - (n & 15) + (!next)];
        }
}

/*
 * The normalized phase resolution is 16-bit, however much table you decide to
 * have needs interpolating or indexing in a reduced number of significant
 * phase bits if it doesn't have the same phase resolution.
 *
 * In this sine table we have a 16 x 15-bit sample quadrant reflected 4 times
 * to make 360 degrees, so 64 accurate sample points, with the rest of the
 * intermediate phases generated by linear interpolation.  That probably would
 * sound a bit funky, but for modulating light dynamically it's more than
 * enough.
 */

lws_led_intensity_t
lws_led_func_sine(lws_led_seq_phase_t n)
{
        /*
         * 2: quadrant
         * 4: table entry in quadrant
         * 10: interp (LSB)
         */

        return (sine_lu(n >> 10, 0) * (0x3ff - (n & 0x3ff)) +
        	sine_lu(n >> 10, 1) * (n & 0x3ff)) / 0x3ff;
}

lws_led_intensity_t
lws_led_func_linear(lws_led_seq_phase_t n)
{
	return (lws_led_intensity_t)n;
}


static lws_led_intensity_t
lws_led_func_static(lws_led_seq_phase_t n)
{
	return ((int)n * LWS_LED_MAX_INTENSITY) / 2;
}

const lws_led_sequence_def_t lws_pwmseq_static_off = {
	.func			= lws_led_func_static,
	.ledphase_offset	= 0,
	.ledphase_total		= 0,
	.ms			= 0
};

const lws_led_sequence_def_t lws_pwmseq_static_half = {
	.func			= lws_led_func_static,
	.ledphase_offset	= 1,
	.ledphase_total		= 0,
	.ms			= 0
};

const lws_led_sequence_def_t lws_pwmseq_static_on = {
	.func			= lws_led_func_static,
	.ledphase_offset	= 2,
	.ledphase_total		= 0,
	.ms			= 0
};

const lws_led_sequence_def_t lws_pwmseq_sine_up = {
	.func			= lws_led_func_sine,
	.ledphase_offset	= 0, /* already at 0 amp at 0 phase */
	.ledphase_total		= LWS_LED_FUNC_PHASE / 2, /* 180 degree ./^ */
	.ms			= 300
};

const lws_led_sequence_def_t lws_pwmseq_sine_down = {
	.func			= lws_led_func_sine,
	.ledphase_offset	= LWS_LED_FUNC_PHASE / 2, /* start at peak */
	.ledphase_total		= LWS_LED_FUNC_PHASE / 2, /* 180 degree ./^ */
	.ms			= 300
};

const lws_led_sequence_def_t lws_pwmseq_linear_wipe = {
	.func			= lws_led_func_linear,
	.ledphase_offset	= 0,
	.ledphase_total		= LWS_LED_FUNC_PHASE - 1,
	.ms			= 300
};

const lws_led_sequence_def_t lws_pwmseq_sine_endless_slow = {
	.func			= lws_led_func_sine,
	.ledphase_offset	= 0, /* already at 0 amp at 0 phase */
	.ledphase_total		= LWS_SEQ_LEDPHASE_TOTAL_ENDLESS,
	.ms			= 1500
};

const lws_led_sequence_def_t lws_pwmseq_sine_endless_fast = {
	.func			= lws_led_func_sine,
	.ledphase_offset	= 0, /* already at 0 amp at 0 phase */
	.ledphase_total		= LWS_SEQ_LEDPHASE_TOTAL_ENDLESS,
	.ms			= 750
};
