/*
 * This file is part of
 *
 * LIBPNET6: a Portable Network Library
 *
 * LIBPNET6 is Copyright (c) 2002, Peter Bozarov
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Peter Bozarov.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * $Id: config.c,v 1.50 2002/12/22 14:13:32 kingofgib Exp $
 */

/*----------------------------------------------------------------------*/
/* This is a quick and dirty configurator. Currently, it checks for	*/
/* files, structs and members in structs, and functions.		*/
/* The three structures, ``files[]'', ``STRUCT_MEM'' ``FUNC'' define 	*/
/* the things you're interested in.					*/ 
/*----------------------------------------------------------------------*/

# include <stdio.h>
# include <string.h>
# ifndef _WIN32
#   include <unistd.h>
#   include <stdlib.h>
#   include <ctype.h>
#   include <sys/types.h>
#   include <sys/stat.h>

# else

#   include <windows.h>

# endif


# define SPACE(p)	( p == ' ' || p == '\t' || p == '\n' || p == '\r' )

typedef struct STRUCT_MEM	STRUCT_MEM;
typedef struct STRUCT		STRUCT;
typedef struct FUNC 		FUNC;

# ifndef _WIN32					/* {{ */

static char * output	= "config.h";
static FILE * fpOut	= NULL;
static const char * CC 	= "/usr/bin/cc";
static const char * dname	= "./_dummy.c";
static const char * oname = 	"./_dummy.o";
static FILE * fDummy	= NULL;


# if 0 
# define UNLINK( f )	 \
	do { \
		printf( "unlinking %s\n", f ); \
		if ( unlink( f ) ) \
		{ \
			printf("unlink(%s) failed on %d\n", f, __LINE__); \
			perror("unlink()"); \
		} \
	} while ( 0 )
# else
# define UNLINK( f )	unlink( f )
# endif

const char * files[] = 
{
    "unistd.h",
    "sys/types.h",
    "time.h",
    "syslog.h",
    "sys/time.h",
    "sys/socket.h",
    "netinet/in.h",
    "arpa/inet.h",
    "netdb.h",
    "net/if_arp.h",
    "net/ethernet.h",
    "sys/ethernet.h",
    "netinet/if_ether.h",
    "sys/if_ether.h",
    "net/bpf.h",
    "sys/utsname.h",
    "sys/uio.h",
    "sys/wait.h",
    "sys/sysctl.h",
    "net/if_dl.h",
    "wait.h",
    "sys/un.h",
    "signal.h",
    "fcntl.h",
    "resolv.h",
    "netinet/icmp6.h",
    "pthread.h",
    "sys/ioctl.h",
    "net/if_types.h",
    "netpacket/packet.h",
    "net/ethernet.h",
    "linux/if_packet.h",
    "linux/if_ether.h",
    "linux/filter.h",
    "net/pfilt.h",
    "sys/dlpi.h",
    NULL,
};

struct STRUCT
{
    int 	fatal;
    char *	name;
    char *	includes[5];
} structList[] = 
{
    { 1, "sockaddr_in", { "sys/types.h", "netinet/in.h", NULL } },
    { 0, "sockaddr_in6", { "sys/types.h", "netinet/in.h", NULL } },
    { 0, "sockaddr_un", { "sys/types.h", "sys/un.h", NULL } },
    { 0, "in6_pktinfo", { "sys/types.h", "netinet/in.h", NULL } },
    { 0, "lifconf", { "sys/types.h", "sys/socket.h", "net/if.h" , NULL } }, 
    { 1, "ifconf", { "sys/types.h", "sys/socket.h", "net/if.h" , NULL } }, 
    { 0, "ipv6_mreq", { "sys/types.h", "netinet/in.h", NULL } }, 
    { 0, "tpacket_stats", { "linux/if_packet.h", NULL } } ,
    { 0, "msghdr", { "sys/types.h", "sys/socket.h", NULL } },
    { 0, "ifdevea", { "sys/types.h", "net/if.h", NULL } },
    { 0, "sockaddr_dl", { "sys/types.h", "net/if_dl.h", NULL } }, 
    { 0, "bpf_program", { "sys/types.h", "sys/time.h","sys/ioctl.h","net/bpf.h", NULL } }, 
    { 0, "bpf_prgm", { "sys/types.h", "sys/time.h","sys/ioctl.h","net/bpf.h", NULL } }, 

    { 0, NULL, { NULL } }
};

struct STRUCT objList[] = 
{
    { 0, "in6addr_any", { "sys/types.h", "netinet/in.h", NULL } }, 

    { 0, NULL, { NULL } }
};

struct STRUCT_MEM
{
    int		fatal;
    char * 	name;
    char *	mem;
    char *	includes[4];
} structMemList[] =
{
    { 0, "msghdr", "msg_control",
	{ "sys/types.h", "sys/socket.h", NULL	} },
    { 0, "msghdr", "msg_flags",
	{ "sys/types.h", "sys/socket.h", NULL	} },
    { 0, "sockaddr"    , "sa_len",
	{ "sys/types.h", "sys/socket.h", NULL	} },
    { 0, "sockaddr_in6", "sin6_scope_id",
	{ "sys/types.h", "netinet/in.h", NULL 			} },
    { 0, "sockaddr_in6", "sin6_flowinfo",
	{ "sys/types.h", "netinet/in.h", NULL 			} },
    { 0, "ifreq", "ifr_hwaddr",
	{ "sys/types.h", "sys/socket.h","net/if.h", NULL } }, 
    { 0, "ifreq", "ifr_ifindex",
	{ "sys/types.h", "sys/socket.h","net/if.h", NULL } }, 
    { 0, "ifreq", "ifr_index",
	{ "sys/types.h", "sys/socket.h","net/if.h", NULL } }, 
    { 0, "ifreq", "ifr_netmask",
	{ "sys/types.h", "sys/socket.h","net/if.h", NULL } }, 
    { 0, "ifreq", "ifr_metric",
	{ "sys/types.h", "sys/socket.h","net/if.h", NULL } }, 
    { 0, "ifreq", "ifr_phys",
	{ "sys/types.h", "sys/socket.h","net/if.h", NULL } }, 
    { 0, "ifreq", "ifr_media",
	{ "sys/types.h", "sys/socket.h","net/if.h", NULL } }, 
    { 0, "ifreq", "ifr_mtu",
	{ "sys/types.h","sys/socket.h", "net/if.h", NULL } }, 

    { 0, NULL, NULL, { NULL } }
};

struct FUNC
{
    int 	fatal;
    char * 	name;
    int		num_args;
    char *	includes[10];
} funcList[] =
{
    { 0, "sysctl", 6,
    	{ "sys/types.h","sys/socket.h",
	  "net/if_types.h", "sys/param.h","net/route.h",
	  "net/if_dl.h", "sys/sysctl.h", NULL } },

    { 0, "fork", 0, { "sys/types.h", "unistd.h",  NULL } }, 

    { 1, "uname", 1, { "sys/utsname.h", NULL } },

    { 0, "inet_ntop", 4, 
    	{ "sys/types.h", "sys/socket.h", "arpa/inet.h", NULL } },
    { 0, "inet_pton", 3, 
    	{ "sys/types.h", "sys/socket.h", "arpa/inet.h", NULL } },

    { 1, "ioctl", 3, { "sys/ioctl.h", NULL } }, 

    { 0, "setuid", 1, { "sys/types.h", "unistd.h" , NULL } } ,

    { 0, "gettimeofday", 2, { "sys/time.h", NULL } },

    { 0, "getaddrinfo", 4, { "sys/types.h", "sys/socket.h", "netdb.h", NULL } },

    { 0, "getnameinfo", 7, { "sys/types.h", "sys/socket.h", "netdb.h", NULL } },

    { 0, "gethostbyname2", 2, { "netdb.h", NULL } },

    { 1, "getenv", 1, { "stdlib.h", NULL } }, 

    { 0, "sendmsg", 3, { "sys/types.h", "sys/socket.h" , NULL } }, 

    { 0, "recvmsg", 3, { "sys/types.h", "sys/socket.h" , NULL } }, 

    { 0, "writev", 3, { "sys/types.h", "sys/uio.h", NULL } },

    { 0, "readv", 3, { "sys/types.h", "sys/uio.h", NULL } },

    { 0, "hstrerror", 1, { "sys/types.h", "netdb.h" , NULL } }, 

    { 0, "getkerninfo", 4, { "sys/ndd_var.h", "sys/kinfo.h", NULL } },

    { 0, "ether_hostton", 2, { "sys/types.h", "sys/socket.h", "net/ethernet.h", NULL } },

    { 0, "ffs", 1, { "strings.h", NULL } },

    { 0, "ffs", 1, { "string.h", NULL } },

    { 0, "vsyslog", 3, { "stdarg.h", "syslog.h", NULL } },

    { 0, "clock_gettime", 2, { "sys/time.h" , NULL }  },

    { 0, NULL, 0, { NULL } }
};


static void 
check_hdr( const char * s )
{
    printf( "%s ... ", s );
    fflush( stdout );
}

static void
define( const char * prefix, const char * s, const char * comment )
{
    fprintf( fpOut, "# define %s%s\t\t1 /* %s */\n", prefix, s, comment );
    return;
}

# define FLAG_W		1
# define FLAG_R		2
# define FLAG_X		4
# define FLAG_D		8

static int
file_exists( const char * fname, int flags )
{
    struct stat s;

    if ( stat( fname, &s ) )	/* No such file */
	return 0;

    if ( (flags & FLAG_D) && ! (s.st_mode & S_IFDIR) )
	return 0;		/* Asked for dir and it wasn't */

    if ( (flags & FLAG_W) && ! (s.st_mode & S_IWUSR ) )
	return 0;		/* Asked for user write and it wasn't */

    if ( (flags & FLAG_R) && ! (s.st_mode & S_IRUSR ) )
	return 0;		/* Asked for user read and it wasn't */

    if ( (flags & FLAG_X) && ! (s.st_mode & S_IXUSR ) )
	return 0;		/* Asked for user exec and it wasn't */

    return 1;
}
static int
find_cc( void )
{
    static const char *	path[] =
	{  "/usr/bin/cc", "/bin/cc", "/usr/local/bin/cc", 
	   "/usr/ucb/cc", "/usr/bin/gcc", NULL };
    const char **	pp;

    check_hdr("Checking for the C compiler ");
    for ( pp = path; *pp; pp++ )
    {
	if ( file_exists( *pp, FLAG_R | FLAG_X ) )
	{
	    printf( "%s\n", *pp );
	    CC = *pp;
	    return 0;
	}
    }
    printf( "\n*** Cannot find full path to C compiler on this system.\n" );
    printf( "*** Using default, cc\n" );
    CC = "cc";
    return 0;
}
static const char *
make_fname( const char * dir, const char * file )
{
    static char buf[1024];
    int 	n = 0;

    if ( dir )
    {
	n = strlen( strcpy( buf, dir ) ); 
	buf[ n++ ] = '/';
    }

    strcpy( buf + n, file );
    return buf;
}

static char *
convert_name( const char * inname )
{
    static char buf[1024];
    char* p;
    char* s;

    strcpy( buf, inname ); 

    s = p = buf;

    while ( *p )
    {
	if ( *p == '/' || *p == '.' || *p == '\\' )
	    *p = '_';
	else
	    *p = toupper( *p );
	p++;
    }
    return s;
}

# define CFF_PRESENT		0x1
# define CFF_READ		0x2
# define CFF_WRITE		0x3

static int
check_file( const char * dir, const char * fname )
{
    FILE *fp;
    const char * tmp;

    tmp = make_fname( dir , fname );
    printf("Checking for %s ... ", tmp );
    fflush( stdout );
    if (!( fp = fopen( tmp, "r" ) ) )
    {
	printf(" NO\n" );
	return -1;
    }
    printf(" yes\n");

# if 0
    fprintf(fpOut, "\n/* include %s */\n", fname );
    fprintf(fpOut, "# include <%s>\n", fname );
# endif

    define( "HAVE_", convert_name( fname ), fname );
    fprintf(fpOut, "\n" );

    return 0;
}

int
open_file( void )
{
    UNLINK( dname );

    fDummy = fopen( dname, "w" );

    if ( ! fDummy )
	return -1;

    return 0;
}
int
close_file( void )
{
    if ( fDummy )
	{ fflush( fDummy); fclose( fDummy ); fDummy = NULL; }
    return 0;
}
int
write_file( const char * text, int newline )
{
    int 	len = strlen( text );
    int		got;

    if ( !fDummy && open_file() )
	{ perror( "write_file()" ); return -1; }

    got = fwrite( text, 1, len, fDummy );

    if ( got != len )
	perror( "write_file()" );

    if ( newline )
	fputc( '\n', fDummy );

    return 0;
}
int
compilefile( void )
{
    FILE * fp;
    char buf[256];
    int ret = 0;

    UNLINK( "./a.out" );

    sprintf( buf, "%s %s -o ./a.out > /dev/null 2>&1", CC, dname ); 
    ret = system( buf );
    if ( ret != 0 )
	{ UNLINK( "./a.out" ); return -1; }

    if ( ( fp = fopen( "./a.out", "r" ) ) )
    {
        fclose( fp );
	UNLINK( "./a.out" );
	return 0;
    }
    UNLINK( "./a.out" );
    return -1;
}
static int
fatal_error( void )
{
    fprintf(stderr,"####################################################\n");
    fprintf(stderr,"             This is a fatal error\n");
    fprintf(stderr,"  Please report this to libpnet6@bozz.demon.nl.\n");
    fprintf(stderr,"Specify your system, and please include the _dummy.c\n");
    fprintf(stderr,"       file that's been left by 'config'.\n");
    fprintf(stderr,"            Thanks for your trouble!\n");
    fprintf(stderr,"####################################################\n");
    exit( 1 );
    return 1;
}
int
check_struct( STRUCT * s, int obj, int no_print )
{
    int r = 0;
    char **inc;

    if ( ! no_print )
    {
	printf( "Checking whether '%s %s' exists ...",
		    obj ? "object" : "struct", s->name );
	fflush( stdout );
    }

    inc = s->includes;

    open_file();

    fprintf( fDummy, "#if defined (hpux)||defined (__osf__)||defined (sun)\n" );
    fprintf( fDummy, "# define _XOPEN_SOURCE		1\n" );
    fprintf( fDummy, "# define _XOPEN_SOURCE_EXTENDED	1\n" );
    fprintf( fDummy, "# define __EXTENSIONS__\n" );
    fprintf( fDummy, "# endif\n" );
    fprintf( fDummy, "# if defined (__osf__)\n" );
    fprintf( fDummy, "# define _SOCKADDR_LEN 		1\n" );
    fprintf( fDummy, "# define _OSF_SOURCE\n" );
    fprintf( fDummy, "# endif\n" );
    

    while ( *inc )
	fprintf(fDummy,"#include <%s>\n", *inc ), inc++;

    if ( obj )
	fprintf(fDummy, "int main() { return sizeof( %s ) != 0; }\n", s->name );
    else
	fprintf( fDummy, "int main(){struct %s s; return sizeof(s)!= 0;}\n",
		s->name );

    close_file();

    if ( compilefile() )
    {
	if ( no_print )
	    { r = 0; goto end; }

	printf( " NO\n" );

	if ( s->fatal )
	    fatal_error();
    }
    else
    {
	if ( no_print )
	    { r = 1; goto end; }

	printf(" yes\n");
	fprintf( fpOut, "# define HAVE_STRUCT_%s\t\t1\n",
			convert_name( s->name ) );
    }
end:
    UNLINK( dname );
    UNLINK( oname );

    return r;
}
int
check_struct_member( STRUCT_MEM * s )
{
    char **inc;

    printf( "Checking whether 'struct %s' has member '%s' ...",
	    s->name, s->mem );
    fflush( stdout );

    inc = s->includes;

    open_file();

    fprintf( fDummy, "# if defined (hpux)||defined (__osf__)||defined (sun)\n" );
    fprintf( fDummy, "# define _XOPEN_SOURCE		1\n" );
    fprintf( fDummy, "# define _XOPEN_SOURCE_EXTENDED	1\n" );
    fprintf( fDummy, "# define __EXTENSIONS__\n" );
    fprintf( fDummy, "# endif\n" );
    fprintf( fDummy, "# if defined (__osf__)\n" );
    fprintf( fDummy, "# define _SOCKADDR_LEN 		1\n" );
    fprintf( fDummy, "# define _OSF_SOURCE\n" );
    fprintf( fDummy, "# endif\n" );

    while ( *inc )
	fprintf(fDummy,"#include <%s>\n", *inc ), inc++;
    fprintf(fDummy, "int main() { struct %s s; return sizeof( s.%s ) > 0; }\n",
	s->name, s->mem);

    close_file();

    if ( compilefile() )
    {
	printf( " NO\n" );
	if ( s->fatal )
	    fatal_error();
    }
    else
    {
	printf(" yes\n");
	fprintf( fpOut, "# define HAVE_%s", convert_name( s->name ) );
	fprintf( fpOut, "_%s\t\t1\n", convert_name( s->mem ) ); 
    }

    UNLINK( dname );

    return 0;
}
int 
check_func( FUNC * func )
{
    char **inc;

    printf( "Checking whether usable '%s' w/ %d args exists ...",
	func->name, func->num_args );
    fflush( stdout );

    inc = func->includes;

    open_file();

    fprintf( fDummy, "# if defined (hpux)||defined (__osf__)||defined (sun)\n" );
    fprintf( fDummy, "# define _XOPEN_SOURCE		1\n" );
    fprintf( fDummy, "# define _XOPEN_SOURCE_EXTENDED	1\n" );
    fprintf( fDummy, "# define __EXTENSIONS__\n" );
    fprintf( fDummy, "# endif\n" );
    fprintf( fDummy, "# if defined (__osf__)\n" );
    fprintf( fDummy, "# define _SOCKADDR_LEN 		1\n" );
    fprintf( fDummy, "# define _OSF_SOURCE\n" );
    fprintf( fDummy, "# endif\n" );

    while ( *inc )
	fprintf(fDummy,"#include <%s>\n", *inc ), inc++;

    fprintf(fDummy, "int main() { %s( ", func->name );

    if( func->num_args > 0 )
    {
	int i;
	for (i = 0; i < func->num_args - 1; i++ )
	    fprintf(fDummy , "0x0,");
	fprintf(fDummy, "0x0");
    }
    fprintf( fDummy ,");\n");

    fprintf( fDummy ,"return 0; }\n");
    close_file();

    if ( compilefile() )
    {
	printf( " NO\n" );
	if ( func->fatal )
	    fatal_error();
	fprintf( fpOut, "# define DONT_HAVE_%s\t\t1\n",
		convert_name( func->name ) );
    }
    else
    {
	printf(" YES\n");
	fprintf( fpOut, "# define HAVE_%s\t\t1\n", convert_name( func->name ) );
    }

    UNLINK( dname );

    return 0;
}

static void
check_endian( void )
{
    int i = 1;
    int lilEndian;

    lilEndian = *(char *)&i;
    fprintf( stdout, "Machine is %s endian\n", lilEndian ? "little" : "big");
    if ( lilEndian )
	fprintf( fpOut, "# define PNET_LITTLE_ENDIAN\t\t1\n");
    else
	fprintf( fpOut, "# define PNET_BIG_ENDIAN\t\t1\n");
}

static void
check_ip( void )
{

}
static void
sys_defs(void)
{
# if defined linux
    define( "","PNET_USE_IFCONF", "For interface access" );
# endif
}
static int get_system_name( FILE * );
static int
print_banner( void )
{
    fprintf(fpOut, "/*-----------------------------------------"
		"-----------------------------*/\n");
    fprintf( fpOut, "/* Configuration file for libpnet6.\t\t\t\t\t*/\n");
    get_system_name( fpOut );
    
    fprintf(fpOut, "/*-----------------------------------------"
		"-----------------------------*/\n");
    fprintf(fpOut, "\n\n# define HAVE_CONFIG_H\t\t\t1\n\n");
    return 0;
}

# endif						/* }} */

# ifndef _WIN32
# include <sys/utsname.h>
# endif
static int
get_system_name( FILE * fp )
{
# ifndef _WIN32
    struct utsname      uts;

    if (uname(&uts) < 0)
        return 0;

    fprintf( fp, "/* hostid: %s, %s-%s-%s */\n",
		uts.nodename, uts.sysname, uts.release, uts.machine );

    fprintf( fp, "\n# define PNET_COMPILE_HOST\t\t\"%s\"\n\n", uts.nodename );
# else
    DWORD               len;
    char                buf[MAX_COMPUTERNAME_LENGTH + 1];

    len = MAX_COMPUTERNAME_LENGTH + 1;

    if (! GetComputerName( buf, &len ))
	sprintf( buf,"Windoos" );
    fprintf( fp, "\n# define PNET_COMPILE_HOST\t\t\"%s\"\n\n", buf );
# endif 
    return 0;
}

# define pvar( type, g, name )	\
fprintf( fp, "typedef %s %s; /* % 2d bits */\n", (type), (name), sizeof(g) << 3 )
static int
check_ip_support( FILE * fp )
{
# ifndef _WIN32
    struct STRUCT s[] =
      {
	{ 1, "sockaddr_in", { "sys/types.h", "netinet/in.h", NULL } },
	{ 0, "sockaddr_in6", { "sys/types.h", "netinet/in.h", NULL } },
	{ 0, "sockaddr_un", { "sys/types.h", "sys/un.h", NULL } },
      };

    FILE * filep;

    printf("/* Internet support found follows */\n");

    if ( check_struct( &s[0], 0, 1 ) )
	fprintf( fp, "# define PNET_HAVE_IPv4 1\n");
    if ( check_struct( &s[1], 0, 1 ) )
	fprintf( fp, "# define PNET_HAVE_IPv6 1\n");
    if ( check_struct( &s[2], 0, 1 ) )
	fprintf( fp, "# define PNET_HAVE_LOCAL 1\n");

if ( filep = fopen ( "/usr/include/pthread.h", "r" ) )
{
	fprintf( fp, "# define PNET_HAVE_PTHREAD_H  1\n" );
	fclose( filep );
}
# endif
    return 0;
}
static int
varsizes( FILE * fp )
{
    get_system_name( stdout );
    check_ip_support( fp );
    pvar( "char", char, "pnet_char" );
    pvar( "unsigned char", unsigned char, "pnet_uchar,pnet_byte" );

    if ( sizeof( int ) == 2 )
    {
	pvar( "int", int, "pnet_short" );
	pvar( "unsigned int", unsigned int, "pnet_ushort" );
    }
    else if ( sizeof( short ) == 2 )
    {
	pvar( "short" , short, "pnet_short" );
	pvar( "unsigned short", unsigned short, "pnet_ushort" );
    }
    else
	{ fprintf( stderr, "*** No 16-bit data type found\n" ); return ( 1 ); }

    if ( sizeof( int ) == 4 )
    {
	pvar( "int", int, "pnet_int" );
	pvar( "unsigned int", unsigned int, "pnet_uint" );
    }
    else if ( sizeof( long ) == 4 )
    {
	pvar( "long", long, "pnet_int" );
	pvar( "unsigned long" , unsigned long, "pnet_uint" );
    }
    else
	{ fprintf( stderr, "*** No 32-bit data type found\n" ); return ( 1 ); }

    pvar( "long", long, "pnet_long" );
    pvar( "unsigned long", unsigned long, "pnet_ulong" );

    if ( sizeof( long ) == 8 )
    {
	fprintf(  stdout, "# define PNET_LONGLONG	1\n");
	pvar( "long", long, "pnet_longlong" );
	pvar( "unsigned long", unsigned long, "pnet_ulonglong" );
    }
    else
    {
	fprintf( stderr, "+++ No 64-bit data type found\n" );
	pvar( "long", long, "pnet_longlong" );
	pvar( "unsigned long", unsigned long, "pnet_ulonglong" );
    }

    return 0;
}
int
main( int argc, char ** argv )
{
    const char ** file;
    STRUCT_MEM* pSM;
    STRUCT * pS;
    FUNC *  pF;

    if ( argc == 2 && ! strcmp( argv[1] ,"-vs" ) )
	return varsizes( stdout );

# ifndef _WIN32		/* { */

    if ( ! file_exists( "/usr/include", FLAG_R | FLAG_D | FLAG_X ) )
    {
	printf("Where's your /usr/include?\n");
	return 1;
    }

    if ( find_cc() )
	return 1;

    if ( ! (fpOut = fopen( output, "w" )) )
	{ printf("Can't open %s\n", output ); exit( 1 ); }

    print_banner();
    check_endian();
    check_ip();
    sys_defs();
    
    for ( file = files; *file; file++ )
	check_file( "/usr/include", *file );
    for ( pS = structList; pS->name; pS++ )
	check_struct( pS, 0, 0 );
    for ( pS = objList; pS->name; pS++ )
	check_struct( pS, 1, 0 );
    for ( pSM= structMemList; pSM->name; pSM++ )
	check_struct_member( pSM );
    for ( pF = funcList; pF->name; pF++ )
	check_func( pF );

    fclose( fpOut );

    UNLINK( dname );
    UNLINK( oname );
# endif

    return 0;
}
