/*
 * This file is part of
 *
 * LIBPNET6: a Portable Network Library
 *
 * LIBPNET6 is Copyright (c) 2002, Peter Bozarov
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Peter Bozarov.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * $Id: signal.c,v 1.7 2002/10/10 12:27:09 kingofgib Exp $
 */


/*----------------------------------------------------------------------*
 * filename:            signal.c
 * created on:          Thu Jul 18 23:37:49 CEST 2002
 * created by:          teddykgb
 * project:             Portable Network Library
 *----------------------------------------------------------------------*/

# include "local.h"

# ifdef HAVE_SIGNAL_H				/* {{ */

/*----------------------------------------------------------------------*/
/* Handle Unix signals							*/
/*----------------------------------------------------------------------*/

PNET_SIGFUNC
pnetSetSighandler( int signum, PNET_SIGFUNC func )
{
    struct sigaction sig,oldsig;

    pdbg(E_INFO,"pnetSetSighandler(signum=%d,func=%lX)\n", signum,XX(func));

    if ( signum < 0 )
    {
	perr(E_FATAL,"pnetSetSighandler(): signum=%d unsupported.\n",signum);
	return PNET_SIG_ERR;
    }

    sig.sa_handler	= func;
    sigemptyset( &sig.sa_mask );
    sig.sa_flags 	= 0;

    /* If SIGALRM is caught, we want to interrupt a possible system call.*/
    /* Some kernels will always restart an interrupted system call, so 	  */
    /* we need to tell them to not do that with SIGALRM			*/	
# ifdef SA_INTERRUPT
    if ( signum == SIGALRM )
    	sig.sa_flags |= SA_INTERRUPT;
# endif

    /* On the other hand, other kernels will always interrupt a system 	*/
    /* call. We put SA_RESTART on all signals, except SIG_ALRM 		*/
# ifdef SA_RESTART 
    if ( signum != SIGALRM )
	sig.sa_flags |= SA_RESTART;
# endif

    if ( sigaction( signum, &sig, &oldsig ) < 0 )
	{ perr(E_FATAL,"pnetSetSighandler(): %s\n",SYSERR()); return PNET_SIG_ERR; }

    return oldsig.sa_handler;
}

void
net_wait_for_child( int signum )
{
    pid_t	pid;
    int		stat;

    while ( (pid = waitpid( -1, &stat, WNOHANG | WUNTRACED )) > 0 )
    {
	if ( WIFEXITED( stat ) )
	{
	    stat = WEXITSTATUS( stat );
	    pdbg(E_DBG1,"Child %d exited with status %d\n",
	    	(int)pid, (int)stat );
	}
	else if ( WIFSIGNALED( stat ) )
	{
	    pdbg(E_DBG1,"Child %d terminated by signal %d\n",
	    	(int)pid, (int)WTERMSIG( stat ));
	}
    }
    return;

    signum = signum;
}

# else						/* }{ */

PNET_SIGFUNC
pnetSetSighandler( int signum, PNET_SIGFUNC func )
{
    perr(E_FATAL,"pnetSetSighandler() not implemented on this system\n");
    return (PNET_SIGFUNC) PNET_SIG_ERR;
}

void
net_wait_for_child( int signum ) { signum = signum; }
# endif						/* }} */
