/*
 * This file is part of
 *
 * LIBPNET6: a Portable Network Library
 *
 * LIBPNET6 is Copyright (c) 2002, Peter Bozarov
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Peter Bozarov.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * $Id: pnetsock.c,v 1.41 2002/12/01 16:41:44 kingofgib Exp $
 */


/*----------------------------------------------------------------------*
 * filename:            pnetsocket.c
 * created on:          Fri May 10 19:28:12 CEST 2002
 * created by:          teddykgb
 * project:             Portable Network Library
 *----------------------------------------------------------------------*/

# include "local.h"

static int net_open_socket(int *,int,int);
static int allocate_auxiliary_info(PNETSOCK);

/*----------------------------------------------------------------------*/
/* Creation of sockets							*/
/*----------------------------------------------------------------------*/

/*
 * Open a socket. Type can be AF_INET, AF_INET6, AF_LOCAL. As long as it's
 * supported of course.
 */

static int
net_open_socket(int *fam,int type,int proto)
{
    int sd = -1;
    int tries = 5;
    int curr;
    int fams[3] = { AF_INET, AF_INET6, 
# ifdef AF_LOCAL
	AF_LOCAL
# endif
	};

# ifdef AF_LOCAL
    if (*fam == AF_LOCAL)
	curr = 2;
    else
# endif
	curr = *fam == AF_INET6;

    if ( ! pnetInitDone )
    {
	if (pnetInit( ))
	    return -1;
    }

    while (tries--)
    {
	/* Try to open a socket of the family they asked for. */
	pdbg(E_INFO,"Trying socket(%s,%s,%d)...\n",
		net_aftoa(fams[curr]),net_sttoa(type),proto);

	sd = socket(fams[curr],type,proto);

	if (sd > 0)
	    { break; }

	pdbg(E_INFO," FAILED\n");

	/* Couldn't. Guess family is not present on this system. */
	/* Try next in list */

	perr(E_WARN,"socket(%s,%s,%d) failed.\n",
	     net_aftoa(fams[curr]),net_sttoa(type),proto);

# ifndef STDWin32
	if (*fam != AF_LOCAL)
# endif
	{
	    curr = ! curr;

	    pdbg(E_WARN,"Trying socket(%s,%s,%d)...\n",
		    net_aftoa(fams[curr]),net_sttoa(type),proto);

	    sd = socket(fams[curr],type,proto);

	    if (sd > 0)
		{ break; }

	    pdbg(E_WARN," FAILED\n");
	}

	/* Try again in a little while */
	sleep(1);
    }

    if (sd < 0)
    {
	perr(E_FATAL,"net_open_socket(fam=%s,type=%s,proto=%d): %s\n",
	      net_aftoa(*fam),net_sttoa(type),proto,neterr());
	return -1;
    }

    *fam = fams[curr];		/* Family might have changed */

    return sd;
}
/*
 * Make a PNETSOCKET from a given socket descriptor if that is larger than > 0.
 * If sd < 0, it's created dynamically.
 */
PNetSocket *
sock_socket(int sd,int fam,int type,int proto)
{
    PNetSocket * ps;
    int		act_fam = fam;

    if (sd < 0)
	sd = net_open_socket(&act_fam,type,proto);
    if (sd < 0)
	return NULL;

    STDMALLOC(ps,sizeof(PNetSocket),NULL);
    ps->sd 	= sd;
    ps->fam	= act_fam;
    ps->req_fam = fam;
    ps->type 	= type;
    ps->proto	= proto;

    if (ps->fam != ps->req_fam)
    {
	perr(E_WARN,"Socket requested family %s not supported, using %s\n",
		net_aftoa(fam),net_aftoa(act_fam));
    }

    return ps;
}
PNetSocket*
pnetDuplicate(PNETSOCK ps)
{
    PNetSocket* cp;

    STDMALLOC(cp,sizeof(PNetSocket),NULL);

    memcpy(cp,ps,sizeof(PNetSocket));

    cp->sd = dup(ps->sd);

# ifdef PNET_HAVE_LOCAL
    cp->copied_path = 1;
# endif

    if (cp->sd < 0)
	neterr();

    return cp;
}
int
pnetClose(PNETSOCK ps)
{
    int r;

    DBG(dbg("pnetClose(ps=%X)\n",(unsigned)ps));

    r = close(ps->sd);

    if (ps->aux_info)
    {
	if ( ps->aux_info->ai_alloc_flag )
	{
	    STDFREE( ps->aux_info->aux_pkt_opts );

	    if ( ps->type == SOCK_RAW )
		STDFREE( ps->aux_info->aux_ip6 );
	}
	STDFREE(ps->aux_info);
    }

# ifdef PNET_HAVE_LOCAL			/* { */
    if ( ps->fam == AF_LOCAL )
    {
	/* Clean up after ourselves 		*/
	LLOG( ps->copied_path );
	if ( ! ps->copied_path )
	{
	    DBG(dbg("Unlinking UNIX socket %s\n", ps->local_addr.pa_unpath ) );
	    if ( unlink( ps->local_addr.pa_unpath ) )
		perr(E_WARN,"pnetClose(): cannot unlink %s: %s\n",
		    ps->local_addr.pa_unpath, neterr() );
	}
	
    }
# endif					/* } */

    STDFREE( ps );
    return r;
}
# ifdef STDWin32 
#   define SHUT_RD		SD_RECEIVE
#   define SHUT_WR		SD_SEND
#   define SHUT_RDWR		SD_BOTH
# endif
int
pnetCloseRead( PNETSOCK ps )
{
    if ( shutdown( ps->sd, SHUT_RD ) )
	{ NETERR("pnetCloseRead()"); return -1; }
    return 0;
}
int
pnetCloseWrite( PNETSOCK ps )
{
    if ( shutdown( ps->sd, SHUT_WR ) )
	{ NETERR("pnetCloseWrite()"); return -1; }
    return 0;
}
int
pnetCloseReadWrite( PNETSOCK ps )
{
    if ( shutdown( ps->sd, SHUT_RDWR ) )
	{ NETERR("pnetCloseReadWrite()"); return -1; }
    return 0;
}

/* Family can be PNET_LOCAL, PNET_IPv4, PNET_IPv6 */

PNETSOCK
pnetTCPSocket2(int pfam)
{
    PNETSOCK ps;
    int on = 1;

    DBG(dbg("pnetTCPSocket2(pfam=%s)\n",net_paftoa(pfam)));

    if ( !(ps = sock_socket(-1,pfam2fam(pfam),SOCK_STREAM,0)) )
	{ XLOG(ps); return NULL; }

    setsockopt(ps->sd,SOL_SOCKET,SO_KEEPALIVE,(char*)&on,sizeof(on));
    setsockopt(ps->sd,SOL_SOCKET,SO_REUSEADDR,(char*)&on,sizeof(on));

    return ps;
}

/* Default, try to open sockets in the IPv6 domain, after all that's 
 * where we're headed 
 */
PNETSOCK pnetTCPSocket( void ) { return pnetTCPSocket2( PNET_IPv6 ); }
PNETSOCK pnetUDPSocket( void ) { return pnetUDPSocket2( PNET_IPv6 ); }

PNETSOCK
pnetUDPSocket2(int pfam)
{
    PNETSOCK ps;

    DBG(dbg("pnetUDPSocket(fam=%s)\n",net_paftoa(pfam)));

    ps = sock_socket(-1,pfam2fam(pfam),SOCK_DGRAM,0);

    return ps;
}
PNETSOCK
pnetUDPSocketX(int pfam)
{
    PNETSOCK 	ps;

    ps = pnetUDPSocket2(pfam);

# ifdef PNET_HAVE_LOCAL
    /* Local sockets don't have auxilary info */
    if ( ps->fam == AF_LOCAL )
	return ps;
# endif

    allocate_auxiliary_info( ps );

    return ps;
}

static int
allocate_auxiliary_info( PNETSOCK ps )
{
    DBG( dbg( "allocate_auxiliary_info( ps = %lX )\n", XX(ps) ) );

    if ( ! ps )
	{ XLOG( ps ); return -1; }

    /* Allocate aux_info buffer         */
    STDMALLOC(ps->aux_info,sizeof(struct pnet_aux_info),-1);
 
    /* Allocate space to receive all packet options     */
    STDMALLOC( ps->aux_info->aux_pkt_opts,
               sizeof( struct pnet_aux_pktoptions ), -1 );

    if ( ps->type == SOCK_RAW )
	/* Allocate space to hold ip headers */
	STDMALLOC(ps->aux_info->aux_ip, MAXIPHDRSIZE, -1 );
 
    ps->aux_info->ai_alloc_flag = 1;
 
    if (ps->fam == AF_INET)
    {
# ifdef IP_PKTINFO
        net_setsockopt_ippktinfo(ps,1);
# endif
# if defined IP_RECVIF
        net_setsockopt_iprecvif(ps,1);
# endif
# if defined IP_RECVDSTADDR
        net_setsockopt_iprecvdstaddr(ps,1);
# endif
    }
# ifdef PNET_HAVE_IPv6
    else if (ps->fam == AF_INET6)
    {
        net_setsockopt_ipv6pktinfo(ps,1);
# ifdef IPV6_HOPLIMIT
        net_setsockopt( ps, IPPROTO_IPV6, IPV6_HOPLIMIT, 1 );
# endif
    }
# endif

    return 0;
}
/*----------------------------------------------------------------------*/
/* Raw socket support 							*/
/*----------------------------------------------------------------------*/

PNETSOCK pnetRAWSocket( int pr ) { return pnetRAWSocket2( PNET_IPv6, pr ); }

PNETSOCK
pnetRAWSocket2(int pfam,int proto)
{
    PNETSOCK ps = NULL;

# ifndef SOCK_RAW			/* {{ */

    perr(E_FATAL,"pnetRAWSocket2(): RAW sockets not supported.\n");

# else					/* }{ */

    DBG(dbg("pnetRAWSocket(fam=%s,proto=%d)\n",net_paftoa(pfam),proto));

    ps = sock_socket(-1,pfam2fam(pfam),SOCK_RAW,proto);

    /* Release root - privs */
# ifdef HAVE_SETUID
    setuid(getuid());
# endif 

    if (ps)
	allocate_auxiliary_info( ps );

# endif					/* }} */

    return ps;
}
/*
 * For RAW sockets one can choose to build their own headers.
 */
int
pnetSockOwnIPHeader(PNETSOCK ps,int on)
{
    DBG(dbg("pnetSockOwnHeader(ps=%lX,on=%d)\n",XX(ps),on));

    /* We can only do this with raw sockets */

    if (ps->type != SOCK_RAW )
	{ TYPEERR("pnetSockOwnHeader()"); return -1; }

    if ( ps->fam == AF_INET && net_setsockopt_iphdrincl(ps,on) )
	{ XLOG(ps); LLOG(on); return -1; }

# ifdef IPV6_FLOWINFO_SEND
    if ( ps->fam == AF_INET6 )
    {
	if ( setsockopt( ps->sd, IPPROTO_IPV6,
			 IPV6_FLOWINFO_SEND, &on, sizeof(on) ) < 0 ) 
	{
	    perr(E_FATAL,"Cannot set IPV6_FLOWINFO_SEND info option \n");
	}
    }
# endif

    ps->own_header = on != 0;

    pdbg(E_INFO,"pnetSockOwnHeader(ps=%lX,on=%s)\n",XX(ps),PONOFF(on));

    return 0;
}

/*
 * Raw sockets of the AF_INET6 family and with a protocol other than
 * ICMPV6, can tell the kernel to compute and fill in the checksum on
 * their behalf.
 * on = turn on or off;
 * offset = byte offset to 16 checksum in application data buffer (IPv6
 *          payload)
 */
int
pnetSockChecksum( PNETSOCK ps, int on, int offset )
{
    DBG(dbg("pnetSockChecksum(ps=%lX,on=%d)\n",XX(ps),on));

    /* We can only do this with raw sockets */
    if (ps->type != SOCK_RAW || ps->fam != AF_INET6)
	{ TYPEERR("pnetSockChecksum()"); return -1; }

    if ( offset < 0 )
    {
	perr(E_FATAL,"pnetSockChecksum(): offset value negative.\n");
	return -1;
    }

    if ( on == 0 )
	offset = -1;

# ifdef IPV6_CHECKSUM
    if ( setsockopt( ps->sd, IPPROTO_IPV6,
		     IPV6_CHECKSUM, &offset, sizeof(offset) ) < 0 ) 
	{ NETERR("pnetSockChecksum()"); return -1; }
# endif

    pdbg(E_INFO,"pnetSockChecksum(ps=%lX,on=%s,offset=%d)\n",
    		 XX(ps),PONOFF(on),offset);

    return 0;
}
/*----------------------------------------------------------------------*/
/* Setting parameters							*/
/*----------------------------------------------------------------------*/

int
pnetSockAddReadcallback(PNETSOCK ps,PNETREADCB cb,void *data)
{

    DBG(dbg("pnetSockAddReadcallback(ps=%X,cb=%X,data=%X)\n",
	(unsigned)ps,(unsigned)cb,(unsigned)data));
    if (!ps)
	{ XLOG(ps); return -1; }

    ps->read_callback = cb;
    ps->callback_data = data;

    return 0;
}
/*----------------------------------------------------------------------*
 * Server-esque routines						*
 *----------------------------------------------------------------------*/
int
sock_bind(PNETSOCK ps,PNETADDR pa)
{
    int ret = 0;

    if ( (ret = bind( ps->sd, pa ? &pa->pa_saddr : NULL,
			      pa ? pa->pa_len    : 0 ) ) )
    {
	char buf[ PNET_ADDR_STRLEN ];
	perr(E_FATAL,"sock_bind(ps=%lX, address='%s'): %s\n",
	     XX( ps ), pnet_ntop( pa, buf, sizeof( buf ) ), neterr());
	DBG( pnetSockDebug( stderr, ps ) );
    }

    return ret;
}
int
sock_getsockname(PNETSOCK ps,PNETADDR pa)
{
    socklen_t	len = PNET_MAX_SOCKADDR;
    int 	ret = 0;
    AddrUnX	addr;

    if ( (ret =  getsockname(ps->sd,&addr.saddr,&len)) < 0 )
	{ NETERR("sock_getsockname()"); }

    addr_from_sockaddr(pa,&addr.saddr);
    return ret;
}
/*----------------------------------------------------------------------*/
/* Get attributes							*/
/*----------------------------------------------------------------------*/

PNETADDR
pnetSockGetPeerAddr(PNETSOCK ps)
{
    return &ps->peer_addr;
}
int
pnetSockSetPeerAddr(PNETSOCK ps,PNETADDR pa)
{
    pnetAddrCopy(&ps->peer_addr,pa);
    return 0;
}
PNETADDR
pnetSockGetLocalAddr(PNETSOCK ps)
{
    return &ps->local_addr;
}

int
pnetSockSetNonBlocking(PNETSOCK ps,int on)
{
    int flags;

    pdbg(E_DBG4,"pnetSockSetNonBlocking(ps=%lX,on=%s)\n",
    		 XX(ps),PYESNO(on != 0));

# ifdef HAVE_FCNTL_H		/* { */

    if ( (flags = fcntl(ps->sd,F_GETFL,0)) < 0 )
	{ FATALERR("pnetSockSetNonBlocking(): fcntl():"); return -1; }

    if (on)
	flags |= O_NONBLOCK;
    else
	flags &= ~O_NONBLOCK;

    if ( fcntl(ps->sd,F_SETFL, flags) < 0 )
	{ FATALERR("pnetSockSetNonBlocking(): fcntl():"); return -1; }
    
    return 0;
# elif defined STDWin32		/* } { */

    if ( ioctlsocket( ps->sd, FIONBIO, (unsigned long*)&on ) == SOCKET_ERROR )
    	{ FATALERR("pnetSockSetNonBlocking()"); return -1; }

    return 0;
# else 					/* } { */
    return -1;
# endif					/* } */
}

int
pnetSockFamily( PNETSOCK ps )
{
    return fam2pfam( ps->fam );
}
/*----------------------------------------------------------------------*/
/* Debug a PNETSOCKET							*/
/*----------------------------------------------------------------------*/

int
pnetSockDebug(FILE *stream,PNETSOCK ps)
{
    char 	buf[ PNET_ADDR_STRLEN ];

    fprintf(stream,"Socket %lX, type = %s:\n",XX(ps),net_sttoa(ps->type));
    fprintf(stream,"   sd = %d, fam = %s, req_fam = %s,\n", 
	    ps->sd,net_aftoa(ps->fam),net_aftoa(ps->req_fam));
    fprintf(stream,"   protocol = %d, own_header = %s, connected = %s\n",
	    ps->proto,PYESNO(ps->own_header),PYESNO(ps->connected));
    fprintf(stream,"   Local Address: %s\n", 
		    pnet_ntop(&ps->local_addr,buf,sizeof(buf)));
    fprintf(stream,"   Peer  Address: %s\n",
    		    pnet_ntop(&ps->peer_addr,buf,sizeof(buf)));
    fprintf(stream,"   aux_info pointer: %lX\n",XX(ps->aux_info));

    return 0;
}
