/*
 * This file is part of
 *
 * LIBPNET6: a Portable Network Library
 *
 * LIBPNET6 is Copyright (c) 2002, Peter Bozarov
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Peter Bozarov.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * $Id: pnetip.c,v 1.30 2002/12/02 08:22:03 kingofgib Exp $
 */


/*----------------------------------------------------------------------*
 * filename:		pnetip.c
 * created on:		Tue Jul 30 00:17:16 CEST 2002
 * created by:		teddykgb
 * project: 		Portable Network Library
 *----------------------------------------------------------------------*/

/*----------------------------------------------------------------------*/
/* IP packets functionality						*/
/*----------------------------------------------------------------------*/
# include "local.h"
# include "pnet6tlp.h"

/*----------------------------------------------------------------------*/
/* The ubuquitous ip_checksum() routine. 				*/
/*----------------------------------------------------------------------*/

pnet_uint
pnetIP_ChecksumAdd( pnet_ushort* buf, pnet_uint len, pnet_uint sum )
{
# include <dbgon.h>
    DBG( dbg("pnetIP_ChecksumAdd(buf=%lX,len=%d,sum=%lX)\n",
    	      XX(buf), len, XX(sum)));

    while ( len  > 1 )
        sum += *buf++, len -= 2;

    if ( len == 1 )
	sum += *(pnet_byte*) buf;
# include <dbgoff.h>

    return sum;
}
pnet_ushort
pnetIP_ChecksumFold( pnet_uint sum )
{
    /*
    sum = ( sum >> 16 ) + ( sum & 0x0000FFFF );
    sum += ( sum >> 16 );
	*/
    while ( sum >>16 )
    	sum = ( sum >> 16 ) + ( sum & 0x0000FFFF );

    return ( pnet_ushort) ~sum;
}

# define ip_get_version( ppip ) ((!ppip) ? -1 : PNET_IPVERSION((pnet_ip*)ppip))

int
pnetIP_Version( void * ppip )
{
    return ip_get_version( ppip );
}
int
pnetIP_Source( void *ppip, PNETADDR pa )
{
    int version = ip_get_version( ppip );

    if ( version < 0 )
    	return version;

    if ( version == 4 )
	return addr_set_ip_addr( pa, AF_INET, &((pnet_ip*)ppip)->ip_src, 4 );
    else if (version == 6 )
	return addr_set_ip_addr( pa, AF_INET6, &((pnet_ip6*)ppip)->ip6_src,16);
    return -1;
}
int
pnetIP_Destination( void *ppip, PNETADDR pa )
{
    int version = ip_get_version( ppip );

    if ( version < 0 )
    	return version;

    if ( version == 4 )
	return addr_set_ip_addr( pa, AF_INET, &((pnet_ip*)ppip)->ip_dst, 4 );
    else if (version == 6 )
	return addr_set_ip_addr( pa, AF_INET6, &((pnet_ip6*)ppip)->ip6_dst,16);
    return -1;
}
pnet_byte*
pnetIP_Payload( void * ppip )
{
    switch ( ip_get_version( ppip ) )
    {
    case -1:
    	return NULL;
    case 4:
	return (byte*)ppip + pnetIP_HeaderLength((pnet_ip*) ppip);
    case 6:
	return (byte*)ppip + sizeof( pnet_ip6 );
    }
    return NULL;
}

int
pnetIP_Protocol( void * ppip )
{
    switch ( ip_get_version( ppip ) )
    {
    case -1:
	return -1;
    case 4:
	return ((pnet_ip*)ppip)->ip_p;
    case 6:
	return ((pnet_ip6*)ppip)->ip6_nxt;
    }
    return -1;
}

int
pnetIP_HeaderLength( void * ppip )
{
    switch ( ip_get_version( ppip ) )
    {
    case -1:
	return -1;
    case 4:
	return PNET_IPHDRLEN( (pnet_ip*)ppip ) << 2;
    case 6:
	return sizeof( pnet_ip6 );
    }
    return -1;
}
int
pnetIP_Length( void * ppip )
{
    switch ( ip_get_version( ppip ) )
    {
    case -1:
	return -1;
    case 4:
	return ntohs( ((pnet_ip* ) ppip)->ip_len );
    case 6:
	return ntohs( ((pnet_ip6*)ppip)->ip6_plen ) + sizeof( pnet_ip6 );
    }
    return -1;
}
int
pnetIP_PayloadLength( void * ppip )
{
    switch ( ip_get_version( ppip ) )
    {
    case -1:
	return -1;
    case 4:
    	{
	    pnet_ip*	ip = (pnet_ip*) ppip;
	    return pnetIP_Length(ppip) - (PNET_IPHDRLEN( ip ) << 2);
	}
    case 6:
	    return ntohs( ((pnet_ip6*)ppip)->ip6_plen );
    }
    return -1;
}

int
pnetIP4_Tos( pnet_ip * ip )
{
    return ip->ip_tos;
}
int
pnetIP4_Id( pnet_ip * ip )
{
    return ntohs( ip->ip_id );
}
int
pnetIP4_FragOffset( pnet_ip * ip )
{
    pnet_ushort off = ntohs(ip->ip_off) & 0x1FFF;

    return off;
    /* return ((pnet_ip* ) ppip)->ip_off; */
}
int
pnetIP4_MoreFrags( pnet_ip * ip )
{
    return (ntohs(ip->ip_off) & PNET_IP_MF) == PNET_IP_MF;
}
int
pnetIP4_DontFrag( pnet_ip * ip )
{
    return (ntohs(ip->ip_off) & PNET_IP_DF) == PNET_IP_DF;
}

int
pnetIP_TTL( void * ppip )
{
    switch ( ip_get_version( ppip ) )
    {
    case -1:
	return -1;
    case 4:
	return ((pnet_ip* ) ppip)->ip_ttl;
    case 6:
	return ((pnet_ip6* ) ppip)->ip6_hlim;
    }
    return -1;
}
int
pnetIP_Hoplimit( void * ppip )
{
    switch ( ip_get_version( ppip ) )
    {
    case -1:
	return -1;
    case 4:
	return ((pnet_ip* ) ppip)->ip_ttl;
    case 6:
	return ((pnet_ip6* ) ppip)->ip6_hlim;
    }
    return -1;
}
int
pnetIP4_Checksum( pnet_ip * ip )
{
    return ntohs( ip->ip_sum );
}
/*
 * IPv6 priority.
 */
int
pnetIP6_Priority( pnet_ip6 * ip6 )
{
    return ((ip6->ip6_vfc & ~IPV6_VERSION_MASK) & 0xFF);
}
/*
 * Traffic Class is the 5th through the 13th bit in the IPv6 header,
 * according to RFC2460.
 */
int
pnetIP6_TrafficClass( pnet_ip6 * ip6 )
{
    return ( (ip6->ip6_flow & htonl( 0x0FF00000 )) >> 20);
}

/*
 * Flow is the last 20 bits of the first IP6 header word,
 * if you go by RFC2460. We return them here in network byte order.
 */
int
pnetIP6_Flow( pnet_ip6 * ip6 )
{
    return ( ip6->ip6_flow & htonl( 0x000FFFFF ));
}
/*----------------------------------------------------------------------*/
/* Construct IP packets from scratch					*/
/*----------------------------------------------------------------------*/

pnet_ip*
pnetIP4_Build( pnet_ip * ip, int tos, int tlen, int id, int frag_off,
	       int ff, int ttl, int proto, PNETADDR src,PNETADDR dst )
{
    int len = 20;

    memset( ip, 0, sizeof( pnet_ip ) );
    ip->ip_vhl = 4 << 4;
    ip->ip_tos = tos;
    ip->ip_len = htons((pnet_ushort) tlen);
    ip->ip_id  = htons((pnet_ushort) id  );
    ip->ip_off = ff | (htons((pnet_ushort)frag_off) & 0x1FFF);
    ip->ip_off = htons( ip->ip_off );
    ip->ip_ttl = ttl;
    ip->ip_p   = proto;
    ip->ip_sum = 0;

    if ( src )
	ip->ip_src = src->pa_s_addr;

    ip->ip_dst = dst->pa_s_addr;

    ip->ip_vhl = (4 << 4 ) | (len >> 2);

    ip->ip_sum = pnetIP4_ComputeChecksum( ip );
    DBG( IPDUMP( ip ) );

    return ip;
}

pnet_ushort
pnetIP4_ComputeChecksum( pnet_ip * ip )
{
    pnet_ushort ipsum = ip->ip_sum;	/* Store original cksum */
    pnet_uint	sum   = 0;

    ip->ip_sum = 0;	/* Clear checksum */

    sum = pnetIP_ChecksumAdd( (pnet_ushort*)ip, (PNET_IPHDRLEN(ip) << 2), 0 );

    ip->ip_sum = ipsum;			/* Restore original cksum */

    return pnetIP_ChecksumFold( sum );
}
pnet_ip6*
pnetIP6_Build( pnet_ip6 *ip, int tc, pnet_uint flow, pnet_ushort len,
		int next_header, int hoplimit, PNETADDR src, PNETADDR dst )
{
    DBG( dbg("pnetIP6_Build(ip=%x,tc=%d,flow=%X,len=%hu,nxt=%d,hlim=%d,"
	     "src=%X,dst=%X)\n",ip,tc,flow,len,next_header,hoplimit,src,dst) );
    memset( ip, 0, sizeof( pnet_ip6 ) );

    ip->ip6_flow = ( 6 << 28 ) | ( (tc&0xFF) << 20 ) | ((flow & 0x000FFFFF));
    ip->ip6_flow = htonl( ip->ip6_flow );
    ip->ip6_plen = htons( len );
    ip->ip6_nxt  = next_header;
    ip->ip6_hlim = hoplimit;
    
    if ( src )
	memcpy( ip->ip6_src, &src->pa_s6_addr , sizeof( ip->ip6_src ) );

    memcpy( ip->ip6_dst, &dst->pa_s6_addr, sizeof( ip->ip6_dst ) );

    DBG( IPDUMP( ip ) );

    return ip;
}
/*----------------------------------------------------------------------*/
/* Send a home-made IP datagram on a socket. The payload of the datagram*/
/* should be some Transport Level proto known to the caller. For TCP    */
/* and UDP, pnetTPC_Send() and pnetUDP_Send() can be used.		*/
/* This is intended for other types of transport level protocols.	*/
/* If a checksum must be computed, then do_chksum must be a index into  */
/* the datagram where the checksum is found (e.g. 16 for TCP, 6 for UDP)*/
/* In addition, if a pseudo-header must be used, the appropriate flag	*/
/* must be set.								*/
/*----------------------------------------------------------------------*/

int
pnetIP_Send( PNETSOCK ps, PNETADDR pa, void *pip,
	     pnet_byte * tlh, int tlh_len,
	     int chk_off, int pseudo_header,
	     pnet_byte * data, int datalen )
{
    struct iovec	iov[3];
    int			i = 0;
    pnet_ushort	*	psum;

    DBG( dbg( "pnetIP_Send( ps=%lX, pa=%lX, pip=%lX, tlh=%lX, "
	      "tlh_len=%d, chk_off=%d, pseudo_header=%d, "
	      "data=%lX, datalen=%lX )\n", XX(ps), XX(pa), XX(pip),
	      XX(tlh), tlh_len, chk_off, pseudo_header, XX(data), datalen ) );

    DBG( IPDUMP( pip ) );

    if ( ! ps->own_header )
    {
	pdbg( E_WARN, "pnetIP_Send(): socket not using own header, checksum\n");
	pdbg( E_WARN, "computation possibly wrong.\n" );
    }
    else
    {
	iov[i].iov_base	= (char*) pip;
	iov[i].iov_len	= pnetIP_HeaderLength( pip );
	i++;
    }

    if ( tlh )
    {
	if ( chk_off > 0 )
	{
	    psum = (pnet_ushort*) ( (byte*)tlh + chk_off );
	    *psum = 0;
	    *psum = pnetIP_TLHComputeChecksum( pip, tlh, tlh_len,
					       data, datalen, 
					       chk_off, pseudo_header );
	}

	iov[i].iov_base	= (char*) tlh;
	iov[i].iov_len	= tlh_len;
	i++;

	if ( data )
	{
 	    iov[i].iov_base	= (char*) data;
	    iov[i].iov_len	= datalen;
	    i++;
	}
    }

    return net_writev( ps, pa, iov, i, 0 );
}
/*----------------------------------------------------------------------*/
/* Compute the checksum for a Transport Level Header (i.e. TCP). 	*/
/* pip is a pointer to the IP header (either IPv4 			*/
/* or IPv6), tlh is a pointer to the upper level header (i.e. TCP, UDP),*/
/* ck offset indicates where the checksum is stored and computed, and 	*/
/* pseudo header means that a IP pseudo header must be compute with 	*/
/* the check sum.							*/
/*----------------------------------------------------------------------*/

pnet_ushort
pnetIP_TLHComputeChecksum( void * pip, byte * tlh, int tlh_len,
			   byte * payload, int payload_len, 
			   int chk_off, int pseudo_header )
{
    pnet_uint 	sum   = 0;		/* guaranteed 32 bits wide */
    pnet_ushort	octets = 0;
    pnet_ushort org_sum;
    pnet_ushort	* psum;
# include <dbgon.h>
    DBG( dbg( "pnetIP_TLHComputeChecksum( pip=%lX, tlh=%lX, tlh_len=%d, "
	"payload=%X, payload_len=%d, "
	"chk_off=%d, pseudo_header=%d )\n",
	XX(pip), XX(tlh), tlh_len, XX( payload ), payload_len, 
	chk_off, pseudo_header ) );

    if ( ! pip )
	return 0;

    if ( pseudo_header )
    {
	/*
	 * Collect the fields making up the pseudo header
	 */
	switch ( pnetIP_Version( pip ) )
	{
	case 4:
	    sum = pnetIP_ChecksumAdd((pnet_ushort*)&((pnet_ip*)pip)->ip_src,8,0);

	    sum += htons( (pnet_ushort) pnetIP_Protocol( pip ) );

	    octets = (pnet_ushort) pnetIP_PayloadLength( pip );

	    sum += htons( octets );

	    break;

	case 6:
	    sum = pnetIP_ChecksumAdd((pnet_ushort*)&((pnet_ip6*)pip)->ip6_src,32,0);

	    sum += (pnet_uint) htonl( (pnet_ushort) pnetIP_Protocol( pip ) );

	    octets = (pnet_ushort) pnetIP_PayloadLength( pip );

	    sum += (pnet_uint) htonl( octets );

	    break;

	default:
	    perr(E_FATAL,"net_compute_checksum(): IP header version corrupt\n");
	    return 0;
	}
    }

    if ( tlh )
    {
	/*
	 * Remember original value of checksum 
	 */
	psum = (pnet_ushort*) (tlh + chk_off);

	org_sum = *psum;
	*psum = 0;

	sum = pnetIP_ChecksumAdd( (pnet_ushort*) tlh, tlh_len, sum );

	if ( payload )
	    sum = pnetIP_ChecksumAdd((pnet_ushort*)payload,payload_len,sum);

	*psum = org_sum;
    }

    return pnetIP_ChecksumFold( sum );
}
