/*
 * This file is part of
 *
 * PNET6: a Portable Network Library
 *
 * PNET6 is Copyright (c) 2002, Peter Bozarov
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Peter Bozarov.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * $Id: pnetpkt.c,v 1.16 2002/12/22 14:13:32 kingofgib Exp $
 */

/*----------------------------------------------------------------------*
 * filename:		pnetpkt.c
 * created on:		Fri Jul  5 16:27:15 CEST 2002
 * created by:		teddykgb
 * project: 		Portable Network Library
 *----------------------------------------------------------------------*/

# include "../local.h"
# include "../pkt.h"
# include "../pnet6pkt.h"

/*----------------------------------------------------------------------*/
/* Just in case the kernel does not support counting packets		*/
/*----------------------------------------------------------------------*/
static pnet_uint 	total_recv;

/*----------------------------------------------------------------------*/
/* Monitoring packets on the network. 					*/
/* We can watch packets coming on a particular device. To reduce the	*/
/* actual amount of packets we need to process, filters can be placed	*/
/* that only select certain packets, and skip others.			*/
/*----------------------------------------------------------------------*/

const char *
pkt_get_ll_name( struct pnet_pktacc *pa )
{
    switch ( pa->pa_iftype )
    {
        case DLT_NULL:   return("None");
# ifdef DLT_LOOP
	case DLT_LOOP:	 return("Loop");
# endif
        case DLT_EN10MB: return("Ethernet (10Mb/100Mb)");
        case DLT_SLIP:   return("Serial Line IP");
        case DLT_PPP:    return("P2P");
        case DLT_FDDI:   return("FDDI");
        case DLT_RAW:    return("Raw IP");
        default:         return("Unknown");
    }

    return "";
}

byte*
pkt_get_ll_info( PNetPacket * pkt, int *plen, int what )
{
    PNetPktAccess *	pa;
    byte *	    	p;

    pa = pkt->pkt_pa;

    if ( ! pa )
	return NULL;
    
    p = pkt->pkt_buf;

    if ( pa->pa_cooked == PKTACC_TRUE_COOKED )
	return NULL;	/* No idea what it is with TRUE cooked mode */

    if ( pa->pa_cooked == PKTACC_PSEUDO_COOKED )
    {
	/* Go back a little, in order to get link layer info */
	p -= pa->pa_llh_len;
    }
    if ( pa->pa_iftype == DLT_NULL )
    {
	*plen = 4;		/* 4 bytes loopback header */

	if ( what == PPKT_LL_LENGTH )
	    return p;

	if (what == PPKT_LL_TYP)
	    { *plen = 1; return p; }

	return p;
    }
    if ( pa->pa_iftype == DLT_EN10MB )
    {
	*plen = 6;
	if ( what == PPKT_LL_LENGTH )
 	    return p;
	if (what == PPKT_LL_SRC)
	    return p;
	if (what == PPKT_LL_DST)
	    return p + 6;
	if (what == PPKT_LL_TYP)
	    { *plen = 2; return p + 12; }
    }
    if ( pa->pa_iftype == DLT_FDDI )
    {
	*plen = 6;
	if ( what == PPKT_LL_LENGTH )
 	    return p;
	if ( what == PPKT_LL_SRC )
	    return p + 7;
	if ( what == PPKT_LL_DST )
	    return p + 1;
	if (what == PPKT_LL_TYP)
	    { *plen = 2; return p + 19; }
    }

    perr( E_FATAL,"Don't know size of link layer header for "
		 "interface type %d\n", pa->pa_iftype );

    return NULL;
}

/*----------------------------------------------------------------------*/
/* Open an interface, and configure it for link-layer packet access.	*/
/*----------------------------------------------------------------------*/
PNET_PKTACCESS
pnetPktOpenInterface( const char *ifname, int prom, int msec, int glen )
{
    PNetPktAccess *pa;

    pa = pkt_open_dev( ifname, 10, glen );

    if ( !pa )
    {
	perr(E_FATAL,"Cannot enter packet access mode.\n");
	return NULL;
    }

# ifdef HAVE_SETUID
    setuid( getuid() );
# endif

    pkt_set_promiscuous( pa, prom );
    pkt_set_read_timeout( pa, msec );

    return pa;
}
/*----------------------------------------------------------------------*/
/* Close the inteface. 							*/
/*----------------------------------------------------------------------*/
void
pnetPktCloseInterface( PNET_PKTACCESS pa )
{
    pkt_set_promiscuous( pa, 0 );
    pkt_close_dev( pa );
}

/*----------------------------------------------------------------------*/
/* Get number of caught and dropped packets 				*/
/*----------------------------------------------------------------------*/
int
pnetPktGetStats( PNET_PKTACCESS pa, pnet_uint * caught, pnet_uint * dropped )
{
    if ( caught )
	*caught = 0;
    if ( dropped )
	*dropped = 0;

    if ( pkt_get_stats( pa, caught, dropped ) )
	return -1;

    /* if ( caught && *caught == 0 )*/
	*caught = total_recv;

    return 0;
}

int
pnetPktSetReadMode( PNET_PKTACCESS pa, int mode )
{
    if ( mode != PNET_READ_RAW && mode != PNET_READ_COOKED )
    {
	perr(E_FATAL,"pnetPktSetReadMode( ): unknown mode %d\n",mode );
	return -1;
    }
    /* 
     * Here, we need to be careful. If a device is in true cooked mode,
     * we can't just switch it into raw mode, since that is not available
     * (TRUE coooked is only set if a kernel or driver does not support
     * actual raw mode).
     */

    if ( mode == PNET_READ_RAW && pa->pa_cooked == PKTACC_TRUE_COOKED )
    {
	perr(E_FATAL,"pnetPktSetReadMode(): device %s does not "
		     "support raw mode\n", pa->pa_ifname );
	return -1;
    }

    /* 
     * Toggle between PSEUDO cooked and raw 
     */
    pa->pa_cooked = mode == PNET_READ_COOKED ? 
			PKTACC_PSEUDO_COOKED : PKTACC_NOT_COOKED; 
    return 0;
}
int
pnetPktSetReadTimeout( PNET_PKTACCESS pa, int msec )
{
    return pkt_set_read_timeout( pa, msec );
}
PNET_PKT
pnetPktMake( void )
{
    PNetPacket * pkt;

    STDMALLOC( pkt, sizeof( PNetPacket ), NULL );

    return pkt;
}
int
pnetPktNextPacket( PNET_PKTACCESS pa, PNET_PKT pkt )
{
    if ( pkt_next_pkt( pa, pkt ) )
    {
	/*
	 * In RAW or TRULY cooked mode, nothing needs to be done
	 */
	if ( pa->pa_cooked == PKTACC_PSEUDO_COOKED )
	{
	    /*
	     * Users want their packets cooked. Since we're doing this
	     * ourselves in PSEUDO mode, we skip past the link layer
	     * header here in this case.
	     */

	    pkt->pkt_buf += pa->pa_llh_len;
	    pkt->pkt_grablen -= pa->pa_llh_len;
	}

	total_recv++;

	return 1;
    }

    return 0;
}
int
pnetPktWrite( PNET_PKTACCESS pa, pnet_byte * buf, pnet_uint blen )
{
    int wrote = pkt_output( pa, buf, blen );

    pdbg( E_DBG1, "pnetPktWrite(): wrote %d bytes\n" , wrote );
    return wrote;
}
void
pnetPktDump(FILE *fp,PNET_PKT pkt)
{
    pnetHexdump( fp, pkt->pkt_buf, pkt->pkt_grablen, 16 );
}

/*----------------------------------------------------------------------*/
/* Access fields of a pnet_pkt						*/
/*----------------------------------------------------------------------*/

int
pnetPktLLLength( PNET_PKTACCESS pa, int * plen )
{
    if ( ! pa )
	return -1;
    
    if ( pa->pa_cooked == PKTACC_TRUE_COOKED )
	return -1;	/* No idea what it is with TRUE cooked mode */

    *plen = pa->pa_llh_len;

    return 0;
}

byte*
pnetPktLLSource( PNetPacket *pkt, int * plen )
{
    return pkt_get_ll_info( pkt, plen, PPKT_LL_SRC );
}
byte*
pnetPktLLDest( PNetPacket *pkt, int * plen )
{
    return pkt_get_ll_info( pkt, plen, PPKT_LL_DST );
}
int
pnetPktLLType( PNetPacket *pkt, int * ptype )
{
    byte * b;
    int len;

    b = pkt_get_ll_info( pkt, &len, PPKT_LL_TYP );

    if ( len == 1 )
	*ptype = (int) (*b);
    if ( len == 2 )
	*ptype = (int) htons( (u_short) *b );

    return - ! ptype;
}
