/*
 * This file is part of
 *
 * PNET6: a Portable Network Library
 *
 * PNET6 is Copyright (c) 2002, Peter Bozarov
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Peter Bozarov.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * $Id: if-win.c,v 1.12 2002/11/07 07:09:45 kingofgib Exp $
 */

/*----------------------------------------------------------------------*
 * filename:            if-win.c
 * created on:          Tue Jun  4 17:49:01 EDT 2002
 * created by:          peter
 * project:             Portable Network Library
 *----------------------------------------------------------------------*/

/*
 * Interface information, MS-Windows style 
 */

# include "../local.h"

# if defined STDWin32 && ! defined STD_WIN98		/* { */

# include "iflocal.h"

# if defined HAVE_IPIFCONS_H || defined STD_WIN98

# include <ipifcons.h>

# endif

static void
win_if_error( int err_code )
{
    switch (err_code)
    {
    case ERROR_BUFFER_OVERFLOW:
	perr(E_FATAL,"get_ifs_info(): %s\n",pneterr( PNETerrSpace ) );
	return;
    case ERROR_INVALID_PARAMETER:
	perr(E_FATAL,"get_ifs_info(): %s\n","INVALID_PARAMETER");
	return;
    case ERROR_NO_DATA:
	perr(E_FATAL,"get_ifs_info(): %s\n","NO_DATA");
	return;
    case ERROR_NOT_SUPPORTED:
	perr(E_FATAL,"get_ifs_info(): %s\n","NOT_SUPPORTED");
	return;
    default:
	perr(E_FATAL,"get_ifs_info(): %s\n",neterr());
	return;
    }
}

# define NULLADDR	"0.0.0.0"

static int
ipAddrStringToInAddr( int fam, IP_ADDR_STRING* ips, PNetIfAddr *pia )
{
    SLOG( ips->IpAddress.String );
    SLOG( ips->IpMask.String );

    while ( ips && ! strcmp(ips->IpAddress.String, NULLADDR) )
	ips = ips->Next;

    if (!ips)
    {
	perr(E_FATAL,"No adapter IP addresses found.\n");
	return -1;
    }

    if (fam == AF_INET)
    {
	InetAddr * ia = (InetAddr*) &pia->pia_addr4;

	ia->sin_family = fam;
	ia->sin_port   = 0x0;

	inet_pton(fam,ips->IpAddress.String,&ia->sin_addr.s_addr);
	inet_pton(fam,ips->IpMask.String,(void*)&pia->pia_netmask[1]);

	pia->pia_netmask[0] = 1;

	SLOG (net_sockaddrtop(ia));
    }
# ifdef PNET_HAVE_IPv6
    else if (fam == AF_INET6)
    {
	InetAddr6 * ia = (InetAddr6*) &pia->pia_addr6;
	ia->sin6_family = fam;
	ia->sin6_port   = 0x0;
	inet_pton(fam,ips->IpAddress.String,&ia->sin6_addr.s6_addr);
	pia->pia_netmask[0] = 0;
	SLOG (net_sockaddrtop(ia));
    }
# endif 
    else
    {
	FAMERR("ipAddrStringToInAddr()");
	return -1; 
    }

    return 0;
}

static int
set_hwaddr(PNetIf * pif,IP_ADAPTER_INFO *pai)
{
    byte	*p;
    pif->hwaddr_len = pai->AddressLength;

    if (pif->hwaddr_len > IF_HWADDRLEN)
    {
	perr(E_FATAL,"set_hwaddr(): hardware address too long\n");
	return -1;
    }

    p = pai->Address;
    memcpy(pif->hwaddr,p,pif->hwaddr_len);

    pdbg(E_INFO,"Hardware address for device %s: "
		"%.2X:%.2X:%.2X:%.2X:%.2X:%.2X\n",
	 pif->pif_name,p[0],p[1],p[2],p[3],p[4],p[5],p[6]);

    return 0;
}

static int
get_num_ifaces( void )
{
    int num = 0;

    if ( GetNumberOfInterfaces( &num ) != NO_ERROR )
    	{ NETERR( "get_num_ifaces()" ); return -1; }

    return num;
}

static void
print_ip_address_list( IP_ADDR_STRING * paddr )
{
    printf(" PADDR: %X\n", XX(paddr));
    for ( ; paddr; paddr = paddr->Next )
    {
	printf(" ADDR: %s\n", paddr->IpAddress.String );
	printf(" MASK: %s\n", paddr->IpMask.String    );
	printf(" CTXT: %d\n", paddr->Context );
    }
}
static void
print_mibifrow( MIB_IFROW * mif )
{
    if (! mif )
	return ;

    printf(" MIF \n");
    printf(" name = %s\n", mif->wszName );
    printf(" index = %d\n", mif->dwIndex );
    printf(" type = %d\n", mif->dwType );
    printf(" mtu = %d\n", mif->dwMtu );
    printf(" speed = %d\n", mif->dwSpeed );
}

static int
get_interface_list( void )
{
    INTERFACE_INFO 	ifaceList[20],*pifinfo;
    pnet_ulong 		len;
    int			num_ifaces,i;
    SOCKET		sd;
    PNETIF		pif;

    sd = WSASocket( AF_INET, SOCK_DGRAM, 0, 0, 0, 0 );

    if ( sd == SOCKET_ERROR )
    {
	NETERR( "get_interface_list() ");
	return -1;
    }


    if ( WSAIoctl( sd, SIO_GET_INTERFACE_LIST, 0, 0, &ifaceList, 
    		   sizeof( ifaceList ) , &len, 0, 0 ) == SOCKET_ERROR )
    {
	NETERR( "get_interface_list() ");
	return -1;
    }

    num_ifaces = len / sizeof( INTERFACE_INFO );

    LLOG( num_ifaces );
    for ( i = 0; i < num_ifaces; i++ )
    {
	pifinfo = &ifaceList[ i ];

	for ( pif = pif_head; pif; pif = pif->next )
	{
	    if ( memcmp( &pif->pif_addr4.sin_addr,
	    		&((SOCKADDR_IN*)&pifinfo->iiAddress)->sin_addr, 4) ) 
	    {
		pif->pif_flags = pifinfo->iiFlags;
	    }

	}
    }

    return 0;
}
int
get_ifs_info( int fam )
{
    struct pnet_if *pif_curr;
    struct pnet_if **ppif;
    PNetIfAddr     **ppia;

    IP_ADAPTER_INFO *	buf;
    IP_ADAPTER_INFO *	pai_curr;
    ULONG		len = 0;
    int			err_code = 0;

    pnetInit( );

    if ( fam == AF_UNSPEC )
	fam = AF_INET;


    LLOG( get_num_ifaces() );

    /* Get size of buffer from kernel */
    buf = NULL;
    len = 0;

    if (GetAdaptersInfo((PIP_ADAPTER_INFO) buf, &len) != ERROR_BUFFER_OVERFLOW)
    {
	NETERR("get_ifs_info(): GetAdaptersInfo()");
	return -1;
    }

    LLOG(len);

    STDMALLOC(buf,len,-1);

    err_code = GetAdaptersInfo((PIP_ADAPTER_INFO) buf,&len);

    if (err_code != ERROR_SUCCESS)
    {
	win_if_error( err_code );
	STDFREE( buf );
	return -1;
    }
    
    LLOG( len );
    LLOG( sizeof( IP_ADAPTER_INFO ) );

    /* Initialize the head of the list */
    pif_curr = pif_head = NULL;
    ppif = &pif_head;

    pai_curr = (IP_ADAPTER_INFO*) buf;
    for ( ; pai_curr; pai_curr = pai_curr->Next ) 
    {
	PNetIfAddr *pia;
	MIB_IFROW   mif;

	SLOG( pai_curr->AdapterName );
	SLOG( pai_curr->Description );
	LLOG( pai_curr->AddressLength );
	XLOG( pai_curr->Next );

	/* if ( pai_curr->AdapterName[0] == 0 ) 
	sprintf( pai_curr->AdapterName, "if%lu", pai_curr->Index ); */

	if ( (pif_curr =
		net_get_if_by_name_bootstrap(fam,pai_curr->AdapterName,1)))
	{
	    pdbg(E_DBG1,"Interface %s seen already\n", pai_curr->AdapterName );

	    pia = NULL;

	    while ( (pia = net_if_get_next_address( pif_curr, pia )) )
		ppia = &pia->pia_next;
	}
	else
	{
	    STDMALLOC(pif_curr,sizeof(PNetIf),-1);

	    *ppif = pif_curr;
	    ppif = &pif_curr->next;
	    ppia = &pif_curr->pif_addrs;

	    pif_curr->pif_flags	= 0x0;

	    LLOG( fam );

	    DBG( print_ip_address_list( pai_curr->CurrentIpAddress ) ) ;
	    DBG( print_ip_address_list( &pai_curr->IpAddressList  ) ) ;
	    DBG( print_ip_address_list( &pai_curr->GatewayList   ) ) ;

	    mif.dwIndex = pai_curr->Index;

	    if ( NO_ERROR != GetIfEntry( &mif ) )
		perr(E_WARN,"Error reading MIB for iface %d\n",
			     pai_curr->Index );
	    else
	    {
		pif_curr->pif_mtu = mif.dwMtu;
	    }
	    /* interface name, index and hardware address */

	    /* strncpy(pif_curr->pif_name,pai_curr->AdapterName,IFNAMSIZ); */
	    /* pif_curr->pif_name[strlen(pif_curr->pif_name)] = 0; */

	    sprintf( pif_curr->pif_name, "if%lu", pai_curr->Index );
	    pif_curr->pif_index = pai_curr->Index;
	    pif_curr->pif_type= pai_curr->Type;

	    SLOG(pif_curr->pif_name);
	    pdbg(E_INFO,"Device index %d\n",pif_curr->pif_index); 
	    pdbg(E_INFO,"Device type  %d\n",pif_curr->pif_type);
	    LLOG( fam );

	    /* Copy hardware address into pif_curr */
	    set_hwaddr(pif_curr,pai_curr);
	    pif_curr->pif_pflags |= PIF_HAS_HWADDR;
	}

	STDMALLOC( pia, sizeof(PNetIfAddr), -1 );
	*ppia = pia;
	ppia  = &pia->pia_next;

	pia->pia_family = fam;
	pia->pia_if = pif_curr;

   	/* Unicast address */
	ipAddrStringToInAddr( fam, &pai_curr->IpAddressList, pia );
	pif_curr->pif_pflags |= PIF_HAS_ADDR;
	pif_curr->pif_pflags |= PIF_HAS_NETMASK;
    }
    STDFREE(buf);
    get_interface_list();

    return 0;
}

# if defined STDWinXPPro || defined STDWinNET_Server  /* { */

static int
set_hwaddr_x(PNetIf * pif,IP_ADAPTER_ADDRESSES *paa)
{
    byte	*p;
    pif->hwaddr_len = (byte)paa->PhysicalAddressLength;

    if (pif->hwaddr_len > IF_HWADDRLEN)
    {
	perr(E_FATAL,"set_hwaddr(): hardware address too long\n");
	return -1;
    }

    p = paa->PhysicalAddress;
    memcpy(pif->hwaddr,p,pif->hwaddr_len);

    pdbg(E_INFO,"Hardware address for device %s: "
		"%.2X:%.2X:%.2X:%.2X:%.2X:%.2X\n",
	 pif->pif_name,p[0],p[1],p[2],p[3],p[4],p[5],p[6]);

    return 0;
}

PNetIf*
get_ifs_info_x( int fam )
{
    struct pnet_if *pif_head;
    struct pnet_if *pif_curr;
    struct pnet_if **ppif;

    IP_ADAPTER_ADDRESSES *	buf;
    IP_ADAPTER_ADDRESSES *	paa;
    ULONG			len = 0;
    int			slen = 0;
    int			err_code = 0;

    if (fam == AF_INET)
    	slen = sizeof(InetAddr);
    else if (fam == AF_INET6 || fam == AF_UNSPEC )
    	slen = sizeof(InetAddr6);
    else
    {
    	FAMERR("get_ifs_info()");
	return NULL;
    }

    LLOG( get_num_ifaces() );

    /* Get size of buffer from kernel */
    buf = NULL;
    len = 0;

    if ( GetAdaptersAddresses( fam,
    	 GAA_FLAG_SKIP_ANYCAST | GAA_FLAG_SKIP_MULTICAST | GAA_FLAG_SKIP_DNS_SERVER,
	 NULL, buf, &len ) !=   ERROR_BUFFER_OVERFLOW )
    {
	NETERR("get_ifs_info(): GetAdaptersAdresses()");
	return NULL;
    }

    LLOG(len);

    STDMALLOC(buf,len,NULL);

    /* Now, fetch adapters addresses */

    if ( GetAdaptersAddresses( fam,
    	 GAA_FLAG_SKIP_ANYCAST | GAA_FLAG_SKIP_MULTICAST | GAA_FLAG_SKIP_DNS_SERVER,
	 NULL, buf, &len ) !=   ERROR_BUFFER_OVERFLOW )
    {
	NETERR("get_ifs_info(): GetAdaptersAdresses()");
	return NULL;
    }

    if (err_code != ERROR_SUCCESS)
    {
	win_if_error( err_code );
	STDFREE( buf );
	return NULL;
    }
    
    LLOG( len );

    /* Initialize the head of the list */
    pif_curr = pif_head = NULL;
    ppif = &pif_head;

    paa = (IP_ADAPTER_ADDRESSES*) buf;
    for ( ; paa; paa = paa->Next ) 
    {
	PIP_ADAPTER_UNICAST_ADDRESS uniaddr;

	STDMALLOC(pif_curr,sizeof(PNetIf),NULL);

	*ppif = pif_curr;
	ppif = &pif_curr->next;

	pif_curr->pif_fam 	= fam;
	pif_curr->pif_flags	= 0x0;

	LLOG( fam );
	SLOG( paa->AdapterName );
	SLOG( paa->Description );
	LLOG( paa->FriendlyName );

	/* interface name, index and hardware address */
	strncpy(pif_curr->pif_name,paa->AdapterName,IFNAMSIZ);
	pif_curr->pif_name[strlen(pif_curr->pif_name)] = 0;
	pif_curr->pif_index = paa->IfIndex;
	pif_curr->pif_type= paa->IfType;

	SLOG(pif_curr->pif_name);

	/* Copy hardware address into pif_curr */
	set_hwaddr_x(pif_curr,paa);
	pif_curr->pif_pflags |= PIF_HAS_HWADDR;

   	/* Unicast address */
	uniaddr = paa->FirstUnicastAddress;
	memcpy( &pif_curr->pif_addr, uniaddr->Address.lpSockaddr, len );

	pif_curr->pif_pflags |= PIF_HAS_ADDR;
	pif_curr->pif_pflags |= PIF_HAS_NETMASK;
    }
    STDFREE(buf);

    return pif_head;
}
# endif						/* } */

const char *
if_get_description( PNetIf * pif )
{
    switch ( pif->pif_type )
    {
    case IF_TYPE_ETHERNET_CSMACD:	return ("Ethernet 10/100MB");
    case IF_TYPE_IS088023_CSMACD: 	return ("ISO88023 CSMACD");
    case IF_TYPE_ISO88024_TOKENBUS:	return ("ISO88024 Token Bus");
    case IF_TYPE_ISO88025_TOKENRING:	return ("ISO88025 Token Ring");
    case IF_TYPE_HYPERCHANNEL:		return ("HyperChannel");
    case IF_TYPE_FDDI:			return ("FDDI");
    case IF_TYPE_PPP:			return ("PPP");
    case IF_TYPE_SOFTWARE_LOOPBACK:	return ("Loopback");
    case IF_TYPE_ETHERNET_3MBIT:	return ("Ethernet 3MB");
    case IF_TYPE_SLIP:			return ("SLIP");
    case IF_TYPE_FRAMERELAY:		return ("Frame Relay");
    case IF_TYPE_PARA:			return ("Parallel Port");
    case IF_TYPE_ARCNET:		return ("ACRnet");
    case IF_TYPE_ARCNET_PLUS:		return ("ACRnet Plus");
    case IF_TYPE_ATM:			return ("ATM");

    case IF_TYPE_HIPPI:
    case IF_TYPE_HIPPIINTERFACE:	return ("HiPPi");
    case IF_TYPE_FASTETHER:		return ("Fast Ethernet (100BaseT)");
    case IF_TYPE_ISDN:			return ("ISDN/X.25");
    case IF_TYPE_ISDN_S:		return ("ISDN S/T");
    case IF_TYPE_ISDN_U:		return ("ISDN U");
    case IF_TYPE_ADSL:			return ("ADSL");

    case IF_TYPE_TUNNEL:		return ("Tunnel Encapsulation");
    }
    return ("Unknown");
}

# ifndef IFF_DEBUG
#   define IFF_DEBUG	0
# endif

# ifndef IFF_SMART
#   define IFF_SMART	0
# endif

# ifndef IFF_RUNNING
#   define IFF_RUNNING	IFF_UP
# endif

# ifndef IFF_NOARP
#   define IFF_NOARP	0
# endif

# ifndef IFF_PROMISC
#   define IFF_PROMISC	0
# endif

# ifndef IFF_SIMPLEX
#   define IFF_SIMPLEX	0
# endif



int pnetIfIsUp( PNETIF pif ) { return pif->pif_flags & IFF_UP; }
int pnetIfIsBroadcast( PNETIF pif ) { return pif->pif_flags & IFF_BROADCAST; }
int pnetIfIsDebug( PNETIF pif ) { return pif->pif_flags & IFF_DEBUG; }
int pnetIfIsLoopback( PNETIF pif ) { return pif->pif_flags & IFF_LOOPBACK; }
int pnetIfIsPointopoint( PNETIF pif ) {return pif->pif_flags&IFF_POINTTOPOINT;}
int pnetIfIsMulticast( PNETIF pif ) { return pif->pif_flags & IFF_MULTICAST; }

int pnetIfIsSmart( PNETIF pif ) { return pif->pif_flags & IFF_SMART; }
int pnetIfIsRunning( PNETIF pif ) { return pif->pif_flags & IFF_RUNNING; }

int pnetIfIsNoArp( PNETIF pif ) { return pif->pif_flags & IFF_NOARP; }
int pnetIfIsPromiscuous( PNETIF pif ) { return pif->pif_flags & IFF_PROMISC; }
int pnetIfIsSimplex( PNETIF pif ) { return pif->pif_flags & IFF_SIMPLEX; }

# endif						/* } */
