/*
 * This file is part of
 *
 * PNET6: a Portable Network Library
 *
 * PNET6 is Copyright (c) 2002, Peter Bozarov
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Peter Bozarov.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * $Id: if-ifconf.c,v 1.28 2002/10/07 06:18:25 kingofgib Exp $
 */

/*----------------------------------------------------------------------*
 * filename:            if-ifconf.c
 * created on:          Wed Aug  7 21:27:55 CEST 2002
 * created by:          peter
 * project:             Portable Network Library
 *----------------------------------------------------------------------*/

/*----------------------------------------------------------------------*/
/* Interface information.						*/
/* We are using IFCONF and IFREQ.					*/
/*----------------------------------------------------------------------*/

# include "../local.h"

# ifdef PNET_USE_IFCONF						/* { */

# include <sys/ioctl.h>
/* # include <net/if.h> */

# ifdef HAVE_NET_IF_ARP_H
# include <net/if_arp.h>
# endif 

# include "iflocal.h"

# ifndef IFF_POINTOPOINT
#   define IFF_POINTOPOINT	0
# endif

/*----------------------------------------------------------------------*/
/* Get hardware parameters for an interface: MAC address, interface 	*/
/* type and interface index 						*/
/*----------------------------------------------------------------------*/

# ifdef PNET_HPUX			/* {{ */

# include <netio.h>
/*
 * On the HPUX, open the driver device, /dev/lan0 or whatever, and read
 * its physical address from there.
 */
static int
if_get_link_address( int sd, PNetIf * pif )
{
    int                 fd;
    struct fis          fis;
    char                dev[sizeof( "/dev/" ) + IFNAMSIZ ];

    /* Skip loopback or pointopoint devices */

    if ( pif->pif_flags & IFF_LOOPBACK || pif->pif_flags & IFF_POINTOPOINT )
        return 0;

    sprintf( dev, "/dev/%s", pif->pif_name );

    SLOG( dev );

    if ( (fd = open( dev, O_RDONLY )) == -1 )
    {
	perr( E_FATAL,"if_get_link_address(): open %s: %s\n",dev,SYSERR() );
	return -1;
    }

    fis.reqtype = LOCAL_ADDRESS;

    if ( if_ioctl( fd, NETSTAT, &fis ) )
    {
	perr( E_FATAL,"if_get_link_address(): ioctl %s: %s\n",dev,SYSERR() );
	return -1;
    }

    LLOG( fis.vtype );

    if ( fis.vtype == 6 )           /* Ethernet */
    {
	pif->pif_type = fis.vtype;
	pif->hwaddr_len = 6;
	memcpy( pif->hwaddr, fis.value.s, 6 );
	pif->pif_pflags |= PIF_HAS_HWADDR;
    }

    return 0;
}
# else					/* }{ */

static int
if_get_link_address( int sd, PNetIf *pif )
{
    struct ifreq	ifr;
    byte *		b = 0;
    int 		i,alen = 0;

# ifdef HAVE_STRUCT_IFDEVEA
    struct ifdevea	ifdev;
# endif

    strncpy( ifr.ifr_name, pif->pif_name, IFNAMSIZ );

# ifdef SIOCGIFHWADDR		/* { */
    if ( if_ioctl( sd, SIOCGIFHWADDR, &ifr ) )
	{ NETERR("ioctl(...,SIOCGIFHWADDR,...)"); return -1; }

# ifdef HAVE_IFREQ_IFR_HWADDR
    alen = 6;
    b 	 = (byte*) ifr.ifr_hwaddr.sa_data;
    pif->pif_type = ifr.ifr_hwaddr.sa_family;
# endif

# endif 			/* } */

# ifdef SIOCGENADDR		/* { */
    if ( if_ioctl( sd, SIOCGENADDR, &ifr ) )
	{ NETERR("ioctl(...,SIOCGENADDR,...)"); return -1; }

    alen = 6;
    b = (byte*) ifr.ifr_enaddr;
# endif 			/* } */

# ifdef SIOCGIFTYPE

    strncpy( ifr.ifr_name, pif->pif_name, IFNAMSIZ );

    if (ioctl( sd,  SIOCGIFTYPE, &ifr ) < 0 )
	{ NETERR("ioctl(...,SIOCGIFTYPE,...)"); return -1; }

    pif->pif_type = ifr.ifr_value;
# endif

# ifdef SIOCRPHYSADDR		/* { */	/* Tru64 and who else? */

    if ( pif->pif_flags & IFF_POINTOPOINT || pif->pif_flags & IFF_LOOPBACK )
	return 0;

    strncpy( ifdev.ifr_name, pif->pif_name, IFNAMSIZ );

    if (ioctl( sd,  SIOCRPHYSADDR, &ifdev ) < 0 )
	{ NETERR("ioctl(...,SIOCRPHYSADDR,...)"); return -1; }

    alen = 6;

    b = (byte*) ifdev.default_pa;
# endif				/* } */


    /* Skip addresses of all 0's */

    for (i = 0; i < alen && !b[i]; i++) /* Empty */;

    if (i == alen)		/* Address is empty */
    {
	perr(E_DBG4,"if_get_hwparams(): hardware address for %s is empty.\n",
		     pif->pif_name);
    }
    else 
    {
	pif->hwaddr_len = alen;
	memcpy( pif->hwaddr, b, alen );
	pif->pif_pflags |= PIF_HAS_HWADDR;
    }

    return 0;
}

# endif					/* }} */

static int
if_get_mtu( int sd, PNetIf* pif )
{
    struct ifreq	ifr;

    strncpy( ifr.ifr_name, pif->pif_name, IFNAMSIZ );

# ifdef SIOCGIFMTU
    if ( if_ioctl( sd, SIOCGIFMTU, &ifr) )
	{ FATALERR("ioctl(...,SIOCGIFMTU,...)"); return -1; }
# endif

# ifdef SIOCRIPMTU
    if ( if_ioctl( sd, SIOCRIPMTU, &ifr) )
	{ FATALERR("ioctl(...,SIOCRIPMTU,...)"); return -1; }
    pif->pif_mtu = ifr.ifr_metric;
# endif

# ifdef HAVE_IFREQ_IFR_MTU
    pif->pif_mtu = ifr.ifr_mtu;
# else
    pif->pif_mtu = ifr.ifr_metric;
# endif

    return 0;
}
static int
if_get_index( int sd, PNetIf *pif )
{
    struct ifreq	ifr;

    strncpy( ifr.ifr_name, pif->pif_name, IFNAMSIZ );

# ifdef SIOCGIFINDEX
    if ( if_ioctl( sd, SIOCGIFINDEX, &ifr) )
	{ FATALERR("ioctl(...,SIOCGIFINDEX,...)"); return -1; }
# endif

# ifdef HAVE_IFREQ_IFR_IFINDEX
    pif->pif_index = ifr.ifr_ifindex;
# endif

# ifdef HAVE_IFREQ_IFR_INDEX
    pif->pif_index = ifr.ifr_index;
# endif

    return 0;
}

static int
if_get_hwparams( PNetIf *pif )
{
    int sd;

    if ( (sd = socket( AF_INET, SOCK_DGRAM, 0 ) ) < 0 )
	{ NETERR("if_get_hwparams"); return -1; }

    if_get_index( sd, pif );
    if_get_mtu( sd, pif );
    if_get_link_address( sd, pif );

    close( sd );

    return 0;
}
/*----------------------------------------------------------------------*/
/* Set an interface's netmask						*/
/*									*/
/* NOTE: pia->pia_family must be set prior to calling this. 		*/
/*----------------------------------------------------------------------*/
static int
if_set_netmask( PNetIfAddr *pia, SockAddr *sa )
{
    if ( sa->sa_family == AF_INET )
	memcpy( pia->pia_netmask + 1,
		&((InetAddr*)sa)->sin_addr.s_addr, sizeof( struct in_addr ) );

    /* For IPv6 addresses, prefix length and scope id are done later on */

    if ( pia->pia_family == AF_INET )
	pia->pia_netmask[0] = 1; /* Trick flag to know if set 	*/

    return 0;
}

# ifdef STDLinux				/* { */

static int char2hex( int c )
{
    c = toupper( c );
    if ( c >= '0' && c <= '9' )
    	return c - '0';
    if ( c >= 'A' && c <= 'F' )
        return c - 'A' + 10;

    return c;
}

/*
 * Read info about a IPv6 enable interface from the /proc.
 * Format of the proc file is:
 * 32 bytes address, interface index, prefix length, scope, flags, name
 */
static PNetIfAddr*
get_ipv6_proc_info( PNetIf ** ppif )
{
    FILE *		fp;
    char 		buf[128];
    PNetIfAddr * 	addr = NULL;
    PNetIfAddr **	paddr= NULL;
    PNetIf*		pif = NULL;

    DBG( dbg("get_ipv6_proc_info(  )\n" ) );

    if ( ! ( fp = fopen( "/proc/net/if_inet6", "r" ) ) )
    	{ XLOG( fp ); return NULL; }

    while ( fgets( buf, sizeof(buf), fp ) )
    {
	int 		i;
	pnet_uint 	idx;
	char		s[3];
	byte * 		b;
	char * 		p = buf + strlen( buf ) - 1;

	*p = 0;
	while ( p &&  isspace( *p ) )
	    p--;
	while ( p && !isspace( *p ) )
	    p--;
	p++;		/* We got the interface name */

	SLOG( p ); 

	if ( ! (pif = net_get_if_by_name_bootstrap( AF_UNSPEC, p, 1 ) ) )
	{
	    struct ifreq	ifr;
	    int 		sd;

	    STDMALLOC( pif, sizeof(PNetIf), NULL );
	    strncpy( pif->pif_name, p, IFNAMSIZ );

	    strncpy( ifr.ifr_name, p, IFNAMSIZ );

	    /* Get interface flags */
	    sd = socket( AF_INET6, SOCK_DGRAM, 0 );

	    if ( if_ioctl( sd, SIOCGIFFLAGS, &ifr) )
		NETERR("get_ipv6_proc_info()");
	    else
		pif->pif_flags = ifr.ifr_flags;

	    if_get_hwparams( pif );

	    if ( sd > 0 )
		close( sd );
	    *ppif = pif;
	    ppif  = &pif->next;
	}

	paddr  = &pif->pif_addrs;
	addr   = pif->pif_addrs;

	/* Find end of address list */
	do
	{
	    if ( addr )
		paddr = &addr->pia_next;
	    addr = net_if_get_next_address( pif, addr );
	} while ( addr );

	STDMALLOC( addr, sizeof(PNetIfAddr), NULL );

	/* Link up to previous address	*/
	*paddr = addr;
	paddr = &addr->pia_next;

	addr->pia_family = AF_INET6;
	addr->pia_next = NULL;
	addr->pia_if   = pif;

	/* Set the address */
	p = buf;
	b = (byte*)&addr->pia_addr6.sin6_addr;
	for ( i = 0; i < 16; i++ )
	    b[i] = (byte) (char2hex(*p++)<<4 | char2hex(*p++));

	/* Read prefix length and scope id */
        sscanf( p,"%x %x %c%c ",
		&idx,(pnet_uint*)&addr->pia_prefixlen,&s[1],&s[0]);
	s[2] = 0;

# ifdef HAVE_SOCKADDR_IN6_SIN6_SCOPE_ID
	addr->pia_addr6.sin6_scope_id = atoi( s );
	if ( IN6_IS_ADDR_LINKLOCAL( &addr->pia_addr6.sin6_addr ) )
	{
	    if ( addr->pia_addr6.sin6_scope_id == 0x0 )
		addr->pia_addr6.sin6_scope_id = pif->pif_index; 
	}
	LLOG( addr->pia_addr6.sin6_scope_id );
# endif
    }

    fclose( fp );
    return addr;
}
# endif						/* } */

/*----------------------------------------------------------------------*/
/* Get the full list of intefaces from the kernel. We read all of them	*/
/* then extract additional information using ioctl() calls, and stored  */
/* the whole lot as a linked list of pnet_if structures.		*/
/*----------------------------------------------------------------------*/

int
get_ifs_info( int fam )
{
    byte *		pnext;
    byte *		pend;
    struct pnet_if *	pif_curr;
    struct pnet_if **	ppif;
    struct ifconf	ifc;
    int 		sd;
    int			cnt = -1;
    PNetIfAddr **	ppia= 0;

    DBG( dbg( "get_ifs_info( fam = %d )\n" ) );

    /* Free old interfaces info first */
    if (pif_head)
	free_ifs_info(pif_head);

    if ( (sd = socket( AF_INET, SOCK_STREAM, 0 ) ) < 0)
	{ NETERR("get_ifs_info()"); return -1; }

# ifdef SIOCGIFCOUNT
    if ( if_ioctl( sd, SIOCGIFCOUNT, &cnt ) )
    {
	pdbg(E_WARN,"get_ifs_info(): SIOCGIFCOUNT not supported.\n");
	cnt = -1;
    }
# endif

    do
    { 
	int	old_len;

	if ( cnt == -1 )
	{
	    old_len = 0;
	    cnt = 10; /* Fall back to old style interface fetching */
	}
	else
	{
	    /* Don't need to guess, we know how many interfaces 	*/
	    /* there are.						*/
	    old_len = cnt * sizeof( struct ifreq );
	}

	/* If cnt was not set by SIOCGIFCOUNT, need to do some guessing.*/
	/* Allocate "enough" space for the buffer.			*/
	/* Some systems fail if the buffer space is not sufficient, or 	*/
	/* else truncate the buffer.					*/
	/* The only way to do it right is to keep reallocating memory	*/
	/* until the new and old buffer sizes are equal.		*/

	ifc.ifc_len = cnt * sizeof(struct ifreq);

	while ( 1 )
	{
	    STDMALLOC(ifc.ifc_buf, ifc.ifc_len, -1);

	    pdbg(E_DBG4,"Trying ioctl(%d,SIOCGIFCONF,...): "
		    "oldbuflen = %d, buflen = %d\n",sd,old_len,ifc.ifc_len);

	    if ( if_ioctl( sd, SIOCGIFCONF, &ifc ) )
		if ( errno != EINVAL || old_len != 0 )
		    { NETERR("get_ifs_info()"); close(sd); return -1; }

	    if ( old_len == ifc.ifc_len )
		break;

	    old_len      = ifc.ifc_len;
	    ifc.ifc_len += cnt * sizeof(struct ifreq);

	    STDFREE( ifc.ifc_buf );
	}
	/* "Count" the configured interfaces */

	cnt = ifc.ifc_len / sizeof( struct ifreq );

    } while ( 0 );

    pdbg(E_INFO,"Found %d configured interfaces\n", cnt );

    if ( cnt < 1 )
	return -1;

    pnext = (byte*)ifc.ifc_buf;
    pend  =  pnext + ifc.ifc_len;

    pif_curr = pif_head = NULL;
    ppif = &pif_head;

    while ( pnext < pend )
    {
	struct ifreq *	ifr = (struct ifreq*) pnext;
	struct ifreq	ifrtmp;
	pnet_uint	len = 0;
	char *		p;
	PNetIfAddr *    pia;

	/* It can happen that the actual struct ifreq is larger/smaller */
	/* than the address stored in it.				*/

	/* Get the actual length of the IP address stored in the ifr 	*/

# ifdef AF_LINK			/* { */
	if ( ifr->ifr_addr.sa_family == AF_LINK )
	{
# ifdef HAVE_STRUCT_SOCKADDR_DL
	    struct sockaddr_dl* 	sdl;
	    sdl = (struct sockaddr_dl*) &ifr->ifr_addr;
	    len = sdl->sdl_len;
# endif
	}
	else
# endif				/* } */
	    len = addr_len( ifr->ifr_addr.sa_family );

	/* Find start of next struct ifreq 				*/
	pnext += sizeof( ifr->ifr_name ) + PMAX( len , sizeof( ifr->ifr_ifru ) );

	/* Skip family if not matching requested family */

	if (fam != AF_UNSPEC && fam != ifr->ifr_addr.sa_family)
	    continue;

	/* It can be that the current entry has a name that we've seen	*/
	/* before, but which contains a new address (or what?)		*/

	if ( (pif_curr = net_get_if_by_name_bootstrap(fam, ifr->ifr_name,1)))
	{
	    pdbg(E_DBG1,"Interface %s seen already\n", ifr->ifr_name);

	    pia = NULL;

	    while ( (pia = net_if_get_next_address( pif_curr, pia )) )
		ppia = &pia->pia_next;
	}
	else
	{
	    /* Allocate a new pif struct and link up the pointers */

	    STDMALLOC(pif_curr,sizeof(PNetIf),-1);
	    *ppif = pif_curr;
	    ppif  = &pif_curr->next;
	    ppia  = &pif_curr->pif_addrs;

	    /* Save interface name and family. */
	    strncpy(pif_curr->pif_name,ifr->ifr_name,IFNAMSIZ);

	    /* Remove semi-colon */
	    if ( (p = strrchr( pif_curr->pif_name, ':' )) )
		*p = 0;

	    ifrtmp = *ifr;

	    /* Get interface flags */
	    if ( if_ioctl( sd, SIOCGIFFLAGS, &ifrtmp ) )
		{ NETERR("get_ifs_info()"); }

	    pif_curr->pif_flags = ifrtmp.ifr_flags;

	    /* Get interface hardware address type and index */

	    if ( if_get_hwparams( pif_curr ) )
	    {
		perr(E_FATAL,"Error reading hardware configuration for "
			     "interface %s\n",pif_curr->pif_name);
	    }

	    /* Get point-to-point destination address */

	    if (pif_curr->pif_flags & IFF_POINTOPOINT)
	    {
# ifdef SIOCGIFDSTADDR	/* { */
		if ( if_ioctl(sd,SIOCGIFDSTADDR,&ifrtmp) == 0 )
		{
		    memcpy(&pif_curr->_pif_daddr,&ifrtmp.ifr_dstaddr,len);
		    pif_curr->pif_pflags |= PIF_HAS_DADDR;
		}
# endif 		/* } */
	    }
	}

	/* Now, finally, get the unicast address, and its netmask 	*/
	/* Allocate an address structure, and link it up to the others	*/
	/* for this interface 						*/


	strncpy( ifrtmp.ifr_name, pif_curr->pif_name, IFNAMSIZ);

	if ( if_ioctl(sd, SIOCGIFADDR, &ifrtmp) != 0 )
	    { NETERR( "pif_curr->pif_name"); continue; }

	STDMALLOC( pia, sizeof( PNetIfAddr ), -1 );

	*ppia = pia;
	pia->pia_if = pif_curr;

	{
	    pia->pia_family     = ifrtmp.ifr_addr.sa_family;

	    memcpy( &pia->pia_address, &ifrtmp.ifr_addr, len);

	    pif_curr->pif_pflags |= PIF_HAS_ADDR;
	}


	/* Get broadcast address */
	strncpy( ifrtmp.ifr_name, pif_curr->pif_name, IFNAMSIZ);
	if (pif_curr->pif_flags & IFF_BROADCAST)
	{
	    if ( if_ioctl(sd,SIOCGIFBRDADDR,&ifrtmp) == 0 )
	    {
		memcpy( &pia->pia_broadaddr[1],
			&((InetAddr*)&ifrtmp.ifr_broadaddr)->sin_addr, 
			sizeof( struct in_addr ) );
		pif_curr->pif_pflags |= PIF_HAS_BADDR;
		pif_curr->pif_addrs->pia_broadaddr[0] = 1;
	    }
	    else
		NETERR( "pif_curr->pif_name");
	}

	strncpy( ifrtmp.ifr_name, pif_curr->pif_name, IFNAMSIZ);

	if ( if_ioctl(sd, SIOCGIFNETMASK, &ifrtmp) == 0 )
	    if_set_netmask( pif_curr->pif_addrs, &ifrtmp.ifr_addr );
	else
	    NETERR( "gif_conf()");

	if_debug( pif_curr );
    }

# ifdef STDLinux
    if ( fam != AF_INET )
	get_ipv6_proc_info( ppif );
# endif 

    close(sd);
    STDFREE(ifc.ifc_buf);

    return 0;
}


# if defined STDLinux|| defined PNET_HPUX || defined PNET_CYGWIN /* { */

const char *
if_get_description( PNetIf * pif )
{
# ifdef STDLinux 
    switch ( pif->pif_type )
    {
    case ARPHRD_ETHER:	return ("Ethernet 10/100MB");
    case ARPHRD_EETHER: return ("Experimental Ethernet");
    case ARPHRD_AX25:	return ("AX.25 Level 2");
    case ARPHRD_PRONET: return ("PROnet token ring");
    case ARPHRD_CHAOS:  return ("Chaosnet");
    case ARPHRD_IEEE802: return ("IEEE 802.2 Ethernet/TR/TB");
    case ARPHRD_ARCNET: return ("ARCnet");
    case ARPHRD_APPLETLK: return ("APPLEtalk");
    case ARPHRD_DLCI: return ("Frame Relay DLCI");
    case ARPHRD_ATM:    return ("ATM");
    case ARPHRD_METRICOM: return ("Metricom STRIP");

    case ARPHRD_SLIP:	return ("SLIP");
    case ARPHRD_CSLIP:  return ("CSLIP");
    case ARPHRD_SLIP6:	return ("SLIP6");
    case ARPHRD_CSLIP6:  return ("CSLIP6");
    case ARPHRD_PPP:	return ("PPP");

    case ARPHRD_TUNNEL:	return ("IP-IP tunnel");
    case ARPHRD_TUNNEL6: return ("IP-IP6 tunnel");
    case ARPHRD_LOOPBACK: return ("Loopback");
    case ARPHRD_SIT:	return ("IPv6-in-IPv4");
    case ARPHRD_IPDDP:  return ("IP-in-DDP tunnel");
    case ARPHRD_HIPPI:	return ("HiPPi");
    }
# endif
    return ("Unknown");
}
# endif

# ifndef IFF_SMART
#   define IFF_SMART	0
# endif

# ifndef IFF_SIMPLEX
#   define IFF_SIMPLEX	0
# endif

# ifndef IFF_MULTICAST
#   define IFF_MULTICAST	0
# endif

# ifndef IFF_DEBUG
#   define IFF_DEBUG		0
# endif

# ifndef IFF_NOARP
#   define IFF_NOARP		0
# endif


int pnetIfIsUp( PNETIF pif ) { return pif->pif_flags & IFF_UP; }
int pnetIfIsBroadcast( PNETIF pif ) { return pif->pif_flags & IFF_BROADCAST; }
int pnetIfIsDebug( PNETIF pif ) { return pif->pif_flags & IFF_DEBUG; }
int pnetIfIsLoopback( PNETIF pif ) { return pif->pif_flags & IFF_LOOPBACK; }
int pnetIfIsPointopoint( PNETIF pif ) { return pif->pif_flags & IFF_POINTOPOINT; }
int pnetIfIsSmart( PNETIF pif ) { return pif->pif_flags & IFF_SMART; }
int pnetIfIsRunning( PNETIF pif ) { return pif->pif_flags & IFF_RUNNING; }
int pnetIfIsNoArp( PNETIF pif ) { return pif->pif_flags & IFF_NOARP; }
int pnetIfIsPromiscuous( PNETIF pif ) { return pif->pif_flags & IFF_PROMISC; }
int pnetIfIsSimplex( PNETIF pif ) { return pif->pif_flags & IFF_SIMPLEX; }
int pnetIfIsMulticast( PNETIF pif ) { return pif->pif_flags & IFF_MULTICAST; }
# endif								/* } */
