#include <cstdlib>

#include <fstream>
#include <list>
#include <string>
#include <sstream>
#include <vector>

#include "ldastoolsal/MemChecker.hh"
#include "ldastoolsal/CommandLineOptions.hh"
#include "ldastoolsal/gpstime.hh"
#include "ldastoolsal/MkDir.hh"
#include "ldastoolsal/regexmatch.hh"
#include "ldastoolsal/Sed.hh"

#include "genericAPI/TCL.hh"
#include "genericAPI/swigexception.hh"
#include "genericAPI/Logging.hh"
#include "genericAPI/LogText.hh"

#include "frameAPI/createRDS.hh"

#include "diskcacheAPI/MetaCommands.hh"
#include "diskcacheAPI/ServerInterface.hh"

/*
 * ./createRDS.exe \
 * --framequery "{ { H1_R H {} 1051388672-1051388927 \
 *                   Chan(H0:PEM-EX_DUST_VEA1_300NM_PCF,H0:PEM-EX_DUST_VEA1_500NM_PCF, ... } }"
 * --usertype { PEM_RDS_A6 \
 * --outputdir /archive/frames/A6/PEM_RDS/LHO/H-PEM_RDS_A6-10513
 * --usejobdirs 0
 * --compressiontype zero_suppress_otherwise_gzip
 * --compressionlevel 1
 * --filechecksum 1
 * --frametimecheck 1
 * --framedatavalid 0
 * --framesperfile 1
 * --secperframe 256
 * --allowshortframes 1
 * --generatechecksum 1
 * --fillmissingdatavalid 0
 * --md5sumregexp "s,frames,meta/frames,i"
 */

using LDASTools::AL::MemChecker;
using LDASTools::Cmd::MkDir;
using LDASTools::Cmd::Sed;

using GenericAPI::queueLogEntry;
using GenericAPI::IsLogging;
using GenericAPI::LogEntryGroup_type;

typedef std::vector< std::string > filenames_type;

typedef LDASTools::AL::CommandLineOptions CommandLineOptions;
typedef CommandLineOptions::Option Option;
typedef CommandLineOptions::OptionSet OptionSet;

using FrameAPI::createRDSSet;

typedef diskCache::MetaCommand::ClientServerInterface::ServerInfo ServerInfo;
typedef diskCache::ServerInterface ServerInterface;

#define POSITION_DESC_TCL 		0
#define POSITION_IFO_TCL 		1
#define POSITION_TIME_RANGE_TCL 	3
#define POSITION_CHANNEL_LIST_TCL 	4
#define POSITION_QUERY_LIST_SIZE_TCL	5

static inline void
depart( int ExitCode )
{
  exit( ExitCode );
}

class query
{
public:
  typedef ::FrameAPI::RDS::channel_container_type channel_container_type;

  query( const std::string );

  std::string			ifo;
  std::string			desc;
  LDASTools::AL::GPSTime		start;
  LDASTools::AL::GPSTime		end;
  std::string			ext;
  channel_container_type	channels;
};

class CommandLine
  : public ServerInfo,
    public FrameAPI::RDS::FileOptions,
    protected CommandLineOptions
{
public:
  using CommandLineOptions::empty;
  using CommandLineOptions::Pop;
  typedef CommandLineOptions::option_type option_type;

  typedef FrameAPI::RDS::Options::seconds_per_frame_type seconds_per_frame_type;

  typedef std::string diskcache_host_type;
  typedef INT_4U diskcache_port_type;
  typedef query frame_query_type;
  typedef std::vector< frame_query_type > frame_queries_type;

  CommandLine( int ArgC, char** ArgV );

  const filenames_type& Filenames( ) const;

  const frame_queries_type& FrameQueries( ) const;

  void Usage( int ExitValue ) const;

  void operator()( );

private:
  enum option_types {
    OPT_ALLOW_SHORT_FRAMES,
    OPT_DESCRIPTION,
    OPT_LOG_DIRECTORY,
    OPT_LOG_DEBUG_LEVEL,
    OPT_DIRECTORY_OUTPUT_FRAMES,
    OPT_DIRECTORY_OUTPUT_MD5SUM,
    OPT_DIRECTORY_OUTPUT_MD5SUM_REGEXP,
    OPT_DISKCACHE_HOST,
    OPT_DISKCACHE_PORT,
    OPT_FILL_MISSING_DATA_VALID_ARRAY,
    OPT_FRAME_COMPRESSION_LEVEL,
    OPT_FRAME_COMPRESSION_METHOD,
    OPT_FRAME_FILES,
    OPT_FRAME_QUERY,
    OPT_FRAMES_PER_FILE,
    OPT_GENERATE_FRAME_CHECKSUM,
    OPT_HELP,
    OPT_NO_HISTORY,
    OPT_SECONDS_PER_FRAME,
    OPT_VERIFY_CHECKSUM_OF_FRAME,
    OPT_VERIFY_CHECKSUM_OF_STREAM,
    OPT_VERIFY_DATA_VALID,
    OPT_VERIFY_FILENAME_METADATA,
    OPT_VERIFY_TIME_RANGE,
    OPT_WITHOUT_HISTORY_RECORD
  };

  OptionSet			options;

  filenames_type		filenames;

  frame_queries_type		frame_queries;

  void parse_frame_files( const std::string& Filenames );

  void parse_frame_query( const std::string& Query );
};

//=======================================================================
//=======================================================================

int
main( int ArgC, char** ArgV )
{
  static const char*	caller = "main";
  MemChecker::Trigger	gc_trigger( true );

  FrameCPP::Initialize( );

  try
  {
    //-------------------------------------------------------------------
    // Variables for logging
    //-------------------------------------------------------------------
    static char cwd_buffer[ 2048 ];
    getcwd( cwd_buffer, sizeof( cwd_buffer ) / sizeof( *cwd_buffer ) );
    std::string	cwd( cwd_buffer );

    GenericAPI::setLogTag( "createRDS" );
    GenericAPI::setLogOutputDirectory( cwd );
    GenericAPI::SetLogFormatter( new GenericAPI::Log::Text( "" ) );

    CommandLine		cl( ArgC, ArgV );
    ServerInterface	server;

    cl();

    if ( cl.Filenames( ).size( ) <= 0 )
    {
      server.Server( cl.Hostname( ), cl.Port( ) );
    }
    
    //-------------------------------------------------------------------
    // Loop over each of the frame queries
    //-------------------------------------------------------------------
    for ( CommandLine::frame_queries_type::const_iterator
	    cur = cl.FrameQueries( ).begin( ),
	    last = cl.FrameQueries( ).end( );
	  cur != last;
	  ++cur )
    {
      filenames_type	filenames;
      bool		resampling = false;

      cl.OutputTimeStart( cur->start.GetSeconds( ) );
      cl.OutputTimeEnd( cur->end.GetSeconds( ) );

      FrameAPI::channel_container_type		channels;

      for ( query::channel_container_type::const_iterator
	      cur_cl_chan = cur->channels.begin( ),
	      last_cl_chan = cur->channels.end( );
	    cur_cl_chan != last_cl_chan;
	    ++cur_cl_chan )
      {
	size_t cne = cur_cl_chan->rfind( '!' );
	if ( cne == std::string::npos )
	{
	  channels.names.push_back( *cur_cl_chan );
	  channels.resampling.push_back( 1 );
	}
	else
	{
	  std::istringstream	sresample;
	  int			resample;
	  std::string		suffix;

	  sresample.str( cur_cl_chan->substr( ( cne + 1 ) ) );
	  sresample >> resample;
	  sresample >> suffix;

	  if ( suffix.length( ) > 0 )
	  {
	    cne = std::string::npos;
	    resample = 1;
	  }
	  else
	  {
	    switch( resample )
	    {
	    case 1:
	      break;
	    case 2:
	    case 4:
	    case 8:
	    case 16:
	      resampling = true;
	      break;
	    default:
	      cne = std::string::npos;
	      resample = 1;
	      break;
	    }
	  }

	  channels.names.push_back( cur_cl_chan->substr( 0, cne ) );
	  channels.resampling.push_back( resample );

	  QUEUE_LOG_MESSAGE( "name: " << cur_cl_chan->substr( 0, cne )
			     << " sampleRate: " << resample,
			     MT_DEBUG,
			     30,
			     caller,
			     "CMD_CREATE_RDS" );
	}
      }
      if ( cl.Filenames( ).size( ) )
      {
	QUEUE_LOG_MESSAGE( "cl.Filenames( ): start",
			   MT_DEBUG,
			   30,
			   caller,
			   "CMD_CREATE_RDS" );
	filenames = cl.Filenames( );
	QUEUE_LOG_MESSAGE( "cl.Filenames( ): stop",
			   MT_DEBUG,
			   30,
			   caller,
			   "CMD_CREATE_RDS" );
      }
      else
      {
	QUEUE_LOG_MESSAGE( "server.FilenamesRDS( ): start"
			   << " ifo: " << cur->ifo
			   << " desc: "  << cur->desc
			   << " time_start:" << cl.OutputTimeStart( )
			   << " time_end: " << cl.OutputTimeEnd( )
			   << " ext: " << cur->ext
			   << " resampling: " << resampling
			   << " filenames size: " << filenames.size( ),
			   MT_DEBUG,
			   30,
			   caller,
			   "CMD_CREATE_RDS" );
	server.FilenamesRDS( filenames,
			     cur->ifo, cur->desc,
			     cl.OutputTimeStart( ),
			     cl.OutputTimeEnd( ),
			     resampling,
			     cur->ext );
	QUEUE_LOG_MESSAGE( " server.FilenamesRDS( ): stop"
			   << " filenames size: " << filenames.size( ),
			   MT_DEBUG,
			   30,
			   caller,
			   "CMD_REATE_RDS" );
      }

      if ( IsLogging( LogEntryGroup_type::MT_DEBUG,
		      30 ) )
      {
	for ( filenames_type::const_iterator
		cur_fname = filenames.begin( ),
		last_fname = filenames.end( );
	      cur_fname != last_fname;
	      ++cur_fname )
	{
	  std::ostringstream	msg;

	  msg << "filename: " << *cur_fname
	    ;
	  queueLogEntry( msg.str( ),
			 LogEntryGroup_type::MT_DEBUG,
			 30,
			 caller,
			 "CMD_CREATE_RDS" );
	}
      }
      if ( cl.SecondsPerFrame( ) == 0 )
      {
	cl.SecondsPerFrame( cl.OutputTimeEnd( )
			    - cl.OutputTimeStart( ) );
      }
      for ( query::channel_container_type::const_iterator
	      cur_cl_chan = cur->channels.begin( ),
	      last_cl_chan = cur->channels.end( );
	    cur_cl_chan != last_cl_chan;
	    ++cur_cl_chan )
      {
	FrameAPI::RDS::ExpandChannelList( channels.names,
					  channels.resampling,
					  filenames.front( ) );
      }
      QUEUE_LOG_MESSAGE( "Description: " << cl.OutputType( )
			 << " start: " << cl.OutputTimeStart( )
			 << " end: " << cl.OutputTimeEnd( ),
			 MT_DEBUG,
			 30,
			 caller,
			 "CMD_CREATE_RDS" );
      createRDSSet( filenames,
		    channels,
		    cl );
    }
  }
  catch( const SwigException& E )
  {
    QUEUE_LOG_MESSAGE( " Caught SwigException:"
		       << " Result: " << E.getResult( )
		       << " Info: " << E.getInfo( ),
		       MT_DEBUG,
		       30,
		       caller,
		       "CMD_CREATE_RDS" );
  }
  catch( const std::exception& E )
  {
    QUEUE_LOG_MESSAGE( " Caught exception: " << E.what( ),
		       MT_DEBUG,
		       30,
		       caller,
		       "CMD_CREATE_RDS" );
  }
  catch( ... )
  {
    QUEUE_LOG_MESSAGE( " Caught an unknown exception",
		       MT_DEBUG,
		       30,
		       caller,
		       "CMD_CREATE_RDS" );
  }

  return 0;
}

//=======================================================================
// Class CommandLine
//=======================================================================
CommandLine::
CommandLine( int ArgC, char** ArgV )
  : CommandLineOptions( ArgC, ArgV )
{
  //---------------------------------------------------------------------
  // Setup the options that will be recognized.
  //---------------------------------------------------------------------
  std::ostringstream	desc;

  options.
    Synopsis( "[options]" );
  options.
    Summary( "This creates a reduced data set." );

  //:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::

  //.....................................................................
  //.....................................................................
  options.
    Add( Option( OPT_HELP,
		 "help",
		 Option::ARG_NONE,
		 "Display this message" ) );


  //.....................................................................
  //.....................................................................
  options.
    Add( Option( OPT_ALLOW_SHORT_FRAMES,
		 "allow-short-frames",
		 Option::ARG_REQUIRED,
		 "Specify if short frames are allowed to be generated",
		 "boolean" ) );
  //.....................................................................
  //.....................................................................
  options.
    Add( Option( OPT_DESCRIPTION,
		 "description",
		 Option::ARG_REQUIRED,
		 "Specify the description portion of filename.",
		 "string" ) );
  //.....................................................................
  //.....................................................................
  options.
    Add( Option( OPT_DIRECTORY_OUTPUT_FRAMES,
		 "directory-output-frames",
		 Option::ARG_REQUIRED,
		 "The output directory where to store the resulting frames.",
		 "directory" ) );
  //.....................................................................
  //.....................................................................
  options.
    Add( Option( OPT_DIRECTORY_OUTPUT_MD5SUM,
		 "directory-output-md5sum",
		 Option::ARG_REQUIRED,
		 "The output directory where to store the md5sum checksum.",
		 "directory" ) );
  //.....................................................................
  //.....................................................................
  options.
    Add( Option( OPT_DIRECTORY_OUTPUT_MD5SUM_REGEXP,
		 "directory-output-md5sum-regexp",
		 Option::ARG_REQUIRED,
		 "Regular expression describing how to modify the --directory-output-frames"
		 " option for outputing the md5sum checksums",
		 "directory" ) );
  //.....................................................................
  //.....................................................................
  options.
    Add( Option( OPT_DISKCACHE_HOST,
		 "diskcache-host",
		 Option::ARG_REQUIRED,
		 "The host name which is running the diskcache server",
		 "hostname" ) );

  //.....................................................................
  //.....................................................................
  options.
    Add( Option( OPT_DISKCACHE_PORT,
		 "diskcache-port",
		 Option::ARG_REQUIRED,
		 "The number of the port for the diskcache server",
		 "int" ) );
  //.....................................................................
  //.....................................................................
  options.
    Add( Option( OPT_FILL_MISSING_DATA_VALID_ARRAY,
		 "fill-missing-data-valid-array",
		 Option::ARG_REQUIRED,
		 "Specify if the data valid array should be generated if it is"
		 " missing from the input data",
		 "boolean" ) );
  //.....................................................................
  //.....................................................................
  options.
    Add( Option( OPT_FRAME_COMPRESSION_METHOD,
		 "compression-type",
		 Option::ARG_REQUIRED,
		 "Compression method to use when writing the frame to the stream",
		 "string" ) );
  //.....................................................................
  //.....................................................................
  options.
    Add( Option( OPT_FRAME_COMPRESSION_LEVEL,
		 "compression-level",
		 Option::ARG_REQUIRED,
		 "Level of compression for compression methods that support it.",
		 "integer" ) );
  //.....................................................................
  //.....................................................................
  options.
    Add( Option( OPT_FRAME_FILES,
		 "frame-files",
		 Option::ARG_REQUIRED,
		 "A comma separated list of frame files",
		 "string" ) );
  //.....................................................................
  //
  // query = { <type> <ifo> <start-time>-<end-time> Chan(<channel-name>[,<channel-name>]) }
  //   query_list = { <query> [<query> ...] }
  //
  // ex: { { H1_R H {} 1051388672-1051388927
  //         Chan(H0:PEM-EX_DUST_VEA1_300NM_PCF,H0:PEM-EX_DUST_VEA1_500NM_PCF, ... } }"
  //.....................................................................
  options.
    Add( Option( OPT_FRAME_QUERY,
		 "frame-query",
		 Option::ARG_REQUIRED,
		 "A TCL formatted list of frame queries"
		 " {{<FrameType> <IFO> <UNKNOWN> <Time Range> Chan(<channel list>)} ...}",
		 "string" ) );

  //.....................................................................
  //.....................................................................
  options.
    Add( Option( OPT_FRAMES_PER_FILE,
		 "frames-per-file",
		 Option::ARG_REQUIRED,
		 "Specify the number of frames to store in each file",
		 "iteger" ) );
  //.....................................................................
  //.....................................................................
  options.
    Add( Option( OPT_GENERATE_FRAME_CHECKSUM,
		 "generate-frame-checksum",
		 Option::ARG_REQUIRED,
		 "Specify if the frame checksum value should be generated",
		 "boolean" ) );
  //.....................................................................
  //.....................................................................
  options.
    Add( Option( OPT_LOG_DIRECTORY,
		 "log-directory",
		 Option::ARG_REQUIRED,
		 "Specify the directory to store logging information.",
		 "direcory" ) );
  //.....................................................................
  //.....................................................................
  options.
    Add( Option( OPT_LOG_DEBUG_LEVEL,
		 "log-debug-level",
		 Option::ARG_REQUIRED,
		 "Specify the level of debugging information to generate.",
		 "int" ) );
  //.....................................................................
  //.....................................................................
  options.
    Add( Option( OPT_SECONDS_PER_FRAME,
		 "seconds-per-frame",
		 Option::ARG_REQUIRED,
		 "Specify the number of seconds to store in each frame",
		 "iteger" ) );
  //.....................................................................
  //.....................................................................
  options.
    Add( Option( OPT_WITHOUT_HISTORY_RECORD,
		 "without-history-record",
		 Option::ARG_NONE,
		 "Suppress the creation of the RDS history record in the output frames." ) );
}

inline const filenames_type& CommandLine::
Filenames( ) const
{
  return filenames;
}

inline const CommandLine::frame_queries_type& CommandLine::
FrameQueries( ) const
{
  return frame_queries;
}

inline void CommandLine::
Usage( int ExitValue ) const
{
  std::cerr << "Usage: "
	    << ProgramName( )
	    << options
	    << std::endl
    ;
  depart( ExitValue );
}

inline void CommandLine::
operator()( )
{
  static const char* caller = "CommandLine::operator()()";

  //---------------------------------------------------------------------
  // Parse the options specified on the command line
  //---------------------------------------------------------------------
  try
  {
    std::string	name;
    std::string value;
    bool parsing( true );
    std::string	log_directory;
    int		log_debug_level = -2;
    std::string	md5sum_regexp;

    //-------------------------------------------------------------------
    // Parse the option flags specified on the command line
    //-------------------------------------------------------------------
    while( parsing )
    {
      const int id( Parse( options, name, value ) );

      switch( id )
      {
      case CommandLineOptions::OPT_END_OF_OPTIONS:
	parsing = false;
	break;
      case OPT_HELP:
	{
	  Usage( 0 );
	}
	break;
      case OPT_ALLOW_SHORT_FRAMES:
	{
	  AllowShortFrames( InterpretBoolean( value ) );
	}
	break;
      case OPT_DESCRIPTION:
	{
	  OutputType( value );
	}
	break;
      case OPT_DIRECTORY_OUTPUT_FRAMES:
	{
	  DirectoryOutputFrames( value );
	}
	break;
      case OPT_DIRECTORY_OUTPUT_MD5SUM:
	{
	  DirectoryOutputMD5Sum( value );
	}
	break;
      case OPT_DIRECTORY_OUTPUT_MD5SUM_REGEXP:
	{
	  md5sum_regexp = value;
	}
	break;
      case OPT_DISKCACHE_HOST:
	{
	  Hostname( value );
	}
	break;
      case OPT_DISKCACHE_PORT:
	{
	  ServerInfo::port_type	port;

	  std::istringstream sport( value );

	  sport >> port;
	  Port( port );
	}
	break;
      case OPT_FILL_MISSING_DATA_VALID_ARRAY:
	{
	  FillMissingDataValidArray( InterpretBoolean( value ) );
	}
	break;
      case OPT_FRAME_COMPRESSION_LEVEL:
	{
	  std::istringstream		svalue( value );
	  compression_level_type	lvalue;

	  svalue >> lvalue;
	  CompressionLevel( lvalue );
	}
	break;
      case OPT_FRAME_COMPRESSION_METHOD:
	{
	  CompressionMethod( value );
	}
	break;
      case OPT_FRAME_FILES:
	{
	  parse_frame_files( value );
	}
	break;
      case OPT_FRAME_QUERY:
	{
	  parse_frame_query( value );
	}
	break;
      case OPT_FRAMES_PER_FILE:
	{
	  std::istringstream	v( value );
	  frames_per_file_type	lvalue;

	  v >> lvalue;
	  FramesPerFile( lvalue );
	}
	break;
      case OPT_GENERATE_FRAME_CHECKSUM:
	{
	  GenerateFrameChecksum( InterpretBoolean( value ) );
	}
	break;
      case OPT_LOG_DEBUG_LEVEL:
	{
	  std::istringstream slevel( value );

	  slevel >> log_debug_level;

	  GenericAPI::setLogDebugLevel( log_debug_level );
	}
	break;
      case OPT_LOG_DIRECTORY:
	{
	  log_directory = value;

	  GenericAPI::setLogOutputDirectory( log_directory );
	  if ( log_debug_level != -2 )
	  {
	    GenericAPI::setLogDebugLevel( log_debug_level );
	  }
	}
	break;
      case OPT_SECONDS_PER_FRAME:
	{
	  seconds_per_frame_type	s;
	  std::istringstream		v( value );

	  v >> s;
	  SecondsPerFrame( s );
	}
	break;
      case OPT_VERIFY_CHECKSUM_OF_FRAME:
	{
	  VerifyChecksumPerFrame( InterpretBoolean( value ) );
	}
	break;
      case OPT_VERIFY_CHECKSUM_OF_STREAM:
	{
	  VerifyChecksumOfStream( InterpretBoolean( value ) );
	}
	break;
      case OPT_VERIFY_DATA_VALID:
	{
	  VerifyDataValid( InterpretBoolean( value ) );
	}
	break;
      case OPT_VERIFY_FILENAME_METADATA:
	{
	  VerifyFilenameMetadata( InterpretBoolean( value ) );
	}
	break;
      case OPT_VERIFY_TIME_RANGE:
	{
	  VerifyTimeRange( InterpretBoolean( value ) );
	}
	break;
      case OPT_WITHOUT_HISTORY_RECORD:
	{
	  HistoryRecord( false );
	}
	break;
      }
    }
    //---------------------------------------------------------------------
    // Check if md5sum directory is described by regex
    //---------------------------------------------------------------------
    if ( ( md5sum_regexp.size( ) > 0 )
	 && ( DirectoryOutputMD5Sum( ).size( ) <= 0 )
	 && ( DirectoryOutputFrames( ).size( ) > 0 ) )
    {
      Sed	sed( md5sum_regexp );

      DirectoryOutputMD5Sum( sed( DirectoryOutputFrames( ) ) );

      QUEUE_LOG_MESSAGE( "sed modified md5sum directory: "
			 << DirectoryOutputMD5Sum( ),
			 MT_DEBUG,
			 30,
			 caller,
			 "CMD_CREATE_RDS" );
    }
  }
  catch( const std::invalid_argument& Error )
  {
    throw;
  }
  catch ( ... )
  {
  }
  //---------------------------------------------------------------------
  // Save the list of files
  //---------------------------------------------------------------------
  while ( empty( ) == false )
  {
    filenames.push_back( Pop( ) );
  }
  //---------------------------------------------------------------------
  // Try to ensure the directories exist
  //---------------------------------------------------------------------
  if ( DirectoryOutputFrames( ).empty( ) == false )
  {
    try
    {

      MkDir	d( 0755, MkDir::OPT_MAKE_PARENT_DIRECTORIES );

      d( DirectoryOutputFrames( ) );
    }
    catch( const std::exception& except )
    {
      QUEUE_LOG_MESSAGE( "Unable to create directory: " << DirectoryOutputFrames( )
			 << " (" << except.what( ) << ")",
			 MT_WARN,
			 0,
			 caller,
			 "CMD_CREATE_RDS" );
    }
  }
  if ( ( DirectoryOutputMD5Sum( ).empty( ) == false )
       && ( DirectoryOutputMD5Sum( ).compare( DirectoryOutputFrames( ) ) != 0 ) )
  {
    try
    {
      MkDir	d( 0755, MkDir::OPT_MAKE_PARENT_DIRECTORIES );

      d( DirectoryOutputMD5Sum( ) );
    }
    catch( const std::exception& except )
    {
      QUEUE_LOG_MESSAGE( "Unable to create directory: " << DirectoryOutputMD5Sum( )
			 << " (" << except.what( ) << ")",
			 MT_WARN,
			 0,
			 caller,
			 "CMD_CREATE_RDS" );
    }
  }
}

void CommandLine::
parse_frame_files( const std::string& Filenames )
{
  static const char* caller = "CommandLine::parse_frame_files";

  if ( Filenames.size( ) )
  {
    size_type	pos( 0 );
    size_type	found_pos;

    while( pos != std::string::npos )
    {
      found_pos = Filenames.find( ',', pos );
      filenames.push_back( Filenames.substr( pos, found_pos - pos )  );
      pos = found_pos;
      if ( pos != std::string::npos )
      {
	++pos;
      }
      QUEUE_LOG_MESSAGE( "filename: " << filenames.back( ),
			 MT_DEBUG,
			 30,
			 caller,
			 "CMD_CREATE_RDS" );
    }
  }
}

void CommandLine::
parse_frame_query( const std::string& Query )
{
  static const char* caller = "CommandLine::parse_frame_query";

  std::list< std::string >	fq;

  QUEUE_LOG_MESSAGE( "parse_frame_query: Query: " << Query,
		     MT_DEBUG,
		     30,
		     caller,
		     "CMD_CREATE_RDS" );
  GenericAPI::TCL::ParseList( Query.c_str( ), fq );

  for ( std::list< std::string >::const_iterator
	  cur = fq.begin( ),
	  last = fq.end( );
	cur != last;
	++cur )
  { 
    QUEUE_LOG_MESSAGE( "parse_frame_query: cur: " << *cur,
		       MT_DEBUG,
		       30,
		       caller,
		       "CMD_CREATE_RDS" );
    frame_queries.push_back( query( *cur ) );
  }
  QUEUE_LOG_MESSAGE( "parse_frame_query: finished",
		     MT_DEBUG,
		     30,
		     caller,
		     "CMD_CREATE_RDS" );
}


//=======================================================================
//
//=======================================================================
query::
query( const std::string Query )
  : ext( ".gwf" )
{
  static const char* caller = "query::query";

  std::list< std::string >	parameters;
  std::vector< std::string >	op;
  int				count = 0;
 
  GenericAPI::TCL::ParseList( Query, parameters );
  op.resize( std::min( parameters.size( ), size_t( POSITION_QUERY_LIST_SIZE_TCL ) ) );
  QUEUE_LOG_MESSAGE( "op.size: " << op.size( ),
		     MT_DEBUG,
		     30,
		     caller,
		     "CMD_CREATE_RDS" );
  for ( std::list< std::string >::const_iterator
	  cur = parameters.begin( ),
	  last = parameters.end( );
	cur != last;
	++cur, ++count )
  {
    if ( count > POSITION_CHANNEL_LIST_TCL )
    {
      op[ POSITION_CHANNEL_LIST_TCL ] += *cur;
    }
    else
    {
      op[ count ] = *cur;
    }
  }
  QUEUE_LOG_MESSAGE( "number of elements in query: " << parameters.size( ),
		     MT_DEBUG,
		     30,
		     caller,
		     "CMD_CREATE_RDS" );
  desc = op[ POSITION_DESC_TCL ];
  ifo = op[ POSITION_IFO_TCL ];

  //---------------------------------------------------------------------
  // Parse the time range
  //---------------------------------------------------------------------
  std::string	s( op[POSITION_TIME_RANGE_TCL ] );
  std::string	e;
  size_t	offset( s.find( '-' ) );

  if ( offset != std::string::npos )
  {
    e = s.substr( offset + 1 );
    s = s.substr( 0, offset );
  }

  LDASTools::AL::GPSTime::seconds_type	sv;
  LDASTools::AL::GPSTime::seconds_type	ev;

  {
    std::istringstream	ss( s );
    ss >> sv;
  }
  {
    std::istringstream	ss( e );
    ss >> ev;
  }

  start = LDASTools::AL::GPSTime( sv, 0 );
  end = LDASTools::AL::GPSTime( ev, 0 );

  //---------------------------------------------------------------------
  // Parse out the channel list
  //---------------------------------------------------------------------
  const std::string& cl( op[ POSITION_CHANNEL_LIST_TCL ] );
  size_t clb( cl.find( '(' ) );
  size_t cle( cl.rfind( ')' ) );

  if ( clb != std::string::npos )
  {
    ++clb;
  }
  QUEUE_LOG_MESSAGE( "cl: " << cl
		     << " clb: " << clb
		     << " cle: " << cle
		     << " len: " << ( cle - clb )
		     << " substr: " << cl.substr( clb, ( cle - clb ) ),
		     MT_DEBUG,
		     30,
		     caller,
		     "CMD_CREATE_RDS" );
  std::string 	cls( cl.substr( clb, ( cle - clb ) ) );
  //---------------------------------------------------------------------
  // Checking to see if the buffer names a file
  //---------------------------------------------------------------------
  {
    QUEUE_LOG_MESSAGE( "Checking if string is a filename",
		       MT_DEBUG,
		       30,
		       caller,
		       "CMD_CREATE_RDS" );
    Regex	filename_regex( "^[[:space:]]*(.*)[[:space:]]*$", REG_EXTENDED );
    Regex	channel_regex( "^[[:space:]]*([^[:space:]]+)[[:space:]]+([[:digit:]]+)[[:space:]]*$", REG_EXTENDED );
    RegexMatch		m( 5 );
    std::ostringstream	channels;


    QUEUE_LOG_MESSAGE( "Checking if string is a filename: "
		       << cls,
		       MT_DEBUG,
		       30,
		       caller,
		       "CMD_CREATE_RDS" );
    if ( m.match( filename_regex, cls.c_str( ) ) )
    {
      QUEUE_LOG_MESSAGE( "Channel name coming from file: "
			 << m.getSubString( 1 ),
			 MT_DEBUG,
			 30,
			 caller,
			 "CMD_CREATE_RDS" );
      std::ifstream	s( m.getSubString( 1 ).c_str( ) );

      if ( s.is_open( ) )
      {
	try
	{
	  bool	first = true;

	  while( s.good( ) )
	  {
	    static char	buffer[256];

	    s.getline( buffer, sizeof( buffer ) );
	    QUEUE_LOG_MESSAGE( "Parsing channel line: "
			       << buffer,
			       MT_DEBUG,
			       30,
			       caller,
			       "CMD_CREATE_RDS" );
	    if ( m.match( channel_regex, buffer ) )
	    {
	      QUEUE_LOG_MESSAGE( "Found Matches: "
				 << m.size( ),
				 MT_DEBUG,
				 30,
				 caller,
				 "CMD_CREATE_RDS" );
	      if ( first )
	      {
		first = false;
	      }
	      else
	      {
		channels << ",";
	      }

	      switch( m.size( ) )
	      {
	      case 2:
		channels << m.getSubString( 1 );
		break;
	      case 3:
		channels << m.getSubString( 1 )
			 << "!" << m.getSubString( 2 );
		break;
	      }
	    }
	  }
	}
	catch( ... )
	{
	}
      }
      
      s.close( );
      cls = channels.str( );
      QUEUE_LOG_MESSAGE( "Channels: "
			 << cls,
			 MT_DEBUG,
			 30,
			 caller,
			 "CMD_CREATE_RDS" );
    }
  }

  offset = 0;

  size_t       	offset_end = cls.find( ',', offset );

  while( offset != std::string::npos )
  {
    channels.push_back( cls.substr( offset,
				    ( offset_end == std::string::npos )
				    ? offset_end
				    : ( offset_end - offset ) ) ) ;
    QUEUE_LOG_MESSAGE( "Added channel: " << channels.back( ),
		       MT_DEBUG,
		       30,
		       caller,
		       "CMD_CREATE_RDS" );
    offset = offset_end;
    if ( offset != std::string::npos )
    {
      ++offset;
      offset_end = cls.find( ',', offset );
    }
  }
  QUEUE_LOG_MESSAGE( "finished query",
		     MT_DEBUG,
		     30,
		     caller,
		     "CMD_CREATE_RDS" );
}
