/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef GENERAL__GPS_TIME_HH
#define GENERAL__GPS_TIME_HH

#if ! defined(SWIGIMPORTED)
#include <stdexcept>
#include <iostream>
#endif /* ! defined(SWIGIMPORTED) */

#include "ldastoolsal/ErrorLog.hh"
#include "ldastoolsal/types.hh"

const REAL_8 NANOSECOND = 10E-9;
const REAL_8 NANOSECOND_MULTIPLIER = 10E+9;
const INT_4U NANOSECOND_INT_MULTIPLIER = 1000000000;
const INT_8U NANOSECOND_INT_64_MULTIPLIER = 1000000000;

namespace LDASTools
{
    namespace AL
    {

	class GPSTime {

	public:
	    typedef INT_4U seconds_type;
	    typedef INT_4U nanoseconds_type;

	    typedef INT_8U gpsnanoseconds_type;

	    typedef enum {
		UTC = 0,
		GPS
	    } unit_type;

	    static const unit_type DEFAULT_TIME_UNIT_TYPE;

	    GPSTime();
	    GPSTime( const seconds_type seconds,
		     const nanoseconds_type nanoseconds,
		     unit_type Units = DEFAULT_TIME_UNIT_TYPE );

	    GPSTime(const GPSTime&);

	    ~GPSTime();

	    GPSTime& operator=(const GPSTime&);

	    seconds_type GetSeconds( ) const;
	    nanoseconds_type GetSeconds( const unit_type Units ) const;
	    nanoseconds_type GetNanoseconds() const;
	    REAL_8 GetTime( ) const;
	    INT_2U GetLeapSeconds() const;

	    //---------------------------------------------------------------
	    /// \brief Return the time in nanosecond units
	    ///
	    /// \return
	    ///      The time value as the number of nanoseconds since the
	    ///      start of GPS time.
	    //---------------------------------------------------------------
	    gpsnanoseconds_type GPSNanoseconds( ) const;

	    GPSTime& operator+=(const double&);
	    GPSTime& operator-=(const double&);

	    /// \brief  Sets the object to the current GPS time.
	    void Now();

	    /// Helper function to make it compatable with Time class of
	    ///   framecpp
	    seconds_type getSec( ) const;

	    /// Helper function to make it compatable with Time class of
	    ///   framecpp
	    nanoseconds_type getNSec( ) const;
        
	    /// \brief  Returns the time "now" as a GPSTime
	    /// \return GPSTime the current time, in GPSTime form
	    static GPSTime NowGPSTime( );

	protected:
	    /// \brief  Returns the time "now" as a GPSTime
	    /// \return GPSTime the current time, in GPSTime form
	    static GPSTime now( );

	private:
	    friend void ErrorLog::operator()( state State,
					      const char* Filename, const int Line,
					      const std::string& Message,
					      bool EOL );
	    seconds_type m_seconds;
	    nanoseconds_type m_nanoseconds;

	}; // class GPSTime

	inline REAL_8 GPSTime::
	GetTime( ) const
	{
	    return ( m_seconds + ( m_nanoseconds / NANOSECOND_MULTIPLIER ) );
	}

	inline GPSTime::nanoseconds_type GPSTime::
	GetNanoseconds() const
	{
	    return m_nanoseconds;
	}

	inline GPSTime::seconds_type GPSTime::
	GetSeconds( ) const
	{
	    return m_seconds;
	}

	inline GPSTime::seconds_type GPSTime::
	getSec( ) const
	{
	    return GetSeconds( );
	}

	/// Helper function to make it compatable with Time class of
	///   framecpp
	inline GPSTime::nanoseconds_type GPSTime::
	getNSec( ) const
	{
	    return GetNanoseconds( );
	}

	//-------------------------------------------------------------------
	/// This routine returns the time value as the number of nanoseconds
	/// that have elapsed since the start of GPS time.
	//-------------------------------------------------------------------
	inline GPSTime::gpsnanoseconds_type GPSTime::
	GPSNanoseconds( ) const
	{
	    gpsnanoseconds_type retval( GetSeconds( ) );
	    return ( ( retval * NANOSECOND_INT_64_MULTIPLIER )
		     + ( GetNanoseconds( ) ) )
		;
	}

	//-------------------------------------------------------------------
	/// Establish the filename which contains the leap seconds
	//-------------------------------------------------------------------
	void SetOffsetTableFilename( const std::string& Filename );

	GPSTime operator+(const GPSTime&, const double&);
	GPSTime operator+(const double&, const GPSTime&);

	GPSTime operator-(const GPSTime&, const double&);
	double operator-(const GPSTime&, const GPSTime&);

	bool operator==(const GPSTime&, const GPSTime&);
	bool operator!=(const GPSTime&, const GPSTime&);

	bool operator<(const GPSTime&, const GPSTime&);
	bool operator>(const GPSTime&, const GPSTime&);

	bool operator<=(const GPSTime&, const GPSTime&);
	bool operator>=(const GPSTime&, const GPSTime&);
    
	std::ostream& operator<<( std::ostream& Stream,
				  const GPSTime& Time );
    } // namespace - AL

} // namespace LDASTools

#endif // GENERAL__GPS_TIME_HH
