#ifndef GENERIC_API__LSTAT_POOL_HH
#define GENERIC_API__LSTAT_POOL_HH

#include <list>

#include "ldastoolsal/ErrorLog.hh"
#include "ldastoolsal/Memory.hh"
#include "ldastoolsal/mutexlock.hh"
#include "ldastoolsal/Singleton.hh"

namespace GenericAPI
{

  template< typename StatInfoType >
  class StatPoolT
    : public LDASTools::AL::Singleton< StatPoolT< StatInfoType > >
  {
  public:
    typedef typename LDASTools::AL::SharedPtr< StatInfoType > info_type;

    static void Cleanup( );

    static bool Interuptable( );

    static info_type Request( );

    static void Release( info_type Buffer );

    static void Destroy( info_type Buffer );

  private:
    typedef std::list< info_type > pool_type;

    LDASTools::AL::MutexLock::baton_type	pool_lock;
    pool_type					available;
    pool_type					in_use;

    void cleanup( );

    void destroy( info_type Buffer );

    void release( info_type Buffer );

    info_type request( );
  };

  template< typename StatInfoType >
  void StatPoolT< StatInfoType >::
  Cleanup( )
  {
    LDASTools::AL::Singleton< StatPoolT< StatInfoType > >::Instance( ).cleanup( );
  }
  
  template< typename StatInfoType >
  bool StatPoolT< StatInfoType >::
  Interuptable( )
  {
    return true;
  }

  template< typename StatInfoType >
  typename StatPoolT< StatInfoType >::info_type StatPoolT< StatInfoType >::
  Request( )
  {
    return LDASTools::AL::Singleton< StatPoolT< StatInfoType > >::Instance( ).request( );
  }

  template< typename StatInfoType >
  void StatPoolT< StatInfoType >::
  Release( info_type Buffer )
  {
    LDASTools::AL::Singleton< StatPoolT< StatInfoType > >::Instance( ).release( Buffer );
  }

  template< typename StatInfoType >
  void StatPoolT< StatInfoType >::
  Destroy( info_type Buffer )
  {
    LDASTools::AL::Singleton< StatPoolT< StatInfoType > >::Instance( ).destroy( Buffer );
  }

  template< typename StatInfoType >
  void StatPoolT< StatInfoType >::
  cleanup( )
  {
    using namespace LDASTools::AL;

    static bool has_been_cleaned = false;

    if ( has_been_cleaned == false )
    {
      MutexLock	lock( pool_lock,
		      __FILE__, __LINE__ );

      if ( has_been_cleaned == false )
      {
	if ( StdErrLog.IsOpen( ) )
	{
	  std::ostringstream	msg;
      
	  msg << "StatPool::~StatPool";
	  StdErrLog( ErrorLog::DEBUG,
		     __FILE__, __LINE__,
		     msg.str( ) );
	}
	available.erase( available.begin( ), available.end( ) );
	in_use.erase( in_use.begin( ), in_use.end( ) );
	has_been_cleaned = true;
      } // if ( has_been_cleaned == false ) (after Mutex lock)
    } // if ( has_been_cleaned == false )
  }

  template< typename StatInfoType >
  void StatPoolT< StatInfoType >::
  destroy( info_type Buffer )
  {
    using namespace LDASTools::AL;

    MutexLock	lock( pool_lock, __FILE__, __LINE__ );

    if ( StdErrLog.IsOpen( ) )
    {
      std::ostringstream	msg;
      
      msg << "StatPool::destroy: Buffer: " << (void*)(Buffer.get( ));
      StdErrLog( ErrorLog::DEBUG,
		 __FILE__, __LINE__,
		 msg.str( ) );
    }
    //-------------------------------------------------------------------
    // Take out of m_in_use list
    //-------------------------------------------------------------------
    typename pool_type::size_type	size = in_use.size( );

    /* erase the element */
    in_use.remove( Buffer );
    if ( size != in_use.size( ) )
    {
      //-----------------------------------------------------------------
      // Destroy the process
      //-----------------------------------------------------------------
      if ( StdErrLog.IsOpen( ) )
      {
	std::ostringstream	msg;
      
	msg << "StatPool::release: deleting: Buffer: " << (void*)(Buffer.get( ));
	StdErrLog( ErrorLog::DEBUG,
		   __FILE__, __LINE__,
		   msg.str( ) );
      }
    }
  }

  template< typename StatInfoType >
  void StatPoolT< StatInfoType >::
  release( info_type Buffer )
  {
    using namespace LDASTools::AL;

    MutexLock	lock( pool_lock, __FILE__, __LINE__ );

    if ( StdErrLog.IsOpen( ) )
    {
      std::ostringstream	msg;
      
      msg << "StatPool::release";
      StdErrLog( ErrorLog::DEBUG,
		 __FILE__, __LINE__,
		 msg.str( ) );
    }
    //-------------------------------------------------------------------
    // Take out of in_use list
    //-------------------------------------------------------------------
    typename pool_type::size_type size = in_use.size( );

    /* erase the element */
    in_use.remove( Buffer );
    if ( size != in_use.size( ) )
    {
      //-----------------------------------------------------------------
      // Put into available queue
      //-----------------------------------------------------------------
      if ( StdErrLog.IsOpen( ) )
      {
	std::ostringstream	msg;
      
	msg << "StatPool::release: returned to a available pool";
	StdErrLog( ErrorLog::DEBUG,
		   __FILE__, __LINE__,
		   msg.str( ) );
      }
      if ( ! MemChecker::IsExiting( ) )
      {
	available.push_back( Buffer );
      }
    }
  }

  template< typename StatInfoType >
  typename StatPoolT< StatInfoType >::info_type StatPoolT< StatInfoType >::
  request( )
  {
    using namespace LDASTools::AL;

    MutexLock	lock( pool_lock, __FILE__, __LINE__ );
    info_type	retval;

    if ( available.size( ) == 0 )
    {
      //-----------------------------------------------------------------
      // Create new thread
      //-----------------------------------------------------------------
      retval.reset( new typename info_type::element_type );

      if ( StdErrLog.IsOpen( ) )
      {
	std::ostringstream	msg;
	
	msg << "StatPool::request: Created new";
	StdErrLog( ErrorLog::DEBUG,
		   __FILE__, __LINE__,
		   msg.str( ) );
      }
    }
    else
    {
      //-----------------------------------------------------------------
      // Take one from the available list
      //-----------------------------------------------------------------
      retval = available.front( );
      available.pop_front( );
      if ( StdErrLog.IsOpen( ) )
      {
	std::ostringstream	msg;
	
	msg << "StatPool::request: Reusing";
	StdErrLog( ErrorLog::DEBUG,
		   __FILE__, __LINE__,
		   msg.str( ) );
      }
    }
    in_use.push_back( retval );
    return retval;
  }

}

#endif /* GENERIC_API__LSTAT_POOL_HH */
