#if HAVE_CONFIG_H
#include <ldas_tools_config.h>
#endif /* HAVE_CONFIG_H */

#include <sstream>
#include <string>

#include "ldastoolsal/unittest.h"

#include "framecpp/Common/FrameFilename.hh"

using FrameCPP::Common::FrameFilename;

LDASTools::Testing::UnitTest Test;

void
test( const std::string& Filename )
{
  try
  {
    FrameFilename	ff( Filename );

    Test.Check( ff.S( ).find( '-' ) == std::string::npos )
      << Filename << ": S does not contain any hypens" << std::endl;
    Test.Check( ff.D( ).find( '-' ) == std::string::npos )
      << Filename << ": D does not contain any hypens" << std::endl;
    Test.Check( ff.G( ) > 0 )
      << Filename << ": GPS start time > 0" << std::endl;
    Test.Check( ff.T( ) > 0 )
      << Filename << ": Total Time > 0" << std::endl;
  }
  catch( const FrameFilename::InvalidFrameFilename& e )
  {
    Test.Check( false )
      << Filename << " ";
    for ( size_t x = 0, end = e.getSize( ) ; x < end; ++x )
    {
      Test.Message( false )
	<< e[ x ].getMessage( )
	<< std::endl;
    }
  }
}

void
valid( const std::string& Frame )
try
{
  // This test simply checks that the class can be constructed.
  FrameFilename	ffn( Frame );
  ffn.D( );
  Test.Check( true ) << "Created class for valid framefile name: "
		      << Frame << std::endl;
}
catch( ... )
{
  Test.Check( false ) << "Caught exception when validating framefile name: "
		      << Frame << std::endl;
}

void
valid( const std::string& Dir,
       const std::string& S,
       const std::string& D,
       const FrameFilename::gps_seconds_type G,
       const FrameFilename::total_time_interval_type T,
       const std::string& Ext )
try
{
  // This test simply checks that the class can be constructed.
  std::ostringstream	filename;
  if ( Dir.length( ) > 0 )
  {
    filename << Dir << "/";
  }
  filename
    << S << "-"
    << D << "-"
    << G << "-"
    << T << "."
    << Ext;

  FrameFilename	ffn( filename.str( ) );
  Test.Check( ( ( Dir.length( ) > 0 ) &&
		( Dir.compare( ffn.Dir( ) ) == 0 ) ) ||
	      ( (Dir.length( ) == 0 ) &&
		( ffn.Dir( ).compare( "." ) == 0 ) ) )
		  << "Validating directory: " << filename.str( ) << std::endl;
  Test.Check( S.compare( ffn.S( ) ) == 0 )
    << "Validating S component: " << filename.str( ) << std::endl;
  Test.Check( D.compare( ffn.D( ) ) == 0 )
    << "Validating D component: " << filename.str( ) << std::endl;
  Test.Check( ffn.G( )  == G )
    << "Validating G component: " << filename.str( ) << std::endl;
  Test.Check( ffn.T( ) == T )
    << "Validating T component: " << filename.str( ) << std::endl;
  Test.Check( Ext.compare( ffn.Ext( ) ) == 0 )
    << "Validating file extension component: " << filename.str( ) << std::endl;
}
catch( ... )
{
  std::ostringstream	filename;
  if ( Dir.length( ) > 0 )
  {
    filename << Dir << "/";
  }
  filename
    << S << "-"
    << D << "-"
    << G << "-"
    << T << "."
    << Ext;

  Test.Check( false ) << "Caught exception when validating framefile name pieces: "
		      << filename.str( ) << std::endl;
}

void
invalid( const std::string& Frame )
try
{
  // These are expected to throw an exception
  FrameFilename	ffn( Frame );
  ffn.D( );
  Test.Check( false )
    << "To throw an exception for the invalid framefile name: "
    << Frame << std::endl;
}
catch( const FrameFilename::InvalidFrameFilename& e )
{
  Test.Check( true )
    << "Caught proper exception for invalid framefile name: "
    << Frame << std::endl;
}
catch( ... )
{
  Test.Check( false )
    << "Threw wrong exception for the invalid framefile name: "
    << Frame << std::endl;
}
  
int
main( int ArgC, char* ArgV[] )
try
{
  Test.Init( ArgC, ArgV );

  //---------------------------------------------------------------------
  // Testing of class 1
  //---------------------------------------------------------------------
  // 1 second of raw GEO data
  test( "G-R-650123123-1.gwf" );
  // 8 seconds of raw LIGO data from Hanford
  test( "H-R-688775627-8.gwf" );
  // One minute of second-trend data from Livingston
  test( "L-T-688775600-60.gwf" );
  // One hour of minute-trend data fro mboth Hanford and Livingston
  test( "HL-M-68877500-3600.gwf" );
  //---------------------------------------------------------------------
  // Testing of class 2
  //---------------------------------------------------------------------
  // 256 seconds of "Level 2" data (e.g. selected channesl, possibly
  //   with some offline processing) from Hanford
  test( "H-level2-693020060-256.gwf" );
  // 1000 seocnds of PEM data from Livingston
  test( "L-PEM-720878444-1000.gwf" );
  // Some simultated data containing inspiral signals
  test( "HL-SimInspiral-730282200-16.gwf" );
  //---------------------------------------------------------------------
  // Testing of class 3
  //---------------------------------------------------------------------
  test( "pss-filter_test-787384722-8.gwf" );
  test( "iul-simset1-694215520-32.gwf" );
  test( "H-calibPEM-720878444-1000.gwf" );

  //---------------------------------------------------------------------
  // Valid frame filename tests
  //---------------------------------------------------------------------
  // Class 1
  valid( "G-R-650123123-1.gwf" );
  valid( "H-R-688775627-8.gwf" );
  valid( "L-T-688775600-60.gwf" );
  valid( "HL-M-688775000-3600.gwf" );

  // Class 2
  valid( "H-level2-693020060-256.gwf" );
  valid( "L-PEM-720878444-1000.gwf" );
  valid( "HL-SimInspiral-730282200-16.gwf" );

  // Class 3
  valid( "pss-filter_test-787384722-8.gwf" );
  valid( "iul-simset1-694215520-32.gwf" );
  valid( "H-calibPEM-720878444-1000.gwf" );

  // Other valid frame files
  valid( "/ldas_outgoing/test/frames/S2/LHO/full/H-R-7307020/H-R-730702080-16.gwf" );

  //---------------------------------------------------------------------
  // Validate the parsing of the components
  //---------------------------------------------------------------------
  valid( "", "G", "R", 650123123, 1, "gwf" );
  valid( "/ldas_outgoing/test/frames/S2/LHO/full/H-R-7307020",
	 "H", "R", 730702080, 16, "gwf" );

  //---------------------------------------------------------------------
  // Invalid frame filename tests
  //---------------------------------------------------------------------
  invalid( "H-calibPEM-720878444-extra-1000.gwf" );
  invalid( "H-720878444-1000.gwf" );

  Test.Exit();
}
catch ( ... )
{
  Test.Check( false ) << "Caught an exception" << std::endl;
  Test.Exit( );
}
