#if HAVE_CONFIG_H
#include <ldas_tools_config.h>
#endif /* HAVE_CONFIG_H */

#include <string.h>
#include <unistd.h>

#include <iomanip>
#include <iostream>
#include <list>
#include <memory>

#include "ldastoolsal/MemChecker.hh"
#include "ldastoolsal/CommandLineOptions.hh"
#include "ldastoolsal/SharedPtr.hh"
#include "ldastoolsal/types.hh"

#include "framecpp/IFrameStream.hh"
#include "framecpp/FrameCPP.hh"
#include "framecpp/FrAdcData.hh"
#include "framecpp/FrEvent.hh"
#include "framecpp/FrProcData.hh"
#include "framecpp/FrSerData.hh"
#include "framecpp/FrSimData.hh"
#include "framecpp/FrSimEvent.hh"
#include "framecpp/FrVect.hh"

using LDASTools::AL::MemChecker;

typedef LDASTools::AL::CommandLineOptions CommandLineOptions;
typedef CommandLineOptions::Option Option;
typedef CommandLineOptions::OptionSet OptionSet;

#if old
typedef const FrVect* vector_type;
#else /* old */
typedef LDASTools::AL::SharedPtr< ::FrameCPP::FrVect > vector_type;
#endif /* old */

typedef FrameCPP::FrVect::data_type data_type;

typedef FrameCPP::FrVect::data_const_pointer_type
data_const_pointer_type;

using std::filebuf;
using namespace FrameCPP;

using FrameCPP::Common::FrameBuffer;

class CommandLine;

enum {
  EXIT_CODE_OK = 0,
  EXIT_CODE_UNCAUGHT_EXCEPTION,
  EXIT_CODE_UNKNOWN_CHANNEL,
  EXIT_CODE_VECTOR_EXPANSION
};

inline void
depart( int ExitCode )
{
  exit( ExitCode );
}

int exit_code = EXIT_CODE_OK;

void dump_vect( vector_type Vect, const CommandLine& Options );

template< class T > void
dump( const CHAR_U* Source, INT_4U NData );

template< > void
dump< CHAR >( const CHAR_U* Source, INT_4U NData )
{
  const CHAR* data( reinterpret_cast< const CHAR* >( Source ) );

  for ( INT_4U x = 0;
	x < NData;
	++x, ++data )
  {
    if ( x )
    {
      std::cout << ", ";
    }
    std::cout << (INT_2S)( *data );
  }
}

template< > void
dump< CHAR_U >( const CHAR_U* Source, INT_4U NData )
{
  const CHAR_U* data( reinterpret_cast< const CHAR_U* >( Source ) );

  for ( INT_4U x = 0;
	x < NData;
	++x, ++data )
  {
    if ( x )
    {
      std::cout << ", ";
    }
    std::cout << (INT_2U)( *data );
  }
}

template< class T > void
dump( const CHAR_U* Source, INT_4U NData )
{
  const T* data( reinterpret_cast< const T* >( Source ) );

  for ( INT_4U x = 0;
	x < NData;
	++x, ++data )
  {
    if ( x )
    {
      std::cout << ", ";
    }
    std::cout << *data;
  }
}

void
dump_raw( data_type Data, INT_8U Size )
{
  std::ostringstream	line;

  INT_8U			cur( 0 );
  data_const_pointer_type	d( Data.get( ) );
  bool				need_nl( false );

  while ( cur != Size )
  {
    ++cur;
    if ( ( cur % 32 ) == 0 )
    {
      if ( need_nl )
      {
	std::cout << std::endl;
      }
      std::cout << line.str( );
      line.str( "" );
      need_nl = true;
    }
    if ( ( cur % 4 ) == 0 )
    {
      line << " ";
    }
    line << std::setfill( '0' ) << std::setw( 2 ) << std::hex << (INT_2U)( *d );
    ++d;
  }
  if ( line.str( ).empty( ) == false )
  {
    if ( need_nl )
    {
      std::cout << std::endl;
    }
    std::cout << line.str( );
  }
}

//-----------------------------------------------------------------------
/// \brief Class to handle command line options for this application
//-----------------------------------------------------------------------
class CommandLine
  : protected CommandLineOptions
{
public:
  typedef std::list< std::string >	channel_container_type;

  CommandLine( int ArgC, char** ArgV );


  inline bool
  BadOption( ) const
  {
    bool	retval = false;

    for ( const_iterator
	    cur = begin( ),
	    last = end( );
	  cur != last;
	  ++cur )
    {
      if ( (*cur)[0] == '-' )
      {
	std::cerr << "ABORT: Bad option: " << *cur
		  << std::endl
	  ;
	retval = true;
      }
    }
    return retval;
  }

  inline const channel_container_type&
  Channels( ) const
  {
    return m_channels;
  }

  inline bool
  Raw( ) const
  {
    return m_raw;
  }

  inline void
  Usage( int ExitValue ) const
  {
    std::cout << "Usage: "
	      << ProgramName( )
	      << m_options
	      << std::endl;
    depart( ExitValue );
  }

  using CommandLineOptions::Pop;
  using CommandLineOptions::empty;
  using CommandLineOptions::size;

private:
  enum option_types {
    OPT_CHANNEL,
    OPT_HELP,
    OPT_RAW
  };

  OptionSet			m_options;
  channel_container_type	m_channels;
  bool				m_raw;
};

CommandLine::
CommandLine( int ArgC, char** ArgV )
  : CommandLineOptions( ArgC, ArgV ),
    m_raw( false )
{
  //---------------------------------------------------------------------
  // Setup the options that will be recognized.
  //---------------------------------------------------------------------
  std::ostringstream	desc;

  m_options.
    Synopsis( "[options] <file> [<file> ...]" );

  m_options.
    Summary( "This command will dump the contents of the selected channels"
	     " for each of the requested files." );

  //:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
  desc.str( "" );
  desc <<
    "Specify a channel."
    ;

  m_options.
    Add( Option( OPT_CHANNEL,
		 "channel",
		 Option::ARG_REQUIRED,
		 desc.str( ),
		 "string" ) );

  //:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
  m_options.
    Add( Option( OPT_HELP,
		 "help",
		 Option::ARG_NONE,
		 "Display this message" ) );

  //:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
  desc.str( "" );
  desc <<
    "Dumps the data without decompressing."
    " (Default: " << ( ( m_raw )
		       ? "enabled"
		       : "disabled" ) << " )"
    ;

  m_options.
    Add( Option( OPT_RAW,
		 "raw",
		 Option::ARG_NONE,
		 desc.str( ) ) );

  //---------------------------------------------------------------------
  // Parse the options specified on the command line
  //---------------------------------------------------------------------

  try
  {
    std::string	arg_name;
    std::string arg_value;
    bool	parsing = true;

    while( parsing )
    {
      const int cmd_id( Parse( m_options, arg_name, arg_value ) );

      switch( cmd_id )
      {
      case CommandLineOptions::OPT_END_OF_OPTIONS:
	parsing = false;
	break;
      case OPT_HELP:
	{
	  Usage( 0 );
	}
	break;
      case OPT_CHANNEL:
	m_channels.push_back( arg_value );
	break;
      case OPT_RAW:
	m_raw = true;
	break;
      }
    }
  }
  catch( ... )
  {
  }
}

//=======================================================================
// Main entry point into the application
//=======================================================================
int
main(int ArgC, char* ArgV[])
try
{
  MemChecker::Trigger	gc_trigger( true );
  CommandLine		cl( ArgC, ArgV );

  if ( cl.empty( )
       || cl.BadOption( ) )
  {
    cl.Usage( 1 );
  }

  FrameCPP::Initialize( );

  while( cl.empty( ) == false )
  {
    const CommandLineOptions::option_type
      filename = cl.Pop( );
    //-------------------------------------------------------------------
    // Creation of the frame structure by reading of frame file
    //-------------------------------------------------------------------
    FrameBuffer< filebuf >*
      ibuf( new FrameBuffer< filebuf >( std::ios::in ) );
    
    ibuf->open( filename.c_str( ), std::ios::in | std::ios::binary );

    IFrameStream	ifs( ibuf );

    std::cout << "Frame file: " << filename << std::endl
      ;
    for ( CommandLine::channel_container_type::const_iterator
	    cur = cl.Channels( ).begin( ),
	    last = cl.Channels( ).end( ); 
	  cur != last;
	  ++cur )
    {
      //-----------------------------------------------------------------
      // FrAdcData
      //-----------------------------------------------------------------
      {
	LDASTools::AL::SharedPtr< FrAdcData >	channel;

	try
	{
	  channel = ifs.ReadFrAdcData( 0, *cur );
	}
	catch( ... )
	{
	}
	if ( channel )
	{
	  dump_vect( channel->RefData( )[ 0 ], cl );
	  continue;
	}
      }
      //-----------------------------------------------------------------
      // FrProcData
      //-----------------------------------------------------------------
      {
	LDASTools::AL::SharedPtr< FrProcData >	channel;

        try
	{
	  channel = ifs.ReadFrProcData( 0, *cur );
	}
	catch( ... )
	{
	}
	if ( channel )
	{
	  dump_vect( channel->RefData( )[ 0 ], cl );
	  continue;
	}
      }
      //-----------------------------------------------------------------
      // FrEvent
      //-----------------------------------------------------------------
      {
	LDASTools::AL::SharedPtr< FrEvent >	channel;

	try
	{
	  channel = ifs.ReadFrEvent( 0, *cur );
	}
	catch( ... )
	{
	}
	if ( channel )
	{
	  dump_vect( channel->RefData( )[ 0 ], cl );
	  continue;
	}
      }
      //-----------------------------------------------------------------
      // FrSerData
      //-----------------------------------------------------------------
      {
	LDASTools::AL::SharedPtr< FrSerData >	channel;

	try
	{
	  channel = ifs.ReadFrSerData( 0, *cur );
	}
	catch( ... )
	{
	}
	if ( channel )
	{
	  dump_vect( channel->RefSerial( )[ 0 ], cl );
	  continue;
	}
      }
      //-----------------------------------------------------------------
      // FrSimData
      //-----------------------------------------------------------------
      {
	LDASTools::AL::SharedPtr< FrSimData >	channel;

	try
	{
	  channel = ifs.ReadFrSimData( 0, *cur );
	}
	catch( ... )
	{
	}
	if ( channel )
	{
	  dump_vect( channel->RefData( )[ 0 ], cl );
	  continue;
	}
      }
      //-----------------------------------------------------------------
      // FrSimEvent
      //-----------------------------------------------------------------
      {
	LDASTools::AL::SharedPtr< FrSimEvent >	channel;

	try
	{
	  channel = ifs.ReadFrSimEvent( 0, *cur );
	}
	catch( ... )
	{
	}
	if ( channel )
	{
	  dump_vect( channel->RefData( )[ 0 ], cl );
	  continue;
	}
      }
      //-----------------------------------------------------------------
      // Case where the channel was not found
      //-----------------------------------------------------------------
      if ( exit_code == EXIT_CODE_OK)
      {
	exit_code = EXIT_CODE_UNKNOWN_CHANNEL;
      }
    }

  }
  //---------------------------------------------------------------------
  // Exit
  //---------------------------------------------------------------------
  depart( exit_code );
}
catch( ... )
{
  depart( EXIT_CODE_UNCAUGHT_EXCEPTION );
}

void
dump_vect( vector_type Vect, const CommandLine& Options )
{
  if ( ! Vect )
  {
    return;
  }
  std::cout
    << "Name:     " << Vect->GetName( ) << std::endl
    << "Compress: " << Vect->GetCompress( )
    << " (Endianness: " << ( ( Vect->GetCompress( ) & 0x100 )
			     ? "Little" : "Big" )
    << " Scheme: " << ( Vect->GetCompress( ) & 0xFF ) << ")"
    << std::endl
    << "Type:     " << Vect->GetType( ) << std::endl
    << "NData:    " << Vect->GetNData( ) << std::endl
    << "NBytes:   " << Vect->GetNBytes( ) << std::endl
    ;
  std::cout << "Data: ";
  if ( Options.Raw( ) )
  {
    dump_raw( Vect->GetDataRaw( ), Vect->GetNBytes( ) );
  }
  else
  {
    try
    {
      FrameCPP::FrVect::data_type	expanded;

      switch ( Vect->GetType( ) )
      {
      case Version::FrVectDataTypes::FR_VECT_C:
	dump< CHAR >( Vect->GetDataUncompressed( expanded ), Vect->GetNData( ) );
	break;
      case Version::FrVectDataTypes::FR_VECT_1U:
	dump< CHAR_U >( Vect->GetDataUncompressed( expanded ), Vect->GetNData( ) );
	break;
      case Version::FrVectDataTypes::FR_VECT_2U:
	dump< INT_2U >( Vect->GetDataUncompressed( expanded ), Vect->GetNData( ) );
	break;
      case Version::FrVectDataTypes::FR_VECT_2S:
	dump< INT_2S >( Vect->GetDataUncompressed( expanded ), Vect->GetNData( ) );
	break;
      case Version::FrVectDataTypes::FR_VECT_4U:
	dump< INT_4U >( Vect->GetDataUncompressed( expanded ), Vect->GetNData( ) );
	break;
      case Version::FrVectDataTypes::FR_VECT_4S:
	dump< INT_4S >( Vect->GetDataUncompressed( expanded ), Vect->GetNData( ) );
	break;
      case Version::FrVectDataTypes::FR_VECT_8U:
	dump< INT_8U >( Vect->GetDataUncompressed( expanded ), Vect->GetNData( ) );
	break;
      case Version::FrVectDataTypes::FR_VECT_8S:
	dump< INT_8S >( Vect->GetDataUncompressed( expanded ), Vect->GetNData( ) );
	break;
      case Version::FrVectDataTypes::FR_VECT_4R:
	dump< REAL_4 >( Vect->GetDataUncompressed( expanded ), Vect->GetNData( ) );
	break;
      case Version::FrVectDataTypes::FR_VECT_8R:
	dump< REAL_8 >( Vect->GetDataUncompressed( expanded ), Vect->GetNData( ) );
	break;
      case Version::FrVectDataTypes::FR_VECT_8C:
	dump< COMPLEX_8 >( Vect->GetDataUncompressed( expanded ), Vect->GetNData( ) );
	break;
      case Version::FrVectDataTypes::FR_VECT_16C:
	dump< COMPLEX_16 >( Vect->GetDataUncompressed( expanded ), Vect->GetNData( ) );
	break;
      }
    }
    catch( const std::exception& Exception )
    {
      std::cerr << "ERROR: "
		<< Exception.what( )
		<< std::endl
	;
      exit_code = EXIT_CODE_VECTOR_EXPANSION;
    }
  }
  std::cout << std::endl;
}
