#ifndef INTERPOLATE_HH
#define INTERPOLATE_HH

#include "LinFilt.hh"

//
// Functions for shifting and interpolation of a vector
//

namespace Filters {

    //: Class which implements Lagrange interpolation of a vector
    template<class T>
    class Interpolate {
    public:
        //: Constructor
        //
        //!param: alpha - interpolation parameter, 0 < alpha < 1
        //!param: order - order of Lagrange polynomial used for interpolation
        //
        //!exc: invalid_argument - Thrown if alpha <= 0 or alpha >= 1
        Interpolate(const double alpha, const size_t order);

        //: Get alpha parameter
        double getAlpha() const;

        //: Get order parameter
        size_t getOrder() const;

        //: Apply the interpolation
        //
        // The data contained in x is interpolated and the result returned
        // in x.
        //
        //!param: x - the data to be interpolated
        //
        //!exc: invalid_argument - Thrown if x.size() == 0
        void apply(std::valarray<T>& x);

    private:
        //: alpha parameter is only stored for getAlpha()
        double m_alpha;

        //: the filter used for interpolation.
        LinFilt<double, T> m_filt;
    };

    template<class T>
    inline
    double Interpolate<T>::getAlpha() const { return m_alpha; }

    template<class T>
    inline
    size_t Interpolate<T>::getOrder() const { return (m_filt.getBSize() - 1); }

    // Non-member functions

    //
    //: Calculate FIR coefficients for an interpolating filter
    //
    // This function designs a FIR filter for interpolating the values in
    // a vector with a Lagrange polynomial of order b.size() - 1.
    //
    //!param: b - the array to fill with Lagrange polynomial coefficients.
    //!param: alpha - interpolation parameter
    //
    //!exc: invalid_argument - Thrown if b.size() == 0
    //!exc: invalid_argument - Thrown if alpha <= 0 or alpha >= 1
    template<class T>
    void designInterpolatingFilter(std::valarray<T>& b, const double alpha);
    
} // namespace Filters

#endif // INTERPOLATE_HH
