#include <ldas_tools_config.h>

#include <fstream>
#include <iomanip>

#include "diskcacheAPI/Streams/ASCII.hh"
#include "diskcacheAPI/Streams/Binary.hh"
#include "diskcacheAPI/Streams/FStream.hh"

#include "Commands.hh"
#include "IO.hh"
#include "MetaCommands.hh"

typedef diskCache::MetaCommand::ClientServerInterface::ServerInfo ServerInfo;

using diskCache::MetaCommand::transfer_helper;
using diskCache::Commands::getFileNames;

namespace
{
  class transfer
    : public transfer_helper
  {
  public:

    //-------------------------------------------------------------------
    /// \brief Default constructor
    //-------------------------------------------------------------------
    transfer( );

    //-------------------------------------------------------------------
    /// \brief read the reponce from the stream
    ///
    /// \param[in] Stream
    ///     The output stream from which to read the responce to the
    ///     request.
    ///
    /// \return
    ///     The stream from which the responce was read.
    //-------------------------------------------------------------------
    std::istream& read( std::istream& Stream );

    //-------------------------------------------------------------------
    /// \brief write the reponce onto the stream
    ///
    /// \param[in] Stream
    ///     The output stream on which to write the responce to the
    ///     request.
    ///
    /// \return
    ///     The stream on which the responce was written.
    //-------------------------------------------------------------------
    std::ostream& write( std::ostream& Stream );

    std::stringstream	answer;
  };

}

namespace diskCache
{
  namespace MetaCommand
  {
    //===================================================================
    // Filenames
    //===================================================================
    OptionSet& Filenames::m_options( Filenames::init_options( ) );

    OptionSet& Filenames::
    init_options( )
    {
      static OptionSet	retval;

      retval.
	Synopsis( "Subcommand: filenames" );

      retval.
	Summary( "The filenames sub command is intended to"
		 " query the memory cache."
		 " Several search options are available to restrict"
		 " the set of filename returned."
		 );

      retval.Add( Option( OPT_EXTENSION,
			  "extension",
			  Option::ARG_REQUIRED,
			  "filename extension to search",
			  "extension" ) );

      retval.Add( Option( OPT_IFO_TYPE_LIST,
			  "ifo-type-list",
			  Option::ARG_REQUIRED,
			  "comma seperated list of <ifo>-<type>- entries to search",
			  "<ifo>-<type>- list" ) );

      retval.Add( Option( OPT_START_TIME,
			  "start-time",
			  Option::ARG_REQUIRED,
			  "GPS start time of interest",
			  "gps_time" ) );

      retval.Add( Option( OPT_END_TIME,
			  "end-time",
			  Option::ARG_REQUIRED,
			  "GPS end time of interest",
			  "gps_time" ) );

      return retval;
    }

    Filenames::
    Filenames( CommandLineOptions& Args,
	       const ServerInfo& Server )
      : ClientServerInterface( Server ),
	m_args( Args ),
	m_ifo_type_list( ".gwf" ),
	m_time_start( 0 ),
	m_time_stop( ~0 )
    {
      if ( m_args.empty( ) == false )
      {
	//---------------------------------------------------------------
	// Parse the commands
	//---------------------------------------------------------------
	std::string	arg_name;
	std::string	arg_value;
	bool 		parsing( true );
	int		opt;

	while( parsing )
	{
	  opt = m_args.Parse( m_options, arg_name, arg_value );

	  switch( opt )
	  {
	  case CommandLineOptions::OPT_END_OF_OPTIONS:
	    parsing = false;
	    break;
	  case OPT_EXTENSION:
	    {
	      m_extension = arg_value;
	    }
	    break;
	  case OPT_IFO_TYPE_LIST:
	    {
	      m_ifo_type_list = arg_value;

	      //-----------------------------------------------------------
	      // Replace the commas with spaces
	      //-----------------------------------------------------------
	      size_t pos = 0;

	      while( ( pos = m_ifo_type_list.find_first_of( ",", pos ) )
		     != std::string::npos )
	      {
		m_ifo_type_list[ pos ] = ' ';
	      }
	    }
	    break;
	  case OPT_START_TIME:
	    {
	      std::istringstream( arg_value ) >> m_time_start;
	    }
	    break;
	  case OPT_END_TIME:
	    {
	      std::istringstream( arg_value ) >> m_time_stop;
	    }
	    break;
	  default:
	    break;
	  }
	}
      }
    }

    const OptionSet& Filenames::
    Options( )
    {
      return m_options;
    }

    void Filenames::
    evalClient( )
    {
      //-----------------------------------------------------------------
      // Construct the request
      //-----------------------------------------------------------------
      std::ostringstream	cmd;
      const Option&		extension( m_options[ OPT_EXTENSION ] );
      const Option&		ifo_type( m_options[ OPT_IFO_TYPE_LIST ] );
      const Option&		start( m_options[ OPT_START_TIME ] );
      const Option&		stop( m_options[ OPT_END_TIME ] );

      cmd << CommandTable::Lookup( CommandTable::CMD_FILENAMES )
	  << " " << extension << " " << ( m_extension.size( )
					  ? m_extension
					  : "unspecified" )
	  << " " << ifo_type << " " << ( m_ifo_type_list.size( )
					 ? m_ifo_type_list
					 : "unspecified" )
	  << " " << start << " " << m_time_start
	  << " " << stop << " " << m_time_stop
	  << std::endl
	;

      //-----------------------------------------------------------------
      // Submit the request
      //-----------------------------------------------------------------

      ServerRequest( cmd.str( ) );
      
      //-----------------------------------------------------------------
      // Retrieve the results of the request
      //-----------------------------------------------------------------
      transfer	responce;
      responce.read( *( serverRequestHandle( ) ) );

      std::cout << responce.answer.str( )
	;
    }

    void Filenames::
    evalServer( )
    {
      transfer	responce;

      //-----------------------------------------------------------------
      //-----------------------------------------------------------------

      process( responce.answer );

      //-----------------------------------------------------------------
      // Send the results back to the client
      //-----------------------------------------------------------------
      responce.write( *( clientHandle( ) ) );
    }

    void Filenames::
    evalStandalone( )
    {
      //-----------------------------------------------------------------
      // Standalone mode
      //-----------------------------------------------------------------
      process( std::cout );
    }

    void Filenames::
    process( std::ostream& Stream )
    {
      typedef diskCache::Cache::QueryAnswer::filename_container_type
	file_container_type;
      diskCache::Cache::QueryAnswer	answer;

      getFileNames( answer,
		    m_ifo_type_list.c_str( ),
		    m_time_start,
		    m_time_stop,
		    m_extension );

      file_container_type	files;

      answer.Complete( );
      answer.Swap( files );
      for ( file_container_type::const_iterator
	      cur = files.begin( ),
	      last = files.end( );
	    cur != last;
	    ++cur )
      {
	Stream << *cur
	       << std::endl
	  ;
      }
    }

  } // namespace - MetaCommand
} // namespace - diskCache

namespace
{
  //=====================================================================
  // transfer
  //=====================================================================
  transfer::
  transfer( )
  {
  }

  std::istream& transfer::
  read( std::istream& Stream )
  {
    bool	available;
    Blob( Stream, available, answer );
    return Stream;
  }

  std::ostream& transfer::
  write( std::ostream& Stream )
  {
    Blob( Stream, true, answer );
    Stream.flush( );
    return Stream;
  }
} // namespace - anonymous
