#include <ldas_tools_config.h>

#include <ctype.h>

#include <algorithm>

#include "ldastoolsal/SignalHandler.hh"

#include "genericAPI/Logging.hh"
#include "genericAPI/MonitorMemory.hh"

#include "diskcacheAPI/Cache/ExcludedDirectoriesSingleton.hh"

#include "diskcacheAPI/MetaCommands.hh"

#include "Commands.hh"
#include "diskcachecmd.hh"
#include "DumpCacheDaemon.hh"
#include "MountPointScanner.hh"
#include "ScanMountPointsDaemon.hh"

using GenericAPI::queueLogEntry;
using LDASTools::AL::CommandLineOptions;

using diskCache::Commands::updateFileExtList;
using diskCache::Commands::updateMountPtList;

typedef diskCache::MetaCommand::client_type	client_type;
typedef diskCache::MetaCommand::ClientServerInterface::ServerInfo ServerInfo;

namespace
{
  CommandLineOptions
  convert_line_to_options( char* Buffer )
  {
    //-------------------------------------------------------------------
    // Create list of options.
    //-------------------------------------------------------------------
    char*	lb_opts[ 64 ];
    int		cur_opt = 0;
    char*	lb_pos = Buffer;
    
    lb_opts[ cur_opt ] = lb_pos;
    while( *lb_pos )
    {
      if ( ::isspace( *lb_pos ) )
      {
	*lb_pos = '\0';
	++lb_pos;
	while( *lb_pos && ::isspace( *lb_pos ) )
	{
	  ++lb_pos;
	}
	if ( *lb_pos )
	{
	  lb_opts[ ++cur_opt ] = lb_pos;
	}
	continue;
      }
      ++lb_pos;
    }
    lb_opts[ ++cur_opt ] = (char*)NULL;
      
    CommandLineOptions	retval( cur_opt, lb_opts );

    return retval;
  }
} // namespace - anonymous
namespace diskCache
{
  namespace MetaCommand
  {
    //===================================================================
    // Daemon::Config_
    //===================================================================
    Daemon::Config_::
    Config_( Daemon& Command )
      : command( Command ),
	state( NON_BLOCK )
    {
    }

    void Daemon::Config_::
    Parse( std::istream& Stream )
    {
      base_type::Parse( Stream );
    }

    void Daemon::Config_::
    ParseBlock( const std::string& Value )
    {
      if ( Value.compare( "DAEMON" ) == 0 )
      {
	state = BLOCK_DAEMON;
      }
      else if ( Value.compare( "EXTENSIONS" ) == 0 )
      {
	state = BLOCK_EXTENSIONS;
      }
      else if ( Value.compare( "EXCLUDED_DIRECTORIES" ) == 0 )
      {
	state = BLOCK_EXCLUDED_DIRECTORIES;
      }
      else if ( Value.compare( "MOUNT_POINTS" ) == 0 )
      {
	state = BLOCK_MOUNT_POINTS;
      }
      else
      {
	//---------------------------------------------------------------
	// Tried everything we know. Flag as unknown block
	//---------------------------------------------------------------
	state = BLOCK_UNKNOWN;
      }
    }

    void Daemon::Config_::
    ParseKeyValue( const std::string& Key,
		   const std::string& Value )
    {
      switch( state )
      {
      case NON_BLOCK:
      case BLOCK_DAEMON:
	if ( Key.compare( "CONCURRENCY" ) == 0 )
	{
	  command.set_concurrency( Value );
	}
	else if ( Key.compare( "LOG" ) == 0 )
	{
	  command.set_log( Value );
	}
	else if ( Key.compare( "LOG_DIRECTORY" ) == 0 )
	{
	  command.set_log_directory( Value );
	}
	else if ( Key.compare( "LOG_ROTATE_ENTRY_COUNT" ) == 0 )
	{
	  command.set_log_rotate_entry_count( Value );
	}
	else if ( Key.compare( "OUTPUT_ASCII" ) == 0 )
	{
	  command.set_output_ascii( Value );
	}
	else if ( Key.compare( "OUTPUT_ASCII_VERSION" ) == 0 )
	{
	  command.set_output_ascii_version( Value );
	}
	else if ( Key.compare( "OUTPUT_BINARY" ) == 0 )
	{
	  command.set_output_binary( Value );
	}
	else if ( Key.compare( "OUTPUT_BINARY_VERSION" ) == 0 )
	{
	  command.set_output_binary_version( Value );
	}
	else if ( Key.compare( "SCAN_INTERVAL" ) == 0 )
	{
	  command.set_scan_interval( Value );
	}
	else
	{
	  if ( state == BLOCK_DAEMON )
	  {
	    //-----------------------------------------------------------
	    /// \todo
	    ///     Need to produce and exception regarding unknown
	    ///     keyword found
	    //-----------------------------------------------------------
	  }
	}
	break;
      case BLOCK_EXCLUDED_DIRECTORIES:
      case BLOCK_EXTENSIONS:
      case BLOCK_MOUNT_POINTS:
	//---------------------------------------------------------------
	/// \todo
	///     Need to produce an exception regarding parse error
	///     in configuration file as a key/value pair was encountered
	///     where a word value was expected.
	//---------------------------------------------------------------
	break;
      default:
	//---------------------------------------------------------------
	// Ignore all other cases
	//---------------------------------------------------------------
	break;
      }
    }

    void Daemon::Config_::
    ParseWord( const std::string& Value )
    {
      if ( Value.size( ) <= 0 )
      {
	//---------------------------------------------------------------
	// Ignore empty words
	//---------------------------------------------------------------
	return;
      }
      //-----------------------------------------------------------------
      // Process the word according to the block it was found in
      //-----------------------------------------------------------------
      switch( state )
      {
      case BLOCK_EXTENSIONS:
	command.push_extension( Value );
	break;
      case BLOCK_EXCLUDED_DIRECTORIES:
	command.push_excluded_directory( Value );
	break;
      case BLOCK_MOUNT_POINTS:
	command.push_mount_point( Value );
	break;
      case BLOCK_DAEMON:
      case BLOCK_UNKNOWN:
      case NON_BLOCK:
	// This shouild never happen.
	/// \todo
	/// Have Daemon::Config_::ParseWord throw exception
	/// if it reaches an unreachable state.
	break;
      }
    }

    //===================================================================
    // Daemon
    //===================================================================
    bool Daemon::daemon_mode;
    OptionSet& Daemon::m_options( Daemon::init_options( ) );

    OptionSet& Daemon::
    init_options( )
    {
      static OptionSet	retval;

      retval.
	Synopsis( "Subcommand: daemon" );

      retval.
	Summary( "The daemon sub command is intended to"
		 " provide a continuous scanning mode." );

      retval.Add( Option( OPT_CONCURRENCY,
			  "concurrency",
			  Option::ARG_REQUIRED,
			  "Number of mount points to scan concurrently",
			  "integer" ) );

      retval.Add( Option( OPT_CONFIGURATION_FILE,
			  "configuration-file",
			  Option::ARG_REQUIRED,
			  "Name of file containing additional configuration information.",
			  "filename" ) );

      retval.Add( Option( OPT_EXCLUDED_DIRECTORIES,
			  "excluded-directories",
			  Option::ARG_REQUIRED,
			  "Comma seperated list of directories not to be searched",
			  "list" ) );

      retval.Add( Option( OPT_EXTENSIONS,
			  "extensions",
			  Option::ARG_REQUIRED,
			  "Comma seperated list of file extensions",
			  "list" ) );

      retval.Add( Option( OPT_SCAN_INTERVAL,
			  "interval",
			  Option::ARG_REQUIRED,
			  "Number of milliseconds to pause between scans.",
			  "integer" ) );

      retval.Add( Option( OPT_LOG,
			  "log",
			  Option::ARG_REQUIRED,
			  "Specify where the log messages should be written.",
			  "filename" ) );

      retval.Add( Option( OPT_LOG_DIRECTORY,
			  "log-directory",
			  Option::ARG_REQUIRED,
			  "Specify the directory to capture logging information",
			  "directory" ) );

      retval.Add( Option( OPT_LOG_ROTATE_ENTRY_COUNT,
			  "log-rotate-entry-count",
			  Option::ARG_REQUIRED,
			  "Specify the point at which to rotate the log files.",
			  "count" ) );

      retval.Add( Option( OPT_MOUNT_POINTS,
			  "mount-points",
			  Option::ARG_REQUIRED,
			  "Comma seperated list of mount points to scan",
			  "list" ) );

      retval.Add( Option( OPT_OUTPUT_ASCII,
			  "output-ascii",
			  Option::ARG_REQUIRED,
			  "Filename for the ascii output; '-' to direct to standard output",
			  "filename" ) );

      retval.Add( Option( OPT_OUTPUT_BINARY,
			  "output-binary",
			  Option::ARG_REQUIRED,
			  "Filename for the binary output",
			  "filename" ) );

      retval.Add( Option( OPT_VERSION_ASCII,
			  "version-ascii",
			  Option::ARG_REQUIRED,
			  "Version of the ascii diskcache dump format to output",
			  "version" ) );

      retval.Add( Option( OPT_VERSION_BINARY,
			  "version-binary",
			  Option::ARG_REQUIRED,
			  "Version of the binary diskcache dump format to output",
			  "version" ) );
      return retval;
    }

    //-------------------------------------------------------------------
    /// Contruct a new instance of Daemon.
    //-------------------------------------------------------------------
    Daemon::
    Daemon( CommandLineOptions& Args,
	    const ClientServerInterface::ServerInfo& Server,
	    const std::string& DefaultConfigurationFilename )
      : m_args( Args ),
	m_concurrency( 4 ),
	log_rotate_entry_count( 800 ),
	scan_interval( 16000 ),
	version_ascii( Streams::Interface::VERSION_NONE ),
	version_binary( Streams::Interface::VERSION_NONE ),
	finished( false ),
	server_info( Server )
    {
      {
	configuration_filename = DefaultConfigurationFilename;
	std::ifstream	stream( configuration_filename.c_str( ) );

	if ( stream.is_open( ) )
	{
	  Config_		cfg( *this );

	  cfg.Parse( stream );
	}
      }
      if ( m_args.empty( ) == false )
      {
	//---------------------------------------------------------------
	// Parse the commands
	//---------------------------------------------------------------
	std::string	arg_name;
	std::string	arg_value;
	bool 		parsing( true );

	while( parsing )
	{
	  switch( m_args.Parse( m_options, arg_name, arg_value ) )
	  {
	  case CommandLineOptions::OPT_END_OF_OPTIONS:
	    parsing = false;
	    break;
	  case OPT_CONCURRENCY:
	    set_concurrency( arg_value );
	    break;
	  case OPT_CONFIGURATION_FILE:
	    {
	      configuration_filename = arg_value;
	      std::ifstream	stream( configuration_filename.c_str( ) );

	      if ( stream.is_open( ) )
	      {
		Config_		cfg( *this );

		cfg.Parse( stream );
	      }
	    }
	    break;
	  case OPT_EXCLUDED_DIRECTORIES:
	    {
	      //-----------------------------------------------------------
	      // Generate list of directories
	      //-----------------------------------------------------------
	      Cache::ExcludedDirectoriesSingleton::directory_container_type
		directories;

	      size_t pos = 0;
	      size_t end = 0;

	      while ( end != std::string::npos )
	      {
		//-------------------------------------------------------
		// Extract each directory
		//-------------------------------------------------------
		end = arg_value.find_first_of( ",", pos );
		const size_t dend( ( end == std::string::npos )
				   ? end
				   : ( end - pos ) );
		push_excluded_directory( arg_value.substr( pos, dend ) );
		pos = end + 1;
	      }
	      //---------------------------------------------------------
	      // Update the list of directories to be excluded
	      //---------------------------------------------------------
	      Cache::ExcludedDirectoriesSingleton::Update( directories );
	    }
	    break;
	  case OPT_EXTENSIONS:
	    {
	      //-----------------------------------------------------------
	      // Generate list of extensions
	      //-----------------------------------------------------------
	      size_t pos = 0;
	      size_t end = 0;

	      while ( end != std::string::npos )
	      {
		end = arg_value.find_first_of( ",", pos );
		push_extension( arg_value.substr( pos,( ( end == std::string::npos )
							? end
							: end - pos ) ) );
		pos = end + 1;
	      }
	    }
	    break;
	  case OPT_SCAN_INTERVAL:
	    set_scan_interval( arg_value );
	    break;
	  case OPT_LOG:
	    set_log( arg_value );
	    break;
	  case OPT_LOG_DIRECTORY:
	    set_log_directory( arg_value );
	    break;
	  case OPT_LOG_ROTATE_ENTRY_COUNT:
	    set_log_rotate_entry_count( arg_value );
	    break;
	  case OPT_MOUNT_POINTS:
	    {
	      //-----------------------------------------------------------
	      // Generate list of mount points
	      //-----------------------------------------------------------
	      size_t pos = 0;
	      size_t end = 0;

	      while ( end != std::string::npos )
	      {
		end = arg_value.find_first_of( ",", pos );
		push_mount_point( arg_value.substr( pos,( ( end == std::string::npos )
							  ? end
							  : end - pos ) ) );
		pos = end + 1;
	      }
	    }
	    break;
	  case OPT_OUTPUT_ASCII:
	    set_output_ascii( arg_value );
	    break;
	  case OPT_OUTPUT_BINARY:
	    set_output_binary( arg_value );
	    break;
	  case OPT_VERSION_ASCII:
	    set_output_ascii_version( arg_value );
	    break;
	  case OPT_VERSION_BINARY:
	    set_output_binary_version( arg_value );
	    break;
	  default:
	    break;
	  }
	}
      }

      GenericAPI::LogFormatter( )->EntriesMax( log_rotate_entry_count );
      {
	using GenericAPI::Log::LDAS;

	LDAS::stream_file_type
	  fs( LDASTools::AL::DynamicPointerCast< LDAS::stream_file_type::element_type >
	      ( GenericAPI::LogFormatter( )->Stream( ) ) );
	if ( fs )
	{
	  std::string	ad;
	  GenericAPI::Symbols::LDAS_ARCHIVE_DIR::Get( ad );
	  
	  fs->ArchiveDirectory( ad );
	  fs->FilenamePattern( "diskcache" );
	  fs->FilenameExtension( GenericAPI::LogFormatter( )->FileExtension( ) );
	}
      }
    }

    //-------------------------------------------------------------------
    /// Return resource back to the system
    //-------------------------------------------------------------------
    Daemon::
    ~Daemon( )
    {
      //-----------------------------------------------------------------
      // Remove the daemon from the list of signal handlers
      //-----------------------------------------------------------------
    }

    //-------------------------------------------------------------------
    //-------------------------------------------------------------------
    const OptionSet& Daemon::
    Options( )
    {
      return m_options;
    }

    //-------------------------------------------------------------------
    /// \brief Register signal handler
    //-------------------------------------------------------------------
    void Daemon::
    ResetOnSignal( bool Value )
    {
      using LDASTools::AL::SignalHandler;

      if ( Value )
      {
	SignalHandler::Register( this, SignalHandler::SIGNAL_HANGUP );
      }
      else
      {
	SignalHandler::Unregister( this, SignalHandler::SIGNAL_HANGUP );
      }
    }

    void Daemon::
    SignalCallback( signal_type Signal )
    {
      if ( Signal == LDASTools::AL::SignalHandler::SIGNAL_HANGUP )
      {
	//---------------------------------------------------------------
	// Start the process of resetting
	//---------------------------------------------------------------
	reset( );
      }
    }

    void Daemon::
    do_client_request( )
    {
      static const char* method_name = "diskCache::MetaCommand::Daemon::do_client_request";

      if ( server->good( ) )
      {
	queueLogEntry( "Server: waiting for client request",
		       GenericAPI::LogEntryGroup_type::MT_DEBUG,
		       30,
		       method_name,
		       "CXX" );
	client = server->accept( );

	if ( client->good( ) )
	{
	  Spawn( );
	  Join( );
	}

	queueLogEntry( "Server: client request completed",
		       GenericAPI::LogEntryGroup_type::MT_DEBUG,
		       30,
		       method_name,
		       "CXX" );
	client.reset( ( server_responce_type::element_type* )NULL );
      }
    }

    bool Daemon::
    process_cmd( CommandLineOptions& Options )
    {
      bool	retval = true;

      switch( CommandTable::Lookup( Options.ProgramName( ) ) )
      {
      case CommandTable::CMD_QUIT:
	{
	  //-------------------------------------------------------------
	  // Quit the current contents of the cache
	  //-------------------------------------------------------------
	  MetaCommand::Quit	cmd( Options, server_info );
	  std::ostringstream	msg;

	  cmd.ClientHandle( client );

	  if ( finished )
	  {
	    msg << "Daemon is already shutting down";
	  }
	  else
	  {
	    msg << "Requested daemon to shut down";
	  }
	  cmd.msg = msg.str( );
	  finished = true;
	}
	break;
      case CommandTable::CMD_DUMP:
	{
	  //-------------------------------------------------------------
	  // Dump the current contents of the cache
	  //-------------------------------------------------------------
	  MetaCommand::Dump	cmd( Options, server_info );

	  cmd.ClientHandle( client );

	  cmd( );
	}
	break;
      case CommandTable::CMD_FILENAMES:
	{
	  MetaCommand::Filenames	cmd( Options, server_info );

	  cmd.ClientHandle( client );

	  cmd( );
	}
	break;
      case CommandTable::CMD_FILENAMES_RDS:
	{
	  MetaCommand::FilenamesRDS	cmd( Options, server_info );
	  
	  cmd.ClientHandle( client );

	  cmd( );
	}
	break;
      case CommandTable::CMD_INTERVALS:
	{
	  MetaCommand::Intervals	cmd( Options, server_info );
	  
	  cmd.ClientHandle( client );

	  cmd( );
	}
	break;
      default:
	{
	  retval = false;
	}
	break;
      }
      return retval;
    }

    bool Daemon::
    read_command( char* Buffer, size_t BufferSize )
    {
      bool	retval = false;

      if ( ! server )
      {
	std::cin.getline( Buffer, BufferSize, '\n' );
	retval = ( ( std::cin.good( ) )
		  ? true
		  : false );
		    
      }
      return retval;
    }

    void Daemon::
    operator()( )
    {
      //-----------------------------------------------------------------
      // 
      //-----------------------------------------------------------------
      daemon_mode = true;

      //-----------------------------------------------------------------
      // Setup the logging stream
      //-----------------------------------------------------------------
      std::ofstream	log_stream_base;
      std::ostream*	log_stream = &std::cout;

      if ( m_log.empty( ) == false )
      {
	log_stream_base.open( m_log.c_str( ) );
	if ( log_stream_base.is_open( ) )
	{
	  log_stream = &log_stream_base;
	}
      }

      //-----------------------------------------------------------------
      // Check to see if the server port has been requested
      //-----------------------------------------------------------------
      if ( server_info.Port( ) >= 0 )
      {
	server = server_type( new server_type::element_type( ) );
	if ( server )
	{
	  using LDASTools::AL::SharedPtr;
	  using GenericAPI::Status::MonitorMemory;
	  using GenericAPI::Status::Recorder;

	  //-------------------------------------------------------------
	  // Setup monitoring of memory resources
	  //-------------------------------------------------------------
	  SharedPtr< MonitorMemory >
	    mm( new MonitorMemory( MonitorMemory::MEMORY_VIRTUAL ) );

	  mm->AddDataQueue( 300, 12 * 24 ); 	// Five minutes for 24 hours
	  mm->AddDataQueue( 1800, 48 );		// 30 minutes for 24 hours
	  mm->AddDataQueue( 3600, 24 );		// hour for 24 hours
	  mm->AddDataQueue( 24 * 3600, 30 );	// day for a month

	  Recorder::Add( mm );

	  //-------------------------------------------------------------
	  // Establish listening port
	  //-------------------------------------------------------------
	  server->open( server_info.Port( ) );
	}
      }

      setup_variables( ~0 );

      ScanMountPointListContinuously( log_stream );
      DumpCacheDaemonStart( log_stream );

      //-----------------------------------------------------------------
      // Read commands to be executed.
      //-----------------------------------------------------------------
      if ( server )
      {
	while ( ( finished == false )
		&& ( server->good( ) ) )
	{
	  do_client_request( );
	}
      }
      else
      {
	char	line_buffer[ 2048 ];

	std::fill( line_buffer, line_buffer + sizeof( line_buffer ), '\0' );
	while ( ( finished == false )
		&& ( read_command( line_buffer, sizeof( line_buffer ) ) ) )
	{
	  ;
	  //-------------------------------------------------------------
	  // Switch on the command specified by the user
	  //-------------------------------------------------------------
	  CommandLineOptions
	    line_options( convert_line_to_options( line_buffer ) );

	  process_cmd( line_options);
	}
      }
      //-----------------------------------------------------------------
      // Get read to terminate
      //-----------------------------------------------------------------
      if ( log_stream_base.is_open( ) )
      {
	log_stream_base.close( );
      }
    }

    void Daemon::
    action( )
    {
      char	line_buffer[ 2048 ];

      INT_4U	bytes;

      (*client) >> bytes;
      if ( bytes >= sizeof( line_buffer) )
      {
      }
      else
      {
	client->read( line_buffer, bytes );
	line_buffer[ bytes ] = '\0';

	//---------------------------------------------------------------
	// Create the command line option
	//---------------------------------------------------------------
	CommandLineOptions
	  line_options( convert_line_to_options( line_buffer ) );

	process_cmd( line_options);
      }
    }

    //-------------------------------------------------------------------
    /// Setup the variables according to the requested configuration.
    /// Some of the variables can be reset by modifying the
    /// configuration file and then signaling the daemon process.
    //-------------------------------------------------------------------
    void Daemon::
    setup_variables( int Mask )
    {
      if ( Mask & VAR_EXCLUDED_DIRECTORIES )
      {
	//---------------------------------------------------------------
	// Update the list of directories to be excluded
	//---------------------------------------------------------------
	Cache::ExcludedDirectoriesSingleton::Update( excluded_directories );
      }
      if ( Mask & VAR_EXTENSIONS )
      {
	//---------------------------------------------------------------
	// Add the extensions
	//---------------------------------------------------------------
	{
	  std::ostringstream	msg;
	  msg << "Adding extension: " << m_extensions.size( )
	    ;
	  queueLogEntry( msg.str( ),
			 GenericAPI::LogEntryGroup_type::MT_NOTE,
			 0,
			 "Daemon::setup_variables",
			 "CXX" );
	}
	updateFileExtList( m_extensions );
      }

      if ( Mask & VAR_MOUNT_POINTS )
      {
	//---------------------------------------------------------------
	// Add the mount points
	//---------------------------------------------------------------
	MountPointManagerSingleton::UpdateResults    status;

	updateMountPtList( status, m_mount_points, false );
      }

      if ( Mask & VAR_CONCURRENCY )
      {
	//---------------------------------------------------------------
	// Setup scanning parameters
	//---------------------------------------------------------------
	ScanConcurrency( m_concurrency );
      }

      if ( Mask & VAR_SCAN_INTERVAL )
      {
	//---------------------------------------------------------------
	// Setup scanning parameters
	//---------------------------------------------------------------
	diskCache::ScanMountPointsDaemon::Interval( scan_interval );
      }

      //-----------------------------------------------------------------
      // Update the cache files
      //-----------------------------------------------------------------
      if ( ( Mask & VAR_OUTPUT_ASCII ) && ( output_ascii.size( ) > 0 ) )
      {
	diskCache::DumpCacheDaemon::FilenameAscii( output_ascii );
      }
      if ( ( Mask & VAR_OUTPUT_BINARY ) && ( output_binary.size( ) > 0 ) )
      {
	diskCache::DumpCacheDaemon::FilenameBinary( output_binary );
      }
      if ( ( Mask & VAR_VERSION_ASCII )
	   && ( version_ascii != Streams::Interface::VERSION_NONE ) )
      {
	diskCache::DumpCacheDaemon::ASCIIVersion( version_ascii );
      }
      if ( ( Mask & VAR_VERSION_BINARY )
	   && ( version_binary != Streams::Interface::VERSION_NONE ) )
      {
	diskCache::DumpCacheDaemon::BinaryVersion( version_binary );
      }
    }

    //-------------------------------------------------------------------
    /// Resetting of the daemon process forces the rereading of
    /// vital configuration information without having to restart
    /// the process.
    //-------------------------------------------------------------------
    void Daemon::
    reset( )
    {
      //-----------------------------------------------------------------
      // Re-Read the configuration information
      //-----------------------------------------------------------------
      if ( ! configuration_filename.empty( ) )
      {
	Config_		cfg( *this );
	std::ifstream	stream( configuration_filename.c_str( ) );
      
	cfg.Parse( stream );
      }
      setup_variables( HOT_VARIABLES );
    }
  } // namespace - MetaCommand
} // namespace - diskCache
