#include "diskcacheAPI/Cache/DirectoryManager.hh"
#include "diskcacheAPI/Cache/HotDirectory.hh"
#include "diskcacheAPI/Cache/QueryAnswer.hh"

using diskCache::Cache::HotDirectory;

namespace
{
  class query_functor
  {
  public:
    typedef diskCache::Cache::Directory Directory;
    typedef diskCache::Cache::DirectoryManager DirectoryManager;

    typedef diskCache::Cache::DirectoryManager::ScanResults
    ScanResults;
    typedef diskCache::Cache::QueryAnswer QueryAnswer;

    query_functor( const DirectoryManager& Manager,
		   QueryAnswer& Answer );

    bool Done( ) const;

    void operator()( Directory::dirref_type Node ) const;

    const DirectoryManager&	m_directory_manager;
    QueryAnswer&		m_answer;
  };

  inline query_functor::
  query_functor( const DirectoryManager& Manager,
		 QueryAnswer& Answer )
    : m_directory_manager( Manager ),
      m_answer( Answer )
  {
  }

  inline bool query_functor::
  Done( ) const
  {
    return ( m_answer.IsCompleted( ) );
  }

  void query_functor::
  operator()( Directory::dirref_type Node ) const
  {
    Node->Find( m_answer );
  }

  class scan_functor
  {
  public:
    typedef diskCache::Cache::Directory Directory;
    typedef diskCache::Cache::Directory::dirref_type return_type;
    typedef diskCache::Cache::DirectoryManager DirectoryManager;

    typedef diskCache::Cache::DirectoryManager::ScanResults
    ScanResults;

    scan_functor( DirectoryManager& Manager,
		  ScanResults& Results,
		  const std::string& Caller,
		  const std::string& JobInfo );

    return_type operator()( Directory::dirref_type Node ) const;

    const std::string	m_caller;
    const std::string	m_job_info;
    DirectoryManager&	m_directory_manager;
    ScanResults&	m_results;
  };

  inline scan_functor::
  scan_functor( DirectoryManager& Manager,
		ScanResults& Results,
		const std::string& Caller,
		const std::string& JobInfo )
    : m_caller( Caller ),
      m_job_info( JobInfo ),
      m_directory_manager( Manager ),
      m_results( Results )
  {
  }

  scan_functor::return_type scan_functor::
  operator()( Directory::dirref_type Node ) const
  {
    Directory::dirref_type modified
      = Node->Scan( m_directory_manager,
		    m_results,
		    m_caller,
		    m_job_info );

    if ( modified )
    {
      //-----------------------------------------------------------------
      /// \todo
      /// All of this needs to be part of the node scanning to ensure
      /// proper and minimal locking of critical sections.
      //-----------------------------------------------------------------
      /* m_directory_manager.OnUpdate( modified, m_results ); */
      Node = modified;

    }
    //-------------------------------------------------------------------
    // Check to see if the directory needs to be registered
    //-------------------------------------------------------------------
    HotDirectory::RegisterIfHot( Node, m_directory_manager, true );
    //-------------------------------------------------------------------
    // Record the scanning statistics once it is known
    //-------------------------------------------------------------------
    m_results.FileCountInc( Node->FileCount( ) );
    m_results.DirectoryCountInc( );
    //-------------------------------------------------------------------
    // Return to the caller
    //-------------------------------------------------------------------
    return Node;
  }
}

namespace diskCache
{
  namespace Cache
  {
    DirectoryManager::
    DirectoryManager( )
    {
    }

    void DirectoryManager::
    Find( const std::string& Root,
	  QueryAnswer& Answer ) const
    {
      query_functor	q( *this, Answer );

      Walk( q, Root );
    }

    void DirectoryManager::
    OnUpdate( directory_ref_type Dir,
	      ScanResults& Results )
    {
      //-----------------------------------------------------------------
      // Replace the previous directory information with the updated
      // information.
      //-----------------------------------------------------------------
      AddDirectory( Dir );
      //-----------------------------------------------------------------
      // Check to see if any child directories have been removed as
      //     nodes need to be removed.
      //-----------------------------------------------------------------
      try
      {
	const std::string basename( Dir->Fullname( ) );
	const ScanResults::filename_container_type&
	  dirs( Results.Removed( basename ) );

	for ( ScanResults::filename_container_type::const_iterator
		cur = dirs.begin( ),
		last = dirs.end( );
	      cur != last;
	      ++cur )
	{
	  std::string fullname( basename );
	  fullname += "/";
	  fullname += *cur;
	  RemoveDirectory( fullname );
	}
      }
      catch ( const std::range_error& Error )
      {
	//---------------------------------------------------------------
	// Nothing was deleted
	//---------------------------------------------------------------
      }
      //-----------------------------------------------------------------
      // Check to see if any child directories have been added as
      //     nodes need to be created.
      //-----------------------------------------------------------------
      try
      {
	const std::string basename( Dir->Fullname( ) );
	const ScanResults::filename_container_type&
	  dirs( Results.Added( basename ) );

	for ( ScanResults::filename_container_type::const_iterator
		cur = dirs.begin( ),
		last = dirs.end( );
	      cur != last;
	      ++cur )
	{
	  std::string fullname( basename );
	  fullname += "/";
	  fullname += *cur;
	  CreateDirectory( fullname, Dir->Root( ) );
	}
      }
      catch ( const std::range_error& Error )
      {
	//---------------------------------------------------------------
	// Nothing was added
	//---------------------------------------------------------------
      }
    }

    void DirectoryManager::
    Scan( directory_ref_type Dir, ScanResults& Results )
    {
      scan_functor	s( *this, Results,
			   "SCAN",
			   "HOT_DIR" );

      directory_container_type::mapped_type
	dir( ReferenceDirectory( Dir->Fullname( ) ) );

      if ( dir )
      {
	s( dir );
      }

    }

    void DirectoryManager::
    Scan( const std::string& Root, ScanResults& Results )
    {
      scan_functor	s( *this, Results,
			   "SCAN",
			   "SCAN_MOUNTPT" );

      directory_container_type::mapped_type
	dir = ReferenceDirectory( Root );

      if ( ! dir )
      {
	dir.reset( new Directory( Root, Root ) );
	AddDirectory( dir );
      }

      Walk( s, Root );
    }
  } // namespace diskCache::Cache
} // namespace - diskCache
