/* -*- c++ -*- */
/*
 * Copyright 2004 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street,
 * Boston, MA 02110-1301, USA.
 */

/*
 * gr-baz by Balint Seeber (http://spench.net/contact)
 * Information, documentation & samples: http://wiki.spench.net/wiki/gr-baz
 */

/*
 * config.h is generated by configure.  It contains the results
 * of probing for features, options etc.  It should be the first
 * file included in your .cc file.
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <baz_peak_detector.h>
#include <gnuradio/io_signature.h>
#include <stdio.h>

/*
 * Create a new instance of baz_peak_detector and return
 * a boost shared_ptr.  This is effectively the public constructor.
 */
baz_peak_detector_sptr  baz_make_peak_detector (float min_diff /*= 0.0*/, int min_len /*= 1*/, int lockout /*= 0*/, float drop /*= 0.0*/, float alpha /*= 1.0*/, int look_ahead /*= 0*/, bool byte_output /*= false*/,bool verbose /*= false*/)
{
	return baz_peak_detector_sptr (new baz_peak_detector (min_diff, min_len, lockout, drop, alpha, look_ahead, byte_output, verbose));
}

/*
 * Specify constraints on number of input and output streams.
 * This info is used to construct the input and output signatures
 * (2nd & 3rd args to gr::block's constructor).  The input and
 * output signatures are used by the runtime system to
 * check that a valid number and type of inputs and outputs
 * are connected to this block.  In this case, we accept
 * only 1 input and 1 output.
 */
static const int MIN_IN = 1;	// mininum number of input streams
static const int MAX_IN = 1;	// maximum number of input streams
static const int MIN_OUT = 1;	// minimum number of output streams
static const int MAX_OUT = 2;	// maximum number of output streams

/*
 * The private constructor
 */
baz_peak_detector::baz_peak_detector (float min_diff /*= 0.0*/, int min_len /*= 1*/, int lockout /*= 0*/, float drop /*= 0.0*/, float alpha /*= 1.0*/, int look_ahead /*= 0*/, bool byte_output /*= false*/, bool verbose /*= false*/)
  : gr::block ("peak_detector",
		gr::io_signature::make (MIN_IN,  MAX_IN,  sizeof(float)),
		gr::io_signature::make2 (MIN_OUT, MAX_OUT, (byte_output ? sizeof(char) : sizeof(float)), sizeof(int)))
	, d_min_diff(min_diff)
	, d_min_len(min_len)
	, d_lockout(lockout)
	, d_drop(drop)
	, d_alpha(alpha)
	, d_look_ahead(look_ahead)
	, d_rising(false)
	, d_rise_count(0)
	, d_lockout_count(1)
	, d_first(0.0f)
	, d_ave(0.0f)
	, d_peak(0.0f)
	, d_look_ahead_count(0)
	, d_peak_idx(-1)
	, d_advance(0)
	, d_verbose(verbose)
	, d_last_peak_idx(-1)
	, d_threshold_set(false)
	, d_threshold(0.0f)
	, d_byte_output(byte_output)
{
	fprintf(stderr, "[%s<%li>] min diff: %f, min len: %d, lockout: %d, drop: %f, alpha: %f, look ahead: %d, verbose: %s\n", name().c_str(), unique_id(), min_diff, min_len, lockout, drop, alpha, look_ahead, (verbose ? "yes" : "no"));
	
	set_history(1+1);

	if (look_ahead > 0)
	{
		//set_min_noutput_items(look_ahead + 1);

		//set_min_output_buffer(0, (look_ahead + 1));	// For when latest peak becomes first sample
		//set_min_output_buffer(look_ahead + 1);
		//fprintf(stderr, "Min outpur buffer size: %d\n", min_output_buffer(0));

		set_output_multiple(look_ahead + 1 + 1);	// FIXME: Just +1
	}
}

baz_peak_detector::~baz_peak_detector ()
{
}
/*
void baz_peak_detector::set_exponent(float exponent)
{
  d_exponent = exponent;
}
*/
void baz_peak_detector::forecast(int noutput_items, gr_vector_int &ninput_items_required)
{
	for (size_t i = 0; i < ninput_items_required.size(); ++i)
	{
		ninput_items_required[i] = noutput_items;
		//ninput_items_required[i] = std::max(noutput_items, d_look_ahead_count);
	}
}

void baz_peak_detector::set_threshold(float threshold)
{
	d_threshold = threshold;
	d_threshold_set = true;
}

void baz_peak_detector::unset_threshold()
{
	d_threshold_set = false;
}

//int baz_peak_detector::work (int noutput_items, gr_vector_const_void_star &input_items, gr_vector_void_star &output_items)
int baz_peak_detector::general_work(int noutput_items, gr_vector_int &ninput_items, gr_vector_const_void_star &input_items, gr_vector_void_star &output_items)
{
	const float *in = (const float *) input_items[0];
	float *out = (float *) output_items[0];
	char *out_char = (char *) output_items[0];
	int *idx_diff_out = NULL;
	if (output_items.size() > 1)
	{
		idx_diff_out = (int *) output_items[1];
		memset(idx_diff_out, 0x00, sizeof(int) * noutput_items);
	}
	
	memset(out, 0x00, (d_byte_output ? sizeof(char) : sizeof(float)) * noutput_items);

	//fprintf(stderr, " %d ", noutput_items);

	if ((d_look_ahead > 0) && (/*ninput_items[0]*/noutput_items < (d_look_ahead + 1 + 1)))
	{
		fprintf(stderr, "Too few items for lookahead: %d\n", /*ninput_items[0]*/noutput_items);
		return 0;
	}

	int advance = d_advance;
	d_advance = 0;
	
	const int offset = 1;	// From history
	for (int i = (0+offset + d_advance); i < (noutput_items+offset); i++)
	{
		d_ave = d_alpha * in[i-1] + (1.0f - d_alpha) * d_ave;	// Updating average from previous sample (not this one!)
		
		if (d_lockout_count > 0)
		{
			--d_lockout_count;
			
			if (d_lockout_count > 0)
				continue;
		}
		
		if (((d_threshold_set == false) || (in[i] >= d_threshold)) &&
			(in[i] > (/*in[i-1]*/d_ave - (/*in[i-1]*/d_ave * d_drop))))	// FIXME: Only apply drop factor if already in rise? Otherwise 0.0
		//if (in[i] > in[i-1])
		{
			//if (!d_byte_output) out[i-offset] = 0.5f;

			bool new_peak = false;
			
			if (d_rising == false)
			{
				d_rising = true;
				d_rise_count = 0;
				d_first = in[i];
				
				new_peak = true;
			}
			else
			{
				if (in[i] > d_peak)
				{
					new_peak = true;
				}
			}
			
			if (new_peak)
			{
				d_peak = in[i];
				d_peak_idx = i - offset;

				//if (!d_byte_output) out[d_peak_idx] = 0.5f;

				if (d_look_ahead > 0)
				{
					d_look_ahead_count = d_look_ahead;

					int to_consume = (i - offset);	// Before the current sample

					//fprintf(stderr, "to_consume: %d\n", to_consume);
					
					if (d_look_ahead > (noutput_items - (to_consume + 1 + 1)))	// FIXME: Just +1
					{
						//if (d_peak_idx >= to_consume)
						//	fprintf(stderr, " %d/%d ", d_peak_idx, to_consume);

						d_peak_idx = 0;
						d_advance = 1;

						if (to_consume == 0)	// First iteration and we already don't have enough items
						{
							fprintf(stderr, "Too few items! (%d left)\n", (noutput_items - to_consume));
						}

						if (to_consume > 0)
							consume(0, to_consume);
						
						return to_consume;
					}
				}
			}
			
			++d_rise_count;

			bool bContinue = true;

			if (d_look_ahead_count > 0)
			{
				--d_look_ahead_count;

				if (d_look_ahead_count == 0)
				{
					if (d_verbose) fprintf(stderr, "Look ahead finished while rising %d/%d\n", i, noutput_items);

					bContinue = false;
				}
			}
			
			if (bContinue)
				continue;
		}

		// Get here if no longer 'rising'

		///////////////////////////////////////////////////////////////////////
		
		if (d_look_ahead_count > 0)
		{
			--d_look_ahead_count;
			
			if (d_look_ahead_count > 0)
			{
				continue;
			}

			//fprintf(stderr, "Look ahead finished %d/%d\n", i, noutput_items);
		}
		
		if (d_rising)	// If here, no longer rising
		{
			if (d_rise_count >= d_min_len)
			{
				// FIXME: Abs or ratio?
				float diff = /*in[i]*/d_peak - d_first;
				//float diff = 0.0f;
				//if (d_first > 0.0f)
				//	diff = /*in[i]*/d_peak / d_first;
				
				if ((d_min_diff == 0.0f) || (diff >= d_min_diff))
				{
					//fprintf(stderr, "Peak idx: %d, %f\n", d_peak_idx, d_peak);

					if (d_byte_output)
						out_char[d_peak_idx] = 1;
					else
						out[d_peak_idx] = 1.0f;

					int64_t idx_diff = 0;
					int64_t idx = nitems_written(0)+d_peak_idx;
					if (d_last_peak_idx > -1)
					{
						int64_t idx_diff = idx - d_last_peak_idx;
						if (idx_diff_out != NULL)
							idx_diff_out[d_peak_idx] = (int)idx_diff;
					}
					d_last_peak_idx = idx;
					
					d_lockout_count = d_lockout;
				}
			}
			
			d_rising = false;
		}
	}

	consume(0, noutput_items);

	return noutput_items;
}
