/* -*- c++ -*- */
/*
 * Copyright 2004 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street,
 * Boston, MA 02110-1301, USA.
 */

/*
 * gr-baz by Balint Seeber (http://spench.net/contact)
 * Information, documentation & samples: http://wiki.spench.net/wiki/gr-baz
 */

/*
 * config.h is generated by configure.  It contains the results
 * of probing for features, options etc.  It should be the first
 * file included in your .cc file.
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <baz_sweep.h>
#include <gnuradio/io_signature.h>

#include <stdio.h>

#include <cmath>

#include <boost/thread/condition_variable.hpp>

/*
 * Create a new instance of baz_pow_cc and return
 * a boost shared_ptr.  This is effectively the public constructor.
 */
baz_sweep_sptr 
baz_make_sweep (float samp_rate, float sweep_rate /*= 0.0*/, bool is_duration /*= false*/)
{
  return baz_sweep_sptr (new baz_sweep (samp_rate, sweep_rate, is_duration));
}

/*
 * Specify constraints on number of input and output streams.
 * This info is used to construct the input and output signatures
 * (2nd & 3rd args to gr::block's constructor).  The input and
 * output signatures are used by the runtime system to
 * check that a valid number and type of inputs and outputs
 * are connected to this block.  In this case, we accept
 * only 1 input and 1 output.
 */
static const int MIN_IN = 0;	// mininum number of input streams
static const int MAX_IN = 0;	// maximum number of input streams
static const int MIN_OUT = 1;	// minimum number of output streams
static const int MAX_OUT = 1;	// maximum number of output streams

/*
 * The private constructor
 */
baz_sweep::baz_sweep (float samp_rate, float sweep_rate, bool is_duration)
	: gr::sync_block ("sweep",
		gr::io_signature::make (MIN_IN, MAX_IN, 0),
		gr::io_signature::make (MIN_OUT, MAX_OUT, sizeof(float)))
	, d_samp_rate(samp_rate)
	, d_default_sweep_rate(sweep_rate)
	, d_default_is_duration(is_duration)
	, d_last_output(0.0)
	, d_target(0.0)
	, d_sweep_rate(0.0)
	, d_start_sample(0)
	, d_start_freq(0.0)
{
	fprintf(stderr, "[%s<%i>] sample rate: %f, default sweep rate: %f, is duration: %s\n", name().c_str(), unique_id(), samp_rate, sweep_rate, (is_duration ? "yes" : "no"));
}

/*
 * Our virtual destructor.
 */
baz_sweep::~baz_sweep ()
{
}

void baz_sweep::sweep(float freq, float rate /*= -1.0f*/, bool is_duration /*= false*/, bool block /*= false*/)
{
	if (rate < 0.0)
	{
		rate = d_default_sweep_rate;
		is_duration = d_default_is_duration;
	}
	
	if (is_duration)
	{
		rate = std::abs(freq - d_last_output) / rate;
	}
	
	//if (rate == 0.0)
	//	return;
	
	fprintf(stderr, "[%s<%i>] beginning sweep to %f at %f\n", name().c_str(), unique_id(), freq, rate);
	
	boost::mutex::scoped_lock lock(d_mutex);
	
	if ((d_sweep_rate == 0.0) && (d_last_output == freq))
	{
		fprintf(stderr, "[%s<%i>] already at %f\n", name().c_str(), unique_id(), freq);
		return;
	}
	
	if (d_sweep_rate != 0.0)
	{
		fprintf(stderr, "[%s<%i>] sweep already in progress (to %f at %f while asking for %s at %s\n", name().c_str(), unique_id(), d_target, d_sweep_rate, freq, rate);
		
		d_sweep_done.notify_all();
		
		d_sweep_rate = 0.0;
	}
	
	if (rate == 0.0)
	{
		fprintf(stderr, "[%s<%i>] jumped to %f\n", name().c_str(), unique_id(), freq);
		
		d_last_output = freq;
		
		return;
	}
	
	d_start_freq = d_last_output;
	d_target = freq;
	
	if (freq > d_last_output)
		d_sweep_rate = rate;
	else
		d_sweep_rate = -rate;
	
	d_start_sample = nitems_written(0);
	
	if (block == false)
		return;
	
	fprintf(stderr, "[%s<%i>] waiting for sweep to %f at %f (starting at %lld)\n", name().c_str(), unique_id(), freq, rate, d_start_sample);
	
	d_sweep_done.wait(lock);
	
	fprintf(stderr, "[%s<%i>] completed sweep to %f at %f\n", name().c_str(), unique_id(), freq, rate);
}

int baz_sweep::work (int noutput_items, gr_vector_const_void_star &input_items, gr_vector_void_star &output_items)
{
	boost::mutex::scoped_lock lock(d_mutex);
	
	float* out = (float*)output_items[0];
	
	for (int i = 0; i < noutput_items; i++)
	{
		if (d_sweep_rate != 0.0)
		{
			uint64_t diff = nitems_written(0) - d_start_sample;
			
			double diff_time = (double)diff / (double)d_samp_rate;
			
			//float freq_diff = d_target - d_start_freq;
			
			//d_last_output = d_start_freq + (freq_diff * (float));
			
			double diff_freq = diff_time * (double)d_sweep_rate;
			
			d_last_output = (float)((double)d_start_freq + diff_freq);
			
			if (((d_sweep_rate > 0) && (d_last_output >= d_target)) ||
				((d_sweep_rate < 0) && (d_last_output <= d_target)))
			{
				d_last_output = d_target;
				d_sweep_rate = 0.0;
				
				fprintf(stderr, "[%s<%i>] sweep complete in work\n", name().c_str(), unique_id());
				
				d_sweep_done.notify_all();
			}
		}
		
		out[i] = d_last_output;
	}

	return noutput_items;
}
