// Copyright 2016-2020, Pulumi Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package engine

import (
	"github.com/pkg/errors"
	"google.golang.org/grpc"

	"github.com/pulumi/pulumi/sdk/v3/go/common/resource/plugin"
	"github.com/pulumi/pulumi/sdk/v3/go/common/util/rpcutil"
	pulumirpc "github.com/pulumi/pulumi/sdk/v3/proto/go"
)

type clientLanguageRuntimeHost struct {
	plugin.Host

	languageRuntime plugin.LanguageRuntime
}

func connectToLanguageRuntime(ctx *plugin.Context, address string) (plugin.Host, error) {
	// Dial the language runtime.
	conn, err := grpc.Dial(address, grpc.WithInsecure(),
		grpc.WithUnaryInterceptor(rpcutil.OpenTracingClientInterceptor()), rpcutil.GrpcChannelOptions())
	if err != nil {
		return nil, errors.Wrap(err, "could not connect to language host")
	}

	client := pulumirpc.NewLanguageRuntimeClient(conn)
	return &clientLanguageRuntimeHost{
		Host:            ctx.Host,
		languageRuntime: plugin.NewLanguageRuntimeClient(ctx, clientRuntimeName, client),
	}, nil
}

func (host *clientLanguageRuntimeHost) LanguageRuntime(runtime string) (plugin.LanguageRuntime, error) {
	return host.languageRuntime, nil
}
