// Copyright 2016-2018, Pulumi Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package integration

import (
	"bytes"
	"encoding/json"
	"fmt"
	"path"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/s3"

	"github.com/pulumi/pulumi/sdk/v2/go/common/resource"
)

// S3Reporter is a TestStatsReporter that publises test data to S3
type S3Reporter struct {
	s3svc     *s3.S3
	bucket    string
	keyPrefix string
}

var _ TestStatsReporter = (*S3Reporter)(nil)

// NewS3Reporter creates a new S3Reporter that puts test results in the given bucket using the keyPrefix.
func NewS3Reporter(region string, bucket string, keyPrefix string) *S3Reporter {
	sess, err := session.NewSession(&aws.Config{
		Region: aws.String(region),
	})
	if err != nil {
		fmt.Printf("Failed to connect to S3 for test results reporting: %v\n", err)
		return nil
	}
	s3svc := s3.New(sess)
	return &S3Reporter{
		s3svc:     s3svc,
		bucket:    bucket,
		keyPrefix: keyPrefix,
	}

}

// ReportCommand uploads the results of running a command to S3
func (r *S3Reporter) ReportCommand(stats TestCommandStats) {
	byts, err := json.Marshal(stats)
	if err != nil {
		fmt.Printf("Failed to serialize report for upload to S3: %v: %v\n", stats, err)
		return
	}
	name, _ := resource.NewUniqueHex(fmt.Sprintf("%v-", time.Now().UnixNano()), -1, -1)
	_, err = r.s3svc.PutObject(&s3.PutObjectInput{
		Bucket: aws.String(r.bucket),
		Key:    aws.String(path.Join(r.keyPrefix, name)),
		Body:   bytes.NewReader(byts),
		ACL:    aws.String(s3.ObjectCannedACLBucketOwnerFullControl),
	})
	if err != nil {
		fmt.Printf("Failed to upload test command report to S3: %v\n", err)
		return
	}
}
