// Copyright 2016-2018, Pulumi Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package integration

import (
	"fmt"
	"io/ioutil"
	"os"
	"path"
	"strings"

	"github.com/pulumi/pulumi/sdk/v2/go/common/testing"
	"github.com/pulumi/pulumi/sdk/v2/go/common/workspace"
	"github.com/stretchr/testify/assert"
)

// CreateBasicPulumiRepo will initialize the environment with a basic Pulumi repository and
// project file definition. Returns the repo owner and name used.
func CreateBasicPulumiRepo(e *testing.Environment) {
	e.RunCommand("git", "init")

	contents := "name: pulumi-test\ndescription: a test\nruntime: nodejs\n"
	filePath := fmt.Sprintf("%s.yaml", workspace.ProjectFile)
	filePath = path.Join(e.CWD, filePath)
	err := ioutil.WriteFile(filePath, []byte(contents), os.ModePerm)
	assert.NoError(e, err, "writing %s file", filePath)
}

// GetStacks returns the list of stacks and current stack by scraping `pulumi stack ls`.
// Assumes .pulumi is in the current working directory. Fails the test on IO errors.
func GetStacks(e *testing.Environment) ([]string, *string) {
	out, err := e.RunCommand("pulumi", "stack", "ls")

	outLines := strings.Split(out, "\n")
	if len(outLines) == 0 {
		e.Fatalf("command didn't output as expected")
	}

	// Confirm header row matches.
	// TODO(pulumi/pulumi/issues/496): Provide structured output for pulumi commands. e.g., so we can avoid this
	// err-prone scraping with just deserializings a JSON object.
	assert.True(e, strings.HasPrefix(outLines[0], "NAME"), "First line was: %q\n--\n%q\n--\n%q\n", outLines[0], out, err)

	var stackNames []string
	var currentStack *string
	stackSummaries := outLines[1:]
	for _, summary := range stackSummaries {
		if summary == "" {
			break
		}
		firstSpace := strings.Index(summary, " ")
		if firstSpace != -1 {
			stackName := strings.TrimSpace(summary[:firstSpace])
			if strings.HasSuffix(stackName, "*") {
				currentStack = &stackName
				stackName = strings.TrimSuffix(stackName, "*")
			}
			stackNames = append(stackNames, stackName)
		}
	}

	return stackNames, currentStack
}
