// Copyright 2016-2018, Pulumi Corporation.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package httpstate

import (
	"net/url"
	"os"
	"path"
	"strings"
)

const (
	// ConsoleDomainEnvVar overrides the way we infer the domain we assume the Pulumi Console will
	// be served from, and instead just use this value. e.g. so links to the stack update go to
	// https://pulumi.example.com/org/project/stack/updates/2 instead.
	ConsoleDomainEnvVar = "PULUMI_CONSOLE_DOMAIN"

	// PulumiCloudURL is the Cloud URL used if no environment or explicit cloud is chosen.
	PulumiCloudURL = "https://" + defaultAPIDomainPrefix + "pulumi.com"

	// defaultAPIDomainPrefix is the assumed Cloud URL prefix for typical Pulumi Cloud API endpoints.
	defaultAPIDomainPrefix = "api."
	// defaultConsoleDomainPrefix is the assumed Cloud URL prefix typically used for the Pulumi Console.
	defaultConsoleDomainPrefix = "app."
)

// cloudConsoleURL returns a URL to the Pulumi Cloud Console, rooted at cloudURL. If there is
// an error, returns "".
func cloudConsoleURL(cloudURL string, paths ...string) string {
	u, err := url.Parse(cloudURL)
	if err != nil {
		return ""
	}

	switch {
	case os.Getenv(ConsoleDomainEnvVar) != "":
		// Honor a PULUMI_CONSOLE_DOMAIN environment variable to override the
		// default behavior. Since we identify a backend by a single URI, we
		// cannot know what the Pulumi Console is hosted at...
		u.Host = os.Getenv(ConsoleDomainEnvVar)
	case strings.HasPrefix(u.Host, defaultAPIDomainPrefix):
		// ... but if the cloudURL (API domain) is "api.", then we assume the
		// console is hosted at "app.".
		u.Host = defaultConsoleDomainPrefix + u.Host[len(defaultAPIDomainPrefix):]
	case u.Host == "localhost:8080":
		// ... or when running locally, on port 3000.
		u.Host = "localhost:3000"
	default:
		// We couldn't figure out how to convert the api hostname into a console hostname.
		// We return "" so that the caller can know to omit the URL rather than just
		// return an incorrect one.
		return ""
	}

	u.Path = path.Join(paths...)
	return u.String()
}
