subroutine uv_shift_mosaic(line,error)
  use clean_def
  use clean_default
  use clean_arrays
  use phys_const
  use gkernel_types
  use gkernel_interfaces
  use gbl_message
  use mapping_interfaces, except_this => uv_shift_mosaic
  !-------------------------------------------------------------------
  ! @ private
  !  MAPPING
  !   Support routine for command
  !       UV_SHIFT  [Offx Offy [Angle]]
  !
  !   Phase shift a Mosaic or Single Dish UV Table to a new
  !   common phase center and orientation
  !     Offx OffY are offsets in ArcSec 
  !               (default 0,0)
  !     Angle     is the final position angle from North in Degree
  !               (default: no change)
  !
  !   Also called implicitely (with no command line arguments, i.e.
  !   no change of Phase Center unless the UV Table is with
  !   Phase Offsets )
  !   by command UV_MAP for Mosaics
  !-------------------------------------------------------------------
  character(len=*), intent(in)    :: line   ! Command line
  logical,          intent(inout) :: error  ! Logical error flag
  ! Local
  character(len=*), parameter :: rname='UV_SHIFT'
  character(len=message_length) :: mess
  real(kind=8) :: abso(2),rela(2),freq,angnew,angold
  real(kind=4) :: cs(2), costheta, sintheta, oldoffx, oldoffy
  real(kind=8), allocatable :: rpos(:,:)
  logical, parameter :: precise=.true.
  integer(kind=4) :: nc,nu,nv,i,iv,ier,loff,moff
  character(len=14) :: cra
  character(len=15) :: cdec
  !
  if (themap%nfields.eq.0) then
    mess = 'UV table is a single field'
  elseif (themap%nfields.lt.0) then
    write(mess,'(A,I0,A)')  &
      'UV table is an observed mosaic of ',-themap%nfields,' fields, phase shifting...'
  elseif (themap%nfields.gt.0) then
    write(mess,'(A,I0,A)')  &
      'UV table is already a phase-shifted mosaic of ',themap%nfields,' fields, recentering...'
  endif
  call map_message(seve%i,rname,mess)
  !
  call uv_shift_center(line,abso,rela,angnew,error)
  if (error)  return
  ! Feedback
  call rad2sexa(abso(1),24,cra)
  call rad2sexa(abso(2),360,cdec)
  write(mess,'(5A,2(F6.2,A))')  'New phase center ',  &
    cra,' ',cdec,' (offsets ',rela(1)*sec_per_rad,' ',rela(2)*sec_per_rad,')'
  call map_message(seve%i,rname,mess)
  !
  if (themap%nfields.ge.0) then
    ! Single field, or mosaic already in pointing offsets
    if (huv%gil%a0.eq.abso(1) .and. huv%gil%d0.eq.abso(2) .and. angnew.eq.huv%gil%pang) then
      call map_message(seve%w,rname,  &
        'UV table has already the desired phase center, nothing done')
      return
    endif
  endif
  !
  ! Store new absolute positions
  huv%gil%a0 = abso(1)
  huv%gil%d0 = abso(2)
  if (themap%nfields.ne.0) then
    ! For mosaics, i.e. UVT with offset columns, these offsets (pointing offsets,
    ! and phase offsets if relevant) are relative to (a0,d0). (ra,dec) is purely
    ! informative => synced with (a0,d0) for clarity
    huv%gil%ra = huv%gil%a0
    huv%gil%dec = huv%gil%d0
    ! For non-mosaics, i.e. UVT without offset columns, (a0,d0) is the phase center,
    ! (ra,dec) is the pointing position => must not change
  endif
  !
  ! Compute observing frequency, and new phase center in wavelengths
  if (precise) then
    nc = huv%gil%nchan
    allocate(rpos(2,nc),stat=ier)
    do i=1,huv%gil%nchan
      freq = gdf_uv_frequency(huv,dble(i))
      rpos(1:2,i) = - freq * f_to_k * rela(1:2)
    enddo
  else
    nc = 1
    allocate(rpos(2,1),stat=ier)
    freq = gdf_uv_frequency(huv)
    rpos(1:2,1) = - freq * f_to_k * rela(1:2)
  endif
  !
  ! Define the rotation
  angold = huv%gil%pang
  huv%gil%pang = angnew
  !
  ! The minus in cs(2) follows from the definition of the postion angle in GILDAS
  ! For the Cosine no change in signal, for the Sine the sign must be reversed.
  cs(1) =  cos(huv%gil%pang-angold)
  cs(2) = -sin(huv%gil%pang-angold)
  !
  ! Recenter all channels, Loop over line table
  ! We work in place - We do not play with UV buffers
  nu = huv%gil%dim(1)
  nv = huv%gil%nvisi
  call shift_uvdata (huv,nu,nv,duv,cs,nc,rpos)
  !
  if (themap%nfields.eq.0) then
    ! Done if Single Field
    return
    !
  elseif (themap%nfields.lt.0) then
    ! Phase offsets mosaics
    !
    ! Offset columns description: convert columns kind
    loff = huv%gil%column_pointer(code_uvt_loff)
    moff = huv%gil%column_pointer(code_uvt_moff)
    !
    huv%gil%column_pointer(code_uvt_xoff) = loff
    huv%gil%column_pointer(code_uvt_yoff) = moff
    huv%gil%column_size(code_uvt_xoff) = 1
    huv%gil%column_size(code_uvt_yoff) = 1
    !
    huv%gil%column_pointer(code_uvt_loff) = 0
    huv%gil%column_pointer(code_uvt_moff) = 0
    huv%gil%column_size(code_uvt_loff) = 0
    huv%gil%column_size(code_uvt_moff) = 0
    !
    ! Offset columns data:
    do iv=1,nv
      duv(loff,iv) = duv(loff,iv) - rela(1)
      duv(moff,iv) = duv(moff,iv) - rela(2)
    enddo
    !
  elseif (themap%nfields.gt.0) then
    ! Pointing offsets mosaics
    !
    ! Offset columns description: does not change
    loff = huv%gil%column_pointer(code_uvt_xoff)
    moff = huv%gil%column_pointer(code_uvt_yoff)
    !
    ! Offset columns data: shifted to new (a0,d0)
    do iv=1,nv
      duv(loff,iv) = duv(loff,iv) - rela(1)
      duv(moff,iv) = duv(moff,iv) - rela(2)
    enddo
    !
  endif
  !
  ! The correct thing to do is a reprojection, this rotation is just an approximation.
  costheta = cs(1)
  sintheta = cs(2)
  if (abs(cs(1)-1.).gt.1e-6) then
     do iv=1,nv
        oldoffx = duv(loff,iv)
        oldoffy = duv(moff,iv)
        duv(loff,iv) = costheta*oldoffx - sintheta*oldoffy
        duv(moff,iv) = sintheta*oldoffx + costheta*oldoffy
     enddo
  endif
  !
  ! Recompute the 'themap' structure
  call check_uvdata_type(huv,duv,themap,error)
  !
end subroutine uv_shift_mosaic
!
subroutine uv_shift_center(line,abso,rela,angle,error)
  use phys_const
  use gbl_message
  use gkernel_types
  use gkernel_interfaces
  use mapping_interfaces, except_this=>uv_shift_center
  use clean_arrays
  !---------------------------------------------------------------------
  ! @ private
  ! Support routine for command UV_SHIFT
  ! Get relative and absolute positions from user inputs
  !---------------------------------------------------------------------
  character(len=*), intent(in)    :: line     !
  real(kind=8),     intent(out)   :: abso(2)  ! [rad] New absolute position
  real(kind=8),     intent(out)   :: rela(2)  ! [rad] New relative position from old absolute position
  real(kind=8),     intent(out)   :: angle    ! [rad] New angle
  logical,          intent(inout) :: error    !
  ! Local
  character(len=*), parameter :: rname='UV_SHIFT'
  character(len=24) :: argum
  integer(kind=4) :: nc
  real(kind=8) :: vars(3),mean(2)
  logical :: dorela,found(3)
  type(projection_t) :: proj
  logical, parameter :: domean=.false.
  !
  ! ZZZ why -pang??? Seems it comes from this comment:
  !  "note Position Angle convention differ in GreG and astronomy"
  call gwcs_projec(huv%gil%a0,huv%gil%d0,-huv%gil%pang,huv%gil%ptyp,proj,error)
  if (error)  return
  !
  if (themap%nfields.eq.0) then
    ! Not a mosaic
    mean(:) = 0.d0  ! Unused
  else
    ! Mosaic
    if (domean) then
      ! Use mean of all fields
      mean(:) = sum(themap%offxy,dim=2)/abs(themap%nfields)
    else
      ! Use center of all fields (center of map which contains all fields)
      mean(:) = (minval(themap%offxy,dim=2) + maxval(themap%offxy,dim=2))/2.d0
    endif
  endif
  !
  ! Get inputs from command line
  if (sic_present(0,1)) then
    ! Rigth ascension
    call sic_ch(line,0,1,argum,nc,.true.,error)
    if (error)  return
    if (index(argum,':').ne.0) then
      ! A sexagesimal string: asbolute RA position
      call sic_decode(argum,abso(1),24,error)
      if (error)  return
      dorela = .true.
    else
      ! Assume relative offset
      call sic_math_dble(argum,nc,rela(1),error)
      if (error)  return
      rela(1) = rela(1)*rad_per_sec
      ! In case of single field or pointing offset mosaic, offsets are
      ! relative to [A0,D0] => no additional correction
      ! In case of phase offset mosaic, offsets are relative from the mean
      ! mosaic center. This makes UV_SHIFT 0 0 behaves like UV_SHIFT (without
      ! arguments) => correct them as offsets relative to [A0,D0]
      if (themap%nfields.lt.0)  rela(1) = rela(1)+mean(1)
      dorela = .false.
    endif
    !
    ! Declination
    call sic_ch(line,0,2,argum,nc,.true.,error)
    if (error)  return
    if (index(argum,':').ne.0) then
      ! A sexagesimal string: asbolute DEC position
      call sic_decode(argum,abso(2),360,error)
      if (error)  return
    else
      ! Assume relative offset
      call sic_math_dble(argum,nc,rela(2),error)
      if (error)  return
      rela(2) = rela(2)*rad_per_sec
      ! See comments for RA offsets
      if (themap%nfields.lt.0)  rela(2) = rela(2)+mean(2)
    endif
    !
    if (dorela) then
      call abs_to_rel(proj,abso(1),abso(2),rela(1),rela(2),1)
    else
      call rel_to_abs(proj,rela(1),rela(2),abso(1),abso(2),1)
    endif
    !
    ! Angle (optional)
    angle = huv%gil%pang*deg_per_rad
    call sic_r8(line,0,3,angle,.false.,error)
    if (error)  return
    angle = angle*rad_per_deg
    !
    return
  endif
  !
  ! Get inputs from MAP_RA, MAP_DEC, MAP_ANGLE
  call sic_get_char('MAP_RA',argum,nc,error)
  if (error)  return
  if (argum.ne.' ') then
    call map_get_radecang(rname,found,vars,error)
    if (error)  return
    if (.not.all(found)) then
      call map_message(seve%e,rname,'MAP_RA and MAP_DEC are not all defined')
      error = .true.
      return
    endif
    !
    abso(1) = vars(1)
    abso(2) = vars(2)
    angle   = vars(3)
    call abs_to_rel(proj,abso(1),abso(2),rela(1),rela(2),1)
    !
    return
  endif
  !
  ! Define automatic inputs from data
  if (themap%nfields.eq.0) then
    call map_message(seve%w,rname,'No automatic shift for single field') 
    ! No shift nor rotation by default
    abso(1) = huv%gil%a0
    abso(2) = huv%gil%d0
    rela(1) = 0.d0
    rela(2) = 0.d0
    angle   = huv%gil%pang
    !
  elseif (themap%nfields.gt.0) then
    ! Already a common phase center: do not change it by default
    call map_message(seve%w,rname,'Mosaic already in pointing offsets, no automatic shift applied')
    ! No shift nor rotation by default
    abso(1) = huv%gil%a0
    abso(2) = huv%gil%d0
    rela(1) = 0.d0
    rela(2) = 0.d0
    angle   = huv%gil%pang
    !
  elseif (themap%nfields.lt.0) then
    ! Compute the mean offset as common phase center position
    call map_message(seve%w,rname,'Phase offset mosaic shifted to mean offset')
    rela(:) = mean(:)
    call rel_to_abs(proj,rela(1),rela(2),abso(1),abso(2),1)
    angle = huv%gil%pang*deg_per_rad
    !
  endif
  !
end subroutine uv_shift_center
