!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!
module cubeio_cube
  use cubetools_parameters
  use cubetools_setup_types
  use cubeio_desc
  use cubeio_memory
  use cubeio_file
  use cubeio_timing
  !
  type :: cubeio_cube_t
    type(cubeio_desc_t)          :: desc  ! [public]  User friendly cube description
    type(cubeio_memory_t)        :: memo  ! [private] Data placeholder (memory mode)
    type(cubeio_file_t), pointer :: file  ! [private] Support for file on disk (disk mode)
    type(cubeio_time_t)          :: time  ! [private] time measurements
  contains
    procedure :: order     => cubeio_get_order
    procedure :: access    => cubeio_get_access
    procedure :: iscplx    => cubeio_get_iscplx
    procedure :: haskind   => cubeio_has_filekind
    procedure :: nbytes    => cubeio_get_nbytes
    procedure :: nentry    => cubeio_get_nentry
    procedure :: ndata     => cubeio_get_ndata
    procedure :: size      => cubeio_get_size
    procedure :: planesize => cubeio_get_planesize
    procedure :: chansize  => cubeio_get_chansize
    procedure :: ysize     => cubeio_get_ysize
    procedure :: memsize   => cubeio_get_memsize
    procedure :: ready     => cubeio_data_ready
    procedure :: open      => cubeio_cube_open
    procedure :: close     => cubeio_cube_close
    procedure :: init      => cubeio_cube_init
    procedure :: free      => cubeio_cube_free
    procedure :: feedback  => cubeio_cube_feedback
  ! final     :: cubeio_cube_final  => crash if implicit (order matters?)
  end type cubeio_cube_t
  !
  public :: cubeio_cube_t
  public :: cubeio_cube_final
  private
  !
contains
  !
  function cubeio_get_order(cub)
    integer(kind=code_k) :: cubeio_get_order
    class(cubeio_cube_t), intent(in) :: cub
    cubeio_get_order = cub%desc%order
  end function cubeio_get_order
  !
  function cubeio_get_access(cub)
    integer(kind=code_k) :: cubeio_get_access
    class(cubeio_cube_t), intent(in) :: cub
    cubeio_get_access = cub%desc%access
  end function cubeio_get_access
  !
  function cubeio_get_iscplx(cub)
    logical :: cubeio_get_iscplx
    class(cubeio_cube_t), intent(in) :: cub
    cubeio_get_iscplx = cub%desc%iscplx
  end function cubeio_get_iscplx
  !
  function cubeio_has_filekind(cub,code_filekind)
    !-------------------------------------------------------------------
    ! Return .true. if the cubeio_cube_t provides the requested kind of
    ! file
    !-------------------------------------------------------------------
    logical :: cubeio_has_filekind
    class(cubeio_cube_t), intent(in) :: cub
    integer(kind=code_k), intent(in) :: code_filekind
    !
    ! The cube can be in memory and/or disk for this cubeio_cube_t.
    ! Request is to know if the given file kind is available on disk.
    cubeio_has_filekind = cub%file%iskind(code_filekind)
  end function cubeio_has_filekind
  !
  function cubeio_get_nbytes(cub)
    integer(kind=4) :: cubeio_get_nbytes
    class(cubeio_cube_t), intent(in) :: cub
    if (cub%iscplx()) then
      cubeio_get_nbytes = 8
    else
      cubeio_get_nbytes = 4
    endif
  end function cubeio_get_nbytes
  !
  function cubeio_get_nentry(cub)
    integer(kind=entr_k) :: cubeio_get_nentry
    class(cubeio_cube_t), intent(in) :: cub
    select case (cub%desc%access)
    case (code_access_imaset)
      cubeio_get_nentry = cub%desc%nc
    case (code_access_speset)
      cubeio_get_nentry = cub%desc%nx*cub%desc%ny
    case (code_access_subset)
      cubeio_get_nentry = cub%desc%n3
    case (code_access_fullset)
      cubeio_get_nentry = cub%desc%n3
    case default
      cubeio_get_nentry = 0
    end select
  end function cubeio_get_nentry
  !
  function cubeio_get_ndata(cub)
    !-------------------------------------------------------------------
    ! Return the number of data values
    !-------------------------------------------------------------------
    integer(kind=data_k) :: cubeio_get_ndata
    class(cubeio_cube_t), intent(in) :: cub
    cubeio_get_ndata = cub%desc%n1 * cub%desc%n2 * cub%desc%n3
  end function cubeio_get_ndata
  !
  function cubeio_get_size(cub)
    !-------------------------------------------------------------------
    ! Return the cube data size in bytes (allocated or not)
    !-------------------------------------------------------------------
    integer(kind=data_k) :: cubeio_get_size
    class(cubeio_cube_t), intent(in) :: cub
    cubeio_get_size = cub%ndata()
    if (cub%desc%iscplx) then
      cubeio_get_size = cubeio_get_size*8
    else
      cubeio_get_size = cubeio_get_size*4
    endif
  end function cubeio_get_size
  !
  function cubeio_get_planesize(cub)
    !-------------------------------------------------------------------
    ! Return the one plane (1st x 2nd dimension) size in bytes
    !-------------------------------------------------------------------
    integer(kind=size_length) :: cubeio_get_planesize
    class(cubeio_cube_t), intent(in) :: cub
    cubeio_get_planesize = cub%desc%n1 * cub%desc%n2
    if (cub%desc%iscplx) then
      cubeio_get_planesize = cubeio_get_planesize*8
    else
      cubeio_get_planesize = cubeio_get_planesize*4
    endif
  end function cubeio_get_planesize
  !
  function cubeio_get_chansize(cub)
    !-------------------------------------------------------------------
    ! Return the one channel (2D plane) size in bytes
    !-------------------------------------------------------------------
    integer(kind=size_length) :: cubeio_get_chansize
    class(cubeio_cube_t), intent(in) :: cub
    cubeio_get_chansize = cub%desc%nx * cub%desc%ny
    if (cub%desc%iscplx) then
      cubeio_get_chansize = cubeio_get_chansize*8
    else
      cubeio_get_chansize = cubeio_get_chansize*4
    endif
  end function cubeio_get_chansize
  !
  function cubeio_get_ysize(cub)
    !-------------------------------------------------------------------
    ! Return the one Y row (2D plane) size in bytes
    !-------------------------------------------------------------------
    integer(kind=size_length) :: cubeio_get_ysize
    class(cubeio_cube_t), intent(in) :: cub
    cubeio_get_ysize = cub%desc%nx * cub%desc%nc
    if (cub%desc%iscplx) then
      cubeio_get_ysize = cubeio_get_ysize*8
    else
      cubeio_get_ysize = cubeio_get_ysize*4
    endif
  end function cubeio_get_ysize
  !
  function cubeio_get_memsize(cub)
    !-------------------------------------------------------------------
    ! Return the memory footprint in bytes
    !-------------------------------------------------------------------
    integer(kind=size_length) :: cubeio_get_memsize
    class(cubeio_cube_t), intent(in) :: cub
    !
    cubeio_get_memsize =  &
      cub%file%memsize() +  &
      cub%memo%memsize()
  end function cubeio_get_memsize
  !
  function cubeio_data_ready(cub)
    !-------------------------------------------------------------------
    ! Return .true. if the data has been prepared for access (memory or
    ! disk). Data is prepared when invoking one of the cubeio_iterate_*
    ! subroutines.
    !-------------------------------------------------------------------
    logical :: cubeio_data_ready
    class(cubeio_cube_t), intent(in) :: cub
    !
    cubeio_data_ready = cub%memo%ready.eq.cub%desc%buffered
  end function cubeio_data_ready
  !
  subroutine cubeio_cube_open(cub,error)
    !-------------------------------------------------------------------
    ! Open (or reopen) a 'cubeio_cube_t' instance for READ-ONLY access,
    ! if not yet opened.
    !-------------------------------------------------------------------
    class(cubeio_cube_t), intent(inout) :: cub    !
    logical,              intent(inout) :: error  !
    !
    call cub%file%open(error)
    if (error)  return
  end subroutine cubeio_cube_open
  !
  subroutine cubeio_cube_close(cub,error)
    !-------------------------------------------------------------------
    ! Close the file of a 'cubeio_cube_t' instance
    !-------------------------------------------------------------------
    class(cubeio_cube_t), intent(inout) :: cub    !
    logical,              intent(inout) :: error  !
    !
    call cub%file%close(error)
    if (error)  return
  end subroutine cubeio_cube_close
  !
  subroutine cubeio_cube_init(cub,error)
    !-------------------------------------------------------------------
    ! Initialize a cubeio_cube_t
    !-------------------------------------------------------------------
    class(cubeio_cube_t), intent(inout) :: cub
    logical,              intent(inout) :: error
    !
    ! Unclear if all access timings should be initialized here.
    call cub%time%init(error)
    if (error)  return
    !
    cub%file => file_allocate_new(error)
    if (error)  return
  end subroutine cubeio_cube_init
  !
  subroutine cubeio_cube_free(cub,error)
    !-------------------------------------------------------------------
    ! Free the memory-consuming components of a 'cubeio_cube_t' instance.
    ! The cubeio_cube_t remains useable after this free. It is the
    ! responsibility of the caller to ensure the data remains available
    ! elsewhere (most likely on disk).
    ! Use cubeio_cube_final to free consistently all the object.
    !-------------------------------------------------------------------
    class(cubeio_cube_t), intent(inout) :: cub
    logical,              intent(inout) :: error
    !
    ! Beware!
    ! If this cube is memory-only, it should be completely discarded.
    ! If it has a file associated on disk, only the data in memory is deleted.
    call cub%file%free(error)
    if (error)  return
    call cub%memo%free(error)
    if (error)  return
    if (cub%desc%buffered.eq.code_buffer_memory) then
      call cubeio_desc_reset(cub%desc,error)
      if (error)  return
    endif
  end subroutine cubeio_cube_free
  !
  subroutine cubeio_cube_final(cub)
    use cubeio_block
    !-------------------------------------------------------------------
    ! Finalize a 'cubeio_cube_t' instance
    !-------------------------------------------------------------------
    type(cubeio_cube_t), intent(inout) :: cub    !
    ! Local
    logical :: error
    !
    error = .false.
    call cub%file%free(error)
    if (error)  continue
    call cub%memo%free(error)
    if (error)  continue
    call cub%close(error)  ! Before cubeio_desc_final
    if (error)  continue
    call cubeio_desc_final(cub%desc) ! Explicit because above component rely on it
    deallocate(cub%file)
  end subroutine cubeio_cube_final

  subroutine cubeio_cube_feedback(cub,cubset)
    !-------------------------------------------------------------------
    !-------------------------------------------------------------------
    class(cubeio_cube_t), intent(in) :: cub
    type(cube_setup_t),   intent(in) :: cubset
    !
    ! Show only if requested
    if (.not.cubset%timing%io)  return
    call cub%time%feedback(cub%size())
  end subroutine cubeio_cube_feedback
end module cubeio_cube
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
