program uv_shift
  !----------------------------------------------------------------------
  ! GILDAS
  ! Shift and Rotate a UV Table for further use by UVMAP
  !----------------------------------------------------------------------
  use gildas_def
  use gkernel_interfaces
  use gbl_constant
  !
  character(len=filename_length) :: name
  character(len=24) :: ra_c,dec_c
  real(4) :: ang
  real(8) :: pos(2)
  logical error,rel,precise
  !
  call gildas_open
  call gildas_char('UVTABLE$',name)
  call gildas_logi('RELATIVE$',rel,1)
  if (rel) then
    call gildas_dble('OFFSET$',pos,2)
  else
    call gildas_char('RA_CENTER$',ra_c)
    call gildas_char('DEC_CENTER$',dec_c)
  endif
  call gildas_real('ANGLE$',ang,1)
  call gildas_logi('PRECISE$',precise,1)
  call gildas_close
  !
  call sub_uv_shift(name,rel,pos,ra_c,dec_c,ang,precise,error)
  if (error) call sysexi(fatale)
end program uv_shift
!
subroutine sub_uv_shift(name,rel,pos,ra_c,dec_c,ang,precise,error)
  use gildas_def
  use image_def
  use gkernel_interfaces
  use gkernel_types
  use phys_const
  use gbl_message
  !----------------------------------------------------------------------
  ! GILDAS
  ! Shift and Rotate a UV Table for further use by UVMAP
  !
  ! S.Guilloteau    LAB
  !----------------------------------------------------------------------
  !
  character(len=*), intent(in) :: name
  logical, intent(in) :: rel
  character(len=*), intent(in) :: ra_c
  character(len=*), intent(in) :: dec_c
  real(4), intent(in) :: ang
  real(8), intent(inout) :: pos(2)
  logical, intent(in) :: precise
  logical, intent(out) :: error
  !
  character(len=*), parameter :: rname='UV_SHIFT'
  character(len=80) :: mess
  integer :: loff, moff
  integer :: nchar, nblock, i, nvisi
  integer :: ier, nc
  real :: cs(2)
  real(8), allocatable :: rpos(:,:)
  real(8) :: o_pang, n_pang, freq, a0,d0
  type (gildas) ::       ht_      ! header of input table
  type(projection_t) :: proj
  logical :: error_1,error_2
  !
  ! 1.0  Support shifting Mosaic UV tables with Pointing offsets only
  ! 2.0  Support shifting Mosaic UV tables with Phase(=Pointing) offsets
  !      (virtual) Pointing center reset to Phase center if the UV table
  !      was initially a Phase offset mosaic UV table.
  !
  ! In a Mosaic UV table, the Offset values are always from the Pointing center.
  ! However, this is unclear for a Single-Field...
  !
  ! character(len=*), parameter :: version='Version 2.0  20-Sep-2016 S.Guilloteau'
  character(len=*), parameter :: version='Version 2.1  21-Sep-2017 S.Guilloteau'
  !
  call gag_message(seve%i,rname,version)
  error = .false.
  nchar = len_trim(name)
  if (nchar.eq.0) then
    call map_message(seve%e,rname,'No input table')
    error = .true.
    return
  endif
  !
  call gildas_null(ht_, 'UVT')
  call gdf_read_gildas (ht_, name, '.uvt', error, data=.false.)
  if (error) then
    call map_message(seve%e,rname,'Cannot read input table')
    return
  endif
  !
  ! Modify header : note Position Angle convention differ in GreG and astronomy
  o_pang = ht_%gil%pang
  n_pang = ang*pi/180.0d0
  call gwcs_projec(ht_%gil%a0,ht_%gil%d0,-ht_%gil%pang,ht_%gil%ptyp,proj,error)
  if (rel) then
    call rel_to_abs (proj,pos(1),pos(2),ht_%gil%a0,ht_%gil%d0,1)
  else
    call sic_decode(ra_c,a0,24,error_1)
    call sic_decode(dec_c,d0,360,error_2)
    error = error_1.or.error_2
    if (error) then
      call map_message(seve%e,rname,'Position conversion error')
      return
    endif
    call abs_to_rel (proj,a0,d0,pos(1),pos(2),1)
    if (pos(1).eq.0.d0 .and. pos(2).eq.0d0) then
      call map_message(seve%w,rname,'Shift is smaller than possible precision')
      !
      ! Return unless a Rotation is also desired
      if (n_pang.eq.o_pang) return
    else
      ht_%gil%a0 = a0
      ht_%gil%d0 = d0
    endif
  endif
  !
  ! Set the Angle AFTER the absolute coordinates of Phase center has been
  ! properly defined...
  ht_%gil%pang = n_pang 
  ht_%gil%posi_words = def_posi_words
  ht_%gil%proj_words = def_proj_words
  !
  ! Compute observing frequency, and new phase center in wavelengths
  if (precise) then
    nc = ht_%gil%nchan
    allocate(rpos(2,nc),stat=ier)
    do i=1,ht_%gil%nchan
      freq = gdf_uv_frequency(ht_,dble(i))
      rpos(1:2,i) = - freq * f_to_k * pos(1:2)
    enddo
  else
    nc = 1
    allocate(rpos(2,1),stat=ier)
    freq = gdf_uv_frequency(ht_)
    rpos(1:2,1) = - freq * f_to_k * pos(1:2)
  endif
  cs(1)  =  cos(n_pang-o_pang)
  cs(2)  = -sin(n_pang-o_pang)
  !
  ! Define blocking factor
  call gdf_nitems('SPACE_GILDAS',nblock,ht_%gil%dim(1)) ! Visibilities at once
  nblock = min(nblock,ht_%gil%dim(2))
  allocate (ht_%r2d(ht_%gil%dim(1),nblock),stat=ier)
  if (ier.ne.0) then
    write(mess,*) 'Memory allocation error ',ht_%gil%dim(1), nblock
    call map_message(seve%e,rname,mess)
    error = .true.
    return
  endif
  !
  !DEBUG call gdf_print_header(ht_)
  !
  ! Recenter all channels, Loop over line table
  ht_%blc = 0
  ht_%trc = 0
  do i=1,ht_%gil%dim(2),nblock
    write(mess,*) i,' / ',ht_%gil%dim(2),nblock
    call map_message(seve%i,rname,mess)
    !
    ht_%blc(2) = i
    ht_%trc(2) = min(ht_%gil%dim(2),i-1+nblock)
    nvisi = ht_%trc(2)-ht_%blc(2)+1
    call gdf_read_data(ht_,ht_%r2d,error)
    call shift_data (ht_,nvisi,cs,nc,rpos,pos)
    call gdf_write_data (ht_,ht_%r2d,error)
    if (error) return
  enddo
  !
  ! Now convert the LOFF - MOFF columns into Pointing columns
  loff = ht_%gil%column_pointer(code_uvt_loff)
  moff = ht_%gil%column_pointer(code_uvt_moff)
  !
  if (loff.ne.0 .or. moff.ne.0) then
    ht_%gil%column_pointer(code_uvt_xoff) = loff
    ht_%gil%column_pointer(code_uvt_yoff) = moff
    ht_%gil%column_size(code_uvt_xoff) = 1
    ht_%gil%column_size(code_uvt_yoff) = 1
    !
    ht_%gil%column_pointer(code_uvt_loff) = 0
    ht_%gil%column_pointer(code_uvt_moff) = 0
    ht_%gil%column_size(code_uvt_loff) = 0
    ht_%gil%column_size(code_uvt_moff) = 0
    !
    ! Reset the Pointing Center at the Phase center
    ht_%gil%ra = ht_%gil%a0
    ht_%gil%dec = ht_%gil%d0
  endif
  !
  call gdf_update_header(ht_,error)
  call gdf_close_image(ht_,error)
  if (error) return
  !
end subroutine sub_uv_shift
!
subroutine shift_data(hx,nvisi,cs,nc,xy,lm)
  use image_def
  use gkernel_interfaces
  use gkernel_types
  use phys_const
  !
  type(gildas), intent(inout) :: hx
  integer, intent(in) :: nvisi
  real, intent(in) :: cs(2)        ! Cos/Sin of Rotation
  integer, intent(in) :: nc
  real(8), intent(in) :: xy(2,nc)  ! Position Shift in Waves
  real(8), intent(in) :: lm(2)     ! Position Shift in Radian
  !  xy = - freq * f_to_k * lm     ! but frequency is channel dependent
  !
  integer(kind=index_length) :: i
  integer :: ix, iu, iv, jc, loff, moff
  real(8) :: phi, sphi, cphi, freq, x, y
  real :: u, v, reel, imag
  real(8), allocatable :: lm_uv(:)
  !
  iu = hx%gil%column_pointer(code_uvt_u)
  iv = hx%gil%column_pointer(code_uvt_v)
  !
  ! If there is a Phase offset column, use it
  !  integer(kind=4), parameter :: code_uvt_loff = 10 ! Phase center Offset
  !  integer(kind=4), parameter :: code_uvt_moff = 11 ! Phase center Offset
  loff = hx%gil%column_pointer(code_uvt_loff)
  moff = hx%gil%column_pointer(code_uvt_moff)
  !
  !Print *,'LOFF ',loff,' MOFF ',moff
  !
  cphi = 1.0  ! To avoid warnings only
  sphi = 0.0  ! ...
  if (loff.ne.0 .or. moff.ne.0) then
    allocate(lm_uv(nc))
    if (nc.eq.1) then
      freq = gdf_uv_frequency(hx)
      lm_uv(1) = freq * f_to_k
    else
      do i=1,hx%gil%nchan
        freq = gdf_uv_frequency(hx,dble(i))
        lm_uv(i) = freq * f_to_k
      enddo
    endif
    !
    do i = 1,nvisi
      !! Print *,'Visi ',i,' Offs ',hx%r2d(loff,i),hx%r2d(moff,i)
      u = hx%r2d(iu,i)
      v = hx%r2d(iv,i)
      hx%r2d(iu,i) = cs(1)*u - cs(2)*v
      hx%r2d(iv,i) = cs(2)*u + cs(1)*v
      !
      if (nc.eq.1) then
        x = xy(1,1) + lm_uv(1) * hx%r2d(loff,i)
        y = xy(2,1) + lm_uv(1) * hx%r2d(moff,i)
        phi = x*u + y*v
        cphi = cos(phi)
        sphi = sin(phi)
      endif
      !
      do jc=1,hx%gil%nchan
        ix = hx%gil%fcol+(jc-1)*hx%gil%natom
        if (nc.gt.1) then
          x = xy(1,jc) + lm_uv(jc) * hx%r2d(loff,i)
          y = xy(2,jc) + lm_uv(jc) * hx%r2d(moff,i)
          phi = x*u + y*v
          cphi = cos(phi)
          sphi = sin(phi)
        endif
        reel = hx%r2d(ix,i) * cphi - hx%r2d(ix+1,i) * sphi
        imag = hx%r2d(ix,i) * sphi + hx%r2d(ix+1,i) * cphi
        hx%r2d(ix,i) = reel
        hx%r2d(ix+1,i) = imag
      enddo
      !
      hx%r2d(loff,i) = hx%r2d(loff,i) - lm(1)
      hx%r2d(moff,i) = hx%r2d(moff,i) - lm(2)
    enddo
    !
  else if (xy(1,1).eq.0. .and. xy(2,1).eq.0.) then
    do i = 1,nvisi
      u = hx%r2d(iu,i)
      v = hx%r2d(iv,i)
      hx%r2d(iu,i) = cs(1)*u - cs(2)*v
      hx%r2d(iv,i) = cs(2)*u + cs(1)*v
    enddo
  else
    do i = 1,nvisi
      u = hx%r2d(iu,i)
      v = hx%r2d(iv,i)
      hx%r2d(iu,i) = cs(1)*u - cs(2)*v
      hx%r2d(iv,i) = cs(2)*u + cs(1)*v
      !
      if (nc.eq.1) then
        phi = xy(1,1)*u + xy(2,1)*v
        cphi = cos(phi)
        sphi = sin(phi)
      endif
      !
      do jc=1,hx%gil%nchan
        ix = hx%gil%fcol+(jc-1)*hx%gil%natom
        if (nc.gt.1) then
          phi = xy(1,jc)*u + xy(2,jc)*v
          cphi = cos(phi)
          sphi = sin(phi)
        endif
        reel = hx%r2d(ix,i) * cphi - hx%r2d(ix+1,i) * sphi
        imag = hx%r2d(ix,i) * sphi + hx%r2d(ix+1,i) * cphi
        hx%r2d(ix,i) = reel
        hx%r2d(ix+1,i) = imag
      enddo
    enddo
  endif
end subroutine shift_data
