!
subroutine uv_baseline(line,error)
  use gkernel_interfaces
  use gildas_def
  use gkernel_types
  use gbl_format
  use gbl_message
  use clean_types
  use clean_arrays
  !----------------------------------------------------------
  ! @ private
  !
  ! MAPPING Support for commands
  !   UV_BASELINE [Degree] /CHANNEL ListVariable [/ZERO]
  !   UV_BASELINE [Degree] /FREQUENCIES ListFreq /WIDTH Width
  !   UV_BASELINE [Degree] /VELOCITIES ListVelo /WIDTH Width
  !
  ! Subtract a continuum baseline, ignoring a list of channels
  ! in the current UV data set.
  !----------------------------------------------------------------
  character(len=*), intent(inout) :: line  ! Command line
  logical, intent(out) :: error            ! Error flag
  !
  character(len=*), parameter :: rname='UV_BASELINE'
  external :: t_baseline
  integer :: degree
  !
  degree = 0
  call sic_i4(line,0,1,degree,.false.,error)
  if (error) return
  if (degree.ne.0 .and. degree.ne.1) then
    call map_message(seve%e,rname,'Only degree 0 or 1 supported')
    error = .true.
    return
  endif
  !
  call uv_filter_base (line,error,rname,t_baseline,degree)
end subroutine uv_baseline
!
subroutine uv_filter(line,error)
  use gkernel_interfaces
  use gildas_def
  use gkernel_types
  use gbl_format
  use gbl_message
  use clean_types
  use clean_arrays
  !----------------------------------------------------------------
  ! @ private
  !
  ! MAPPING  support for
  !   UV_FILTER  /CHANNEL ListVariable [/ZERO]
  !   UV_FILTER  /FREQUENCIES ListFreq /WIDTH Width [UNIT]
  !   UV_FILTER  /VELOCITY ListVelo /WIDTH Width [UNIT]
  !
  ! "Filter", i.e. flag, a list of channels in the current UV
  ! data set. Flagging is reversible, unless the /ZERO option is
  ! present. With /ZERO, the "filtered" visibilities are set to zero.
  !----------------------------------------------------------------
  character(len=*), intent(inout) :: line  ! Command line
  logical, intent(out) :: error            ! Error flag
  !
  character(len=*), parameter :: rname='UV_FILTER'
  external :: t_filter
  !
  integer, parameter :: opt_chan=1
  integer, parameter :: opt_freq=2
  integer, parameter :: opt_velo=3
  integer, parameter :: opt_width=4
  integer, parameter :: opt_zero=5
  integer :: zero
  !
  if (sic_present(opt_zero,0)) then
    zero = 0
  else
    zero = 1
  endif
  !
  call uv_filter_base (line,error,rname,t_filter,zero)
end subroutine uv_filter
!
subroutine t_channel_sampling(rname,huv,nident,msize)
  use image_def
  use clean_default
  use gbl_message
  !----------------------------------------------------------
  ! @ public-mandatory
  !
  ! MAPPING
  !   Check how many channels can have the same beam
  !----------------------------------------------------------
  character(len=*), intent(in) :: rname      ! Caller's name
  type(gildas), intent(in) :: huv            ! UV data header
  integer, intent(out) :: nident             ! returned step
  integer, intent(in) :: msize               ! Current map_size
  !
  real :: df, dc
  character(len=80) :: chain
  !
  !
  ! Allow Cell_Precis (default 10 %) pixel difference at map edge
  ! Relative Delta Frequency per channel is
  df = abs(huv%gil%fres/huv%gil%freq)
  !
  ! so scale error at map edge is, in pixel units
  dc = df * msize / 2
  !
  ! so allowed number of channels with same beam is
  nident = max(nint(default_map%precis/dc),1)
  !
  write(chain,'(A,I6,A,F10.1)') 'Maximum number of channels '// &
    & 'for same beam ',nident,' Bandwidth ',nident*huv%gil%fres
  call map_message(seve%i,rname,chain)
  !
  ! With random Frequency axis, use only 1 channel
  if (huv%gil%nfreq.gt.1) nident = 1
end subroutine t_channel_sampling
!
subroutine uv_line2cont(line,error)
  use gkernel_interfaces
  use mapping_interfaces, except_this=>uv_line2cont
  use clean_def
  use clean_default
  use clean_types
  use clean_arrays
  use gbl_message
  !----------------------------------------------------------
  ! @ private
  !
  ! MAPPING
  !   Support routine for command UV_CONT /INDEX Alpha [Frequency]
  !   Create a continuum UV table from a Line one
  !----------------------------------------------------------
  character(len=*), intent(inout) :: line  ! Command line
  logical, intent(out) :: error            ! Error flag
  !
  integer :: channels(3)
  integer :: uvcode
  integer :: nchan, mc, nu, nv, ier, i
  type (gildas) :: hcuv
  character(len=*), parameter :: rname='UV_CONT'
  real, pointer :: duv_previous(:,:), duv_next(:,:)
  real :: alpha, uvmax, uvmin
  real(8) :: freq
  integer :: msize
  type (uvmap_par) :: map
  !
  real(kind=8), parameter :: pi=3.14159265358979323846d0
  real(kind=8), parameter :: f_to_k = 2.d0*pi/299792458.d-6
  !
  uvcode = 1 ! code_uvt
  !
  msize = maxval(default_map%size)
  if (msize.eq.0) then
    map = default_map
    freq = gdf_uv_frequency(huv)
    call uvgmax (huv,duv,uvmax,uvmin)
    ! Now transform UVMAX in kiloWavelength (including 2 pi factor)
    uvmax = uvmax*freq*f_to_k
    uvmin = uvmin*freq*f_to_k
    error = .false.
    call map_parameters(rname,map,freq,uvmax,uvmin,error)
    if (error) return
    msize = maxval(map%size)
  endif
  channels = 0
  call t_channel_sampling (rname,huv,channels(3),msize)
  !
  call sic_i4(line,0,1,channels(3),.false.,error)
  call sic_i4(line,0,2,channels(1),.false.,error)
  call sic_i4(line,0,3,channels(2),.false.,error)
  !
  nchan = huv%gil%nchan
  ! Use the <0 convention to start from the end
  if (channels(1).lt.0) then
     channels(1) = max(1,nchan+channels(1))
  else
     channels(1) = min(max(1,channels(1)),nchan)
  endif
  if (channels(2).lt.0) then
     channels(2) = max(channels(1),nchan+channels(2))
  else if (channels(2).eq.0) then
     channels(2) = nchan
  else
     channels(2) = max(channels(1),min(nchan,channels(2)))
  endif
  nchan = channels(2)-channels(1)+1
  !
  ! Make sure things remain tractable for "random"
  ! Frequency axis...
  if (huv%gil%nfreq.ne.0 .and. nchan.ne.1) then
    if (channels(3).ne.1) then
      !
      ! Ultimately, one may check here pseudo-linear axes...
      call map_message(seve%e,rname,'UV tables with random Frequency axis ' &
      & //' can only be converted with channel step = 1')
      error = .true.
      return
    endif
  endif
  !
  call gildas_null(hcuv, type = 'UVT')
  call gdf_copy_header(huv,hcuv,error)
  !
  mc = nchan/channels(3)
  if (mc*channels(3).ne.nchan) mc = mc+1
  hcuv%gil%dim(2) = huv%gil%dim(2)*mc
  hcuv%gil%dim(1) = 10+huv%gil%ntrail
  hcuv%gil%nchan = 1
  hcuv%gil%nvisi = hcuv%gil%dim(2)
  hcuv%gil%inc(1) = huv%gil%inc(1)*mc
  hcuv%gil%fres = channels(3)*hcuv%gil%fres
  hcuv%gil%vres = channels(3)*hcuv%gil%vres
  !
  hcuv%gil%nfreq = 0 ! Reset this
  !
  ! Shift trailing columns to handle Mosaicing
  nu = hcuv%gil%dim(1)
  nv = hcuv%gil%dim(2)
  do i=1,code_uvt_last
    if (huv%gil%column_pointer(i).gt.huv%gil%lcol) then
      hcuv%gil%column_pointer(i) = huv%gil%column_pointer(i)-huv%gil%lcol+10
    endif
  enddo
  hcuv%gil%lcol = 10
  !
  nullify (duv_previous, duv_next)
  call uv_find_buffers (rname,nu,nv,duv_previous,duv_next,error)
  if (error) return
  !
  hcuv%r2d => duv_next
  huv%r2d => duv_previous
  if (sic_present(1,0)) then
    call sic_r4(line,1,1,alpha,.true.,error)
    if (error) return
    call t_continuum(huv,hcuv,channels,uvcode,error,alpha)
  else
    call t_continuum(huv,hcuv,channels,uvcode,error)
  endif
  call uv_clean_buffers (duv_previous, duv_next,error)
  if (error) return
  !
  hcuv%gil%ref(1) = 1.D0
  !
  ! Copy back to UV data set
  call gdf_copy_header(hcuv,huv,error)
  !
  ! Indicate Weights have changed, optimization and save status
  do_weig = .true.
  optimize(code_save_uv)%change = optimize(code_save_uv)%change + 1
  save_data(code_save_uv) = .true.
  !
  ! Redefine SIC variables (mandatory)
  call sic_delvariable ('UV',.false.,error)
  call sic_mapgildas ('UV',huv,error,duv)
  !
end subroutine uv_line2cont
!
subroutine t_continuum(hluv,hcuv,channels,uvcode,error,alpha)
  use gkernel_interfaces
  use mapping_interfaces, only : map_message
  use gildas_def
  use image_def
  use uvmap_def
  use gbl_message
  !$ use omp_lib
  !----------------------------------------------------------
  ! @ public-mandatory
  !
  ! MAPPING
  !   Support routine for UV_CONT / UV_LINE2CONT
  !   create a continuum UV table from a Line one
  !----------------------------------------------------------
  !
  type (gildas), intent(in) :: hluv          ! Line UV header
  type (gildas), intent(inout) :: hcuv       ! Continuum UV header
  integer, intent(in) :: channels(3)      ! Channel selection
  integer, intent(in) :: uvcode           ! Type of UV data
  logical, intent(out) :: error
  real, intent(in), optional :: alpha     ! Spectral index
  !
  character(len=*), parameter :: rname='UV_CONT'
  real(8) :: freq,fval,scale_uv,scale_flux,scale_w
  real, allocatable :: are(:), aim(:), awe(:), adw(:)
  real :: re, im, we, dw
  integer :: ifi,ila,nv,ic,jc,kc,ier
  integer :: iv,ov,nt,ft
  integer :: othread, nthread, ov_num
  real(8), allocatable :: uv_scale(:)
  !
  error = .false.
  !
  ! Define number of Visibilities and Channels...
  nv = hluv%gil%nvisi
  nt = hluv%gil%ntrail
  ft = hluv%gil%lcol+1
  !
  ov_num =  (channels(2)-channels(1))/channels(3)+1
  !
  allocate(uv_scale(channels(2)),stat=ier)
  fval = hluv%gil%val(hluv%gil%faxi)
  do ic=channels(1),channels(2),channels(3)
    freq = gdf_uv_frequency(hluv, dble (2*ic+channels(3)-1)/2.d0 )
    !
    ! The angular scale goes as Lambda/B, i.e. 1/(B nu)
    ! so to preserve a constant angular scale, the "apparent"
    ! B must scale as \nu
    !
    ! so
    uv_scale(ic) = freq/fval ! This is the right formula...
    scale_uv = uv_scale(ic)
    Print *,'Scale UV ',ic, (ic+channels(3))/2.0, scale_uv, fval, freq
  enddo
  !
  if (present(alpha)) then
    if (uvcode.eq.code_uvt) then  ! UVT order
      !
      !$OMP PARALLEL DEFAULT(none) &
      !$OMP   & SHARED(nv, ov_num, uv_scale, hluv, hcuv, channels, alpha, nt, ft) &
      !$OMP   & PRIVATE(iv,ov,fval,ic,freq,re,im,we,kc,jc,dw) &
      !$OMP   & PRIVATE(scale_uv,scale_flux,scale_w)
      !
      !$OMP DO
      do iv=1,nv
        ov = (iv-1)*ov_num+1
        !
        ! Fill in, channel after channel
        do ic=channels(1),channels(2),channels(3)
          scale_uv = uv_scale(ic)
          scale_flux = scale_uv**(-alpha)
          scale_w = scale_uv**(2*alpha)
          !
          hcuv%r2d(1:3,ov) = hluv%r2d(1:3,iv)*scale_uv
          hcuv%r2d(4:7,ov) = hluv%r2d(4:7,iv)
          !
          ! Compact the channels first
          if (channels(3).gt.1) then
            re = 0
            im = 0
            we = 0
            kc = 5+3*ic
            do jc = ic,min(ic+channels(3)-1,channels(2))
              dw = max(0.,hluv%r2d(kc+2,iv))
              re = re + hluv%r2d(kc,iv)*dw
              kc = kc+1
              im = im + hluv%r2d(kc,iv)*dw
              we = we+dw
              kc = kc+2
            enddo
            if (we.ne.0.0) then
              re = re/we
              im = im/we
            end if
            hcuv%r2d(8,ov) = re*scale_flux
            hcuv%r2d(9,ov) = im*scale_flux
            hcuv%r2d(10,ov) = we
          else
            hcuv%r2d(8:9,ov)  = hluv%r2d(5+3*ic:6+3*ic,iv)*scale_flux
            hcuv%r2d(10,ov)  = hluv%r2d(7+3*ic,iv)*scale_w
          endif
          if (nt.gt.0) then
            hcuv%r2d(11:,ov) = hluv%r2d(ft:,iv)
          endif
          ov = ov+1
        enddo
        if (ov.ne.iv*ov_num+1) print *,'Programming error ',iv,ov,ov_num
      enddo
      !$OMP END DO
      !$OMP END PARALLEL
      !
    else if (uvcode.eq.code_tuv) then ! TUV order
      Print *,'TUV order '
      if (channels(3).gt.1) then
        allocate (are(nv),aim(nv),awe(nv),adw(nv),stat=ier)
        if (ier.ne.0) then
          call map_message(seve%e,rname,'Channels allocation error')
          error = .true.
          return
        endif
      endif
      !
      ! Fill in, channel after channel
      othread = 1
      nthread = 1
      !$ othread = omp_get_max_threads()
      !$ nthread = min(othread,ov_num)
      !$ call omp_set_num_threads(nthread)
      !
      !$OMP PARALLEL DEFAULT(none) &
      !$OMP   & SHARED(nv,uv_scale,hluv,hcuv,channels,alpha,nt,ft) &
      !$OMP   & PRIVATE(ic,ifi,ila,freq) &
      !$OMP   & PRIVATE(are,aim,awe,jc,adw) &
      !$OMP   & PRIVATE(scale_uv,scale_flux,scale_w)
      !
      !$OMP DO
      do ic=channels(1),channels(2),channels(3)
        ifi = (ic-1)*nv+1
        ila = ifi+nv-1
        scale_uv = uv_scale(ic)
        scale_flux = scale_uv**(-alpha)
        scale_w = scale_uv**(2*alpha)
        !
        hcuv%r2d(ifi:ila,1:3) = hluv%r2d(1:nv,1:3)*scale_uv
        hcuv%r2d(ifi:ila,4:7) = hluv%r2d(1:nv,4:7)
        !
        ! Compact the channels first
        if (channels(3).gt.1) then
          are = 0
          aim = 0
          awe = 0
          do jc = ic,min(ic+channels(3)-1,channels(2))
            adw(:) = max(0.,hluv%r2d(1:nv,7+3*jc))
            are(:) = are + hluv%r2d(1:nv,5+3*jc)*adw
            aim(:) = aim + hluv%r2d(1:nv,6+3*jc)*adw
            awe(:) = awe+adw
          enddo
          where (awe.ne.0.)
            are(:) = are/awe
            aim(:) = aim/awe
          end where
          hcuv%r2d(ifi:ila,8) = are*scale_flux
          hcuv%r2d(ifi:ila,9) = aim*scale_flux
          hcuv%r2d(ifi:ila,10) = awe*scale_w
        else
          hcuv%r2d(ifi:ila,8:9) = hluv%r2d(1:nv,5+3*ic:6+3*ic)*scale_flux
          hcuv%r2d(ifi:ila,10) = hluv%r2d(1:nv,7+3*ic)*scale_w
        endif
        if (nt.gt.0) then
          hcuv%r2d(ifi:ila,11:) = hluv%r2d(1:nv,ft:)
        endif
      enddo
      !$OMP END DO
      !$OMP END PARALLEL
      !$ call omp_set_num_threads(othread)
      if (channels(3).gt.1) then
        deallocate (are,aim,awe,stat=ier)
      endif
    endif
    !
  else
    ! Simple code with no spectral index
    if (uvcode.eq.code_uvt) then  ! UVT order
      !
      !$OMP PARALLEL DEFAULT(none) &
      !$OMP   & SHARED(nv, ov_num, uv_scale, hluv, hcuv, channels, nt, ft) &
      !$OMP   & PRIVATE(iv,ov,fval,ic,freq,scale_uv,re,im,we,kc,jc,dw)
      !
      !$OMP DO
      do iv=1,nv
        ov = (iv-1)*ov_num+1
        !
        ! Fill in, channel after channel
        do ic=channels(1),channels(2),channels(3)
          scale_uv = uv_scale(ic)
          !
          hcuv%r2d(1:3,ov) = hluv%r2d(1:3,iv)*scale_uv
          hcuv%r2d(4:7,ov) = hluv%r2d(4:7,iv)
          !
          ! Compact the channels first
          if (channels(3).gt.1) then
            re = 0
            im = 0
            we = 0
            kc = 5+3*ic
            do jc = ic,min(ic+channels(3)-1,channels(2))
              dw = max(0.,hluv%r2d(kc+2,iv))
              re = re + hluv%r2d(kc,iv)*dw
              kc = kc+1
              im = im + hluv%r2d(kc,iv)*dw
              we = we+dw
              kc = kc+2
            enddo
            if (we.ne.0.0) then
              re = re/we
              im = im/we
            end if
            hcuv%r2d(8,ov) = re
            hcuv%r2d(9,ov) = im
            hcuv%r2d(10,ov) = we
          else
            hcuv%r2d(8:10,ov)  = hluv%r2d(5+3*ic:7+3*ic,iv)
          endif
          if (nt.gt.0) then
            hcuv%r2d(11:,ov) = hluv%r2d(ft:,iv)
          endif
          ov = ov+1
        enddo
        if (ov.ne.iv*ov_num+1) print *,'Programming error ',iv,ov,ov_num
      enddo
      !$OMP END DO
      !$OMP END PARALLEL
      !
    else if (uvcode.eq.code_tuv) then ! TUV order
      Print *,'TUV order '
      if (channels(3).gt.1) then
        allocate (are(nv),aim(nv),awe(nv),adw(nv),stat=ier)
        if (ier.ne.0) then
          call map_message(seve%e,rname,'Channels allocation error')
          error = .true.
          return
        endif
      endif
      !
      ! Fill in, channel after channel
      othread = 1
      nthread = 1
      !$ othread = omp_get_max_threads()
      !$ nthread = min(othread,ov_num)
      !$ call omp_set_num_threads(nthread)
      !
      !$OMP PARALLEL DEFAULT(none) &
      !$OMP   & SHARED(nv,uv_scale,hluv,hcuv,channels,nt,ft) &
      !$OMP   & PRIVATE(ic,ifi,ila,freq,scale_uv) &
      !$OMP   & PRIVATE(are,aim,awe,jc,adw)
      !
      !$OMP DO
      do ic=channels(1),channels(2),channels(3)
        ifi = (ic-1)*nv+1
        ila = ifi+nv-1
        scale_uv = uv_scale(ic)
        !
        hcuv%r2d(ifi:ila,1:3) = hluv%r2d(1:nv,1:3)*scale_uv
        hcuv%r2d(ifi:ila,4:7) = hluv%r2d(1:nv,4:7)
        !
        ! Compact the channels first
        if (channels(3).gt.1) then
          are = 0
          aim = 0
          awe = 0
          do jc = ic,min(ic+channels(3)-1,channels(2))
            adw(:) = max(0.,hluv%r2d(1:nv,7+3*jc))
            are(:) = are + hluv%r2d(1:nv,5+3*jc)*adw
            aim(:) = aim + hluv%r2d(1:nv,6+3*jc)*adw
            awe(:) = awe+adw
          enddo
          where (awe.ne.0.)
            are(:) = are/awe
            aim(:) = aim/awe
          end where
          hcuv%r2d(ifi:ila,8) = are
          hcuv%r2d(ifi:ila,9) = aim
          hcuv%r2d(ifi:ila,10) = awe
        else
          hcuv%r2d(ifi:ila,8:10) = hluv%r2d(1:nv,5+3*ic:7+3*ic)
        endif
        if (nt.gt.0) then
          hcuv%r2d(ifi:ila,11:) = hluv%r2d(1:nv,ft:)
        endif
      enddo
      !$OMP END DO
      !$OMP END PARALLEL
      !$ call omp_set_num_threads(othread)
      if (channels(3).gt.1) then
        deallocate (are,aim,awe,stat=ier)
      endif
    endif
    !
  endif
  !
end subroutine t_continuum
!
subroutine uv_filter_base(line,error,rname,t_routine,zero)
  use gkernel_interfaces
  use gildas_def
  use gkernel_types
  use gbl_format
  use gbl_message
  use clean_types
  use clean_arrays
  !----------------------------------------------------------------
  ! @ private
  !
  ! MAPPING  support for
  !   UV_FILTER or UV_BASELINE  /CHANNEL ListVariable [/ZERO]
  !   UV_FILTER or UV_BASELINE  /FREQUENCIES ListFreq /WIDTH Width [UNIT]
  !   UV_FILTER or UV_BASELINE  /VELOCITY ListVelo /WIDTH Width [UNIT]
  !
  ! "Filter", i.e. flag, a list of channels in the current UV
  ! data set. Flagging is reversible, unless the /ZERO option is
  ! present. With /ZERO, the "filtered" visibilities are set to zero.
  !----------------------------------------------------------------
  character(len=*), intent(inout) :: line  ! Command line
  logical, intent(out) :: error            ! Error flag
  external :: t_routine
  character(len=*), intent(in) :: rname
  integer, intent(in) :: zero
  !
  real(kind=8), parameter :: pi=3.14159265358979323846d0
  real(kind=8), parameter :: f_to_k = 2.d0*pi/299792458.d-6
  !
  include 'gbl_memory.inc'
  !
  character(len=64) :: listname
  logical :: found
  integer(kind=address_length) :: jpd
  type(sic_descriptor_t) :: desc
  integer :: nf, k, i, j, l, m, ichan, jchan, narg, ier
  real(8) :: freq, velo
  real(4) :: width
  integer, allocatable :: channels(:)
  integer :: na, nstyle
  integer, parameter :: mstyle=3
  character(len=12) :: vstyle(mstyle), astyle, argu
  data vstyle/'CHANNEL','FREQUENCY','VELOCITY'/
  integer, parameter :: opt_chan=1
  integer, parameter :: opt_freq=2
  integer, parameter :: opt_velo=3
  integer, parameter :: opt_width=4
  !
  if (sic_present(opt_chan,0)) then
    narg = sic_narg(opt_chan)
    if (narg.gt.1) then
      allocate(channels(narg),stat=ier)
      do i=1,narg
        call sic_i4(line,opt_chan,i,channels(i),.true.,error)
        if (error) return
      enddo
      call t_routine (narg,channels,zero,error)
      !
    else
      call sic_ch(line,opt_chan,1,listname,na,.true.,error)
      call sic_descriptor(listname,desc,found)
      if (.not.found) then
        call sic_i4(line,opt_chan,1,ichan,.true.,error)
        if (error) then
          call map_message(seve%e,rname,'Variable '//trim(listname)//' does not exists.')
          error = .true.
        endif
        call t_routine (1,ichan,zero,error)
      endif
      if (desc%type.ne.fmt_i4) then
        call map_message(seve%e,rname,'Variable '//trim(listname)//' must be Integer ')
        error = .true.
        return
      endif
      if (desc%ndim.ne.1) then
        call map_message(seve%e,rname,'Variable '//trim(listname)//' must have rank 1')
        error = .true.
        return
      endif
      jpd = gag_pointer(desc%addr,memory)
      nf = desc%dims(1)
      call t_routine (nf,memory(jpd),zero,error)
      if (error) call map_message(seve%e,rname,'Memory allocation error')
    endif
    !
  else if (sic_present(opt_freq,0)) then
    width = abs(huv%gil%fres)  ! In MHz
    call sic_r4(line,opt_width,1,width,.false.,error)
    if (sic_present(opt_width,2)) then
      call sic_ke(line,opt_width,2,argu,na,.true.,error)
      call sic_ambigs (rname,argu,astyle,nstyle,vstyle,mstyle,error)
      select case(astyle)
        case ('CHANNEL')
          width = width*abs(huv%gil%fres)
        case ('VELOCITY')
          width = width*abs(huv%gil%fres/huv%gil%vres)
      end select
    endif
    width = 0.5*width
    !
    narg = sic_narg(opt_freq)
    allocate(channels(huv%gil%nchan),stat=ier)
    if (ier.ne.0) then
      call map_message(seve%e,rname,'Channels allocation error')
      error = .true.
      return
    endif
    k = 0
    !
    do i=1,narg
      call sic_r8(line,opt_freq,i,freq,.true.,error)
      if (error) return
      if (huv%gil%fres.gt.0) then
        ichan = (freq-huv%gil%freq-width)/huv%gil%fres + huv%gil%ref(1)
        jchan = (freq-huv%gil%freq+width)/huv%gil%fres + huv%gil%ref(1)
      else
        ichan = (freq-huv%gil%freq+width)/huv%gil%fres + huv%gil%ref(1)
        jchan = (freq-huv%gil%freq-width)/huv%gil%fres + huv%gil%ref(1)
      endif
      !
      ! Set channels only once...
      do j=ichan,jchan
        if (j.lt.0 .or. j.gt.huv%gil%nchan) then
          cycle
        else
          m = 0
          do l=1,k
            if (channels(l).eq.j) then
              m = l
              exit
            endif
          enddo
          if (m.eq.0) then
            k = k+1
            channels(k) = j
          endif
        endif
      enddo
    enddo
    call t_routine(k,channels,zero,error)
    deallocate(channels)
    if (error) call map_message(seve%e,rname,'Memory allocation error')
    !
  else if (sic_present(opt_velo,0)) then
    width = abs(huv%gil%vres)  ! In km/s
    call sic_r4(line,opt_width,1,width,.false.,error)
    if (sic_present(opt_width,2)) then
      call sic_ke(line,opt_width,2,argu,na,.true.,error)
      call sic_ambigs (rname,argu,astyle,nstyle,vstyle,mstyle,error)
      select case(astyle)
        case ('CHANNEL')
          width = width*abs(huv%gil%vres)
        case ('FREQUENCY')
          width = width*abs(huv%gil%vres/huv%gil%fres)
      end select
    endif
    width = 0.5*width
    !
    narg = sic_narg(opt_velo)
    allocate(channels(huv%gil%nchan),stat=ier)
    if (ier.ne.0) then
      call map_message(seve%e,rname,'Channels allocation error')
      error = .true.
      return
    endif
    k = 0
    !
    do i=1,narg
      call sic_r8(line,opt_velo,i,velo,.true.,error)
      if (error) return
      if (huv%gil%vres.gt.0) then
        ichan = (velo-huv%gil%voff-width)/huv%gil%vres + huv%gil%ref(1)
        jchan = (velo-huv%gil%voff+width)/huv%gil%vres + huv%gil%ref(1)
      else
        ichan = (velo-huv%gil%voff+width)/huv%gil%vres + huv%gil%ref(1)
        jchan = (velo-huv%gil%voff-width)/huv%gil%vres + huv%gil%ref(1)
      endif
      !
      ! Set channels only once...
      do j=ichan,jchan
        if (j.lt.0 .or. j.gt.huv%gil%nchan) then
          cycle
        else
          m = 0
          do l=1,k
            if (channels(l).eq.j) then
              m = l
              exit
            endif
          enddo
          if (m.eq.0) then
            k = k+1
            channels(k) = j
          endif
        endif
      enddo
    enddo
    call t_routine(k,channels,zero,error)
    deallocate(channels)
    if (error) call map_message(seve%e,rname,'Memory allocation error')
  else
    call map_message(seve%e,rname,'Missing option /CHANNEL or /FREQUENCY')
    error = .true.
    return
  endif
  !
  save_data(code_save_uv) = .true.
  !
end subroutine uv_filter_base
!
subroutine t_filter(mf,filter,zero,error)
  !----------------------------------------------------------
  ! @ private
  !
  ! MAPPING
  !   Support routine for UV_FILTER
  !   Filter / Flag a list of channels
  !----------------------------------------------------------
  use gildas_def
  use image_def
  use clean_arrays
  !
  integer, intent(in) :: mf          ! Number of values
  integer, intent(in) :: filter(mf)  ! Channel list
  integer, intent(in) :: zero        ! Zero or not
  logical, intent(out) :: error
  !
  integer :: nf, nv, nc, iv, i, ier
  integer, allocatable :: filtre(:)
  !
  nv = huv%gil%dim(2)
  nc = huv%gil%nchan
  !
  allocate(filtre(mf),stat=ier)
  error = ier.ne.0
  if (error) return
  !
  nf = 0
  do i=1,mf
    if (filter(i).gt.0 .and. filter(i).le.nc) then
      nf = nf+1
      filtre(nf) = filter(i)
    endif
  enddo
  !
  if (zero.eq.0) then
    do iv=1,nv
      do i=1,nf
        duv(5+3*filtre(i):7+3*filtre(i),iv) = 0
      enddo
    enddo
  else
    do iv=1,nv
      do i=1,nf
        duv(7+3*filtre(i),iv) = -abs(duv(7+3*filtre(i),iv))
      enddo
    enddo
  endif
  deallocate (filtre)
end subroutine t_filter
!
subroutine t_baseline(mf,filter,degree,error)
  use gildas_def
  use image_def
  use clean_arrays
  !----------------------------------------------------------
  ! @ private
  !
  ! MAPPING
  !    Support routine for UV_BASELINE
  !    Subtract a baseline with a list of channels to be ignored
  !----------------------------------------------------------
  integer, intent(in) :: mf          ! Number of values
  integer, intent(in) :: filter(mf)  ! Channel list
  integer, intent(in) :: degree      ! Polynomial degree
  logical, intent(out) :: error
  !
  integer :: nf, nv, nc, iv, i, j, ier
  integer, allocatable :: filtre(:)
  real, allocatable :: wreal(:), wimag(:), wx(:), wxx(:), wxy(:)
  real :: mreal, mimag, sx, sy, sxx, sxy, a, b, delta
  !
  error = .false.
  nv = huv%gil%dim(2)
  nc = huv%gil%nchan
  !
  ! FILTER contains the ones to be ignored.
  ! We must build the list of valid ones...
  allocate(filtre(nc),stat=ier)
  error = (ier.ne.0)
  if (error) return
  !
  nf = 0
  do i=1,nc
    nf = nf+1
    filtre(nf) = i
    do j=1,mf
      if (filter(j).eq.i) then
        nf = nf-1
        exit
      endif
    enddo
  enddo
  !
  ! Allocate a work array for Real and Imaginary parts
  !
  if (degree.eq.0) then
    allocate(wreal(nf),wimag(nf),stat=ier)
  else
    allocate(wreal(nf),wimag(nf),wx(nf),wxx(nf),wxy(nf),stat=ier)
    wx = filtre(1:nf)
  endif
  error = (ier.ne.0)
  if (error) return
  !
  do iv=1,nv
    do i=1,nf
      wreal(i) = duv(5+3*filtre(i),iv)
      wimag(i) = duv(6+3*filtre(i),iv)
    enddo
    !
    ! Compute the fitting polynomials.
    ! Filtre(:) is the channel number which can be used as abscissa
    ! For the time being, only mean value...
    if (degree.eq.0) then
      mreal = sum(wreal)/nf
      mimag = sum(wimag)/nf
      !
      ! Subtract
      do i=1,nc
        duv(5+3*i,iv) = duv(5+3*i,iv)-mreal
        duv(6+3*i,iv) = duv(6+3*i,iv)-mimag
      enddo
    else if (degree.eq.1) then
      ! This code actually does a poor job. There is not enough
      ! Signal to noise ratio to fit a linear baseline per Visibility
      sx = sum(wx)
      sy = sum(wreal)
      wxx = wx**2
      sxx = sum(wxx)
      wxy = wx*wreal
      sxy = sum(wxy)
      delta = (nf*sxx-sx**2)
      a = (sxx*sy-sx*sxy)/delta
      b = (nf*sxy-sx*sy)/delta
      do i=1,nc
        duv(5+3*i,iv) = duv(5+3*i,iv)-(a+b*i)
      enddo
      !
      sy = sum(wimag)
      wxx = wx**2
      sxx = sum(wxx)
      wxy = wx*wimag ! Bug correction 15-Dec-2014
      sxy = sum(wxy)
      delta = (nf*sxx-sx**2)
      a = (sxx*sy-sx*sxy)/delta
      b = (nf*sxy-sx*sy)/delta
      do i=1,nc
        duv(6+3*i,iv) = duv(6+3*i,iv)-(a+b*i)
      enddo
    endif
  enddo
  if (degree.eq.0) then
    deallocate (filtre,wreal,wimag)
  else
    deallocate (filtre,wreal,wimag,wx,wxx,wxy)
  endif
  error = .false.
end subroutine t_baseline
!
