subroutine map_continuum(line,error)
  use image_def
  use clean_def
  use clean_arrays
  use clean_types
  use gkernel_interfaces
  use imager_interfaces, except_this => map_continuum 
  use gbl_message
  !---------------------------------------------------------------------
  ! @ private
  !   IMAGER
  !
  !   Support for MAP_CONTINUUM  /METHOD command
  !---------------------------------------------------------------------
  character(len=*), intent(in) :: line  ! command line
  logical, intent(inout) :: error       ! logical error flag
  ! Global
  real(kind=8), parameter :: pi=3.14159265358979323846d0
  character(len=*), parameter :: rname='MAP_CONTINUUM'
  integer, parameter :: o_method=1
  integer, parameter :: i_global=1
  integer, parameter :: i_gauss=2
  integer, parameter :: i_scm=3
  integer, parameter :: i_egm=4
  integer, parameter :: i_cscm=5
  integer, parameter :: mmeth=5
  character(len=8) :: vmeth(mmeth), cmeth
  data vmeth/'GLOBAL','GAUSS', 'SCM','EGM','C-SCM'/
  ! Local
  !! logical :: debug=.false.
  integer, save :: i_method=i_global  ! Method
  integer :: imeth
  real, save :: thremu=3.0   ! Rejection Threshold
  !
  character(len=16) :: name
  character(len=80) :: chain
  integer :: n, nchan, ier, na
  integer(kind=index_length) :: nhist, mhist, dim(2)
  real :: clip_value 
  type(gildas) :: hmap
  real, allocatable, target, save :: spectrum(:,:)
  real :: xsize, ysize, factor, cx, cy, rms
  integer :: imin,imax,jmin,jmax,i1,i2, box(4), nx,ny
  logical :: do_mask
  !
  call sic_delvariable('CONTINUUM',.false.,error)
  if (allocated(dcont)) deallocate(dcont)
  call sic_delvariable('CLIPPED',.false.,error)
  if (allocated(spectrum)) deallocate(spectrum)
  !
  error = .false.
  call gildas_null(hmap)
  call gildas_null(hcont)
  !
  name = 'CLEAN'
  call sic_ke(line,0,1,name,n,.false.,error)
  select case (name)
  case ('CLEAN')
    if (hclean%loca%size.eq.0) then
      call map_message(seve%e,rname,'No CLEAN data loaded')
      error = .true.
      return
    endif
    call gdf_copy_header(hclean,hmap,error)
    hmap%r3d => dclean
    ! Number of pixels per beam
    factor = abs(pi*hmap%gil%majo*hmap%gil%mino/hmap%gil%inc(1)/hmap%gil%inc(2))
  case ('SKY')
    if (hsky%loca%size.eq.0) then
      call map_message(seve%e,rname,'No SKY data loaded')
      error = .true.
      return
    endif
    call gdf_copy_header(hsky,hmap,error)
    hmap%r3d => dsky
    ! Number of pixels per beam
    factor = abs(pi*hmap%gil%majo*hmap%gil%mino/hmap%gil%inc(1)/hmap%gil%inc(2))
  case ('DIRTY')
    if (hdirty%loca%size.eq.0) then
      call map_message(seve%e,rname,'No DIRTY data loaded')
      error = .true.
      return
    endif
    call gdf_copy_header(hdirty,hmap,error)
    hmap%r3d => ddirty
    factor = 1.0
  case default
    call map_message(seve%e,rname,'Only work on CLEAN or DIRTY images')
    error = .true.
    return
  end select
  !
  if (sic_present(o_method,0)) then
    call sic_ke(line,o_method,1,name,na,.true.,error)
    if (error) return
    call sic_ambigs (rname,name,cmeth,imeth,vmeth,mmeth,error)
    if (error) return
    i_method = imeth
    call map_message(seve%w,rname,'Switching to method '//vmeth(i_method),3)
  else  
    call map_message(seve%i,rname,'Using method '//vmeth(i_method))
  endif
  !
  ! Copy header to Continuum
  call gdf_copy_header(hmap,hcont,error)
  rms = hmap%gil%noise
  !
  ! Define continuum data
  hcont%gil%ndim = 2
  hcont%gil%dim(3) = 1
  allocate(dcont(hcont%gil%dim(1),hcont%gil%dim(2)), &
    &   spectrum(hmap%gil%dim(3),4),stat=ier)
  if (ier.ne.0) then
    call map_message(seve%e,rname,'Memory allocation error')
    error = .true.
    return
  endif
  hcont%r2d => dcont
  !
  clip_value = 2.5
  call sic_r4(line,0,2,clip_value,.false.,error)
  if (clip_value.lt.1.0 .or. clip_value.gt.5.0) then
    call map_message(seve%e,rname,'Clipping value outside of recommended [1-5] range')
    error = .true.
    return
  endif
  !
  nchan = hmap%gil%dim(3)
  !
  mhist = max(8,nint(sqrt(real(nchan))))
  mhist = min(mhist,32)
  nhist = max(8,mhist)
  call sic_i8(line,0,3,nhist,.false.,error) 
  if (nchan.lt.32) then
    write(chain,'(A,I0,A)') 'Too few channels [',nchan,'], display only'
    call map_message(seve%w,rname,chain)
    nhist = 1
  else if (nhist.lt.8 .or. nhist.gt.max(8,nchan/8)) then
    write(chain,'(A,I0,A)') 'Histogram size out of allowed range [8-',nchan/8,']'
    call map_message(seve%e,rname,chain)
    error = .true.
    return
  else if (nhist.gt.mhist)  then
    write(chain,'(A,I0,A)') 'Histogram size out of recommended range [8-',mhist,']'
    call map_message(seve%w,rname,chain)
  endif
  ! 
  !
  ! We use the mechanics of MASK and SUPPORT as in the STATISTIC command
  nx = hmap%gil%dim(1)
  ny = hmap%gil%dim(2)
  do_mask = method%do_mask  ! Bug before 25-Nov-2021
  !
  if (do_mask) then
    call copy_method(user_method,method)
    !
    if (.not.allocated(d_mask)) then
      allocate(d_mask(nx,ny),d_list(nx*ny),stat=ier)
      if (ier.ne.0) then
        call map_message(seve%e,rname,'Error getting support memory')
        error = .true.
        return
      endif
      method%do_mask = .true.
      method%nlist = 0
    endif
    method%box = [nx,ny,1,1] 
    call check_area(method,hmap,.true.) ! Needed ?
    call check_mask(method,hmap)
    box(1) = min(box(1),method%box(1))
    box(2) = min(box(2),method%box(2))
    box(3) = max(box(3),method%box(3))
    box(4) = max(box(4),method%box(4))
    call map_message(seve%i,rname,'Using current support')
    !
    if (i_method.eq.1) then
      call compute_continuum(hmap,hcont,spectrum,nhist,clip_value,error,box,d_mask)
    else
      call gauss_continuum(i_method,hmap,hcont,spectrum,nhist,clip_value,error, &
      & rms,thremu,box,d_mask)
    endif
    if (error) return  
  else
    !
    xsize = 0.
    ysize = 0.
    call sic_get_real('SIZE[1]',xsize,error)
    call sic_get_real('SIZE[2]',ysize,error)
    !
    if (xsize.eq.0. .and. ysize.eq.0.) then
      box = [1,1,nx,ny]
    else
      cx = 0
      call sic_get_real('CENTER[1]',cx,error)
      cy = 0
      call sic_get_real('CENTER[1]',cy,error)
      !
      call map_message(seve%i,rname,'Using current SIZE')
      xsize = xsize*pi/180.d0/3600d0
      ysize = ysize*pi/180.d0/3600d0
      cx = cx*pi/180.d0/3600.d0
      i1 = (cx-0.5*xsize-hmap%gil%val(1))/hmap%gil%inc(1) + hmap%gil%ref(1)
      i2 = (cx+0.5*xsize-hmap%gil%val(1))/hmap%gil%inc(1) + hmap%gil%ref(1)
      imin = min(i1,i2)
      imax = max(i1,i2)
      imin = max(1,imin)
      cy = cy*pi/180.d0/3600.d0
      i1 = (cy-0.5*ysize-hmap%gil%val(2))/hmap%gil%inc(2) + hmap%gil%ref(2)
      i2 = (cy+0.5*ysize-hmap%gil%val(2))/hmap%gil%inc(2) + hmap%gil%ref(2)
      jmin = min(i1,i2)
      jmax = max(i1,i2)
      jmin = max(1,jmin)
      !
      imin = min(imin,hmap%gil%dim(1))
      imax = min(imax,hmap%gil%dim(1))
      jmin = min(jmin,hmap%gil%dim(2))
      jmax = min(jmax,hmap%gil%dim(2))
      !
      box(1) = imin
      box(2) = jmin
      box(3) = imax
      box(4) = jmax
    endif
    !
    if (i_method.eq.1) then
      call compute_continuum(hmap,hcont,spectrum,nhist,clip_value,error,box)
    else
      call gauss_continuum(i_method,hmap,hcont,spectrum,nhist,clip_value,error, &
        & rms,thremu,box)
    endif
    if (error) return  
  endif
  !
  hcont%loca%addr = locwrd(hcont%r2d) 
  hcont%loca%size = hcont%gil%dim(1) * hcont%gil%dim(2)
  call gdf_get_extrema (hcont,error)
  !
  call sic_mapgildas('CONTINUUM',hcont,error,dcont)
  dim(1) = hmap%gil%dim(3)
  dim(2) = 4
  !
  ! Scale to Jy if possible (CLEAN data)
  !! Print *,'Normalizing by ',factor
  spectrum(:,2) = spectrum(:,2)/factor
  spectrum(:,3) = spectrum(:,3)/factor
  where(spectrum(:,4).ne.0) 
    spectrum(:,4) = 1.0
  end where
  call sic_def_real('CLIPPED',spectrum,2,dim,.false.,error)
  !
  call exec_program('@ p_continuum')   
  !
end subroutine map_continuum
!
subroutine compute_continuum(hmap,hcont,spectrum,nhist,clip_value,error,box,mask)
  use image_def
  use gbl_message
  use gkernel_interfaces
  use imager_interfaces, except_this => compute_continuum
  !---------------------------------------------------------------------
  ! @ public-mandatory
  !
  !   IMAGER
  !
  !     Support routine to derive the continuum image
  !   The method is to find a global set of windows (along 3rd axis)
  !   based on the mean spectrum in a selected region (defined by 
  !   a mask), and to compute the continuum as the average
  !   of the "line-free" channels outside of these windows.
  !
  !     The method uses a space-invariant spectral map. The error
  !   map of the derived continuum is thus constant. The method may
  !   fail to find weak continuum sources in fields which contain
  !   bright, spectrally confused, sources 
  !---------------------------------------------------------------------
  type(gildas), intent(in) :: hmap      ! Input data cube
  type(gildas), intent(inout) :: hcont  ! Output continuum image
  real, intent(out) :: spectrum(:,:)    ! Output spectrum
  integer(kind=index_length), intent(in) :: nhist ! Number of Histograms
  real, intent(in) :: clip_value        ! Clipping value
  logical, intent(out) :: error         ! Error flag
  integer(kind=4), intent(in) :: box(4) ! Bounding box of mask
  logical, intent(in), optional :: mask(:,:)  ! Optional mask
  !
  character(len=*), parameter :: rname='MAP_CONTINUUM'
  integer, allocatable :: mylines(:), oulines(:)
  real, pointer :: rmask(:,:)
  real :: blank, val
  integer(kind=index_length) :: nchan,kxy
  integer :: nx,ny,ix,iy,ier,ic,kc,ib,nb,inb,iextent
  logical :: debug=.false.
  !
  error = .false.
  !
  nchan = hmap%gil%dim(3)
  nx = hmap%gil%dim(1)
  ny = hmap%gil%dim(2)
  !
  allocate(mylines(nchan),oulines(nchan),stat=ier)
  if (ier.ne.0) then
    call map_message(seve%e,rname,'Memory allocation error')
    error = .true.
    return
  endif
  spectrum = 0.0
  do ic=1,nchan
    spectrum(ic,1) = (ic-hmap%gil%ref(3))*hmap%gil%vres + hmap%gil%voff
  enddo
  ! 
  if (hcont%gil%blan_words.ne.2 .or. hcont%gil%eval.lt.0) then
    !    
    ! We need to define a Blanking value outside of the range
    blank = minval(hmap%r3d)
    if (blank.gt.0) then
      hcont%gil%bval = -2.0*blank*nchan
      hcont%gil%eval = 0.0
    else if (blank.lt.0) then
      hcont%gil%bval = 3.0*blank*nchan
      hcont%gil%eval = 0.0
    else
      hcont%gil%bval = -1.0
      hcont%gil%eval = 0.0
    endif
  endif
  !
  ! Compute the mean spectrum
  !
  if (present(mask)) then
    !
    if (hmap%gil%eval.ge.0.0) then
      kxy = 0
      do iy=box(2),box(4)
        do ix=box(1),box(3)
          if (mask(ix,iy)) then
            kxy = kxy+1
            do ic=1,nchan
              val = hmap%r3d(ix,iy,ic)
              if (abs(val-hmap%gil%bval).gt.hmap%gil%eval) then
                spectrum(ic,2) = spectrum(ic,2) + val
              endif
            enddo
          endif
        enddo
      enddo
    else
      kxy = 0
      do iy=box(2),box(4)
        do ix=box(1),box(3)
          if (mask(ix,iy)) then
            kxy = kxy+1
            do ic=1,nchan
              spectrum(ic,2) = spectrum(ic,2) + hmap%r3d(ix,iy,ic)
            enddo
          endif
        enddo
      enddo
    endif
  else
    if (all(box.eq.0)) then
      kxy =nx*ny
      !
      do ic = 1,nchan
        call gr4_mean(hmap%r3d(:,:,ic),hmap%gil%dim(1)*hmap%gil%dim(2),  &
          & hmap%gil%bval,hmap%gil%eval,spectrum(ic,2)) 
      enddo
    else
      kxy =(box(4)-box(2)+1)*(box(3)-box(1)+1)
      !
      if (hmap%gil%eval.ge.0.0) then
        do ic=1,nchan
          do iy=box(2),box(4)
            do ix=box(1),box(3)
              val = hmap%r3d(ix,iy,ic)
              if (abs(val-hmap%gil%bval).gt.hmap%gil%eval) then
                spectrum(ic,2) = spectrum(ic,2) + val
              endif
            enddo
          enddo
        enddo
      else
        do ic=1,nchan
          do iy=box(2),box(4)
            do ix=box(1),box(3)
              spectrum(ic,2) = spectrum(ic,2) + hmap%r3d(ix,iy,ic)
            enddo
          enddo
        enddo
      endif
    endif
  endif
  !
  spectrum(:,3) = spectrum(:,2)
  !
  if (nhist.gt.1) then
    !
    ! Find out the line free regions
    call clip_lineregions(rname,spectrum(:,3),nchan,nhist,  &
      & hcont%gil%bval,hcont%gil%eval,clip_value,debug,0.0,1.0)
    nb = 0
    call guess_lineregions(spectrum(:,3),nchan,       &
      & hcont%gil%bval,hcont%gil%eval,            &
      & mylines,nb)
    !
    ! Expand line regions by continuity
    inb = nb
    iextent = 8
    call clip_expand(hmap%gil%dim(3),mylines,inb,oulines,nb,iextent)
    !
    ! These are the line-contaminated channels
    do ib=1,nb
      spectrum(oulines(ib),4) = 1
    enddo
  endif
  !
  ! Compute the continuum image 
  kc = 0
  do ic=1,nchan
    if (spectrum(ic,4).ne.0) cycle ! Skip the Line Contaminated channels
    kc = kc+1
    if (kc.eq.1) then
      hcont%r2d = hmap%r3d(:,:,ic)
    else
      rmask => hmap%r3d(:,:,ic)
      if (hmap%gil%eval.ge.0) then
        where ( (abs(rmask-hmap%gil%bval).gt.hmap%gil%eval).and. &
          & (abs(hcont%r2d-hmap%gil%bval).gt.hmap%gil%eval) ) &
          & hcont%r2d = hcont%r2d + rmask
      else
        hcont%r2d = hcont%r2d + rmask
      endif
    endif
  enddo
  !
  if (kc.ne.0) then
    where (abs(hcont%r2d-hcont%gil%bval).gt.hcont%gil%eval) hcont%r2d = hcont%r2d/kc
    hcont%gil%rms = hmap%gil%rms /sqrt(real(kc))
    hcont%gil%noise = hmap%gil%noise /sqrt(real(kc))
  endif
  !
  ! Return the number of pixels here. It will serve
  ! to normalize the spectrum after that.
  spectrum(:,4) = kxy*spectrum(:,4)
end subroutine compute_continuum
!
!====================================================================================
!
subroutine gauss_continuum(i_code,hmap,hcont,spectrum,nhist,clip_value,error, &
  & rms, thremu, box, mask)
  use image_def
  use gbl_message
  use gkernel_interfaces
  use imager_interfaces, except_this => gauss_continuum
  !---------------------------------------------------------------------
  ! @ public-mandatory
  !
  !   IMAGER
  !
  !   Support for command
  !     MAP_CONTINUUM /METHOD GAUSS [Threshold]
  !
  !     Support routine to derive the continuum image, pixel per pixel
  !   The noise is no longer space-invariant, but the method is able to
  !   identify weak continuum sources in the presence of strong, spectrally
  !   confused sources.
  !
  !     The method is a variant of the Sigma Clipping Median, 
  !   that works if there is no strong confusion. It avoids re-computing
  !   medians (a slow operation) by taking into account that in this
  !   case, the mean value can be found by fitting a parabola 
  !   in the histogram of the channel values. The fit is made stable
  !   against outliers by adjusting the Min Max of the histogram
  !   so that the parabolic fit converges.
  !
  !   Two methods are available:
  !   -  a c-SCM (corrected Sigma Clipping Median), as in STATCONT
  !     and UV_PREVIEW
  !
  !   -  That of Jordan Mollet et al 2018
  !     Fit a modified Exponentially modified Gaussian function
  !     to the Histogram of intensity values for all channels,
  !---------------------------------------------------------------------
  integer, intent(in) :: i_code         ! Code of method
  type(gildas), intent(in) :: hmap      ! Input data cube
  type(gildas), intent(inout) :: hcont  ! Output continuum image
  real, intent(out) :: spectrum(:,:)    ! Output spectrum
  integer(kind=index_length), intent(in) :: nhist ! Number of Histograms
  real, intent(in) :: clip_value        ! Clipping value
  logical, intent(out) :: error         ! Error flag
  real, intent(in) :: rms               ! Expected RMS (from theoretical noise)
  real, intent(in) :: thremu            ! Rejection threshold
  integer(kind=4), intent(in) :: box(4) ! Bounding box of mask
  logical, intent(in), optional :: mask(:,:)  ! Optional mask
  !
  real, parameter :: sec_to_rad = acos(-1.0)/3600/180
  integer, parameter :: max_count=4 ! < 8 for sure
  integer, parameter :: i_gauss=2, i_scm=3, i_egm=4, i_cscm=5
  character(len=*), parameter :: rname='GAUSS'
  !
  real, allocatable :: array(:,:), noises(:,:)
  real, allocatable :: temp(:), tmp(:)
  real, allocatable :: hist(:,:), logv(:), logf(:)
  integer :: ier, nx,ny,nc,ix,jy,jc
  integer(kind=index_length) :: nchan
  !
  integer :: mloc(2), count, mhist, i, nxy, kc, mxy, next, percentage_step
  real :: a,b,c,d, amin, amax, arms, amed, aoff, mu, peak, sigma, val
  character(len=64) :: ch, st, mess
  !
  real :: bval, eval, blank, brms, crms, noise, tnoise
  logical :: goon, outside_mask
  logical :: debug, corrected, do_more, do_mask, gbl_error
  !
!  real :: cx,cy,sx,sy, xmin, xmax, ymin, ymax
!  integer :: i1,i2, 
  integer :: jmin,jmax, imin,imax, mcount, code
  integer :: ihollow, iout, ilong, ndata, npos, nneg
  !
  real :: logwings
  !
  mcount = 8
  call sic_get_inte('COUNTS',mcount,error)
  logwings = 3.5
  call sic_get_real('WINGS',logwings,error)
  !
  debug = .false.
  call sic_get_logi('DEBUG',debug,error)
  if (.not.debug) then
    call map_message(seve%w,rname,'This part of the code is not yet operational')
!    error = .true.
!    return
  endif
  !
  percentage_step = 10.
  nx = hmap%gil%dim(1)
  ny = hmap%gil%dim(2)
  nchan = hmap%gil%dim(3)
  nc = hmap%gil%dim(3)
  !
  mhist = nhist 
  !
  allocate(array(nx,ny), noises(nx,ny), temp(nc), tmp(nc), hist(mhist,2), logv(mhist), logf(mhist), stat=ier)
  if (ier.ne.0) then
    call map_message(seve%e,rname,'Memory allocation error')
    error = .true.
    return
  endif
  !
  spectrum = 0.0
  do kc=1,nchan
    spectrum(kc,1) = (kc-hmap%gil%ref(3))*hmap%gil%vres + hmap%gil%voff
  enddo
  !
  ! RMS is the expected RMS noise
  ! We could check that it is actually reached outside of the
  ! area to be analyzed in a first pass..
  !
  imin = box(1)
  imax = box(3)
  jmin = box(2)
  jmax = box(4)
  !
  blank = hmap%gil%rmin
  if (blank.gt.0) then
    bval = -2.0*blank
    eval = blank
  else if (blank.lt.0) then
    bval = 3.0*blank
    eval = -blank
  else
    bval = -1.0
    eval = 0.0
  endif
  write(ch,'(A,1PG11.3,1X,1PG11.3)') '/BLANK ',bval,eval 
  !
  array = bval
  !
  i = 0
  nxy = 0
  ihollow = 0
  iout = 0
  ilong = 0
  tnoise = 0.
  do_mask = present(mask)
  gbl_error = .false.
  mxy = (jmax-jmin+1)*(imax-imin+1)
  !
  next = imax-imin+1
  call progress_report('Computing',1,next,mxy,percentage_step)
  !
  !$OMP PARALLEL DEFAULT(NONE) &
  !$OMP & SHARED (hmap,array,noises,mask, spectrum, percentage_step) &
  !$OMP & SHARED (jmin,jmax,imin,imax,do_mask,nchan,i_code,bval,eval,thremu,rms) &
  !$OMP & SHARED (logwings, debug, nhist, mhist, mcount, ch, i, next) & 
  !$OMP & SHARED (mxy, nxy, ihollow, iout, ilong, tnoise, gbl_error) &
  !$OMP & PRIVATE(val,amin,amax,arms,aoff,amed,goon,count,corrected,code, error) &
  !$OMP & PRIVATE(ix,jy,outside_mask,kc,jc,a,b,c,d,mu,sigma,mloc,noise,do_more) &
  !$OMP & PRIVATE(npos, nneg, ndata, peak, brms, crms) &
  !$OMP & PRIVATE(temp,tmp,hist,logv,logf)
  !$OMP DO COLLAPSE(2) REDUCTION(+:ihollow,iout,ilong,tnoise) 
  do jy=jmin,jmax
    do ix=imin,imax
      if (gbl_error) cycle
      !
      ! Test if inside or outside the mask
      outside_mask = .false.
      if (do_mask) then
        outside_mask = .not.mask(ix,jy)
      endif
      !
      ! Ignore if outside
      if (outside_mask) cycle
      !
      error = .false. 
      !
      ! Get the spectrum
      temp(:) = hmap%r3d(ix,jy,:)
      noise = 0
      !
      ! increment the integrated spectrum "On-the-Fly"
      !$OMP CRITICAL
      nxy = nxy+1
      !$OMP END CRITICAL
      do kc=1,nchan
        val = hmap%r3d(ix,jy,kc)
        if (abs(val-hmap%gil%bval).gt.hmap%gil%eval) then
          spectrum(kc,2) = spectrum(kc,2) + val     ! Worry in Parallel mode, see GLOBAL code
        endif
      enddo
      !
      !
      select case(i_code)
      case (i_gauss)
        ! Compute the histogram of channel values 
        !
        ! Define the Histogram range around the median
        call median_filter (temp,nchan,bval,eval,aoff,amin,amax,arms,thremu)
        if (debug) print *,' Amin ',amin,', Amax ',amax,', Aoff ',aoff,' RMS ',rms, arms
        goon = (amin.lt.amax)     ! Protect against fully constant data (0 in models...)
        !        
        if (goon .and. rms.ne.0) then
          ! Protect against High S/N lines with both negative and positive features
          if (arms.gt.3*rms) then
            amin = max(-3.0*rms,amin)
            amax = -amin
          endif
        endif
        !
        amed = (amin+amax)/2.0
        count = 0
        !
        !
        !   Fit this histogram with a Gaussian
        !
        !   another way is to compute the histogram between the true Min and Max,
        ! and eliminate the values that contain less than about a few % of 
        ! the channels to recompute another Min - Max. In other words, ignore
        ! the first N % on one side, the last N % on the other side, by
        ! having the cumulative count, where N = 3-5 or so.
        !
        !   We combine here both methods, by ensuring that the histograms has no "holes" 
        ! before fitting. That helps to make the code more robust, and still fast.
        !
        corrected = .false.   ! Assume the method will work
        do while (goon) 
          !
          ! Make the histogram betweeen the current Min Max
          call histo44(temp,nchan,hist,nhist,2,amin,amax,bval,eval)  
          ! Check it has no hole
          code = 0
          call check_logv(logv,logwings,temp,nchan,hist,nhist,amin,amax,bval,eval,mcount,code)
          if (code.ne.0) then
            ihollow = ihollow+1
            ! Print *,'Hollow Histogram after ',mcount,' iteration ',ix,jy
          endif
          !
          ! Use MHIST there, not NHIST  (I*4 vs I*8 to be fixed)
          ! Apply Caruana algorithm : the Log of a Gaussian is a Parabola
          ! The small bias does not affect the result too much
          call fit_parabola(mhist,hist(1:mhist,2),logv,a,b,c,d)
          if (debug) print *,' A B C D ',a,b,c,d
          !
          ! Now back to Gaussian characteristics
          if (c.eq.0) then
            ! Means out of bound, either too small (flat histogram) or to broad  (1 pixel histogram)
            ! This should no longer happens, but play safe.
            !!print *,' A B C D ',a,b,c,d,' at ',ix,jy
            !!print *,minval(hist(:1:mhist,2)), maxval(hist(:1:mhist,2)),  amin, amax
            !!print *,hist(1:mhist,2)
            if (d.gt.0) then    ! Histogram is too narrow, restrict range
              mloc = maxloc(hist,1)
              mu = hist(mloc(1),2)
              Print *,' Peaky histogram Pixel ',ix,jy,' Solution ',mu,' out of ',amin,amax
              !
              sigma = (amax-amin)/4.0
            else                ! Histogram is flat... Should either extend range of use value
              mu = hist(1,2)
              Print *,'Flat histogram  Pixel ',ix,jy,' Solution ',mu,' out of ',amin,amax
              sigma = -(amax-amin)/4.0
            endif
            d = 1.0  ! Force it "not converged"
          else
            ! There is a solution
            mu = -b/(2.0*c)             ! Position
            sigma = sqrt(-1.0/(2.0*c))  ! Sigma
            peak = exp(a-b**2/(4.0*c))  ! Intensity (biased)
            if (do_more) print *,'Normal mu ',mu,sigma,peak, d
          endif
          !
          corrected = .false.
          if ((mu.gt.amax).or.(mu.lt.amin)) then
            corrected = .true.
            !! Print *,'Pixel ',ix,jy,' Solution ',mu,' out of ',amin,amax,' NVal ',sum(hist(1:mhist,1))
            code = 1
            call check_logv(logv,logwings,temp,nchan,hist,nhist,amin,amax,bval,eval,2,code)
            if (code.ne.1) then
              iout = iout+1
              !! Print *,' Min  Max reset to ',amin,amax,maxval(hist(1:mhist,2)), ' IX JY ',ix,jy
            endif
          endif
          !
          if (debug) then
            count= count+10
            do jc=1,mhist
              write(count,*) hist(jc,2), logv(jc), a+b*hist(jc,2)+c*hist(jc,2)**2, d
            enddo
            close(unit=count)
            count = count-10
          endif    
          !
          ! Treat the confusion case: the Gaussian  peak should be close 
          ! to the maximum of the Histogram - if not, this must be line 
          ! contamination
          if (debug) Print *,'Peak ',log(peak),' MaxVal ',maxval(logv),' D ',d
          !
          ! Now get the "continuum" value
          amed = mu                   ! Define the position of maximum 
          if (count.eq.max_count) then
            ! MAX_COUNT iterations, this is too much...
            goon = .false.
          else if (d.gt.0.25 .or. log(peak).lt.maxval(logv)-d) then
            !! Print *,'D ',d,' Logs ',log(peak),' > ', maxval(logv)-d
            ! If not converged, the Maximum can be wrong - iterate
            mloc = maxloc(hist,1)
            amed = hist(mloc(1),2)
            if (.not.corrected) then
              amin = amed-3.0*sigma  
              amax = amed+3.0*sigma
            endif
            if (debug) Print *,count,': Min ',amin,' Median ',aoff,' Max ',amax,' Rms ',sigma
            count = count+1
          else
            ! Converged, fitted...
            goon = .false.
          endif
          if (corrected) then
            if (.not.goon) then
              logf(:) = a+b*hist(:,2)+c*hist(:,2)**2
              call debug_plot(ix,jy,nhist,hist,logv,logf,nchan,spectrum,temp, i, ch)
              if (i.eq.-1) then
                error = .true.
              endif
            endif
          endif
        enddo
        !
        ! Correct for the Offset
        ! print *,'Amed Aoff ',amed,aoff
        amed = amed+aoff
!!      if (rms.ne.0.0 .and. sigma.gt.rms) then
!!        write(chain,'(A,1PG10.2,A,1PG10.2)')  'Expected noise ',rms, &
!!          & ' is smaller than measured noise ',sigma
!!        call map_message(seve%w,rname,chain,3)
!!      endif
        !
        ! Evaluate the noise
        ndata = sum(hist(1:nhist,1))
        noise = sigma/sqrt(ndata-1.0)
        !
      case (i_scm,i_cscm)
        !
        ! Define the Histogram range around the median
        goon = .true.
        brms = rms
        crms = 0.1*rms
        count = 0
        amed = 0
        do while (goon)
          call median_filter (temp,nchan,bval,eval,aoff,amin,amax,arms,thremu)
          if (debug) print *,' Amin ',amin,', Amax ',amax,', Aoff ',aoff,' RMS ',rms, arms
          goon = (amin.lt.amax)     ! Protect against fully constant data (0 in models...)
          goon = goon .and. abs(arms-brms).gt.crms
          brms = arms
          crms = 0.1*brms
          count = count+1
          if (count.gt.mcount) ilong = ilong+1
          amed = amed+aoff
        enddo
        !
        ! Evaluate the noise - How many data points ?
        tmp = 0
        where(temp.gt.amax) tmp = 1.0
        npos = sum(tmp)
        tmp = 0
        where (temp.lt.amin) tmp = 1.0
        nneg = sum(tmp)
        ndata = nchan-npos-nneg
        noise = arms/sqrt(ndata-1.0)
        !
        ! Debias by % of Positive or Negative outliers
        ! if (icode.eq.i_bcm) then
      case (i_egm)
        error = .true.
      end select
      ! OK, we got the Histogram, now we can analyze it
      ! hist(:,2) are the values of flux in the various channels
      ! hist(:,1) are the number of channels with these values
      !
      ! Now we can fit the distribution by some analytic function
      ! to find out the most likely value, i.e. the "continuum" flux
      !
      ! Depending on the SkewNess, this function can be a Gaussian
      ! or a much more complex one
      array(ix,jy) = amed
      noises(ix,jy) = noise
      tnoise = tnoise+noise
      !
      if (error) gbl_error = .true.
      if (sic_ctrlc()) gbl_error = .true.
      call progress_report(' ',nxy,next,mxy,percentage_step)
    enddo ! IX
  enddo ! JY
  !$OMP END DO
  !$OMP END PARALLEL
  write(6,'(A)') '...  Done.'
  !
  ! Interpolate blanked pixels ?
  error = .false.
  hcont%r2d = 0
  where(array.ne.bval) hcont%r2d  = array
  hcont%gil%blan_words = 2
  !
  spectrum(:,3) = spectrum(:,2)
  spectrum(:,4) = 0
  !
  if (ihollow.ne.0) then
    write(mess,'(A,I0,A)') 'Corrected ',ihollow,' Pixels with hollow histograms'
    call map_message(seve%w,rname,mess,3)
  endif
  if (iout.ne.0) then
    write(mess,'(A,I0,A)') 'Corrected ',iout,' Pixels with out of range solutions'
    call map_message(seve%w,rname,mess,3)
  endif
  if (ilong.ne.0) then
    write(mess,'(A,I0,A)') 'Found ',ilong,' Pixels with non converged median'
    call map_message(seve%w,rname,mess,3)
  endif
  !
  tnoise = tnoise/nxy
  hcont%gil%rms = hcont%gil%rms /sqrt(nchan-1.0)
  hcont%gil%noise = tnoise
  !
end subroutine gauss_continuum
!  
subroutine comp_r4_shape_blank (x,n,mean,rms,skew,vblank4,eblank4)
  use gildas_def
  !---------------------------------------------------------------------
  ! @ private-mandatory 
  !
  ! Compute Mean, Rms and Skewness of an array
  ! Protected against NaNs and Using Blanking Values. If Blanking
  ! enabled, will return blanking value if no valid result. If Blanking
  ! is not enabled, will return NaN if no valid result.
  !
  !---------------------------------------------------------------------
  real(4),          intent(in)  :: x(*)  ! Data values to compute
  integer(kind=size_length), intent(in)  :: n     ! Number of data values
  real(4),          intent(out) :: mean  ! Output scalar value
  real(4),          intent(out) :: rms   ! Output scalar value
  real(4),          intent(out) :: skew  ! Output scalar value
  real(4),          intent(in)  :: vblank4, eblank4
  ! Local
  integer(kind=size_length) :: i,count
  real(4) :: s0,s1,s2,y,z, num, out
  !
  !!Print *,'N ',n,' Bval ',vblank4,eblank4
  out = vblank4
  if (n.lt.1) return
  out = 0.0
  !
  s0 = 0.
  s1 = 0.
  s2 = 0.
  count = 0
  do i = 1,n
    if (x(i).eq.x(i)) then
      if (eblank4.ge.0) then
        if (abs(x(i)-vblank4).gt.eblank4) then
          s0 = s0+x(i)
          count = count+1
        endif
      else
        s0 = s0+x(i)
        count = count+1
      endif
    endif
  enddo
  if (count.gt.0) then
    s0 = s0/dble(count)
  else
    out = vblank4 ! lastchanceretval
    return
  endif
  !
  ! Note philosophique: Count is the same in the 2 loops
  do i = 1,n
    if (x(i).eq.x(i)) then
      if (eblank4.lt.0) then
        y = x(i)-s0
        z = y*y
        s1 = s1 + z 
        s2 = s2 + z*y
      else
        if (abs(x(i)-vblank4).gt.eblank4) then
          y = x(i)-s0
          z = y*y
          s1 = s1 + z 
          s2 = s2 + z*y
        endif
      endif
    endif
  enddo
  num = real(count)
  s1 = s1/(num-1)
  rms = sqrt (s1)
  !
  s2 = s2/num
  ! Correct for sample size
  skew = s2/(rms*rms*rms)*sqrt(num*(num-1))/(num-2)
end subroutine comp_r4_shape_blank
!
function continuum_emg(x, amp, mu, sig, lamb)
  real(4) :: continuum_emg ! intent(out)
  real(4), intent(in) :: x
  real(4), intent(in) :: amp ! Amplitude of function
  real(4), intent(in) :: mu  ! Position
  real(4), intent(in) :: sig ! Width
  real(4), intent(in) :: lamb ! Exponential factor
  !
  continuum_emg = &
  & amp*exp(lamb*(2.*mu+lamb*sig*sig-2.*x)/2.)*(1.-erf((mu+lamb*sig*sig-x)/(sqrt(2.)*sig)))
end function continuum_emg
!
function continuum_gauss(x, amp, mu, sig)
  real(4) :: continuum_gauss ! intent(out)
  real(4), intent(in) :: x
  real(4), intent(in) :: amp ! Amplitude of function
  real(4), intent(in) :: mu  ! Position
  real(4), intent(in) :: sig ! Width
  !
  continuum_gauss = &
  & amp*exp(-0.5*((x-mu)/sig)**2)
end function continuum_gauss
!
subroutine get_logv(logv,logwings,temp,nchan,hist,nhist,mcount,amin,amax,bval,eval)
  use gildas_def
  !-------------------------------------------------------------------------
  ! @ private
  !   IMAGER
  !     Support for command MAP_CONTINUUM 
  !   Derive the histgram of the intensity distribution
  !   and make sure its logarithm is suitable for Gaussian fitting 
  !   (i.e. of parabolic shape)
  !-------------------------------------------------------------------------
  integer(kind=size_length), intent(in) :: nchan  ! Number of data points
  integer(kind=size_length), intent(in) :: nhist  ! Number of histogram bins
  real, intent(out) :: logv(nhist)                ! Log of bin contents
  real, intent(in) :: logwings                    ! Truncation threshold
  real, intent(in) :: temp(nchan)                 ! Data points
  real, intent(out) :: hist(nhist,2)              ! Histogram
  real, intent(out) :: amin, amax                 ! Range of selected values
  real, intent(in) :: bval, eval                  ! Blanking
  integer, intent(in) :: mcount                   ! Maximum number of loops
  !
  integer :: code                  ! Operation code
  !
  ! Derive histogram
  call histo44(temp,nchan,hist,nhist,2,amin,amax,bval,eval)  
  !
  ! Verify its shape
  code = 0
  call check_logv(logv,logwings,temp,nchan,hist,nhist,amin,amax,bval,eval,mcount,code)
end subroutine get_logv
!
subroutine check_logv(logv,logwings,temp,nchan,hist,nhist,amin,amax,bval,eval,mcount,code)
  use gildas_def
  !-------------------------------------------------------------------------
  ! @ private
  !   IMAGER
  !     Support for command MAP_CONTINUUM 
  !
  !   Make sure its logarithm of the Intensity distribution histogram
  !   is suitable for Gaussian fitting  (i.e. of parabolic shape)
  !   Re-adjust range and (optionally) recompute histogram if needed.
  !-------------------------------------------------------------------------
  integer(kind=size_length), intent(in) :: nchan  ! Number of data points
  integer(kind=size_length), intent(in) :: nhist  ! Number of histogram bins
  real, intent(out) :: logv(nhist)                ! Log of bin contents
  real, intent(in) :: logwings                    ! Truncation threshold
  real, intent(in) :: temp(nchan)                 ! Data points
  real, intent(out) :: hist(nhist,2)              ! Histogram
  real, intent(out) :: amin, amax                 ! Range of selected values
  real, intent(in) :: bval, eval                  ! Blanking
  integer, intent(in) :: mcount                   ! Maximum number of loops
  integer, intent(inout) :: code                  ! Operation code
  !
  integer :: jh,lh,kh,nh,mloc(2), jcount
  real :: logbase, val
  !
  ! Verify Histogram shape - Ignore outliers beyond the main peak
  ! Log of a Gaussian is a parabola
  logv(:) = log(hist(:,1)+1.0)           ! Bias by 1 to avoid Log(0)
  logbase = maxval(logv)-logwings        ! Truncate wings
  where (logv.lt.logbase) logv=0.0
  !
  jcount = 1
  do while (jcount.lt.mcount)
    mloc = maxloc(logv,1) 
    nh = mloc(1)
    kh = 1
    do jh=nh,1,-1
      if (logv(jh).eq.0) then
        kh = jh+1
        exit
      endif
    enddo
    lh = nhist
    do jh=nh,nhist,1
      if (logv(jh).eq.0) then
        lh = jh-1
        exit
      endif
    enddo
    !
    if (code.eq.0) then
      if ((kh.gt.1) .or. (lh.lt.nhist)) then
        ! Histogram is too peaked,
        !   Restrict range to avoid outliers
        amin = hist(kh,2)
        amax = hist(lh,2)
        call histo44(temp,nchan,hist,nhist,2,amin,amax,bval,eval)  
        logv(:) = log(hist(:,1)+1.0)           ! Bias by 1 to avoid Log(0)
        logbase = maxval(logv)-logwings        ! Truncate wings
        where (logv.lt.logbase) logv=0.0
        jcount = jcount+1
        if (jcount.eq.mcount) then
          if (any(logv.eq.0)) code = -1
          exit
        endif
      else 
        exit    ! Converged histogram
      endif
    else 
      if ((kh.eq.1).or.(lh.eq.nhist)) then 
        ! Histogram is too flat
        !   Just extend range to a symmetric interval
        val = max(-amin,amax)
        amin = -val
        amax = val
        code = -1
        exit
      endif
    endif
  enddo
  !
end subroutine check_logv
!
subroutine debug_plot(ix,jy,nhist,hist,logv,logf,nchan,spectrum,temp, i, ch)
  use gildas_def
  integer, intent(in) :: ix,jy
  integer(kind=size_length), intent(in) :: nhist
  integer(kind=size_length), intent(in) :: nchan
  real, intent(in) :: spectrum(nchan,2), temp(nchan)
  real, intent(in) :: hist(nhist,2), logv(nhist), logf(nhist)
  integer, intent(inout) :: i
  character(len=*) :: ch
  !
  character(len=64) :: st
  !
  call gr_exec('CLEAR')
  !
  call gr4_give('X',nchan,spectrum(:,1))
  call gr4_give('Y',nchan,temp)
  call gr_exec('SET BOX  16 26 4 18')
  call gr_exec('LIMITS '//ch)
  call gr_exec('BOX')
  call gr_exec('HISTO')
  !
  call gr_exec('SET BOX 4 12 4 18')  
  write(st,'(A,I0,A,I0,A)') '"IX ',ix,'  IY ',jy,'"'
  call gr4_give('X',nhist,hist(:,2))
  call gr4_give('Y',nhist,logv)
  call gr_exec('LIMITS')
  call gr_exec('BOX')
  call gr_exec('HISTO')
  call gr4_give('Y',nhist,logf)
  call gr_exec('CONNECT')
  call gr_exec('DRAW TEXT 0 1 '//trim(st)//' /CHAR 8') 
  if (i.eq.0) then
    write(*,*) 'Enter an integer to continue (0 Question, 1 Ignore next, -1 abort'
    read(5,*) i
  endif
end subroutine debug_plot
!
