subroutine beam_jvm(hbeam,dbeam,jvm,circle,print,error)
  use image_def
  !-------------------------------------------------------------
  ! IMAGER
  !   @ private
  !   Support for FIT /JVM commands
  !
  !   Evaluate the Jorsater - van Moorsel residual scaling factor.
  !   from the dirty beam shape.
  ! 
  !-------------------------------------------------------------
  type(gildas), intent(in) :: hbeam     ! Beam header
  real, intent(in) :: dbeam(:,:)        ! 2-D beam
  real, intent(out) :: jvm              ! Scaling factor
  logical, intent(in) :: circle         ! Method of circular averaging
  logical, intent(in) :: print          ! Save results for display
  logical, intent(out) :: error         ! Error return
  !
  integer :: npix, nx, ny, ix, iy, ir, ier, io
  real(8) :: cospa, sinpa, ux, uy, vx, vy, zz, ctot, dtot
  real(8) :: radius, major, minor, angle, zero, epsilon
  real(8), allocatable :: axis(:), dc(:), dd(:), cc(:), cd(:), area(:)
  integer, allocatable :: count(:)
  !
  error = .false.
  nx = hbeam%gil%dim(1)
  ny = hbeam%gil%dim(2)
  npix = min(nx,ny)/2
  !
  allocate (axis(npix), dd(npix), dc(npix), cc(npix), cd(npix), count(npix), area(npix), stat=ier)
  if (ier.ne.0) then
    error = .true.
    return
  endif
  !
  ! Compute Clean beam
  major = hbeam%gil%majo
  minor = hbeam%gil%mino
  angle = hbeam%gil%posa
  cospa = cos(angle) 
  sinpa = sin(angle)
  !
  count = 0
  dc = 0
  dd = 0
  axis = 0
  !
  ! Compute the azimutal averages of the Clean and Dirty beams,
  ! after circularization of the Clean beam, or not (according to "circle" 
  do iy=1,ny
    uy = (iy-hbeam%gil%convert(1,2))*hbeam%gil%convert(3,2)
    do ix=1,nx
      ux = (ix-hbeam%gil%convert(1,1))*hbeam%gil%convert(3,1)
      !
      vx = cospa * ux - sinpa * uy
      vy = sinpa * ux + cospa * uy
      !
      ! zz is the Square of the "deprojected" distance in Clean beam axis size
      ! so we might directly use that for the distance instead of the coordinate
      !
      zz = (vx/major)**2 + (vy/minor)**2
      zz = zz * 4.d0*log(2d0) ! FWHM to sigma conversion
      !
      if (circle) then
        radius = sqrt(zz)*major     ! Circularized beam
      else
        !
        radius = sqrt(ux**2+uy**2)  ! Elliptical beam
      endif
      ir = nint(radius/abs(hbeam%gil%convert(3,1)))
      if (ir.gt.0 .and. ir.le.npix) then
        count(ir) = count(ir)+1
        dc(ir) = dc(ir) + exp(-zz)
        dd(ir) = dd(ir) + dbeam(ix,iy) 
        axis(ir) = axis(ir) + radius
      endif
    enddo
  enddo
  !
  io = 0
  do ir=1,npix
    if (count(ir).ne.0) then
      io = io+1
      axis(io) = axis(ir)/count(ir)
      dc(io) = dc(ir)/count(ir)
      dd(io) = dd(ir)/count(ir)
      area(io) = count(ir)
    endif
  enddo
  !
  npix = io
  call gr8_trie(axis,count,npix,error)
  call gr8_sort(dc,cc,count,npix)
  call gr8_sort(dd,cc,count,npix)
  call gr8_sort(area,cc,count,npix)
  !
  ! Compute cumulative areas as a function of radius.
  ! Identify first null of the Dirty beam.
  ! Derive JvM factor as ratio of Clean to Dirty beam areas at first null.
  ctot = 0
  dtot = 0
  zero = 0
  epsilon = 0
  do ir=1,npix
    cc(ir) = ctot
    cd(ir) = dtot
    ctot = ctot + dc(ir)*area(ir)
    dtot = dtot + dd(ir)*area(ir)
    !!Print *,'IR ',ir,' Zero ',zero,' DD ',dd(ir),' Epsilon ',epsilon
    if ((zero.eq.0).and.(dd(ir).lt.zero)) then
      zero = axis(min(npix,ir+5))
      epsilon = cc(ir-1)/cd(ir-1) 
    endif
  enddo
  jvm = epsilon
  if (.not.print) return
  !
  ! This is to debug the behaviour. It is based on the value of the 
  ! JVM_PRINT SIC variable decoded in the "clean_beam" routine.
  write(1,*) zero, epsilon
  write(1,*) axis(1),dc(1),dd(1),cc(1),cd(1),1.0d0,area(ir)
  do ir=2,npix
    write(1,*) axis(ir),dc(ir),dd(ir),cc(ir),cd(ir),cc(ir)/cd(ir),area(ir)
  enddo
  close(unit=1)
  !
end subroutine beam_jvm
!
subroutine clean_beam (line,error)
  use gkernel_interfaces
  use clean_def
  use clean_arrays
  use gbl_message
  use imager_interfaces, only : map_message, get_clean, beam_jvm
  !-------------------------------------------------------------
  ! IMAGER
  !   @ private
  !
  ! Code for COMMAND
  !   FIT [First Last] [PLANE|CHANNEL|FIELD First Last] 
  !       [PLANE|CHANNEL|FIELD First Last] [/JVM_FACTOR  [NoCircle]]
  !       [/THRESHOLD Value]
  !-------------------------------------------------------------
  character(len=*), intent(in) :: line
  logical, intent(inout) :: error
  real, parameter :: sec=acos(-1.0)/180./3600.
  !
  character(len=*), parameter :: rname = 'FIT'
  real(kind=8), parameter :: pi=acos(-1d0)
  integer, parameter :: o_thresh=2
  integer, parameter :: o_jvm=1
  !
  logical, save :: first_call=.True.
  !
  type(gildas) :: head
  integer :: nfield, cfirst, clast, ffirst, flast, nchan, k, iarg, itype, narg
  integer :: jplane, jfield, iw, iz
  real, save, target :: fit_major, fit_minor, fit_angle, fit_jvm 
  real :: jvm
  logical :: do_jvm, do_circle, jvm_print
  !
  integer, parameter :: mtype=3
  character(len=12) :: name
  character(len=8) :: ctype(mtype), argum, ktype
  data ctype /'CHANNEL','PLANE','FIELD'/
  !
  if (huv%loca%size.eq.0) then
    call map_message(seve%e,rname,'No UV data')
    error = .true.
    return
  endif
  !
  if (hbeam%loca%size.eq.0) then
    call map_message(seve%e,rname,'No Dirty Beam')
    error = .true.
    return
  endif
  !
  nchan = hbeam%gil%dim(hbeam%gil%faxi)
  !
  nfield = 1
  if (hbeam%char%code(3).eq.'FIELD') then
    nfield = hbeam%gil%dim(3)
  endif
  if (hbeam%gil%ndim.gt.3) then
    if (hbeam%char%code(4).eq.'FIELD') then
      nfield = hbeam%gil%dim(4)
    endif
  endif
  !
  ffirst = 1
  flast = 0
  cfirst = 1
  clast = 0
  narg = sic_narg(0)
  !
  if ((nfield.gt.1).and.(nchan.gt.1)) then
    !
    ! Required Keyword FIELD or PLANE or CHANNEL
    iarg = 1
    do while (iarg.le.narg)
      argum = ' '
      call sic_ke(line,0,1,argum,k,.true.,error)
      if (error) return
      call sic_ambigs('FIT',argum,ktype,itype,ctype,mtype,error)
      if (error) return
      iarg = iarg+1
      !
      if (itype.eq.3) then ! FIELD
        ffirst = 1
        call sic_i4 (line,0,iarg,ffirst,.false.,error)
        if (iarg.eq.narg) then
          flast = ffirst
        else
          iarg = iarg+1
          flast = 0
          call sic_i4 (line,0,iarg,flast,.false.,error)
          if (error) return
        endif
      else
        cfirst = 1
        call sic_i4 (line,0,iarg,cfirst,.false.,error)
        if (iarg.eq.narg) then
          clast = cfirst
        else
          iarg = iarg+1
          clast = 0
          call sic_i4 (line,0,iarg,clast,.false.,error)
          if (error) return
        endif
        if (error) return
      endif
      iarg = iarg+1
    enddo
    !
  else if (narg.gt.0) then     
    ! Here, keywords are optional
    argum = ' '
    call sic_ke(line,0,1,argum,k,.false.,error)
    if (nchan.gt.1) then
      !  only CHANNEL or PLANE is allowed
      if ((argum(1:1).ne.'P').and.(argum(1:1).ne.'C')) then
        iarg = 1
      else
        call sic_ambigs(' ',argum,ktype,itype,ctype,2,error)
        if (.not.error) iarg = 2
        error = .false.
      endif
      cfirst = 1
      call sic_i4 (line,0,iarg,cfirst,.false.,error)
      iarg = iarg+1
      clast = 0
      call sic_i4 (line,0,iarg,clast,.false.,error)
      if (error) return
    else
      ! only FIELD is allowed
      if (argum(1:1).ne.'F') then
        iarg = 1
      else
        call sic_ambigs(' ',argum,ktype,itype,ctype(3),1,error)
        if (.not.error) iarg = 2
        error = .false.
      endif
      ffirst = 1
      call sic_i4 (line,0,iarg,ffirst,.false.,error)
      iarg = iarg+1
      flast = 0
      call sic_i4 (line,0,iarg,flast,.false.,error)
      if (error) return
    endif    
    !
  endif
  if (flast.le.0) flast = nfield+flast
  if (clast.le.0) clast = nchan+clast
  !
  cfirst = min(max(1,cfirst),nchan)
  clast = max(cfirst,min(clast,nchan))
  ffirst = min(max(1,ffirst),nfield)
  flast = max(ffirst,min(flast,nfield))
  !
  method%thresh = user_method%thresh
  call sic_r4(line,o_thresh,1,method%thresh,.false.,error)
  if (error) return
  !
  do_jvm = sic_present(o_jvm,0)
  if (do_jvm) then
    call gildas_null(head)
    call gdf_copy_header(hbeam,head,error)
    do_circle = .not.sic_present(o_jvm,1)
    jvm_print = .false.
    call sic_get_logi('JVM_PRINT',jvm_print,error)
    error = .false.
  endif
  !
  fit_major = 0
  fit_minor = 0
  fit_angle = 0
  fit_jvm = 0
  jvm = 0 ! Required...
  !
  k = 0
  !
  name = method%method
  method%method = '  '
  do jfield = ffirst, flast
    do jplane = cfirst, clast
      method%major = 0.0
      method%minor = 0.0
      method%angle = 0.0
      !
      if (hbeam%gil%faxi.eq.3) then
        iz = jplane
        iw = jfield
        call get_clean (method, hbeam, dbeam(:,:,jplane,jfield),error)
      else
        iz = jfield
        iw = jplane
      endif
      call get_clean (method, hbeam, dbeam(:,:,iz,iw),error)
      if (do_jvm) then 
        head%gil%majo = method%major
        head%gil%mino = method%minor
        head%gil%posa = method%angle*pi/180
        call beam_jvm(head,dbeam(:,:,iz,iw),jvm,do_circle,jvm_print,error)
        fit_jvm = fit_jvm + jvm
      endif
      call pribeam('FIT',method%major,method%minor,method%angle,jfield,jplane,jvm)
      !
      fit_major = fit_major + method%major/sec
      fit_minor = fit_minor + method%minor/sec
      fit_angle = fit_angle + method%angle ! In Degree ?
      k = k+1
    enddo
  enddo
  method%method = name
  fit_jvm = fit_jvm/k
  fit_major = fit_major/k
  fit_minor = fit_minor/k
  fit_angle = fit_angle/k
  if (fit_jvm.eq.0) fit_jvm = 1.0
  !
  ! Put in Beam Header
  hbeam%gil%reso_words = 3
  hbeam%gil%majo = fit_major*pi/180/3600
  hbeam%gil%mino = fit_minor*pi/180/3600
  hbeam%gil%posa = fit_angle*pi/180
  !
  if (first_call) then
    call sic_def_real('BEAM_MAJOR',fit_major,0,0,.true.,error)
    call sic_def_real('BEAM_MINOR',fit_minor,0,0,.true.,error)
    call sic_def_real('BEAM_ANGLE',fit_angle,0,0,.true.,error)
    call sic_def_real('BEAM_JVM',fit_jvm,0,0,.true.,error)
    first_call = .false.
  endif
end subroutine clean_beam
!
subroutine new_dirty_beam
  use gkernel_interfaces
  use clean_def
  use clean_arrays
  !
  ! needed when a new dirty map is computed by command uv_map:
  logical :: error
  error = .false.
  !
  if (allocated(dclean)) deallocate(dclean)
  call sic_delvariable ('CLEAN',.false.,error)
  hclean%loca%size = 0
  !
  if (allocated(dresid)) deallocate(dresid)
  call sic_delvariable ('RESIDUAL',.false.,error)
  hresid%loca%size = 0
  !
  if (allocated(dcct)) deallocate(dcct)
  call sic_delvariable ('CCT',.false.,error)
  hcct%loca%size = 0
  !
  if (allocated(dsky)) deallocate(dsky)
  call sic_delvariable ('SKY',.false.,error)
  hsky%loca%size = 0
end subroutine new_dirty_beam
