/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-subapp-thread.c,v 1.14 2004/12/12 12:38:57 hoa Exp $
 */

#include "etpan-subapp-thread.h"
#include "etpan-thread-manager.h"
#include "etpan-errors.h"
#include <stdlib.h>
#include "etpan-app.h"
#include "etpan-msg-params.h"
#include <libetpan/libetpan.h>
#include <string.h>

struct thread_op_callback {
  int app_op_type;
  struct etpan_thread_op * op;
  void (* callback)(struct etpan_subapp *,
      struct etpan_thread_op *, int, void *);
  void * data;
  void (* handle_cancel)(struct etpan_subapp *,
      struct etpan_thread_op *, int, void *);
};

/*
  when no callback is provided, the action cannot be cancelled.
  WARNING : This feature is highly used in other parts.
*/

int etpan_subapp_thread_op_add(struct etpan_subapp * app,
    int app_op_type,
    int cmd,
    struct mailstorage * storage,
    struct mailfolder * folder,
    mailmessage * msg,
    struct mailmime * mime,
    void * arg,
    void (* callback)(struct etpan_subapp *,
        struct etpan_thread_op *, int, void *),
    void * data,
    void (* handle_cancel)(struct etpan_subapp *,
        struct etpan_thread_op *, int, void *))
{
  struct etpan_thread_op * op;
  int r;
  unsigned int pos;
  struct thread_op_callback * op_callback;
  
  if (callback != NULL) {
    /* allocate cell */
    op_callback = malloc(sizeof(* op_callback));
    if (op_callback == NULL) {
      goto err;
    }
    op_callback->app_op_type = app_op_type;
    op_callback->op = NULL;
    op_callback->callback = callback;
    op_callback->data = data;
    op_callback->handle_cancel = handle_cancel;
    
    r = carray_add(app->thread_list, op_callback, &pos);
    if (r < 0) {
      free(op_callback);
      goto err;
    }
    /* run attached */
    
    op = etpan_thread_op_add(app->app->thread_manager,
        cmd, storage, folder, msg, mime, arg, 0);
    if (op == NULL) {
      free(op_callback);
      carray_delete(app->thread_list, pos);
      goto err;
    }
    
    op_callback->op = op;
  }
  else {
    /* run detached */
    
    op = etpan_thread_op_add(app->app->thread_manager,
        cmd, storage, folder, msg, mime, arg, 1);
    if (op == NULL) {
      goto err;
    }
  }
  
  return NO_ERROR;
  
 err:
  ETPAN_APP_LOG((app->app,
                    "thread manager - could not create operation, not enough memory"));
  return ERROR_MEMORY;
}

void etpan_subapp_thread_cancel_all(struct etpan_subapp * app)
{
  unsigned int i;
  
  ETPAN_APP_DEBUG((app->app, "cancel all thread of %s", app->driver->name));
  for(i = 0 ; i < carray_count(app->thread_list) ; i ++) {
    struct thread_op_callback * op_callback;
    
    op_callback = carray_get(app->thread_list, i);
    if (op_callback->handle_cancel != NULL)
      op_callback->handle_cancel(app, op_callback->op,
          op_callback->app_op_type, op_callback->data);
    etpan_thread_op_cancel(op_callback->op);
    free(op_callback);
  }
  carray_set_size(app->thread_list, 0);
}

void etpan_subapp_thread_set_fd(struct etpan_subapp * app,
    fd_set * fds, int * maxfd)
{
  unsigned int i;
  
  for(i = 0 ; i < carray_count(app->thread_list) ; i ++) {
    struct thread_op_callback * op_callback;
    
    op_callback = carray_get(app->thread_list, i);
    etpan_thread_op_set_fds(op_callback->op, fds, maxfd);
  }
}

void etpan_subapp_thread_handle_fd(struct etpan_subapp * app,
    fd_set * fds)
{
  unsigned int i;
  
  i = 0;
  while(i < carray_count(app->thread_list)) {
    struct thread_op_callback * op_callback;
    
    op_callback = carray_get(app->thread_list, i);
    if (etpan_thread_op_is_finished(op_callback->op)) {
      struct etpan_thread_op * op;
      void (* callback)(struct etpan_subapp *,
          struct etpan_thread_op *, int, void *);
      int app_op_type;
      void * data;
      
      op = op_callback->op;
      app_op_type = op_callback->app_op_type;
      callback = op_callback->callback;
      data = op_callback->data;
      
      free(op_callback);
      carray_delete(app->thread_list, i);
      
      etpan_thread_op_wait(op);
      callback(app, op, app_op_type, data);
      etpan_thread_op_ack(op);
    }
    else {
      i ++;
    }
  }
}

int etpan_subapp_thread_has_match_op(struct etpan_subapp * app,
    int app_op_type,
    struct mailstorage * storage, struct mailfolder * folder,
    mailmessage * msg, struct mailmime * mime)
{
  unsigned int i;
  
  for(i = 0 ; i < carray_count(app->thread_list) ; i ++) {
    struct thread_op_callback * op_callback;
    struct etpan_thread_op * op;
    int match;
    
    op_callback = carray_get(app->thread_list, i);
    op = op_callback->op;
    
    match = 1;
    
    if (app_op_type != -1)
      if (app_op_type != op_callback->app_op_type)
        match = 0;
    
    if (storage != NULL)
      if (storage != op->data.mailaccess.storage)
        match = 0;

    if (folder != NULL)
      if (folder != op->data.mailaccess.folder)
        match = 0;

    if (msg != NULL)
      if (msg != op->data.mailaccess.msg)
        match = 0;
    
    if (mime != NULL)
      if (mime != op->data.mailaccess.mime)
        match = 0;
    
    if (match)
      return 1;
  }
  
  return 0;
}


/* helper functions for message reference */

/*
  etpan_queue_ref_msg_mime()
*/

#if 0
int etpan_queue_ref_msg_mime(struct etpan_subapp * app,
    struct mailfolder * folder,
    mailmessage * msg)
{
  struct mailstorage * storage;
  int r;

  storage = NULL;
  if (folder != NULL)
    storage = folder->storage;
  
  r = etpan_subapp_thread_op_add(app, THREAD_ID_REFCOUNTER_REF_MSG_MIME,
      ETPAN_THREAD_MESSAGE_GET_BODYSTRUCTURE,
      storage, folder, msg, NULL, NULL,
      NULL, NULL, NULL);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app->app, "ref mime - not enough memory"));
    return r;
  }
  
  return NO_ERROR;
}
#endif

int etpan_queue_ref_msg_mime(struct etpan_subapp * app,
    mailmessage * msg)
{
  int r;
  
  r = etpan_subapp_thread_msg_op_add(app, THREAD_ID_REFCOUNTER_REF_MSG_MIME,
      ETPAN_THREAD_MESSAGE_GET_BODYSTRUCTURE,
      msg, NULL, NULL,
      NULL, NULL, NULL);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app->app, "ref mime - not enough memory"));
    return r;
  }
  
  return NO_ERROR;
}

/*
  etpan_queue_unref_msg_mime()
  
  result can be ignore, if this fails, this means that memory is low,
  and there will be some leak.
*/

#if 0
int etpan_queue_unref_msg_mime(struct etpan_subapp * app,
    struct mailfolder * folder,
    mailmessage * msg)
{
  struct mailstorage * storage;
  int r;

  storage = NULL;
  if (folder != NULL)
    storage = folder->storage;
  
  r = etpan_subapp_thread_op_add(app, THREAD_ID_REFCOUNTER_UNREF_MSG_MIME,
      ETPAN_THREAD_MESSAGE_FLUSH,
      storage, folder, msg, NULL, NULL,
      NULL, NULL, NULL);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app->app, "ref mime - not enough memory"));
    return r;
  }
  
  return NO_ERROR;
}
#endif

int etpan_queue_unref_msg_mime(struct etpan_subapp * app,
    mailmessage * msg)
{
  int r;

  r = etpan_subapp_thread_msg_op_add(app, THREAD_ID_REFCOUNTER_UNREF_MSG_MIME,
      ETPAN_THREAD_MESSAGE_FLUSH,
      msg, NULL, NULL,
      NULL, NULL, NULL);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app->app, "ref mime - not enough memory"));
    return r;
  }
  
  return NO_ERROR;
}

/*
  etpan_queue_ref_msg()
*/

#if 0
int etpan_queue_ref_msg(struct etpan_subapp * app,
    struct mailfolder * folder,
    mailmessage * msg)
{
  struct mailstorage * storage;
  int r;

  storage = NULL;
  if (folder != NULL)
    storage = folder->storage;
  
  r = etpan_subapp_thread_op_add(app, THREAD_ID_REFCOUNTER_REF_MSG,
      ETPAN_THREAD_MESSAGE_REF,
      storage, folder, msg, NULL, NULL,
      NULL, NULL, NULL);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app->app, "ref msg - not enough memory"));
    return r;
  }
  
  return NO_ERROR;
}
#endif

int etpan_queue_ref_msg(struct etpan_subapp * app,
    mailmessage * msg)
{
  int r;

  r = etpan_subapp_thread_msg_op_add(app, THREAD_ID_REFCOUNTER_REF_MSG,
      ETPAN_THREAD_MESSAGE_REF,
      msg, NULL, NULL,
      NULL, NULL, NULL);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app->app, "ref msg - not enough memory"));
    return r;
  }
  
  return NO_ERROR;
}


/*
  etpan_queue_unref_msg()
  
  result can be ignore, if this fails, this means that memory is low,
  and there will be some leak.
*/

#if 0
int etpan_queue_unref_msg(struct etpan_subapp * app,
    struct mailfolder * folder,
    mailmessage * msg)
{
  struct mailstorage * storage;
  int r;

  storage = NULL;
  if (folder != NULL)
    storage = folder->storage;
  
  r = etpan_subapp_thread_op_add(app, THREAD_ID_REFCOUNTER_UNREF_MSG,
      ETPAN_THREAD_MESSAGE_UNREF,
      storage, folder, msg, NULL, NULL,
      NULL, NULL, NULL);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app->app, "ref msg - not enough memory"));
    return r;
  }
  
  return NO_ERROR;
}
#endif

int etpan_queue_unref_msg(struct etpan_subapp * app,
    mailmessage * msg)
{
  int r;

  r = etpan_subapp_thread_msg_op_add(app, THREAD_ID_REFCOUNTER_UNREF_MSG,
      ETPAN_THREAD_MESSAGE_UNREF,
      msg, NULL, NULL,
      NULL, NULL, NULL);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app->app, "ref msg - not enough memory"));
    return r;
  }
  
  return NO_ERROR;
}


/* helper functions to add thread operations */

int etpan_subapp_thread_msg_op_add(struct etpan_subapp * app,
    int app_op_type,
    int cmd,
    mailmessage * msg,
    struct mailmime * mime,
    void * arg,
    void (* callback)(struct etpan_subapp *,
        struct etpan_thread_op *, int, void *),
    void * data,
    void (* handle_cancel)(struct etpan_subapp *,
        struct etpan_thread_op *, int, void *))
{
  struct mailfolder * folder;

#if 0
  chashdatum key;
  chashdatum value;
  int r;
  
  key.data = (char *) &msg;
  key.len = sizeof(msg);
  
  r = chash_get(app->app->thread_manager->message_hash, &key, &value);
  if (r == 0)
    folder = (struct mailfolder *) value.data;
  else
    folder = NULL;
#endif
  folder = libetpan_message_get_folder(app->app->thread_manager->engine, msg);
  
  return etpan_subapp_thread_folder_op_add(app, app_op_type, cmd,
    folder, msg, mime, arg, callback, data, handle_cancel);
}


int etpan_subapp_thread_folder_op_add(struct etpan_subapp * app,
    int app_op_type,
    int cmd,
    struct mailfolder * folder,
    mailmessage * msg,
    struct mailmime * mime,
    void * arg,
    void (* callback)(struct etpan_subapp *,
        struct etpan_thread_op *, int, void *),
    void * data,
    void (* handle_cancel)(struct etpan_subapp *,
        struct etpan_thread_op *, int, void *))
{
  struct mailstorage * storage;
  
  if (folder != NULL)
    storage = folder->fld_storage;
  else
    storage = NULL;
  
  return etpan_subapp_thread_op_add(app, app_op_type, cmd,
    storage, folder, msg, mime, arg, callback, data, handle_cancel);
}


static inline int collect_single_node_msg(struct etpan_subapp * app,
    chash * hash_msg_tree, unsigned int child_pos,
    struct mailmessage_tree * node)
{
  mailmessage * msg;
  struct mailmessage_tree * root;
  unsigned int index;
  chashdatum key;
  chashdatum value;
  struct mailfolder * folder;
  int r;
  
  msg = node->node_msg;
  if (msg == NULL)
    return NO_ERROR;
  
#if 0
  key.data = (char *) &msg;
  key.len = sizeof(msg);
  r = chash_get(app->app->thread_manager->message_hash, &key, &value);
  if (r < 0) {
    ETPAN_APP_DEBUG((app->app, "BUG detected - message does not belong to a folder"));
    return ERROR_INVAL;
  }
  
  folder = (struct mailfolder *) value.data;
#endif
  folder = libetpan_message_get_folder(app->app->thread_manager->engine, msg);
  
  key.data = &folder;
  key.len = sizeof(folder);
  r = chash_get(hash_msg_tree, &key, &value);
  if (r < 0) {
    /* set the root */
    
    value.data = node;
    value.len = 0;
    r = chash_set(hash_msg_tree, &key, &value, NULL);
    if (r < 0)
      return ERROR_MEMORY;
    
    if (node->node_parent != NULL)
      carray_set(node->node_parent->node_children, child_pos, NULL);
    
    node->node_parent = NULL;
  }
  else {
    /* add the node as a child */
    root = value.data;
    
    r = carray_add(root->node_children, node, &index);
    if (r < 0)
      return ERROR_MEMORY;
    
    if (node->node_parent != NULL)
      carray_set(node->node_parent->node_children, child_pos, NULL);
    
    node->node_parent = root;
  }
  
  return NO_ERROR;
}

static int recursive_collect_by_folder(struct etpan_subapp * app,
    chash * hash_msg_tree, unsigned int child_pos,
    struct mailmessage_tree * node)
{
  unsigned int i;
  int r;
  
  r = collect_single_node_msg(app, hash_msg_tree, child_pos, node);
  if (r != NO_ERROR)
    goto err;
  
  for(i = 0 ; i < carray_count(node->node_children) ; i ++) {
    struct mailmessage_tree * child;
    
    child = carray_get(node->node_children, i);
    
    r = recursive_collect_by_folder(app, hash_msg_tree, i, child);
    if (r != NO_ERROR)
      goto err;
    
    child = carray_get(node->node_children, i);
    
    if (child != NULL)
      mailmessage_tree_free(child);
  }
  
  /* empties children list */
  carray_set_size(node->node_children, 0);
  
  return NO_ERROR;
  
 err:
  return MAIL_ERROR_MEMORY;
}


void etpan_free_msg_list(struct etpan_subapp * app,
    struct mailmessage_tree * env_tree,
    struct etpan_node_msg_params * node_params)
{
  int r;
  chash * hash_msg_tree;
#if 0
  uint32_t i;
#endif
  chashiter * cur;
  
  if (node_params != NULL)
    etpan_node_msg_params_free(node_params);
  
  hash_msg_tree = chash_new(CHASH_DEFAULTSIZE, CHASH_COPYKEY);
  if (hash_msg_tree == NULL)
    goto err;
  
  r = recursive_collect_by_folder(app, hash_msg_tree, 0, env_tree);
  if (r != NO_ERROR)
    goto free_hash;

  mailmessage_tree_free(env_tree);
  
  /* run threads */
  
  for(cur = chash_begin(hash_msg_tree) ; cur != NULL ;
      cur = chash_next(hash_msg_tree, cur)) {
    struct mailfolder * folder;
    chashdatum key;
    chashdatum value;
    struct etpan_folder_free_msg_list_arg * arg;
    struct mailmessage_tree * tree;
    
    chash_key(cur, &key);
    memcpy(&folder, key.data, sizeof(folder));
    chash_value(cur, &value);
    tree = value.data;
    
    arg = malloc(sizeof(* arg));
    if (arg == NULL) {
      goto err;
    }
    
    arg->env_tree = tree;
    arg->node_params = NULL;
    r = etpan_subapp_thread_folder_op_add(app,
        THREAD_ID_MSGLIST_FREE_MSG_LIST,
        ETPAN_THREAD_FOLDER_FREE_MSG_LIST,
        folder,
        NULL, NULL,
        arg,
        NULL, NULL, NULL);
    if (r != NO_ERROR) {
      free(arg);
      goto err;
    }
  }
  
  chash_free(hash_msg_tree);
  
  return;
  
 free_hash:
  chash_free(hash_msg_tree);
 err:
  ETPAN_APP_DEBUG((app->app, "free msg list - not enough memory"));
}


int etpan_subapp_thread_abook_op_add(struct etpan_subapp * app,
    int app_op_type,
    int cmd,
    struct etpan_abook * abook,
    void * arg,
    void (* callback)(struct etpan_subapp *,
        struct etpan_thread_op *, int, void *),
    void * data,
    void (* handle_cancel)(struct etpan_subapp *,
        struct etpan_thread_op *, int, void *))
{
  struct etpan_thread_op * op;
  int r;
  unsigned int pos;
  struct thread_op_callback * op_callback;
  
  if (callback != NULL) {
    /* allocate cell */
    op_callback = malloc(sizeof(* op_callback));
    if (op_callback == NULL) {
      goto err;
    }
    op_callback->app_op_type = app_op_type;
    op_callback->op = NULL;
    op_callback->callback = callback;
    op_callback->data = data;
    op_callback->handle_cancel = handle_cancel;
    
    r = carray_add(app->thread_list, op_callback, &pos);
    if (r < 0) {
      free(op_callback);
      goto err;
    }
    /* run attached */
    
    op = etpan_thread_abook_op_add(app->app->thread_manager,
        cmd, abook, arg, 0);
    if (op == NULL) {
      free(op_callback);
      carray_delete(app->thread_list, pos);
      goto err;
    }
    
    op_callback->op = op;
  }
  else {
    /* run detached */
    
    op = etpan_thread_abook_op_add(app->app->thread_manager,
        cmd, abook, arg, 1);
    if (op == NULL) {
      goto err;
    }
  }
  
  return NO_ERROR;
  
 err:
  ETPAN_APP_LOG((app->app,
                    "thread manager - could not create operation, not enough memory"));
  return ERROR_MEMORY;
}

int etpan_subapp_thread_sender_op_add(struct etpan_subapp * app,
    int app_op_type,
    int cmd,
    struct etpan_sender_item * sender,
    void * arg,
    void (* callback)(struct etpan_subapp *,
        struct etpan_thread_op *, int, void *),
    void * data,
    void (* handle_cancel)(struct etpan_subapp *,
        struct etpan_thread_op *, int, void *))
{
  struct etpan_thread_op * op;
  int r;
  unsigned int pos;
  struct thread_op_callback * op_callback;
  
  if (callback != NULL) {
    /* allocate cell */
    op_callback = malloc(sizeof(* op_callback));
    if (op_callback == NULL) {
      goto err;
    }
    op_callback->app_op_type = app_op_type;
    op_callback->op = NULL;
    op_callback->callback = callback;
    op_callback->data = data;
    op_callback->handle_cancel = handle_cancel;
    
    r = carray_add(app->thread_list, op_callback, &pos);
    if (r < 0) {
      free(op_callback);
      goto err;
    }
    /* run attached */
    
    op = etpan_thread_sender_op_add(app->app->thread_manager,
        cmd, sender, arg, 0);
    if (op == NULL) {
      free(op_callback);
      carray_delete(app->thread_list, pos);
      goto err;
    }
    
    op_callback->op = op;
  }
  else {
    /* run detached */
    
    op = etpan_thread_sender_op_add(app->app->thread_manager,
        cmd, sender, arg, 1);
    if (op == NULL) {
      goto err;
    }
  }
  
  return NO_ERROR;
  
 err:
  ETPAN_APP_LOG((app->app,
                    "thread manager - could not create operation, not enough memory"));
  return ERROR_MEMORY;
}
