/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.rest;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Supplier;
import java.util.function.UnaryOperator;
import java.util.stream.Collectors;
import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.client.node.NodeClient;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.breaker.CircuitBreaker;
import org.elasticsearch.common.bytes.BytesArray;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.io.stream.BytesStreamOutput;
import org.elasticsearch.common.logging.DeprecationLogger;
import org.elasticsearch.common.path.PathTrie;
import org.elasticsearch.common.util.concurrent.ThreadContext;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.core.internal.io.Streams;
import org.elasticsearch.http.HttpServerTransport;
import org.elasticsearch.indices.breaker.CircuitBreakerService;
import org.elasticsearch.rest.BaseRestHandler;
import org.elasticsearch.rest.BytesRestResponse;
import org.elasticsearch.rest.DeprecationRestHandler;
import org.elasticsearch.rest.MethodHandlers;
import org.elasticsearch.rest.RestChannel;
import org.elasticsearch.rest.RestHandler;
import org.elasticsearch.rest.RestHeaderDefinition;
import org.elasticsearch.rest.RestRequest;
import org.elasticsearch.rest.RestResponse;
import org.elasticsearch.rest.RestStatus;
import org.elasticsearch.rest.RestUtils;
import org.elasticsearch.usage.UsageService;
import org.elasticsearch.xcontent.XContentBuilder;
import org.elasticsearch.xcontent.XContentType;

public class RestController
implements HttpServerTransport.Dispatcher {
    private static final Logger logger = LogManager.getLogger(RestController.class);
    private static final DeprecationLogger deprecationLogger = DeprecationLogger.getLogger(RestController.class);
    static final String ELASTIC_PRODUCT_HTTP_HEADER = "X-elastic-product";
    static final String ELASTIC_PRODUCT_HTTP_HEADER_VALUE = "Elasticsearch";
    private static final BytesReference FAVICON_RESPONSE;
    private final PathTrie<MethodHandlers> handlers = new PathTrie(RestUtils.REST_DECODER);
    private final UnaryOperator<RestHandler> handlerWrapper;
    private final NodeClient client;
    private final CircuitBreakerService circuitBreakerService;
    private final Set<RestHeaderDefinition> headersToCopy;
    private final UsageService usageService;

    public RestController(Set<RestHeaderDefinition> headersToCopy, UnaryOperator<RestHandler> handlerWrapper, NodeClient client, CircuitBreakerService circuitBreakerService, UsageService usageService) {
        this.headersToCopy = headersToCopy;
        this.usageService = usageService;
        if (handlerWrapper == null) {
            handlerWrapper = h -> h;
        }
        this.handlerWrapper = handlerWrapper;
        this.client = client;
        this.circuitBreakerService = circuitBreakerService;
        this.registerHandlerNoWrap(RestRequest.Method.GET, "/favicon.ico", (request, channel, clnt) -> channel.sendResponse(new BytesRestResponse(RestStatus.OK, "image/x-icon", FAVICON_RESPONSE)));
    }

    protected void registerAsDeprecatedHandler(RestRequest.Method method, String path, RestHandler handler, String deprecationMessage) {
        this.registerAsDeprecatedHandler(method, path, handler, deprecationMessage, null);
    }

    protected void registerAsDeprecatedHandler(RestRequest.Method method, String path, RestHandler handler, String deprecationMessage, @Nullable Level deprecationLevel) {
        assert (!(handler instanceof DeprecationRestHandler));
        this.registerHandler(method, path, new DeprecationRestHandler(handler, method, path, deprecationLevel, deprecationMessage, deprecationLogger));
    }

    protected void registerAsReplacedHandler(RestRequest.Method method, String path, RestHandler handler, RestRequest.Method replacedMethod, String replacedPath) {
        String replacedMessage = "[" + replacedMethod.name() + " " + replacedPath + "] is deprecated! Use [" + method.name() + " " + path + "] instead.";
        this.registerHandler(method, path, handler);
        this.registerAsDeprecatedHandler(replacedMethod, replacedPath, handler, replacedMessage);
    }

    protected void registerHandler(RestRequest.Method method, String path, RestHandler handler) {
        if (handler instanceof BaseRestHandler) {
            this.usageService.addRestHandler((BaseRestHandler)handler);
        }
        this.registerHandlerNoWrap(method, path, (RestHandler)this.handlerWrapper.apply(handler));
    }

    private void registerHandlerNoWrap(RestRequest.Method method, String path, RestHandler maybeWrappedHandler) {
        this.handlers.insertOrUpdate(path, new MethodHandlers(path).addMethod(method, maybeWrappedHandler), (mHandlers, newMHandler) -> mHandlers.addMethod(method, maybeWrappedHandler));
    }

    public void registerHandler(RestHandler.Route route, RestHandler handler) {
        if (route.isReplacement()) {
            RestHandler.Route replaced = route.getReplacedRoute();
            this.registerAsReplacedHandler(route.getMethod(), route.getPath(), handler, replaced.getMethod(), replaced.getPath());
        } else if (route.isDeprecated()) {
            this.registerAsDeprecatedHandler(route.getMethod(), route.getPath(), handler, route.getDeprecationMessage(), route.getDeprecationLevel());
        } else {
            this.registerHandler(route.getMethod(), route.getPath(), handler);
        }
    }

    public void registerHandler(RestHandler handler) {
        handler.routes().forEach(route -> this.registerHandler((RestHandler.Route)route, handler));
    }

    @Override
    public void dispatchRequest(RestRequest request, RestChannel channel, ThreadContext threadContext) {
        threadContext.addResponseHeader(ELASTIC_PRODUCT_HTTP_HEADER, ELASTIC_PRODUCT_HTTP_HEADER_VALUE);
        try {
            this.tryAllHandlers(request, channel, threadContext);
        }
        catch (Exception e) {
            try {
                channel.sendResponse(new BytesRestResponse(channel, e));
            }
            catch (Exception inner) {
                inner.addSuppressed(e);
                logger.error(() -> new ParameterizedMessage("failed to send failure response for uri [{}]", (Object)request.uri()), (Throwable)inner);
            }
        }
    }

    @Override
    public void dispatchBadRequest(RestChannel channel, ThreadContext threadContext, Throwable cause) {
        threadContext.addResponseHeader(ELASTIC_PRODUCT_HTTP_HEADER, ELASTIC_PRODUCT_HTTP_HEADER_VALUE);
        try {
            Exception e = cause == null ? new ElasticsearchException("unknown cause", new Object[0]) : (cause instanceof Exception ? (Exception)cause : new ElasticsearchException(cause));
            channel.sendResponse(new BytesRestResponse(channel, RestStatus.BAD_REQUEST, e));
        }
        catch (IOException e) {
            if (cause != null) {
                e.addSuppressed(cause);
            }
            logger.warn("failed to send bad request response", (Throwable)e);
            channel.sendResponse(new BytesRestResponse(RestStatus.INTERNAL_SERVER_ERROR, "text/plain; charset=UTF-8", BytesArray.EMPTY));
        }
    }

    private void dispatchRequest(RestRequest request, RestChannel channel, RestHandler handler, ThreadContext threadContext) throws Exception {
        int contentLength = request.contentLength();
        if (contentLength > 0) {
            XContentType xContentType = request.getXContentType();
            if (xContentType == null) {
                this.sendContentTypeErrorMessage(request.getAllHeaderValues("Content-Type"), channel);
                return;
            }
            if (handler.supportsContentStream() && xContentType != XContentType.JSON && xContentType != XContentType.SMILE) {
                channel.sendResponse(BytesRestResponse.createSimpleErrorResponse(channel, RestStatus.NOT_ACCEPTABLE, "Content-Type [" + (Object)((Object)xContentType) + "] does not support stream parsing. Use JSON or SMILE instead"));
                return;
            }
        }
        RestChannel responseChannel = channel;
        try {
            if (handler.canTripCircuitBreaker()) {
                RestController.inFlightRequestsBreaker(this.circuitBreakerService).addEstimateBytesAndMaybeBreak(contentLength, "<http_request>");
            } else {
                RestController.inFlightRequestsBreaker(this.circuitBreakerService).addWithoutBreaking(contentLength);
            }
            responseChannel = new ResourceHandlingHttpChannel(channel, this.circuitBreakerService, contentLength);
            if (!handler.allowsUnsafeBuffers()) {
                request.ensureSafeBuffers();
            }
            if (!handler.allowSystemIndexAccessByDefault()) {
                String prodOriginValue = request.header("X-elastic-product-origin");
                if (prodOriginValue != null) {
                    threadContext.putHeader("_system_index_access_allowed", Boolean.TRUE.toString());
                    threadContext.putHeader("_external_system_index_access_origin", prodOriginValue);
                } else {
                    threadContext.putHeader("_system_index_access_allowed", Boolean.FALSE.toString());
                }
            } else {
                threadContext.putHeader("_system_index_access_allowed", Boolean.TRUE.toString());
            }
            handler.handleRequest(request, responseChannel, this.client);
        }
        catch (Exception e) {
            responseChannel.sendResponse(new BytesRestResponse(responseChannel, e));
        }
    }

    private boolean handleNoHandlerFound(String rawPath, RestRequest.Method method, String uri, RestChannel channel) {
        Set<RestRequest.Method> validMethodSet = this.getValidHandlerMethodSet(rawPath);
        if (!validMethodSet.contains((Object)method)) {
            if (method == RestRequest.Method.OPTIONS) {
                this.handleOptionsRequest(channel, validMethodSet);
                return true;
            }
            if (!validMethodSet.isEmpty()) {
                this.handleUnsupportedHttpMethod(uri, method, channel, validMethodSet, null);
                return true;
            }
        }
        return false;
    }

    private void sendContentTypeErrorMessage(@Nullable List<String> contentTypeHeader, RestChannel channel) throws IOException {
        String errorMessage = contentTypeHeader == null ? "Content-Type header is missing" : "Content-Type header [" + Strings.collectionToCommaDelimitedString(contentTypeHeader) + "] is not supported";
        channel.sendResponse(BytesRestResponse.createSimpleErrorResponse(channel, RestStatus.NOT_ACCEPTABLE, errorMessage));
    }

    private void tryAllHandlers(RestRequest request, RestChannel channel, ThreadContext threadContext) throws Exception {
        RestRequest.Method requestMethod;
        try {
            this.copyRestHeaders(request, threadContext);
            this.validateErrorTrace(request, channel);
        }
        catch (IllegalArgumentException e) {
            channel.sendResponse(BytesRestResponse.createSimpleErrorResponse(channel, RestStatus.BAD_REQUEST, e.getMessage()));
            return;
        }
        String rawPath = request.rawPath();
        String uri = request.uri();
        try {
            requestMethod = request.method();
            Iterator<MethodHandlers> allHandlers = this.getAllHandlers(request.params(), rawPath);
            while (allHandlers.hasNext()) {
                MethodHandlers handlers = allHandlers.next();
                RestHandler handler = handlers == null ? null : handlers.getHandler(requestMethod);
                if (handler == null) {
                    if (!this.handleNoHandlerFound(rawPath, requestMethod, uri, channel)) continue;
                    return;
                }
                this.dispatchRequest(request, channel, handler, threadContext);
                return;
            }
        }
        catch (IllegalArgumentException e) {
            this.handleUnsupportedHttpMethod(uri, null, channel, this.getValidHandlerMethodSet(rawPath), e);
            return;
        }
        this.handleBadRequest(uri, requestMethod, channel);
    }

    private void validateErrorTrace(RestRequest request, RestChannel channel) {
        if (request.paramAsBoolean("error_trace", false) && !channel.detailedErrorsEnabled()) {
            throw new IllegalArgumentException("error traces in responses are disabled.");
        }
    }

    private void copyRestHeaders(RestRequest request, ThreadContext threadContext) throws IOException {
        for (RestHeaderDefinition restHeader : this.headersToCopy) {
            String name = restHeader.getName();
            List<String> headerValues = request.getAllHeaderValues(name);
            if (headerValues == null || headerValues.isEmpty()) continue;
            List distinctHeaderValues = headerValues.stream().distinct().collect(Collectors.toList());
            if (!restHeader.isMultiValueAllowed() && distinctHeaderValues.size() > 1) {
                throw new IllegalArgumentException("multiple values for single-valued header [" + name + "].");
            }
            if (name.equals("traceparent")) {
                String traceparent = (String)distinctHeaderValues.get(0);
                if (traceparent.length() < 55) continue;
                threadContext.putHeader("trace.id", traceparent.substring(3, 35));
                continue;
            }
            threadContext.putHeader(name, String.join((CharSequence)",", distinctHeaderValues));
        }
    }

    Iterator<MethodHandlers> getAllHandlers(@Nullable Map<String, String> requestParamsRef, String rawPath) {
        Supplier<Map<String, String>> paramsSupplier;
        if (requestParamsRef == null) {
            paramsSupplier = () -> null;
        } else {
            HashMap<String, String> originalParams = new HashMap<String, String>(requestParamsRef);
            paramsSupplier = () -> {
                requestParamsRef.clear();
                requestParamsRef.putAll(originalParams);
                return requestParamsRef;
            };
        }
        return this.handlers.retrieveAll(rawPath, paramsSupplier);
    }

    private void handleUnsupportedHttpMethod(String uri, @Nullable RestRequest.Method method, RestChannel channel, Set<RestRequest.Method> validMethodSet, @Nullable IllegalArgumentException exception) {
        try {
            StringBuilder msg = new StringBuilder();
            if (exception == null) {
                msg.append("Incorrect HTTP method for uri [").append(uri);
                msg.append("] and method [").append((Object)method).append("]");
            } else {
                msg.append(exception.getMessage());
            }
            if (!validMethodSet.isEmpty()) {
                msg.append(", allowed: ").append(validMethodSet);
            }
            BytesRestResponse bytesRestResponse = BytesRestResponse.createSimpleErrorResponse(channel, RestStatus.METHOD_NOT_ALLOWED, msg.toString());
            if (!validMethodSet.isEmpty()) {
                bytesRestResponse.addHeader("Allow", Strings.collectionToDelimitedString(validMethodSet, ","));
            }
            channel.sendResponse(bytesRestResponse);
        }
        catch (IOException e) {
            logger.warn("failed to send bad request response", (Throwable)e);
            channel.sendResponse(new BytesRestResponse(RestStatus.INTERNAL_SERVER_ERROR, "text/plain; charset=UTF-8", BytesArray.EMPTY));
        }
    }

    private void handleOptionsRequest(RestChannel channel, Set<RestRequest.Method> validMethodSet) {
        BytesRestResponse bytesRestResponse = new BytesRestResponse(RestStatus.OK, "text/plain; charset=UTF-8", BytesArray.EMPTY);
        if (!validMethodSet.isEmpty()) {
            bytesRestResponse.addHeader("Allow", Strings.collectionToDelimitedString(validMethodSet, ","));
        }
        channel.sendResponse(bytesRestResponse);
    }

    private void handleBadRequest(String uri, RestRequest.Method method, RestChannel channel) throws IOException {
        try (XContentBuilder builder = channel.newErrorBuilder();){
            builder.startObject();
            builder.field("error", "no handler found for uri [" + uri + "] and method [" + (Object)((Object)method) + "]");
            builder.endObject();
            channel.sendResponse(new BytesRestResponse(RestStatus.BAD_REQUEST, builder));
        }
    }

    private Set<RestRequest.Method> getValidHandlerMethodSet(String rawPath) {
        HashSet<RestRequest.Method> validMethods = new HashSet<RestRequest.Method>();
        Iterator<MethodHandlers> allHandlers = this.getAllHandlers(null, rawPath);
        while (allHandlers.hasNext()) {
            MethodHandlers methodHandlers = allHandlers.next();
            if (methodHandlers == null) continue;
            validMethods.addAll(methodHandlers.getValidMethods());
        }
        return validMethods;
    }

    private static CircuitBreaker inFlightRequestsBreaker(CircuitBreakerService circuitBreakerService) {
        return circuitBreakerService.getBreaker("in_flight_requests");
    }

    static {
        try (InputStream stream = RestController.class.getResourceAsStream("/config/favicon.ico");){
            ByteArrayOutputStream out = new ByteArrayOutputStream();
            Streams.copy(stream, out);
            FAVICON_RESPONSE = new BytesArray(out.toByteArray());
        }
        catch (IOException e) {
            throw new AssertionError((Object)e);
        }
    }

    private static final class ResourceHandlingHttpChannel
    implements RestChannel {
        private final RestChannel delegate;
        private final CircuitBreakerService circuitBreakerService;
        private final int contentLength;
        private final AtomicBoolean closed = new AtomicBoolean();

        ResourceHandlingHttpChannel(RestChannel delegate, CircuitBreakerService circuitBreakerService, int contentLength) {
            this.delegate = delegate;
            this.circuitBreakerService = circuitBreakerService;
            this.contentLength = contentLength;
        }

        @Override
        public XContentBuilder newBuilder() throws IOException {
            return this.delegate.newBuilder();
        }

        @Override
        public XContentBuilder newErrorBuilder() throws IOException {
            return this.delegate.newErrorBuilder();
        }

        @Override
        public XContentBuilder newBuilder(@Nullable XContentType xContentType, boolean useFiltering) throws IOException {
            return this.delegate.newBuilder(xContentType, useFiltering);
        }

        @Override
        public XContentBuilder newBuilder(XContentType xContentType, XContentType responseContentType, boolean useFiltering) throws IOException {
            return this.delegate.newBuilder(xContentType, responseContentType, useFiltering);
        }

        @Override
        public BytesStreamOutput bytesOutput() {
            return this.delegate.bytesOutput();
        }

        @Override
        public RestRequest request() {
            return this.delegate.request();
        }

        @Override
        public boolean detailedErrorsEnabled() {
            return this.delegate.detailedErrorsEnabled();
        }

        @Override
        public void sendResponse(RestResponse response) {
            this.close();
            this.delegate.sendResponse(response);
        }

        private void close() {
            if (!this.closed.compareAndSet(false, true)) {
                throw new IllegalStateException("Channel is already closed");
            }
            RestController.inFlightRequestsBreaker(this.circuitBreakerService).addWithoutBreaking(-this.contentLength);
        }
    }
}

