use std::collections::HashMap;

use crate::Result;
use hexpm::version::{Range, Version};
use itertools::Itertools;

#[derive(Debug, Clone, serde::Serialize, serde::Deserialize, PartialEq, Eq)]
pub struct Manifest {
    #[serde(serialize_with = "ordered_map")]
    pub requirements: HashMap<String, Range>,
    #[serde(serialize_with = "sorted_vec")]
    pub packages: Vec<ManifestPackage>,
}

impl Manifest {
    // Rather than using the toml library to do serialization we implement it
    // manually so that we can control the formatting.
    // We want to keep entries on a single line each so that they are more
    // resistant to merge conflicts and are easier to fix when it does happen.
    pub fn write_to<W: std::fmt::Write>(&self, mut buffer: W) -> std::fmt::Result {
        let Self {
            requirements,
            packages,
        } = self;
        write!(
            buffer,
            "# This file was generated by Gleam
# You typically do not need to edit this file

",
        )?;

        // Packages
        writeln!(buffer, "packages = [")?;
        for ManifestPackage {
            name,
            source,
            version,
            otp_app,
            build_tools,
            requirements,
        } in packages.iter().sorted_by(|a, b| a.name.cmp(&b.name))
        {
            write!(buffer, r#"  {{"#)?;
            write!(buffer, r#" name = "{}""#, name)?;
            write!(buffer, r#", version = "{}""#, version)?;
            write!(buffer, r#", build_tools = ["#)?;
            for (i, tool) in build_tools.iter().enumerate() {
                if i != 0 {
                    write!(buffer, ", ")?;
                }
                write!(buffer, "\"{}\"", tool)?;
            }

            write!(buffer, "], requirements = [")?;
            for (i, package) in requirements.iter().enumerate() {
                if i != 0 {
                    write!(buffer, ", ")?;
                }
                write!(buffer, "\"{}\"", package)?;
            }
            write!(buffer, "]")?;

            if let Some(app) = otp_app {
                write!(buffer, ", otp_app = \"{}\"", app)?;
            }

            match source {
                ManifestPackageSource::Hex { outer_checksum } => {
                    write!(buffer, r#", source = "hex""#)?;
                    write!(
                        buffer,
                        r#", outer_checksum = "{}""#,
                        outer_checksum.to_string()
                    )?;
                }
            };

            writeln!(buffer, " }},")?;
        }
        write!(buffer, "]\n\n")?;

        // Requirements
        writeln!(buffer, "[requirements]")?;
        for (name, range) in requirements.iter().sorted_by(|a, b| a.0.cmp(b.0)) {
            writeln!(buffer, "{} = \"{}\"", name, range)?;
        }

        Ok(())
    }
}

#[test]
fn manifest_toml_format() {
    let mut buffer = String::new();
    let mut manifest = Manifest {
        requirements: [
            ("zzz".to_string(), Range::new("> 0.0.0".to_string())),
            ("aaa".to_string(), Range::new("> 0.0.0".to_string())),
            (
                "gleam_stdlib".to_string(),
                Range::new("~> 0.17".to_string()),
            ),
            ("gleeunit".to_string(), Range::new("~> 0.1".to_string())),
        ]
        .into(),
        packages: vec![
            ManifestPackage {
                name: "gleam_stdlib".to_string(),
                version: Version::new(0, 17, 1),
                build_tools: ["gleam".into()].into(),
                otp_app: None,
                requirements: vec![],
                source: ManifestPackageSource::Hex {
                    outer_checksum: Base16Checksum(vec![1, 22]),
                },
            },
            ManifestPackage {
                name: "aaa".to_string(),
                version: Version::new(0, 4, 0),
                build_tools: ["rebar3".into(), "make".into()].into(),
                otp_app: Some("aaa_app".into()),
                requirements: vec!["zzz".into(), "gleam_stdlib".into()],
                source: ManifestPackageSource::Hex {
                    outer_checksum: Base16Checksum(vec![3, 22]),
                },
            },
            ManifestPackage {
                name: "zzz".to_string(),
                version: Version::new(0, 4, 0),
                build_tools: ["mix".into()].into(),
                otp_app: None,
                requirements: vec![],
                source: ManifestPackageSource::Hex {
                    outer_checksum: Base16Checksum(vec![3, 22]),
                },
            },
            ManifestPackage {
                name: "gleeunit".to_string(),
                version: Version::new(0, 4, 0),
                build_tools: ["gleam".into()].into(),
                otp_app: None,
                requirements: vec!["gleam_stdlib".into()],
                source: ManifestPackageSource::Hex {
                    outer_checksum: Base16Checksum(vec![3, 46]),
                },
            },
        ],
    };
    manifest.write_to(&mut buffer).unwrap();
    assert_eq!(
        buffer,
        r#"# This file was generated by Gleam
# You typically do not need to edit this file

packages = [
  { name = "aaa", version = "0.4.0", build_tools = ["rebar3", "make"], requirements = ["zzz", "gleam_stdlib"], otp_app = "aaa_app", source = "hex", outer_checksum = "0316" },
  { name = "gleam_stdlib", version = "0.17.1", build_tools = ["gleam"], requirements = [], source = "hex", outer_checksum = "0116" },
  { name = "gleeunit", version = "0.4.0", build_tools = ["gleam"], requirements = ["gleam_stdlib"], source = "hex", outer_checksum = "032E" },
  { name = "zzz", version = "0.4.0", build_tools = ["mix"], requirements = [], source = "hex", outer_checksum = "0316" },
]

[requirements]
aaa = "> 0.0.0"
gleam_stdlib = "~> 0.17"
gleeunit = "~> 0.1"
zzz = "> 0.0.0"
"#
    );
    let deserialised: Manifest = toml::from_str(&buffer).unwrap();
    manifest.packages.sort_by(|a, b| a.name.cmp(&b.name));
    assert_eq!(deserialised, manifest);
}

#[derive(Debug, Clone, PartialEq, Eq, PartialOrd, Ord)]
pub struct Base16Checksum(pub Vec<u8>);

impl ToString for Base16Checksum {
    fn to_string(&self) -> String {
        base16::encode_upper(&self.0)
    }
}

impl serde::Serialize for Base16Checksum {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: serde::Serializer,
    {
        serializer.serialize_str(&base16::encode_upper(&self.0))
    }
}

impl<'de> serde::Deserialize<'de> for Base16Checksum {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error>
    where
        D: serde::Deserializer<'de>,
    {
        let s: &str = serde::de::Deserialize::deserialize(deserializer)?;
        base16::decode(s)
            .map(Base16Checksum)
            .map_err(serde::de::Error::custom)
    }
}
#[derive(Debug, Clone, PartialEq, Eq, PartialOrd, Ord, serde::Serialize, serde::Deserialize)]
pub struct ManifestPackage {
    pub name: String,
    pub version: Version,
    pub build_tools: Vec<String>,
    #[serde(default)]
    pub otp_app: Option<String>,
    #[serde(serialize_with = "sorted_vec")]
    pub requirements: Vec<String>,
    #[serde(flatten)]
    pub source: ManifestPackageSource,
}

#[derive(Debug, Clone, PartialEq, Eq, PartialOrd, Ord, serde::Serialize, serde::Deserialize)]
#[serde(tag = "source")]
pub enum ManifestPackageSource {
    #[serde(rename = "hex")]
    Hex { outer_checksum: Base16Checksum },
}

fn ordered_map<S, K, V>(value: &HashMap<K, V>, serializer: S) -> Result<S::Ok, S::Error>
where
    S: serde::Serializer,
    K: serde::Serialize + Ord,
    V: serde::Serialize,
{
    use serde::Serialize;
    let ordered: std::collections::BTreeMap<_, _> = value.iter().collect();
    ordered.serialize(serializer)
}

fn sorted_vec<S, T>(value: &[T], serializer: S) -> Result<S::Ok, S::Error>
where
    S: serde::Serializer,
    T: serde::Serialize + Ord,
{
    use serde::Serialize;
    let mut value: Vec<&T> = value.iter().collect();
    value.sort();
    value.serialize(serializer)
}
