!=================================================================================
!
! Routines:
!
! (1) epscopy_mpi()      modified from epscopy_disk()      Last Modified 3/3/2010 (CHP)
!
!     Copy dielectric matrices form eps0mat/epsmat files to memory.
!
!     We no longer symmetrize the q->0 dielectric matrix (esp0mat) using symmetries.
!
!     This routine reads the eps0mat and epsmat files (units 10 and 11)
!     and copies the relevant information about those dielectric matrices
!     to memory (epsmpi).  Processor 0 will actually read data and redistribute
!     them to other processors.
!
!     The routine actually modifies two of the input arguments:  nq and q(:,:).
!     These are read from unit 11 (file epsmat) so they should be the same
!     as what was read in.  But just to be safe, they are broadcasted
!     to other units at the end.
!
!     nq actually increases by 1 after going through this routine.
!
!==================================================================================

#include "f_defs.h"

subroutine epscopy_mpi(crys,gvec,nq,q,sig,neps,epsmpi,epshead)

  use global_m
  use misc_m
  implicit none

  type (crystal), intent(in) :: crys
  type (gspace), intent(in) :: gvec
  integer, intent(inout) :: nq
  real(DP), intent(inout) :: q(3,nq+1)
  type (siginfo), intent(inout) :: sig
  integer, intent(in) :: neps
  type (epsmpiinfo), intent(inout) :: epsmpi
  SCALAR, intent(out) :: epshead

  integer :: nq0,ngq,ngq0,nmtx,nqpp
  integer, allocatable :: isrtq(:),isrtqi(:)
  real(DP) :: qk(3)
  real(DP), allocatable :: ekold(:)
  SCALAR, allocatable :: eps(:)
  complex(DPC), allocatable :: epsRDyn(:,:),epsADyn(:,:)

  integer :: iq,i,j,jj,iout,nold,nold0,gx,gy,gz, &
    nFreq,qgrid(3),dest,tag
  integer, allocatable :: oldx(:),oldy(:),oldz(:),isrtold(:)
  real(DP) :: gmax_in,q0(3),ekin
  character :: ajname*6,adate*11,ajname2*6,adate2*11
  character (len=100) :: tmpstr


!--------------------- Begin Calculation -----------------------------------------
!
! CHP:  Even if you are not node 0, you have to receive data allocated to you.
!       Life is somewhat more fair than in epscopy.

! Otherwise, this is node 0, so let`s get some space to do some work

  PUSH_SUB(epscopy_mpi)

  if (peinf%inode .eq. 0) then
    write(6,*) ' '
    write(6,*) ' Allocating epsilon matrices: ', neps
    write(6,*) ' '
  endif
  
  if (peinf%inode .eq. 0) then
    SAFE_ALLOCATE(isrtq, (gvec%ng))
    SAFE_ALLOCATE(isrtqi, (gvec%ng))
  endif

!---------------------------
! SIB: Read information for inverse dielectric matrix for q->0 tape10 (eps0mat)

  if(peinf%inode.eq.0) then
    read(10)
    read(10)
    read(10)
    read(10)
    read(10)
    read(10)
    read(10)
    read(10)
    read(10) nold0
    read(10) ngq0
    call close_file(10)
    call open_file(unit=10,file='eps0mat',form='unformatted',status='old')
    
    SAFE_ALLOCATE(oldx, (nold0))
    SAFE_ALLOCATE(oldy, (nold0))
    SAFE_ALLOCATE(oldz, (nold0))
    SAFE_ALLOCATE(isrtold, (ngq0))
    SAFE_ALLOCATE(ekold, (ngq0))
    
    read(10) ajname,adate
    read(10) i,sig%nFreq
    
  endif

  if (sig%freq_dep .eq. 2) then
#ifdef MPI
    call MPI_Bcast(sig%nFreq,1,MPI_INTEGER,0,MPI_COMM_WORLD,mpierr)
#endif
    SAFE_ALLOCATE(sig%dFreqGrid,(sig%nFreq))
    SAFE_ALLOCATE(sig%dFreqBrd,(sig%nFreq))
  endif

! JRD Now that we have actually read sig%nFreq, we may use it to allocate epsDyn
  
  if (sig%freq_dep.eq.0.or.sig%freq_dep.eq.1) then
    SAFE_ALLOCATE(eps, (neps))
    SAFE_ALLOCATE(epsmpi%eps, (neps,epsmpi%ngown,nq+1))
  endif
  if (sig%freq_dep.eq.2) then
    SAFE_ALLOCATE(epsRDyn, (sig%nFreq,neps))
    SAFE_ALLOCATE(epsmpi%epsR, (sig%nFreq,neps,epsmpi%ngown,nq+1))
#ifdef CPLX
    SAFE_ALLOCATE(epsADyn, (sig%nFreq,neps))
    SAFE_ALLOCATE(epsmpi%epsA, (sig%nFreq,neps,epsmpi%ngown,nq+1))
#endif
  endif
  
  if(peinf%inode.eq.0) then
    if ((i.eq.0.and.sig%freq_dep.eq.2).or. &
      (i.eq.2.and.sig%freq_dep.eq.0).or. &
      (i.eq.2.and.sig%freq_dep.eq.1)) then
      call die('epscopy: eps0mat freq_dep')
    end if
    
    read(10) (qgrid(i),i=1,3)
    if(sig%igamma.ne.0) then
      if (all(sig%qgrid(1:3) == 0)) then
        sig%qgrid(:)=qgrid(:)
      endif
    endif
  endif
  
  if(peinf%inode.eq.0) then
    if (sig%freq_dep .eq. 2) then
      read(10) (sig%dFreqGrid(i),i=1,sig%nFreq),(sig%dFreqBrd(i),i=1,sig%nFreq)
      sig%dFreqBrd = sig%dFreqBrd/dble(sig%ninter)
    else
      read(10)
    endif
    
    read(10)
    read(10)
    read(10) gmax_in
    read(10) nq0,(q0(i),i=1,3)
    read(10) nold0,(oldx(i),oldy(i),oldz(i),i=1,nold0)
    if (nq0 /= 1) then
      call die('epscopy: nq0 /= 1')
    end if
    
!        if (dot_product(q0,q0) .lt. TOL_Small) then
!          call die('epscopy: q0 too small')
!        end if

    write(6,3003) ajname,adate,nq0,(q0(i),i=1,3)
3003 format(/,6x,'eps0mat read from ',a,2x,a,/, &
       6x,'nq0= ',i3, ' q0= ',3f10.3)
    
    if(sig%igamma == 0) then

!------------------------
! Read data for dielectric matrices from tape11 for q!=0
! tape11: 'epsmat'

      read(11) ajname2,adate2
      read(11) i,nFreq
      if ((i.eq.0.and.sig%freq_dep.eq.2).or. &
        (i.eq.2.and.sig%freq_dep.eq.0).or. &
        (i.eq.2.and.sig%freq_dep.eq.1)) then
        call die('epscopy: epsmat freq_dep')
      end if
      if (nFreq.ne.sig%nFreq) then
        call die('epscopy: epsmat freq_scale different between espmat and eps0mat')
      end if
      read(11) (qgrid(i),i=1,3)
      if (all(sig%qgrid(1:3) == 0)) then
        sig%qgrid(:)=qgrid(:)
      endif
      read(11)
      read(11)
      read(11)
      read(11) gmax_in
      read(11) nq,((q(i,j+1),i=1,3),j=1,nq)
      read(11) nold
      read(11) ngq
      
      write(6,3005) ajname2,adate2,nq,gmax_in
      write(6,3006) ((q(i,j+1),i=1,3),j=1,nq)
3005  format(/,6x,'epsmat read from',1x,a,2x,a, &
        /,7x,'nq=',i4,1x,'gmax=',f11.3)
3006  format(12x,3f10.3)

    endif

    q(:,1) = 0.0d0
    nqpp = nq + 1

  endif  ! if (peinf%inode.eq.0)

#ifdef MPI
  call MPI_Bcast(nq,1,MPI_INTEGER,0,MPI_COMM_WORLD,mpierr)
  call MPI_Bcast(nqpp,1,MPI_INTEGER,0,MPI_COMM_WORLD,mpierr)
#endif


!---------------------------
! Read q->0 dielectric matrix and distribute it to each processor

  if (peinf%inode.eq.0) then
    write(tmpstr,*) 'Reading ngq0=',ngq0,', nmtx, isortold(:)'
    call logit(tmpstr)
    read(10) ngq0,nmtx,(isrtold(i),j,i=1,ngq0)
    if (nmtx.gt.neps) then
      call die('epscopy: nmtx > neps')
    endif
    write(tmpstr,*) 'Reading ekold and eps, nmtx=',nmtx
    call logit(tmpstr)
    read(10) (ekold(i),i=1,ngq0)
    read(10) (qk(i),i=1,3)
    
    call logit(' Storing eps0 to memory')
    
    isrtq=0
    isrtqi=0
    call logit(' Matching oldxyz')
    do i=1,ngq0
      if (ekold(isrtold(i)).le.sig%ecutb) then
        gx=oldx(isrtold(i))
        gy=oldy(isrtold(i))
        gz=oldz(isrtold(i))
        call findvector(iout,gx,gy,gz,gvec)
        isrtq(i)=iout
        isrtqi(iout)=i
        if(i == 1) then
          ! just check the first so we do not waste time
          ekin = DOT_PRODUCT(dble(gvec%k(:,iout)), MATMUL(crys%bdot, dble(gvec%k(:,iout))))
          if(abs(ekin - ekold(isrtold(i))) > TOL_Zero) then
            write(0,*) 'eps0mat: ekold(isrtold(i)) = ', ekold(isrtold(i)), ' ekin = ', ekin
            call die('Incorrect kinetic energies in eps0mat.')
          endif
        endif
      endif
    enddo
    SAFE_DEALLOCATE(oldx)
    SAFE_DEALLOCATE(oldy)
    SAFE_DEALLOCATE(oldz)
    SAFE_DEALLOCATE(ekold)
    SAFE_DEALLOCATE(isrtold)
  endif
  
#ifdef MPI
  if (sig%freq_dep .eq. 2) then
    call MPI_Bcast(sig%dFreqGrid,sig%nFreq,MPI_DOUBLE_PRECISION,0,MPI_COMM_WORLD,mpierr)
    call MPI_Bcast(sig%dFreqBRD,sig%nFreq,MPI_DOUBLE_COMPLEX,0,MPI_COMM_WORLD,mpierr)
  endif
#endif

  if (peinf%inode .eq. 0) then
    epsmpi%isrtq(:,1) = isrtq(:)
    epsmpi%isrtqi(:,1) = isrtqi(:)
  endif
  
#ifdef MPI
  call MPI_Bcast(nmtx,1,MPI_INTEGER,0,MPI_COMM_WORLD,mpierr)
#endif

  epsmpi%nmtx(1) = nmtx

  do j=1,nmtx
    dest = epsmpi%igown(j)
    tag=j+1000
    
    if (peinf%inode.eq.0) then
      if (sig%freq_dep.eq.0.or.sig%freq_dep.eq.1) then
        read(10) (eps(i),i=1,nmtx)

! JRD: For debugging, set head to 0D0
!            if (j .eq. 1) eps(1) = 0D0

        if (j .eq. 1) epshead = eps(1)
        
        if(epsmpi%igown(j).eq.0) then
          epsmpi%eps(:,epsmpi%iggown(j),1) = eps(:)
        else
#ifdef MPI
          call MPI_SEND(eps,neps,MPI_SCALAR,dest,tag,MPI_COMM_WORLD,mpierr)
#endif
        endif ! if(epsmpi%igown(j).eq.0)
      endif ! if (sig%freq_dep.eq.0.or.sig%freq_dep.eq.1)
      
      if (sig%freq_dep.eq.2) then
        do i=1,nmtx
          read(10) (epsRDyn(jj,i),jj=1,sig%nFreq) ! Retarded part
        enddo
#ifdef CPLX
        do i=1,nmtx
          read(10) (epsADyn(jj,i),jj=1,sig%nFreq) ! Advanced part
        enddo
#endif
        
        if(epsmpi%igown(j).eq.0) then
          epsmpi%epsR(:,:,epsmpi%iggown(j),1) = epsRDyn(:,:)
#ifdef CPLX
          epsmpi%epsA(:,:,epsmpi%iggown(j),1) = epsADyn(:,:)
#endif
        else if (epsmpi%igown(j).ne.0) then
#ifdef MPI
          call MPI_SEND(epsRDyn,sig%nFreq*neps,MPI_COMPLEX_DPC, &
            dest,tag,MPI_COMM_WORLD,mpierr)
#ifdef CPLX
          call MPI_SEND(epsADyn,sig%nFreq*neps,MPI_COMPLEX_DPC, &
            dest,tag,MPI_COMM_WORLD,mpierr)
#endif
#endif
        endif  ! if(epsmpi%igown(j).ne.0)
      endif  ! if (sig%freq_dep.eq.2)
    endif  ! if (peinf%inode.eq.0)

    if (peinf%inode.eq.dest.and.dest.ne.0) then
      if (sig%freq_dep.eq.0.or.sig%freq_dep.eq.1) then
#ifdef MPI
        call MPI_RECV(eps,neps,MPI_SCALAR,0,tag,MPI_COMM_WORLD,mpistatus,mpierr)
#endif
        epsmpi%eps(:,epsmpi%iggown(j),1) = eps(:)
      endif
      
      if (sig%freq_dep.eq.2) then
#ifdef MPI
        call MPI_RECV(epsRDyn,sig%nFreq*neps,MPI_COMPLEX_DPC,0,tag,MPI_COMM_WORLD,mpistatus,mpierr)
        epsmpi%epsR(:,:,epsmpi%iggown(j),1) = epsRDyn(:,:)
#ifdef CPLX
        call MPI_RECV(epsADyn,sig%nFreq*neps,MPI_COMPLEX_DPC,0,tag,MPI_COMM_WORLD,mpistatus,mpierr)
        epsmpi%epsA(:,:,epsmpi%iggown(j),1) = epsADyn(:,:)
#endif
#endif
      endif  ! if (sig%freq_dep.eq.2)
    endif  ! if (peinf%inode.eq.dest.and.dest.ne.0)
    
  enddo  ! do j=1,nmtx

  if(sig%igamma == 0) then

!--------------------------
! Read inverse dielectric matrices from tape11 for q.ne.0

    if (peinf%inode.eq.0) then
      SAFE_ALLOCATE(oldx, (nold))
      SAFE_ALLOCATE(oldy, (nold))
      SAFE_ALLOCATE(oldz, (nold))
      SAFE_ALLOCATE(isrtold, (ngq))
      SAFE_ALLOCATE(ekold, (ngq))
      
      call close_file(11)
      call open_file(unit=11,file='epsmat',form='unformatted',status='old')
      read(11)
      read(11)
      read(11)
      read(11)
      read(11)
      read(11)
      read(11)
      read(11)
      read(11) nold,(oldx(i),oldy(i),oldz(i),i=1,nold)
    endif
    
    do iq=1,nq
      
      call logit(' Storing eps to memory')

      if (peinf%inode.eq.0) then
        read(11) ngq,nmtx,(isrtold(i),j,i=1,ngq)
        read(11) (ekold(i),i=1,ngq)
        read(11) (qk(i),i=1,3)
        
        isrtq=0
        isrtqi=0
        do i=1,ngq
          if (ekold(isrtold(i)).le.sig%ecutb) then
            gx=oldx(isrtold(i))
            gy=oldy(isrtold(i))
            gz=oldz(isrtold(i))
            call findvector(iout,gx,gy,gz,gvec)
            isrtq(i)=iout
            isrtqi(iout)=i
            if(i == 1) then
              ! just check the first so we do not waste time
              ekin = DOT_PRODUCT(gvec%k(:,iout)+qk(:), MATMUL(crys%bdot, gvec%k(:,iout)+qk(:)))
              if(abs(ekin - ekold(isrtold(i))) > TOL_Zero) then
                write(0,*) 'epsmat: ekold(isrtold(i)) = ', ekold(isrtold(i)), ' ekin = ', ekin
                call die("Incorrect kinetic energies in epsmat.")
              endif
            endif
          endif  ! if (ekold(isrtold(i)).le.sig%ecutb) 
        enddo  ! do i=1,ngq
      endif  ! if (peinf%inode.eq.0)

      if (peinf%inode .eq. 0) then
        epsmpi%isrtq(:,iq+1) = isrtq(:)
        epsmpi%isrtqi(:,iq+1) = isrtqi(:)
      endif

#ifdef MPI
      call MPI_Bcast(nmtx,1,MPI_INTEGER,0,MPI_COMM_WORLD,mpierr)
      call MPI_Bcast(epshead,1,MPI_SCALAR,0,MPI_COMM_WORLD,mpierr)
#endif
      
      epsmpi%nmtx(iq+1) = nmtx
      
      do j=1,nmtx
        dest = epsmpi%igown(j)
        tag=j+1000
        
        if (peinf%inode.eq.0) then
          if (sig%freq_dep.eq.0.or.sig%freq_dep.eq.1) then
            read(11) (eps(i),i=1,nmtx)
            if(epsmpi%igown(j).eq.0) then
              epsmpi%eps(:,epsmpi%iggown(j),iq+1) = eps(:)
            else
#ifdef MPI
              call MPI_SEND(eps,neps,MPI_SCALAR,dest,tag,MPI_COMM_WORLD,mpierr)
#endif
            endif ! if(epsmpi%igown(j).eq.0)
          endif ! if (sig%freq_dep.eq.0.or.sig%freq_dep.eq.1)
          
          if (sig%freq_dep.eq.2) then
            do i=1,nmtx
              read(11) (epsRDyn(jj,i),jj=1,sig%nFreq) ! Retarded part
            enddo
#ifdef CPLX
            do i=1,nmtx
              read(11) (epsADyn(jj,i),jj=1,sig%nFreq) ! Advanced part
            enddo
#endif
            
            if(epsmpi%igown(j).eq.0) then
              epsmpi%epsR(:,:,epsmpi%iggown(j),iq+1) = epsRDyn(:,:)
#ifdef CPLX
              epsmpi%epsA(:,:,epsmpi%iggown(j),iq+1) = epsADyn(:,:)
#endif
            else
#ifdef MPI
              call MPI_SEND(epsRDyn,sig%nFreq*neps,MPI_COMPLEX_DPC, &
                dest,tag,MPI_COMM_WORLD,mpierr)
#ifdef CPLX
              call MPI_SEND(epsADyn,sig%nFreq*neps,MPI_COMPLEX_DPC, &
                dest,tag,MPI_COMM_WORLD,mpierr)
#endif
#endif
            endif  ! if(epsmpi%igown(j).eq.0) 
          endif  ! if (sig%freq_dep.eq.2)
        endif  ! if (peinf%inode.eq.0)
        
        
        if (peinf%inode.eq.dest.and.dest.ne.0) then
          if (sig%freq_dep.eq.0.or.sig%freq_dep.eq.1) then
#ifdef MPI
            call MPI_RECV(eps,neps,MPI_SCALAR,0,tag,MPI_COMM_WORLD,mpistatus,mpierr)
#endif
            epsmpi%eps(:,epsmpi%iggown(j),iq+1) = eps(:)
          endif
          
          if (sig%freq_dep.eq.2) then
#ifdef MPI
            call MPI_RECV(epsRDyn,sig%nFreq*neps,MPI_COMPLEX_DPC,0,tag,MPI_COMM_WORLD,mpistatus,mpierr)
            epsmpi%epsR(:,:,epsmpi%iggown(j),iq+1) = epsRDyn(:,:)
#ifdef CPLX
            call MPI_RECV(epsADyn,sig%nFreq*neps,MPI_COMPLEX_DPC,0,tag,MPI_COMM_WORLD,mpistatus,mpierr)
            epsmpi%epsA(:,:,epsmpi%iggown(j),iq+1) = epsADyn(:,:)
#endif
#endif
          endif  ! if (sig%freq_dep.eq.2)
        endif  ! if (peinf%inode.eq.dest.and.dest.ne.0)
      enddo  ! do j=1,nmtx
      
    enddo ! iq
    
    if (peinf%inode.eq.0) then
      SAFE_DEALLOCATE(oldx)
      SAFE_DEALLOCATE(oldy)
      SAFE_DEALLOCATE(oldz)
      SAFE_DEALLOCATE(ekold)
      SAFE_DEALLOCATE(isrtold)
    endif
    
  endif
  
  nq = nqpp
  
  if (sig%freq_dep.eq.0.or.sig%freq_dep.eq.1) then
    SAFE_DEALLOCATE(eps)
  endif
  if (sig%freq_dep.eq.2) then
    SAFE_DEALLOCATE(epsRDyn)
#ifdef CPLX
    SAFE_DEALLOCATE(epsADyn)
#endif
  endif
  if (peinf%inode .eq. 0) then
    SAFE_DEALLOCATE(isrtq)
    SAFE_DEALLOCATE(isrtqi)
  endif

! Write out info about job

  if (peinf%inode.eq.0) then
    write(6,250) ajname,adate
    write(7,260) ajname,adate
250 format(/,5x,'q->0',2x,a,1x,a)
260 format(5x,'q->0',2x,a,1x,a)
    if (nq>1) then
      write(6,270) nq,ajname2,adate2
      write(7,280) nq,ajname2,adate2
280   format(5x,'nq=',i3,2x,a,1x,a)
270   format(/,5x,'nq=',i3,2x,a,1x,a)
    endif
  endif

! SIB: This is where we come to if we are NOT node zero
! Have everyone wait for processor zero to finish, and then broadcast
! the two values that might have changed.
!

#ifdef MPI
  call MPI_Bcast(nq,1,MPI_INTEGER,0,MPI_COMM_WORLD,mpierr)
  call MPI_Bcast(q(1,1),3*nq,MPI_REAL_DP,0,MPI_COMM_WORLD,mpierr)
  call MPI_Bcast(q0,3,MPI_REAL_DP,0,MPI_COMM_WORLD,mpierr)
  call MPI_Bcast(sig%qgrid,3,MPI_INTEGER,0,MPI_COMM_WORLD,mpierr)
#endif
  
  epsmpi%qk(:,:) = q(:,:)
  
  if (peinf%inode .eq. 0) then
    write(6,*) 'Using q0vec read from eps0mat'
    write(6,*) q0
    write(6,*)
  endif
  sig%q0vec(:)=q0(:)
  
! We are done with epsmat/eps0mat files, so we can close them.

  if(peinf%inode.eq.0) then
    call close_file(10)
  if(sig%igamma == 0) call close_file(11)
  endif

  POP_SUB(epscopy_mpi)

  return
end subroutine epscopy_mpi
