!=================================================================================
!
! Routines:
!
! (1) epscopy_disk()          Originally By ?            Last Modified 5/1/2008 (JRD)
!
!     Copy dielectric matrices from eps0mat/epsmat files to temporary
!     INT_EPS file.
!
!     We no longer symmetrize the q->0 dielectric matrix (esp0mat) using symmetries.
!
!     This routine reads the eps0mat and epsmat files (units 10 and 11)
!     and copies the relevant information about those dielectric matrices
!     to temporary INT_EPS file (unit itpe).  Only node 0 will actually
!     read data and write it to itpe; the other nodes will simply hang around,
!     and wait until node 0 has finished and closed the file.
!
!     The routine actually modifies two of the input arguments:  nq and q(:,:).
!     These are read from unit 11 (file epsmat) so they should be the same
!     as what was read in.  But just to be safe, they are broadcasted
!     to other units at the end.
!
!     nq actually increases by 1 after going through this routine.
!
!==================================================================================

#include "f_defs.h"

subroutine epscopy_disk(crys,gvec,nq,q,sig,neps,itpe,fne,epshead)

  use global_m
  use misc_m
  implicit none

  type (crystal), intent(in) :: crys
  type (gspace), intent(in) :: gvec
  integer,intent(inout) :: nq
  real(DP), intent(inout) :: q(3,nq+1)
  type (siginfo), intent(inout) :: sig
  integer, intent(in) :: neps,itpe
  character*20, intent(in) :: fne
  SCALAR, intent(out) :: epshead

  integer :: nq0,ngq,ngq0,nmtx,nqpp
  integer, allocatable :: isrtq(:),isrtqi(:)
  real(DP) :: qk(3)
  real(DP), allocatable :: eknq(:),ekin(:),ekold(:)
  SCALAR, allocatable :: eps(:)
  complex(DPC), allocatable :: epsDyn(:)

  integer :: iq,i,j,jj,iout,nold,nold0,gx,gy,gz,nFreq,qgrid(3)
  integer, allocatable :: oldx(:),oldy(:),oldz(:),isrtold(:)
  real(DP) :: gmax_in,q0(3),ekin_test
  character :: ajname*6,adate*11,ajname2*6,adate2*11
  character (len=100) :: tmpstr

  PUSH_SUB(epscopy_disk)

!--------------------- Begin Calculation -----------------------------------------
!
! SIB:  If we are not node 0, then we have no business doing anything;
! we will just go to the end and wait for node 0 to get done.

  if (peinf%inode == 0) then

! Otherwise, this is node 0, so let`s get some space to do some work

    write(6,*) ' '
    write(6,*) ' Allocating epsilon matrices: ', neps
    write(6,*) ' '
    
    SAFE_ALLOCATE(isrtq, (gvec%ng))
    SAFE_ALLOCATE(isrtqi, (gvec%ng))
    SAFE_ALLOCATE(eknq, (gvec%ng))
    SAFE_ALLOCATE(ekin, (gvec%ng))
    
!--------------------------
! Compute |g|^2

    do j=1,gvec%ng
      ekin(j)=DOT_PRODUCT(gvec%k(:,j),MATMUL(crys%bdot,gvec%k(:,j)))
    enddo

!---------------------------
! SIB: Read information for inverse dielectric matrix for q->0 tape10 (eps0mat)

    read(10)
    read(10)
    read(10)
    read(10)
    read(10)
    read(10)
    read(10)
    read(10)
    read(10) nold0
    read(10) ngq0
    call close_file(10)
    call open_file(unit=10,file='eps0mat',form='unformatted',status='old')
    
    SAFE_ALLOCATE(oldx, (nold0))
    SAFE_ALLOCATE(oldy, (nold0))
    SAFE_ALLOCATE(oldz, (nold0))
    SAFE_ALLOCATE(isrtold, (ngq0))
    SAFE_ALLOCATE(ekold, (ngq0))
    
    read(10) ajname,adate
    read(10) i,sig%nFreq
    
    if (sig%freq_dep .eq. 2) then
      SAFE_ALLOCATE(sig%dFreqGrid,(sig%nFreq))
      SAFE_ALLOCATE(sig%dFreqBrd,(sig%nFreq))
    endif
    
! JRD Now that we have actually read sig%nFreq, we may use it to allocate epsDyn

    if (sig%freq_dep.eq.0.or.sig%freq_dep.eq.1) then
      SAFE_ALLOCATE(eps, (neps))
    endif
    if (sig%freq_dep.eq.2) then
      SAFE_ALLOCATE(epsDyn, (sig%nFreq))
    endif
    
    if ((i.eq.0.and.sig%freq_dep.eq.2).or. &
      (i.eq.2.and.sig%freq_dep.eq.0).or. &
      (i.eq.2.and.sig%freq_dep.eq.1)) then
      call die('epscopy: eps0mat freq_dep')
    end if
    read(10) (qgrid(i),i=1,3)
    if(sig%igamma.ne.0) then
      if (sig%qgrid(1).eq.0.and.sig%qgrid(2).eq.0.and.sig%qgrid(3).eq.0) then
        sig%qgrid(:)=qgrid(:)
      endif
    endif
    if (sig%freq_dep .eq. 2) then
      read(10) (sig%dFreqGrid(i),i=1,sig%nFreq),(sig%dFreqBrd(i),i=1,sig%nFreq)
      sig%dFreqBrd=sig%dFreqBrd/dble(sig%ninter)  
    else
      read(10)
    endif
    read(10)
    read(10)
    read(10) gmax_in
    read(10) nq0,(q0(i),i=1,3)
    read(10) nold0,(oldx(i),oldy(i),oldz(i),i=1,nold0)
    if (nq0 /= 1) then
      call die('epscopy: nq0 /= 1')
    end if
    
!      if (dot_product(q0,q0) .lt. TOL_Small) then
!        call die('epscopy: q0 too small')
!      end if

    write(6,3003) ajname,adate,nq0,(q0(i),i=1,3)
3003 format(/,6x,'eps0mat read from ',a,2x,a,/, &
       6x,'nq0= ',i3, ' q0= ',3f10.3)

    if(sig%igamma == 0) then
      
!------------------------
! Read data for dielectric matrices from tape11 for q!=0
! tape11: 'epsmat'

      read(11) ajname2,adate2
      read(11) i,nFreq
      if ((i.eq.0.and.sig%freq_dep.eq.2).or. &
        (i.eq.2.and.sig%freq_dep.eq.0).or. &
        (i.eq.2.and.sig%freq_dep.eq.1)) then
        call die('epscopy: epsmat freq_dep')
      end if
      if (nFreq.ne.sig%nFreq ) then
        call die('epscopy: epsmat freq_scale')
      end if
      read(11) (qgrid(i),i=1,3)
      if (sig%qgrid(1).eq.0.and.sig%qgrid(2).eq.0.and.sig%qgrid(3).eq.0) then
        sig%qgrid(:)=qgrid(:)
      endif
      read(11)
      read(11)
      read(11)
      read(11) gmax_in
      read(11) nq,((q(i,j+1),i=1,3),j=1,nq)
      read(11) nold
      read(11) ngq
      
      write(6,3005) ajname2,adate2,nq,gmax_in
      write(6,3006) ((q(i,j+1),i=1,3),j=1,nq)
3005  format(/,6x,'epsmat read from',1x,a,2x,a, &
        /,7x,'nq=',i4,1x,'gmax=',f11.3)
3006  format(12x,3f10.3)
      
    endif
    
! Initialize tape itpe
! Write q(:,1)=0.0d0 to itpe. Though we are no longer symmetrizing

    q(:,1) = 0.0d0
    nqpp = nq + 1
    call open_file(itpe,file=fne,form='unformatted',status='replace')
    write(itpe) nqpp,((q(i,j),i=1,3),j=1,nqpp)

!---------------------------
! Read q->0 dielectric matrix and write to unit itpe

    write(tmpstr,*) 'Reading ngq0=',ngq0,', nmtx, isortold(:)'
    call logit(tmpstr)
    read(10) ngq0,nmtx,(isrtold(i),j,i=1,ngq0)
    if (nmtx.gt.neps) then
      call die('epscopy: nmtx > neps')
    endif
    write(tmpstr,*) 'Reading ekold and eps, nmtx=',nmtx
    call logit(tmpstr)
    read(10) (ekold(i),i=1,ngq0)
    read(10) (qk(i),i=1,3)
    
    call logit('Writing eps0 to INT_EPS')
    
    isrtq=0
    isrtqi=0
    call logit('  Matching oldxyz')
    do i=1,ngq0
      if (ekold(isrtold(i)).le.sig%ecutb) then
        gx=oldx(isrtold(i))
        gy=oldy(isrtold(i))
        gz=oldz(isrtold(i))
        call findvector(iout,gx,gy,gz,gvec)
        isrtq(i)=iout
        isrtqi(iout)=i
! we could check here that kinetic energies are correct as for epsmat and epscopy_mpi,
! but we are going to replace them with our own anyway, so we will not worry. --DAS
      endif
    enddo
    SAFE_DEALLOCATE(oldx)
    SAFE_DEALLOCATE(oldy)
    SAFE_DEALLOCATE(oldz)
    SAFE_DEALLOCATE(ekold)
    SAFE_DEALLOCATE(isrtold)
    
    write(itpe) gvec%ng,nmtx, &
      isrtq(1:gvec%ng), &
      isrtqi(1:gvec%ng), &
      ekin(1:gvec%ng), &
      (q(i,1),i=1,3)
    
    if (sig%freq_dep.eq.0.or.sig%freq_dep.eq.1) then
      do j=1,nmtx
        read(10) (eps(i),i=1,nmtx)
        write(itpe) (eps(i),i=1,nmtx)
        if (j .eq. 1) then
          epshead=eps(1)
        endif
      enddo
    endif
    if (sig%freq_dep.eq.2) then
      do j=1,nmtx
        do i=1,nmtx
          read(10) (epsDyn(jj),jj=1,sig%nFreq) ! Retarded part
          write(itpe) (epsDyn(jj),jj=1,sig%nFreq)
        enddo
#ifdef CPLX
        do i=1,nmtx
          read(10) (epsDyn(jj),jj=1,sig%nFreq) ! Advanced part
          write(itpe) (epsDyn(jj),jj=1,sig%nFreq)
        enddo
#endif
      enddo
    endif

! JRD: Copy q->0 eps that was read from file into eps0. We now
! keep the wings because it isn`t too hard to deal with them
! and setting them to zero is wrong for metals and graphene
!
!      eps0 = eps
!      if (sig%icutv .eq. 0) then
!        if (peinf%inode.eq.0) write(6,*) 'Fixing wings'
!! find g=0 in main gvec list
!        call findvector(iout,0,0,0,gvec)
!! find g=0 in eps gvector list
!        iout = isrtqi(iout)
!! fix wing divergence for semiconductors and graphene
!        do i=1,nmtx
!          if (i.ne.iout) then
!            if (sig%iscreen .eq. 0) then
!              eps0(i,iout) = eps0(i,out)*qsz/(2d0*q0len)
!              eps0(iout,i) = eps0(iout,i)*qsz/(2d0*q0len)
!            endif
!            if (sig%iscreen .eq. 1) then
!              eps0(i,iout) = eps0(i,out)*qsz*2d0/(3d0*q0len)
!              eps0(iout,i) = eps0(iout,i)*qsz*2d0/(3d0*q0len)
!            endif
!          endif
!        enddo
!      endif !no truncation

    if(sig%igamma == 0) then

!--------------------------
! Read inverse dielectric matrices from tape11 for q.ne.0
! and write it to unit itpe

      SAFE_ALLOCATE(oldx, (nold))
      SAFE_ALLOCATE(oldy, (nold))
      SAFE_ALLOCATE(oldz, (nold))
      SAFE_ALLOCATE(isrtold, (ngq))
      SAFE_ALLOCATE(ekold, (ngq))
      
      call close_file(11)
      call open_file(unit=11,file='epsmat',form='unformatted',status='old')
      read(11)
      read(11)
      read(11)
      read(11)
      read(11)
      read(11)
      read(11)
      read(11)
      read(11) nold,(oldx(i),oldy(i),oldz(i),i=1,nold)
      
      do iq=1,nq
        read(11) ngq,nmtx,(isrtold(i),j,i=1,ngq)
        read(11) (ekold(i),i=1,ngq)
        read(11) (qk(i),i=1,3)
        
        isrtq=0
        isrtqi=0
        do i=1,ngq
          if (ekold(isrtold(i)).le.sig%ecutb) then
            gx=oldx(isrtold(i))
            gy=oldy(isrtold(i))
            gz=oldz(isrtold(i))
            call findvector(iout,gx,gy,gz,gvec)
            isrtq(i)=iout
            isrtqi(iout)=i
            eknq(iout)=ekold(isrtold(i))
            if(i == 1) then
              ! just check the first so we do not waste time
              ekin_test = DOT_PRODUCT(gvec%k(:,iout)+qk(:), MATMUL(crys%bdot, gvec%k(:,iout)+qk(:)))
              if(abs(ekin_test - ekold(isrtold(i))) > TOL_Zero) then
                write(0,*) 'epsmat has ekin = ', ekold(isrtold(i)), ' should be ', ekin_test
                call die("Incorrect kinetic energies in epsmat.")
              endif
            endif
          endif
        enddo
        
        write(itpe) gvec%ng,nmtx, &
          isrtq(1:gvec%ng), &
          isrtqi(1:gvec%ng), &
          eknq(1:gvec%ng), &
          (qk(i),i=1,3)
        
        if (sig%freq_dep.eq.0.or.sig%freq_dep.eq.1) then
          do j=1,nmtx
            read(11) (eps(i),i=1,nmtx)
            write(itpe) (eps(i),i=1,nmtx)
          enddo
        endif
        if (sig%freq_dep.eq.2) then
          do j=1,nmtx
            do i=1,nmtx
              read(11) (epsDyn(jj),jj=1,sig%nFreq) ! Retarded part
              write(itpe) (epsDyn(jj),jj=1,sig%nFreq)
            enddo
#ifdef CPLX
            do i=1,nmtx
              read(11) (epsDyn(jj),jj=1,sig%nFreq) ! Advanced part
              write(itpe) (epsDyn(jj),jj=1,sig%nFreq)
            enddo
#endif
          enddo
        endif
        
      enddo ! iq
      
      SAFE_DEALLOCATE(oldx)
      SAFE_DEALLOCATE(oldy)
      SAFE_DEALLOCATE(oldz)
      SAFE_DEALLOCATE(ekold)
      SAFE_DEALLOCATE(isrtold)
      
    endif
    
    nq = nqpp
    
    if (sig%freq_dep.eq.0.or.sig%freq_dep.eq.1) then
      SAFE_DEALLOCATE(eps)
    endif
    if (sig%freq_dep.eq.2) then
      SAFE_DEALLOCATE(epsDyn)
    endif
    SAFE_DEALLOCATE(isrtq)
    SAFE_DEALLOCATE(isrtqi)
    SAFE_DEALLOCATE(eknq)
    SAFE_DEALLOCATE(ekin)

! SIB: close unit itpe (for flushing purposes)

    call close_file(itpe)

! Write out info about job

    write(6,250) ajname,adate
    write(7,260) ajname,adate
250 format(/,5x,'q->0',2x,a,1x,a)
260 format(5x,'q->0',2x,a,1x,a)
    if (nq>1) then
      write(6,270) nq,ajname2,adate2
      write(7,280) nq,ajname2,adate2
280   format(5x,'nq=',i3,2x,a,1x,a)
270   format(/,5x,'nq=',i3,2x,a,1x,a)
    endif

! SIB: This is where we come to if we are NOT node zero
! Have everyone wait for processor zero to finish, and then broadcast
! the two values that might have changed.
!
  endif

#ifdef MPI
  call MPI_Bcast(nq,1,MPI_INTEGER,0,MPI_COMM_WORLD,mpierr)
  call MPI_Bcast(epshead,1,MPI_SCALAR,0,MPI_COMM_WORLD,mpierr)
  call MPI_Bcast(q(1,1),3*nq,MPI_REAL_DP,0,MPI_COMM_WORLD,mpierr)
  call MPI_Bcast(q0,3,MPI_REAL_DP,0,MPI_COMM_WORLD,mpierr)
  call MPI_Bcast(sig%qgrid,3,MPI_INTEGER,0,MPI_COMM_WORLD,mpierr)
  if (sig%freq_dep .eq. 2) then
    call MPI_Bcast(sig%nFreq,1,MPI_INTEGER,0,MPI_COMM_WORLD,mpierr)
    if (peinf%inode .ne. 0) then
      SAFE_ALLOCATE(sig%dFreqGrid,(sig%nFreq))
      SAFE_ALLOCATE(sig%dFreqBrd,(sig%nFreq))
    endif
    call MPI_Bcast(sig%dFreqGrid,sig%nFreq,MPI_DOUBLE_PRECISION,0,MPI_COMM_WORLD,mpierr)
    call MPI_Bcast(sig%dFreqBrd,sig%nFreq,MPI_DOUBLE_COMPLEX,0,MPI_COMM_WORLD,mpierr)
  endif
#endif
  
  if (peinf%inode .eq. 0) then
    write(6,*) 'Using q0vec read from eps0mat'
    write(6,*) q0
    write(6,*)
  endif
  sig%q0vec(:)=q0(:)
  
! We are done with epsmat/eps0mat files, so we can close them.

  if(peinf%inode.eq.0) then
    call close_file(10) ! eps0mat
    if(sig%igamma == 0) call close_file(11) ! epsmat
  endif

  POP_SUB(epscopy_disk)

  return
end subroutine epscopy_disk
